/* $Header: /g1/users/staff/gore/exp/notes/src/lib/RCS/config.c,v 2.0 89/04/16 00:43:55 gore Exp $ */

/* 
 * Functions to extract configuration information from the default
 * configurations file.  There are two functions; one returns a string
 * for string valued configuration items, the other returns an integer
 * for integer valued or boolean items.
 */

#include <stdio.h>
#include <pwd.h>
#include <errno.h>
#include <ctype.h>	/* for isspace(c)    */
#include "macros.h"	/* for forcelower(c) */
#include "dumpmsg.h"
#include "parms.h"
#include "structs.h"

extern char *sys_errlist[];

/* name of network special file, used only for RFA */
char remote_netfile[WDLEN];

#ifdef	TEST
#define libdir "/usr/notes/lib"
#define CONFIGFILE "config"
#endif	TEST

#define KEY_LENGTH 32
#define VAL_LENGTH 192

static struct conf_line {
    char key[KEY_LENGTH];
    char val[VAL_LENGTH];
    struct conf_line *next;
};

/*
 * This is where the data is kept during the run.  It's local to this file,
 * since it's nobody else's business.  For convenience, it starts with a 
 * header link, so the first useful element is in *(Configuration.next).
 *
 * As long as it's there, we use its first byte to be "~" before init_config
 * is called and "#" after.
 */
static struct conf_line Configuration = {"~", "#", (struct conf_line *)NULL};
#define CONFIG_INITIALIZED	  (Configuration.key[0] == '#')
#define MARK_CONFIG_INITIALIZED   (Configuration.key[0] = '#')
#define MARK_CONFIG_UNINITIALIZED (Configuration.key[0] = '~')

static  struct conf_line *last_conf = &Configuration;

/*
 * Loads the configuration file into a list in memory, to make the
 * job of getconfig()'s job easier.
 *
 * In the config file, lines starting with "#" or " " are assumed to
 * be comments, and are ignored.  Other lines must be of the format:
 *
 *	 KeyName: ValuePart
 *
 * The string starting with "ValuePart" is what is returned by getconfig().
 * Blanks are stripped.
 *
 * Returns 0 on success, non-0 on failure.
 */
int
init_config(libdir)
    char *libdir;
{
    FILE *configfile;
    int line_num = 0;
    char fname[256];
    char buf[192];
    char *from, *to;
    int limiter;
    extern char *malloc();
    extern char *index();
    
    sprintf (fname, "%s/%s", libdir, CONFIGFILE);
    if (NULL == (configfile = fopen (fname, "r"))) {
        return 1;
    }

    while (NULL != fgets (buf, sizeof (buf), configfile)) {

	++line_num;

	if ((buf[0] == '#') || (buf[0] == ' ')) {
	    continue;			/* ignore comments */
	}

	buf[strlen(buf)-1] = '\0';	/* nuke newline */

	if ((struct conf_line *)NULL ==
	       (last_conf->next =
	          (struct conf_line *)malloc(sizeof(struct conf_line)))) {
	    return 1;
	}
	last_conf = last_conf->next;
	last_conf->next = NULL;

	from = buf;			/* copy the key */
	to = last_conf->key;
	limiter = KEY_LENGTH;
	while(*from && *from != ':' && limiter--) {
	    *to++ = *from++;
	}
	if (*from != ':') {		/* there must be a ':' */
	    fprintf(stderr, "Error in configuation file '%s', line %d\n",
		    configfile, line_num);
	    exit(1);
	}

	++from;				/* skip over the ':' */

	while (*from && isspace(*from)) {	/* skip white space */
	    ++from;
	}
	if (!(*from)) {			/* there must be a ValuePart */
	    fprintf(stderr, "Error in configuation file '%s', line %d\n",
		    configfile, line_num);
	    exit(1);
	}

	to = last_conf->val;		/* get the ValuePart */
	limiter = VAL_LENGTH-1;		/* reserve last char for '\0' */
	while(*from && limiter--) {
	    *to++ = *from++;
	}

	while (isspace(*to)) {		/* trim white space at the end */
	    --to;
	}
	*++to = '\0';
    }
    close(configfile);
    MARK_CONFIG_INITIALIZED;
    return 0;
}


/*
 * This function actually does the lookup of strings in the
 * config list; since this list is assumed to be short, we
 * just do a sequential search.  This returns the value part of
 * the config (if any) or NULL if not found.  Blanks are stripped.
 */

char *getconfig (precious_name)
    char *precious_name;
{
    extern char *index();
    struct conf_line *line;
    int name_length;
    char *name;

    name = strsave(precious_name);

    if (!CONFIG_INITIALIZED) {	/* init_config() hasn't been called?  */
	if(init_config(libdir)) {
	    return NULL;	/* Couldn't load configuration! */
	}
    }

    name_length = strlen(name);	/* Be forgiving of an extra ':' at the end */
    if (name[--name_length] == ':') {
	name[name_length] = '\0';
    }

    for (line = Configuration.next; line; line = line->next) {
	if (0 == strcasecmp(line->key, name)) {
	    return (strsave(line->val));
	}
    }
    return (NULL);
}


/*
 * This function looks up boolean or integer valued items in the
 * config file; returns -1 if nothing was found, otherwise the
 * value.  "YES" or "TRUE" or "NO" or "FALSE" are recognized as
 * valid configuration values for this, and will be returned as
 * either 0 or 1, as appropriate.
 */

int igetconfig(name)
char *name;
{
    char *value;
    int ivalue;

    value = getconfig(name);
    if (value == NULL) {
	return -1;
    }

    /* first, try for boolean values */
    switch(forcelower(value[0])) {
      case 'y':
      case 't':
	return 1;
      case 'n':
      case 'f':
	return 0;
    }

    /* if that fails, then assume it is an integer */
    if (0 == sscanf (value, "%d", &ivalue))
        return (-1);	/* failed! */
    else
        return (ivalue);
}


#ifndef	TEST

/*
 *  Utility routines for loading values from config file.
 *
 *  name      - name the value is stored under in the config file
 *  dest      - the config variable itself
 *  size      - maximum string size (set_dflt_string only)
 *  set_dflts - controls whether the default value (dflt) is used
 *                 or ignored
 *  dflt      - default value
 */

static void
set_dflt_int(name, dest, set_dflts, dflt)
char *name;
int *dest;
int set_dflts;
int dflt;
{
    int value = igetconfig(name);

    if (value != -1)
	*dest = value;
    else if (set_dflts)
	*dest = dflt;
}

static void
set_dflt_string(name, dest, size, set_dflts, dflt)
char *name;
char *dest;
int size;
int set_dflts;
char *dflt;
{
    char *value = getconfig(name);

    if (value != NULL)
	safecpy(dest, value, size);
    else if (set_dflts)
	safecpy(dest, dflt, size);
}

/*
 *  Load defaults from the config file and the hard defaults.
 *
 *  The config variables are set from the config file.  If there is
 *  no entry in the config file for that variable, the argument
 *  "set_dflts" controls the behavior.
 *
 *     If set_dflts is zero, the config variable is left alone.
 *     If set_dflts is non-zero, the config variable will be over-written
 *       with the hard default.
 *
 *  This behavior is actually implemented inside the routines set_dflt_int
 *  and set_dflt_string, defined above.
 */
static void
load_config(set_dflts)
int set_dflts;
{
    char *value;
    char mailp_tmp[WDLEN];

    if(!CONFIG_INITIALIZED && 0 != init_config(libdir)) {
	fprintf(stderr,
		"load_config: Could not load configuration file %s/%s\n",
		libdir, CONFIGFILE);
	exit(1);
    }

    /*
     * First, check if this config file defers to another LIBDIR.
     * If it does, load the config file in that directory first,
     * so that this config file can overrule items in it.
     *
     * NOTE that it does not matter where in this config file the
     * "Real-LIBDIR:" line occurs -- the inclusion of $Real-LIBDIR/config
     * is always done first.
     */
       
    value = getconfig("Real-LIBDIR");
    if (NULL != value) {
	/*
	 * We just let init_config append the lines from the other
	 * config file to the list.  Since getconfig() always searches
	 * the list from the beginning, a value for from this config file
	 * for a specific key will overrule the value from the other config
	 * file for that key, simply because it appears first in the list.
	 *
	 * Note that "Real-LIBDIR" is not checked a second time, so you
	 * cannot defer LIBDIR more than once.  On the bright side, we
	 * won't get into a loop, either.
	 */
	if (0 == init_config(value)) {
	    safecpy(libdir, value, WDLEN);
	} else {
	    fprintf(stderr,
		    "load_config: Could not load configuration file %s/%s\n",
		    value, CONFIGFILE);
	    /*
	     * Proceed without changing LIBDIR.
	     *  Maybe this should be fatal instead?
	     */
	}
    }

    set_dflt_string("Prompt", prompt_str, SYSSZ, set_dflts, "");
    set_dflt_string("Organization", myorg, NNLEN, set_dflts, DFLTORG);
    set_dflt_string("Spool-Directory",
		    spooldir, WDLEN, set_dflts, DFLTSPOOLDIR);
    set_dflt_string("Archive-Directory",
		    archdir, WDLEN, set_dflts, DFLTARCHDIR);
    set_dflt_string("Editor", editor, WDLEN, set_dflts, DFLTEDITOR);
    sprintf(mailp_tmp, "%s/%s", libdir, DFLTMAILPOSTER);
    set_dflt_string("Mailposter", mailposter, WDLEN, set_dflts, mailp_tmp);
    set_dflt_string("Pager", pager, WDLEN, set_dflts, DFLTPAGER);
    set_dflt_string("Write", writeprog, WDLEN, set_dflts, DFLTWRITE);
    set_dflt_string("Shell", shell, WDLEN, set_dflts, DFLTSHELL);

/* Now we do some acrobatics to determine the hostname(s) */

    /* See if "Full-Hostname:" was supplied in the config file */
    value = getconfig("Full-Hostname");
    if (NULL != value) {
	/* Yes, so use it */
	safecpy (full_hostname, value, SYSSZ);
    } else {
	/*
	 * No, so compose it of "this_system" (dynamically determined)
	 * and "mydomain".  NOTE: no dot is inserted between "this_system"
	 * and "mydomain"!
	 */
	set_dflt_string("Domain", mydomain, SYSSZ, set_dflts, DFLTDOMAIN);
	safecpy (full_hostname, this_system, SYSSZ);
	safecpy (full_hostname+strlen(full_hostname),
		 mydomain, SYSSZ - strlen(full_hostname));
    }

    set_dflt_string("Hostname-In-Path",
		    hostname_in_path, SYSSZ, this_system);

    if (NULL == replyto_address) {
	value = getconfig("Reply-To-Hostname");
	if (NULL != value) {
	    safecpy(replyto_hostname, value, SYSSZ);
	}
    }

    set_dflt_int("Verbose", &verbose_mode, set_dflts, 0);
    set_dflt_int("Verbose-log", &verbose_log, set_dflts, 0);
    set_dflt_int("UUCP-mailer", &uucp_mailer, set_dflts, 0);
    set_dflt_int("Group-timeout", &group_timeout, set_dflts, 0);
    set_dflt_int("Manual-rmgroup", &manual_rmgroup, set_dflts, 0);
    set_dflt_int("Manual-Mod-Change", &manual_mod_change, set_dflts, 0);
    set_dflt_string("Default-Moderator", default_subm_addr, ADDRLEN,
		    set_dflts, "notes");
    {
	char tmp[WDLEN];
	sprintf(tmp, "%s/%s", libdir, "canon-addr");
	set_dflt_string("Address-Canonizer", address_canonizer, WDLEN,
			set_dflts, tmp);
    }
    {
	char tmp[16];
	set_dflt_string("Default-Save-Format", tmp, 16,
			set_dflts, STR_DFLT);
	default_dump_format = dump_format(tmp);
	if (default_dump_format == DMP_INVALID) {
	    /* Don't know this one.  Use default instead. */
	    default_dump_format = DMP_DFLT;
	}
    }
}

/*
 *  Load assorted default configuration stuff at initialization time.
 *  Performs the delicate dance between the local and remote config
 *  files for Notes-over-RFA.  For non-RFA installations, this just
 *  calls load_config().  Honest.
 */

load_default_config()
{
#ifdef RFA
    extern struct passwd *fgetpwent();
    extern char *strtok();
    FILE *fp;
    char *cp;
    extern char remote_netfile[WDLEN];
    char remote_login[WDLEN];
    char remote_passwdfile[WDLEN];
    char remote_accessfile[WDLEN];
    char remote_libdir[WDLEN];
    char local_libdir[WDLEN];
    char buff[BUFSIZ];
    struct passwd *pwent;

    /* Save a copy of the local libdir, we will need it later */
	safecpy(local_libdir, libdir, WDLEN);

    /* Change libdir to the remote directory if the remote */
    /* entries are present in the local config file */
    safecpy(remote_netfile, getconfig("Remote-Network-file"), WDLEN);
    if (remote_netfile[0] != '\0')
    {
	/*
	 * Prime the config list with entries from the local config
	 * file.
	 */
	init_config(local_libdir);
	safecpy(remote_login, getconfig("Remote-Login"), WDLEN);
	x ((netunam(remote_netfile, remote_login) == -1),
	   "load_default_config: netunam call failed (%s) on %s",
	   sys_errlist[errno], remote_netfile);

	/*
	 * Look up the home directory for the notes user on the
	 * remote machine.
	 */
	sprintf(remote_passwdfile, "%s/etc/passwd", remote_netfile);
	x ((fp = fopen(remote_passwdfile, "r")) == NULL,
	   "load_default_config: fopen call failed (%s) on %s",
	   sys_errlist[errno], remote_passwdfile);

	/* We have to use fgetpwent since there isn't an fgetpwnam */
	while ((pwent = fgetpwent(fp)) != NULL)
	{
	    if (strcmp(pwent->pw_name, NOTESUID) == 0)
	    {
		sprintf(remote_libdir, "%s/%s",
			remote_netfile, pwent->pw_dir);
		safecpy(libdir, remote_libdir);
		break;
	    }
	}

	fclose(fp);

	/* See if the hub system has allowed us access */
	/* A missing access file implies unlimited access */
	sprintf(remote_accessfile, "%s/lnotes.allow", libdir);
	if ((fp = fopen(remote_accessfile, "r")) != NULL)
	{
	    while (fgets(buff, BUFSIZ, fp) != NULL)
	    {
		char *cp = strtok(buff, " \t\n");

		if (match(cp, this_system))
		    goto ok;
	    }

	    fprintf(stderr, "Sorry, remote notes access denied by hub system\n");
	    exit(BAD);

	ok: 
	    fclose(fp);
	}
    }
    
    /*
     * Let the hubsystem forge the remote hostname.
     * Normally, this would be a boolean flag to forge the
     * the remote hostname to the hubsystem name, but there's
     * no way to get the hubsystem's hostname via RFA (the name
     * stored in the /net file isn't necessarily the hostname.
     */
    set_dflt_string("Forged-Remote-Hostname", this_system, WDLEN,
	    FALSE, "");
    /*
     * Make sure that entries from the hub's config file are appended
     * to the config list.
     */
    MARK_CONFIG_UNINITIALIZED;
#endif RFA

    /* Now load the config file values from libdir/config, */
    /* wherever that is by now. */
    load_config(/* set_dflts = */ TRUE);

#ifdef RFA
    /*
     * Relax.  init_config() has already stuck the original config
     * file's entries to the front of the config list.  They will
     * therefore be chosen over their duplicates in the remote
     * config file.
     */
#endif RFA
}

#else	TEST

char *testvalues[] = {
    "Archive-Days:",
    "Organization:",
    "Domain:",
    "Allow-Old:",
    "Bogus-Item:",
    "Verbose:",
    "Prompt:",
    "Spool-Directory:",
    "Spool-Directory",
    "Spo",
    "Archive-Directory:",
    "No-Such-Field:",
    "Editor:",
    "Pager:",
    "Auto-Create:",
    "Auto-Delete:",
    "Path-Map:",
    "END" };

main ()
{
    char *p;
    int i;
    
    for (i=0; 0 != strcmp ("END", testvalues[i]); i++) {
	printf ("%s %s [%d]\n", testvalues[i],
	    getconfig (testvalues[i]),
	    igetconfig (testvalues[i]));
    }
}
#endif	TEST
