/* $Header: compress.c,v 1.1 85/12/20 16:47:44 notes Exp $ */

#include "parms.h"
#include "structs.h"
#include <sys/stat.h>

/*
 *	compress(io) struct io_f
 *	compresses the notefile specified. All wasted space
 *	reclaimed. The process is a simple one which, like dcheck,
 *	does not work so well on active file systems.
 *	As a consequence, the director options (which call this)
 *	require the notefile to be closed before allowing compression
 *	to take place. 
 *	The code generates 3 scratch files, corresponding with the
 *	two index files and the text file. These are made to
 *	represent a virgin notefile. The descriptor is copied over
 *	with the appropriate fields zapped, and then we go through
 *	a cycle of (read note; write note; (read resp; write resp))
 *	until all the notes and responses are moved over.
 *	the new files are then copied back into place.
 *	
 *	Returns:	0 - all successful
 *			-1 - if notesfiles compressed already
 *
 *			otherwise will core dump with the notefile
 *			in a shambles from the users view point,
 *			but still recoverable by a hotshot-pro.
 *
 *	Original Coding:	Ray Essick	January 1981
 */

compress(io, lockflag, numnotes, numresps)
struct io_f *io;
int lockflag;
int *numnotes, *numresps;
{
	struct io_f tmpio;		/* scratch notefile */
	struct note_f note;		/* hold a note record */
	struct resp_f resp;		/* hold the response format */
	char fn1[WDLEN], fn2[WDLEN], fn3[WDLEN];
	char on1[WDLEN], on2[WDLEN], on3[WDLEN];
	char txtfn[WDLEN];		/* hold text going between files */
	int i;
	int nnotes, nresps, dint, roffset, num, rblock;
	register int newnum, presps, rnum;
	struct daddr_f daddr;
	struct stat sb;
	time_t tb[2];


	/* build names of files - in notefile directory */
	sprintf(fn1, "%s/%s/tmp.%s", spooldir, mapnfname(io->nf), INDEXN); /* new files */
	sprintf(fn2, "%s/%s/tmp.%s", spooldir, mapnfname(io->nf), INDEXR);
	sprintf(fn3, "%s/%s/tmp.%s", spooldir, mapnfname(io->nf), TEXT);
	sprintf(on1, "%s/%s/%s", spooldir, mapnfname(io->nf), INDEXN);/* old files */
	sprintf(on2, "%s/%s/%s", spooldir, mapnfname(io->nf), INDEXR);
	sprintf(on3, "%s/%s/%s", spooldir, mapnfname(io->nf), TEXT);

	if (stat(on3, &sb) < 0 || sb.st_size <= (off_t)4)
		return(-1);	/* if notesfile empty, don't bother */
	tb[0] = sb.st_atime;
	tb[1] = sb.st_mtime;
	/*
	 * if the notesfile is not already locked,
	 * do so now.  This also insures signals will
	 * be ignored
	 */
	if (lockflag)
		lock(io, 'n');			/* lock up the notefile */
	x ((tmpio.fidndx = creat(fn1, 0660)) < 0, "compress: create nindex");
	x ((tmpio.fidrdx = creat(fn2, 0660)) < 0, "compress: create rindex");
	x ((i = creat(fn3, 0660)) < 0, "compress: create txt");
	x ((tmpio.fidtxt = fdopen(i, "w")) == NULL, "compress: fdopen txt");

	dint = 0;				/* resp index free pointer */
	daddr.addr = sizeof daddr;		/* and for text file */
	x (write(tmpio.fidrdx, (char *)&dint, sizeof(dint)) != sizeof(dint),
		"compress: resp ptr");
	x (fwrite((char *)&daddr, sizeof(daddr), 1, tmpio.fidtxt) != 1,
		"Compress: text ptr");


	closenf(&tmpio);				/* close them up */

	x ((tmpio.fidndx = open(fn1, 2)) < 0, "compress: reopen 1");
	x ((tmpio.fidrdx = open(fn2, 2)) < 0, "compress: reopen 2");
	x ((tmpio.fidtxt = fopen(fn3, "r+")) == NULL, "compress: reopen 3");

	strcpy(tmpio.nf, io->nf);	/* copy over notefile name */
	getdscr(io, &tmpio.descr);		/* grab descriptor */
	if (io->descr.d_stat & NFINVALID) {
		/* fix to clean up things, from Steve Potter, Tektronix */
		closenf(&tmpio);
		x(unlink(fn1) < 0, "compress: remove tmp 1");
		x(unlink(fn2) < 0, "compress: remove tmp 2");
		x(unlink(fn3) < 0, "compress: remove tmp 3");
		/* end of fix */

		if (lockflag)
			unlock(io, 'n');
		return(-1);
	}
	nnotes = nresps = 0;		/* init counts of living notes */
	sprintf(txtfn, "%s/nfcXXXXXX", TMPDIR);
	mktemp(txtfn);
	tmpio.descr.d_nnote = 0;		/* reset note count */
	putdscr(&tmpio, &tmpio.descr);	/* place it into the file */

	if (io->descr.d_plcy) {		/* copy the policy note over */
		getnrec(io, 0, &note);
		copymsg(io, &tmpio, &note.n_msg);
		putnote(&tmpio, &note, POLICY, NOADDTIME);
	}

	for (num = 1; num <= io->descr.d_nnote; num++) {
		getnrec(io, num, &note);
		if (note.n_stat & DELETED)
			continue;	/* deleted - we throw away */
		copymsg(io, &tmpio, &note.n_msg);
		/* save max number of responses */
		presps = note.n_nresp;
		newnum = putnote(&tmpio, &note, NOPOLICY, NOADDTIME);
		nnotes++;				/* add a note */

		for (rnum = 1; rnum <= presps; rnum++) {
			/* process responses */
			if (lrsp(io, num, rnum, &resp, &roffset, &rblock) != 0) {
				/* bad response chain - drop rest */
				break;
			}
			copymsg(io, &tmpio, &resp.r_msg[roffset]);
			putresp(&tmpio, newnum, &note, &resp.r_msg[roffset],
			    NOADDTIME);
			nresps++;		/* count responses */
		}
	}

	/*
	 * well, we have now copied the entire notefile over,
	 * so the time has come to move it back into the correct
	 * file names - we will do this by linking
	 */
	closenf(&tmpio);				/* close the new one */

	getdscr(io, &io->descr);
	io->descr.d_stat |= NFINVALID;		/* mark it bad */
	putdscr(io, &io->descr);
	closenf(io);				/* close the old one */

	x(rename(fn1,on1) < 0, "compress: rename 1");
	x(rename(fn2,on2) < 0, "compress: rename 2");
	x(rename(fn3,on3) < 0, "compress: rename 3");
	unlink(txtfn);
	utime(on3, tb);			/* preserve modification times */

	opennf(io, io->nf);			/* relink to new one */
	getdscr(io, &io->descr);		/* get the new descriptor */

	if (lockflag)
		unlock(io, 'n');
	*numnotes = nnotes;
	*numresps = nresps;

	return(0);					/* return ok */
}

copymsg(oldio, newio, msgp)
	struct io_f *oldio, *newio;
	struct msg_f *msgp;
{
	struct txthead_f txthead;

	gethrec(oldio, &msgp->m_head, &txthead);
	puttrec(newio, oldio->fidtxt, &msgp->m_head, txthead.textlen);
	gethrec(oldio, &msgp->m_addr, &txthead);
	puttrec(newio, oldio->fidtxt, &msgp->m_addr, txthead.textlen);
}
