/* $Header: acl.c,v 1.0 87/06/10 20:06:18 notes Exp $ */

/*
 *	acl - access control list support
 *
 *	(Much of this was once done by doaccess(), but it also read input
 *	from the keyboard, and that should be a separate layer of code.
 *	Much of the rest was done by functions in acssort.c, but there is
 *	very little, if anything, left of that code.)
 *
 *	Original coding: Jacob Gore 88/03/02
 */

#include "parms.h"
#include "structs.h"
#include "acl.h"

#ifdef __STDC__

extern FILE *fopen(const char*, const char*);
int ace_compare(ace_t*, ace_t*);

#else /* Old C */

extern FILE *fopen();
int ace_compare();

#endif __STDC__

#define is_deleted(ptype) ( (ptype) >= PERMDELETED )
#define force_undeleted(ptype) \
            ( is_deleted(ptype) ? ((ptype) - PERMDELETED) : (ptype) )

/*
 * Creates a new, empty acl.  Returns a pointer to it.
 *
 * NOTE: an empty acl will not be saved on disk, since acl_open expects
 * a non-empty list (must have at least group:Other and system:Other).
 */
acl_t*
acl_create(io)
    struct io_f *io;
{
    acl_t *aclp;

    aclp = (acl_t *)malloc(sizeof(acl_t));

    sprintf(aclp->filename, "%s/%s/%s", spooldir, mapnfname(io->nf), ACCESS);

    aclp->length = 0;
    aclp->changed = FALSE;
    aclp->num_deleted = 0;

    return aclp;
}
/*
 * Like acl_create, except takes on good faith the name of the future
 * ACCESS.LIST file.
 */
acl_t*
acl_create_quick(filename)
    char *filename;
{
    acl_t *aclp;

    aclp = (acl_t *)malloc(sizeof(acl_t));

    safecpy(aclp->filename, filename, strlen(aclp->filename));

    aclp->length = 0;
    aclp->changed = FALSE;
    aclp->num_deleted = 0;

    return aclp;
}

/*
 * Returns a pointer to the acl.
 */
acl_t*
acl_open(io)
    struct io_f *io;
{
    FILE *acl_file;
    acl_t *aclp;

    aclp = (acl_t *)malloc(sizeof(acl_t));

    sprintf(aclp->filename, "%s/%s/%s", spooldir, mapnfname(io->nf), ACCESS);
    x ((acl_file = fopen(aclp->filename, "r")) == NULL,
       "acl_open: can't open access file \"%s\"", aclp->filename);

    x ((aclp->length = fread((char *)aclp->items, sizeof(ace_t),
			     NPERMS, acl_file)) == 0,
       "acl_open: access list file \"%s\" is empty, should not be",
       aclp->filename);

    aclp->changed = FALSE;
    aclp->num_deleted = 0;

    fclose(acl_file);
    return aclp;
}

/*
 * Sorts the acl, gets rid of deleted entries, and writes it out
 * to the database.
 *
 * Returns 1 on success, crashes on failure.
 *
 * Assumes that the current working directory is the same as it was when
 * acl_open() was called.
 */
int
acl_close(aclp)
    acl_t *aclp;
{
    FILE *acl_file;

    if (aclp->changed) {
	acl_sort(aclp);
	ignsigs++;
	x ((acl_file = fopen(aclp->filename, "w")) == NULL,
	   "doaccess: failed to reopen \"%s\"", aclp->filename);

	x (fwrite((char *)aclp->items,
		  sizeof(ace_t), aclp->length, acl_file) != aclp->length,
	   "doaccess: update write to \"%s\" failed", aclp->filename);
	fclose(acl_file);
	ignsigs--;

    }
    return 1;
}

/*
 * Returns the handle to the first element of an acl, or ACL_NOHANDLE
 * if the acl is empty.
 */
acl_h
acl_first(aclp)
    acl_t *aclp;
{
    if (aclp->length > 0) {
	return 0;
    } else {
	return ACL_NOHANDLE;
    }
}

/*
 * Returns the handle to the element of an acl that follows the element at
 * the given handle.
 *
 * Returns ACL_NOHANDLE if there is no next emement.
 */
acl_h
acl_next(aclp, aclh)
    acl_t *aclp;
    acl_h aclh;
{
    if (aclh >= 0 && aclh < aclp->length - 1) {
	return aclh + 1;
    } else {
	return ACL_NOHANDLE;
    }
}

/*
 * Returns a handle to the n-th entry in the acl (the first entry being
 * number 1).  
 *
 * Returns ACL_NOHANDLE if there is no n-th element.
 *
 * This should only be used on a sorted acl.  
 */
acl_h
acl_nth(aclp, n)
    acl_t* aclp;
    int n;
{
    --n;
    if (n >= 0 && n < aclp->length) {
	return n;
    } else {
	return ACL_NOHANDLE;
    }
}

/*
 * Finds an access control entry in an acl.
 *
 * If the entry is deleted, but still in the list, it is found anyway.
 *
 * Returns its handle if found, ACL_NOHANDLE if not.
 */
acl_h
acl_find(aclp, acep)
    acl_t *aclp;
    ace_t *acep;
{
    acl_h handle;

    /* Locate the entry in the list */
    for (handle = acl_first(aclp);
	 handle != ACL_NOHANDLE;
	 handle = acl_next(aclp, handle)) {
	if (ace_same(acl_ace(aclp,handle), acep)) {
	    return handle;
	}
    }
    return ACL_NOHANDLE;
}

/*
 * Returns a pointer to the entry at the given handle in the acl.
 *
 * Returns NULL if the handle is invalid.
 */
ace_t*
acl_ace(aclp, aclh)
    acl_t *aclp;
    acl_h aclh;
{
    if (aclh >= 0 && aclh < aclp->length) {
	return &aclp->items[aclh];
    } else {
	return (ace_t *)0;
    }
}

/*
 * Finds a deleted access control entry in an acl.
 * Returns its handle if found, ACL_NOHANDLE if not.
 *
 * This is an implementation detail, so it's local to this file.
 */
static acl_h
acl_find_deleted(aclp)
    acl_t *aclp;
{
    acl_h handle;

    /* Locate the entry in the list */
    for (handle = acl_first(aclp);
	 handle != ACL_NOHANDLE;
	 handle = acl_next(aclp, handle)) {
	if (is_deleted(acl_ace(aclp,handle)->ptype)) {
	    return handle;
	}
    }
    return ACL_NOHANDLE;
}

static acl_h
acl_newslot(aclp)
    acl_t *aclp;
{
    if (aclp->length < NPERMS) {
	return aclp->length++;
    } else {
	return ACL_NOHANDLE;
    }
}

/*
 * Replaces the element at the given handle with another entry.
 *
 * Returns the same handle on success, ACL_NOHANDLE on failure.
 */
acl_h
acl_replace(aclp, aclh, acep)
    acl_t *aclp;
    acl_h  aclh;
    ace_t *acep;
{
    if (aclh >= 0 && aclh < aclp->length) {
	int was_deleted = is_deleted((aclp->items[aclh]).ptype);
	if (ace_copy(&(aclp->items[aclh]), acep)) {
	    if (was_deleted) {
		aclp->num_deleted--;
	    }
	    aclp->changed = TRUE;
	    return aclh;
	}
    }
    return ACL_NOHANDLE;
}

/*
 * Insert an access control entry into a list.
 *
 * If there is already an entry in the list for the same entity,
 * the existing entry is modified instead of a new one being added.
 *
 * Returns its handle upon success, ACLNOHANDLE upon failure.
 */
acl_h
acl_add(aclp, acep)
    acl_t *aclp;
    ace_t *acep;
{
    acl_h destination;

    destination = acl_find(aclp, acep); /* if entity in list already, use it */
    if (destination == ACL_NOHANDLE) {

	destination = acl_find_deleted(aclp); /* else try using an empty one */
	if (destination == ACL_NOHANDLE) {

	    destination = acl_newslot(aclp);   /* else try to append */
	}
    }

    if (destination == ACL_NOHANDLE) {
	return ACL_NOHANDLE;			/* No room... */
    }

    aclp->changed = TRUE;
    return acl_replace(aclp, destination, acep);
}

/*
 * Deletes the element at the given handle in the acl.
 *
 * Returns 1 if the element was found, 0 otherwise.
 */
int
acl_delete (aclp, aclh)
    acl_t *aclp;
    acl_h aclh;
{
    ace_t *acep;

    if (aclh < 0 || aclh > aclp->length) {
	return 0;
    }

    acep = acl_ace(aclp,aclh);

    if (is_deleted(acep->ptype)) {
	return 0;
    } else {
	acep->ptype += PERMDELETED;
	aclp->changed = TRUE;
	aclp->num_deleted++;
	return 1;
    }
}

/*
 * Deletes the element matching the given given entity in the acl.
 *
 * Returns 1 if the element was found, 0 otherwise.
 */
int
acl_delete_ace (aclp, acep)
    acl_t *aclp;
    ace_t *acep;
{
    return acl_delete(aclp, acl_find(aclp,acep));
}

/*
 * Sorts an access control list, discarding deleted entries.
 *
 * See ace_compare() for lexicographic ordering used.
 */
acl_sort(aclp)
    acl_t *aclp;
{
    qsort((char *)aclp->items, aclp->length, sizeof(ace_t), ace_compare);

    /* All deleted entries have bubbled to the end; chop them off */
    aclp->length -= aclp->num_deleted;
    aclp->num_deleted = 0;
}

/*
 * Returns non-0 if the acl has changed since it was opened, 0 otherwise.
 *
 * Note that since an acl is assumed to be sorted before it is opened,
 * sorting the list alone is assumed not to change it.
 */
int
acl_was_changed(aclp)
    acl_t *aclp;
{
    return aclp->changed;
}

/*
 *	Compares two access control entries.
 *	The lexicographic ordering is:
 *	  + people before groups before systems
 *	  + alphabetical within each class, but "Other" always last
 *	  + deleted entries last
 *	The ordering is maintained through constants defined in acl.h.
 */
int
ace_compare (a, b)
    ace_t *a, *b;
{
    if (a->ptype < b->ptype) {
	return -1;
    }

    if (a->ptype > b->ptype) {
	return 1;
    }

    if (strcmp ("Other", a->name) == 0) {
	return (strcmp ("Other", b->name) == 0);	/* put "Other" last */
    }

    if (strcmp ("Other", b->name) == 0) {
	return -1;					/* is correct */
    }

    return strcmp(a->name, b->name);
}

/*
 *	Returns 1 if two access list entries refer to the same entity,
 *	0 otherwise.
 */
ace_same (a, b)
    ace_t *a, *b;
{
    return (force_undeleted(a->ptype) == force_undeleted(b->ptype) &&
	    strcmp(a->name, b->name) == 0);
}

/*
 * Copies one access control entry into another.
 *
 * Returns 1 on success, 0 on failure.
 *
 * (Actually, since the current implementation uses bcopy(), it will return
 * 1 on success and crash on failure.)
 */
int
ace_copy(to_acep, from_acep)
    ace_t *to_acep, *from_acep;
{
    bcopy(from_acep, to_acep, sizeof(ace_t));
    return 1;
}
