/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

/* LINTLIBRARY */

#include "hostenv.h"
#include "mailer.h"
#ifdef	USE_RESOLV

#undef	RFC974		/* MX/WKS/A processing according to RFC974 */

#include <netdb.h>

/* do we have a normal nameserver? */
#ifdef	TRY_AGAIN
#ifdef	BSDTYPES_H
#include BSDTYPES_H
#endif	/* BSDTYPES_H */
#include <netinet/in.h>
#ifdef NOERROR
#undef NOERROR
#endif
#include <arpa/nameser.h>
#include <resolv.h>
#include "search.h"

/*
 * Query a BIND standard (version 4.7 or later) nameserver.
 */


/* For use by WKS lookup, we need to know the SMTP port number. */

#ifndef	IPPORT_SMTP
#define	IPPORT_SMTP	25
#endif	/* !IPPORT_SMTP */

#ifndef	MAXNAME
#define	MAXNAME		BUFSIZ
#endif	/* !MAXNAME */

#define	MAXVALIDTTL	(60*60*24*365)	/* any ttl over this is ignored */

#ifndef	BIND_VER
#ifdef	GETLONG
/* 4.7.3 introduced the {GET,PUT}{LONG,SHORT} macros in nameser.h */
#define	BIND_VER	473
#else	/* !GETLONG */
#define	BIND_VER	472
#endif	/* GETLONG */
#endif	/* !BIND_VER */

#if	defined(BIND_VER) && (BIND_VER >= 473)
char	*conffile;
typedef u_char msgdata;
#else	/* !defined(BIND_VER) || (BIND_VER < 473) */
typedef char msgdata;
#endif	/* defined(BIND_VER) && (BIND_VER >= 473) */

extern int D_bind, D_resolv;

extern int	h_errno;
char h_errhost[MAXNAME];

struct qtypes {
	char	*typename;
	u_short	value;
} qt[] = {
	{	"cname",	T_CNAME		},
	{	"mx",		T_MX		},
	{	"a",		T_A		},
#ifdef	T_MP
	{	"mp",		T_MP		},
#endif	/* T_MP */
#ifdef	T_UNAME
	{	"uname",	T_UNAME		},
#endif	/* T_UNAME */
#ifdef	T_TXT
	{	"txt",		T_TXT		},
#endif	/* T_TXT */
	{	"wks",		T_WKS		},
	{	"ptr",		T_PTR		},
	{	0,		0		}
};

char *zh_errlist[] = {
	"Error 0",
	"unknown host",				/* 1 - HOST_NOT_FOUND */
	"host name lookup",			/* 2 - TRY_AGAIN */
	"unknown server",			/* 3 - NO_RECOVERY */
	"no address associated with name"	/* 4 - NO_ADDRESS */
};

extern struct conscell *getmxrr(), *getcrrtype(), *getrrtype();

void
zherror(s)
	char *s;
{
	extern int h_errno;

	if (h_errno >= 0 && h_errno < (sizeof zh_errlist/sizeof zh_errlist[0]))
		(void) fprintf(stderr, "%s: resolver error: %s\n",
				       s, zh_errlist[h_errno]);
	else
		(void) fprintf(stderr, "%s: resolver error: %d\n", s, h_errno);
}

struct conscell *
search_res(sip)
	struct search_info *sip;
{
	struct qtypes *qtp;
	struct conscell *rval;
	char *host, buf[BUFSIZ];
	extern char *conffile, *myhostname;
	extern int deferit, res_init(), cistrcmp();
	extern void die(), v_set();

	if (!(_res.options & RES_INIT)) {
		if (sip->file != NULL)
			conffile = sip->file;
		if (res_init() < 0) {
			fprintf(stderr,
			    "search_res: nameserver initialization failure!\n");
			die(1, "res_init failure");
		}
	}
	if (D_resolv)
		_res.options |= RES_DEBUG;
	else
		_res.options &= ~RES_DEBUG;
	/*
	 * Turn down the resolver's retry count.
	 * The default (4) will cause the router to fall hopelessly
	 * behind in the face of broken DNS data.
	 */
	_res.retry = 2;
	if (sip->subtype == NULL || *(sip->subtype) == '\0') {
		fprintf(stderr, "search_res: missing subtype to BIND query!\n");
		return NULL;
	}
	for (qtp = qt; qtp->value != 0; ++qtp)
		if (CISTREQ(qtp->typename, sip->subtype))
			break;
	if (qtp->value == 0) {
		fprintf(stderr,
			"search_res: unknown subtype '%s' to BIND query!\n",
				sip->subtype);
		return NULL;
	}
	h_errno = 0;
	h_errhost[0] = '\0';
	switch (qtp->value) {
	case T_MX:
		rval = getmxrr(sip->key, myhostname, &sip->ttl, 0);
		break;
	case T_WKS:
	case T_PTR:
	case T_A:
#ifdef	T_MP
	case T_MP:
#endif	/* T_MP */
#ifdef	T_UNAME
	case T_UNAME:
#endif	/* T_UNAME */
#ifdef	T_TXT
	case T_TXT:
#endif	/* T_TXT */
		rval = getrrtype(sip->key, qtp->value, &sip->ttl);
		break;
	case T_CNAME:
		rval = getcrrtype(sip->key, qtp->value, &sip->ttl);
		break;
	default:
		rval = NULL;
	}
	if (h_errno > 0) {
		if (h_errhost[0] == '\0')
			host = (char *)sip->key;
		else
			host = h_errhost;

		deferit++;

		if (h_errno >= 0 &&
		    h_errno < (sizeof zh_errlist/sizeof zh_errlist[0]))
			(void) fprintf(stderr,
			       "search_res: deferred: %s: %s (%s) error\n",
			       host, qtp->typename, zh_errlist[h_errno]);
		else
			(void) fprintf(stderr,
				"search_res: deferred: %s: %s (%d) error\n",
				host, qtp->typename, h_errno);

		(void) sprintf(buf, "NS:%s/%s", host, qtp->typename);
		v_set(DEFER, buf);
	}
	return rval;
}


typedef union {
	HEADER qb1;
	char qb2[PACKETSZ];
} querybuf;

struct mxdata {
	msgdata *host;
	u_short	pref;
	time_t	ttl;
};

struct conscell *
getmxrr(host, localhost, ttlp, depth)
	char *host;
	char *localhost;
	time_t *ttlp;
	int depth;
{
	HEADER *hp;
	msgdata *eom, *cp;
	querybuf buf, answer;
	int n, nmx, i, ancount, qdcount, maxpref;
	struct conscell *lhead, *l, *tmp;
	struct mxdata mx[20];
	msgdata hbuf[MAXNAME], realname[MAXNAME];
	u_short type;
	time_t ttl, maxttl;
	extern int res_mkquery(), res_send(), dn_skipname(), dn_expand();
	extern int cistrcmp();

	if (depth > 2) {
		(void) fprintf(stderr,
			"search_res: CNAME chain length exceeded (%s)\n",
			host);
		strcpy(h_errhost, host);	/* use strcat on purpose */
		h_errno = TRY_AGAIN;
		return NULL;
	}
	n = res_mkquery(QUERY, host, C_IN, T_MX, (char *)NULL, 0, NULL,
		(char *)&buf, sizeof(buf));
	if (n < 0) {
		fprintf(stderr, "search_res: res_mkquery (%s) failed\n", host);
		strcpy(h_errhost, host);	/* use strcat on purpose */
		h_errno = NO_RECOVERY;
		return NULL;
	}
	n = res_send((char *)&buf, n, (char *)&answer, sizeof(answer));
	if (n < 0) {
		if (D_bind || _res.options & RES_DEBUG)
			fprintf(stderr,
				"search_res: res_send (%s) failed\n", host);
		strcpy(h_errhost, host);	/* use strcat on purpose */
		h_errno = TRY_AGAIN;
		return NULL;
	}
	eom = (msgdata *)&answer + n;
	/*
	 * find first satisfactory answer
	 */
	hp = (HEADER *) &answer;
	ancount = ntohs(hp->ancount);
	qdcount = ntohs(hp->qdcount);
	if (hp->rcode != NOERROR || ancount == 0) {
		if (D_bind || _res.options & RES_DEBUG)
			fprintf(stderr,
				"search_res: rcode = %d, ancount=%d, aa=%d\n",
				hp->rcode, ancount, hp->aa);
		switch (hp->rcode) {
			case NXDOMAIN:
				return NULL;
			case SERVFAIL:
				strcpy(h_errhost, host);
				h_errno = TRY_AGAIN;
				return NULL;
#ifdef	OLDJEEVES
			/*
			 * Jeeves (TOPS-20 server) still does not
			 * support MX records.  For the time being,
			 * we must accept FORMERRs as the same as
			 * NOERROR.
			 */
			case FORMERR:
#endif	/* OLDJEEVES */
			case NOERROR:
				/* if we got this, then ancount == 0! */
				return NULL /*getrrtype(host, T_A, ttlp)*/;
#ifndef	OLDJEEVES
			case FORMERR:
#endif	/* !OLDJEEVES */
			case NOTIMP:
			case REFUSED:
				strcpy(h_errhost, host);
				h_errno = NO_RECOVERY;
				return NULL;
		}
		return NULL;
	}
	cp = (msgdata *)&answer + sizeof(HEADER);
	for (; qdcount > 0; --qdcount)
#if	defined(BIND_VER) && (BIND_VER >= 473)
		cp += dn_skipname(cp, eom) + QFIXEDSZ;
#else	/* !defined(BIND_VER) || (BIND_VER < 473) */
		cp += dn_skip(cp) + QFIXEDSZ;
#endif	/* defined(BIND_VER) && (BIND_VER >= 473) */
	nmx = 0;
	maxttl = ttl = 0;
	realname[0] = '\0';
	maxpref = -1;
	/* assert: stickymem == MEM_MALLOC;  for storing RHS of MX RR's */
	while (--ancount >= 0 && cp < eom && nmx < (sizeof mx/sizeof mx[0])) {
		n = dn_expand((msgdata *)&answer, eom, cp, hbuf, sizeof hbuf);
		if (n < 0)
			break;
		cp += n;
		type = _getshort(cp);
 		cp += sizeof(u_short);				/* type */
 		cp += sizeof(u_short);				/* class */
		ttl = _getlong(cp);
		cp += sizeof(u_long);				/* ttl */
		n = _getshort(cp);
		cp += sizeof(u_short);				/* dlen */
		if (type == T_CNAME) {
			cp += dn_expand((msgdata *)&answer, eom, cp,
					realname, sizeof realname);
			continue;
		} else if (type != T_MX)  {
			if (D_bind || _res.options & RES_DEBUG)
				fprintf(stderr,
					"search_res: bad mx query answer type %d, size %d\n",
					type, n);
			cp += n;
			continue;
		}
		mx[nmx].ttl = ttl;
		mx[nmx].pref = _getshort(cp);
		cp += sizeof(u_short);		/* MX preference value */
		n = dn_expand((msgdata *)&answer, eom, cp, hbuf, sizeof hbuf);
		if (n < 0)
			break;
		cp += n;
		mx[nmx].host = (msgdata *)strnsave((char *)hbuf, strlen((char *)hbuf));
		if (localhost != NULL && CISTREQ(hbuf, localhost))
			maxpref = mx[nmx].pref;
		++nmx;
	}
	if (nmx == 0 && realname[0] != '\0' && !CISTREQ(host,(char*)realname)) {
		/* do it recursively for the real name */
		return getmxrr((char *)realname, localhost, ttlp, depth+1);
	} else if (nmx == 0)
		return NULL;
	/* discard MX RRs with a value >= that of localdomain */
	if (maxpref >= 0) {
		for (n = 0; n < nmx; ++n) {
			if (mx[n].pref >= maxpref && mx[n].host != NULL) {
				(void) free((char *)mx[n].host);
				mx[n].host = NULL;
			}
		}
	}
#ifdef	RFC974
	/* discard MX's that do not support SMTP service */
	for (n = 0; n < nmx; ++n) {
		if (mx[n].host == NULL)
			continue;
		(void) strcpy(hbuf, mx[n].host);
		if (!getrrtype(hbuf, T_WKS, ttlp)) {
			(void) free(mx[n].host);
			mx[n].host = NULL;
		}
	}
#endif	/* RFC974 */
	/* determine how many are left, and their max ttl */
	n = 0;
	lhead = l = NULL;
	for (i = 0; i < nmx; ++i) {
		if (mx[i].host == NULL)
			continue;
		++n; /* found one! */
		if (mx[i].ttl > maxttl && mx[i].ttl < MAXVALIDTTL)
			maxttl = ttl;
		if (D_bind || _res.options & RES_DEBUG)
			fprintf(stderr, "search_res: %s: mx[%d] = %s\n",
					host, n, mx[i].host);
		if (lhead == NULL)
			lhead = l = newstring((u_char *)mx[i].host);
		else {
			cdr(l) = newstring((u_char *)mx[i].host);
			l = cdr(l);
		}
	}
	if (D_bind || _res.options & RES_DEBUG)
		fprintf(stderr, "search_res: %s: %d valid MX RR's\n", host, n);
	if (n == 0) /* MX's exist, but their WKS's show no TCP smtp service */
		return NULL;
	else if (maxttl > 0)
		*ttlp = maxttl;
	return ncons(lhead);
}

struct conscell *
getcrrtype(host, rrtype, ttlp)		/* getrrtype with completion */
	char *host;
	u_short rrtype;
	time_t *ttlp;
{
	struct conscell *rval;
	char buf[BUFSIZ], **domain;

	if (strchr(host, '.') != NULL) {
		if ((rval = getrrtype(host, rrtype, ttlp)) != NULL)
			return rval;
		if (*host != '\0' && *(host+strlen(host)-1) == '.')
			return NULL;
	}
	for (domain = _res.dnsrch; h_errno == 0 && *domain != NULL; domain++) {
		(void) sprintf(buf, "%s.%s", host, *domain);
		if ((rval = getrrtype(buf, rrtype, ttlp)) != NULL)
			return rval;
	}
	strcpy(h_errhost, host);
	return NULL;
}

struct conscell *
getrrtype(host, rrtype, ttlp)
	char *host;
	u_short rrtype;
	time_t *ttlp;
{
	struct conscell *lhead, *l, *tmp;
	char *s;
	HEADER *hp;
	msgdata *eom, *cp, *nextcp;
	querybuf buf, answer;
	int n, ancount, qdcount, ok, first;
	struct in_addr inaddr;
	time_t maxttl, ttl;
	u_short type;
	msgdata nbuf[BUFSIZ];
	char hb[MAXNAME];
	extern char *dottedquad();
	extern int res_mkquery(), res_send(), dn_expand(), dn_skipname();
	extern int cistrcmp();

	n = res_mkquery(QUERY, host, C_IN, rrtype, (char *)NULL, 0, NULL,
		(char *)&buf, sizeof(buf));
	if (n < 0) {
		if (D_bind || _res.options & RES_DEBUG)
			(void) fprintf(stderr,
				"search_res: res_mkquery (%s) failed\n", host);
		strcpy(h_errhost, host);
		h_errno = NO_RECOVERY;
		return NULL;
	}
	n = res_send((char *)&buf, n, (char *)&answer, sizeof(answer));
	if (n < 0) {
		if (D_bind || _res.options & RES_DEBUG)
			fprintf(stderr,
				"search_res: res_send (%s) failed\n", host);
		strcpy(h_errhost, host);
		h_errno = TRY_AGAIN;
		return NULL;
	}
	eom = (msgdata *)&answer + n;
	/*
	 * find first satisfactory answer
	 */
	hp = (HEADER *) &answer;
	ancount = ntohs(hp->ancount);
	qdcount = ntohs(hp->qdcount);
	/*
	 * We don't care about errors here, only if we got an answer
	 */
	if (ancount == 0) {
		if (D_bind || _res.options & RES_DEBUG)
			fprintf(stderr,
				"search_res: rcode=%d, ancount=%d, rrtype=%d\n",
				hp->rcode, ancount, rrtype);
		if (rrtype == T_CNAME && hp->rcode == NOERROR) {
			if (qdcount > 0 && strchr(host, '.') == NULL) {
				cp = (msgdata *)&answer + sizeof(HEADER);
				if (dn_expand((msgdata *)&answer,eom,cp,hb,sizeof hb)>=0) {
					if (hb[0] == '\0') {
						hb[0] = '.'; hb[1] = '\0';
					}
					return newstring((u_char *)strnsave(hb,
							    strlen(hb)));
				}
			}
			return newstring((u_char *)strnsave(host,strlen(host)));
		}
		if (rrtype == T_WKS) /* absence of WKS means YES ... */
			return newstring((u_char *)strnsave(host,strlen(host)));
		return NULL;
	}
	cp = (msgdata *)&answer + sizeof(HEADER);
	for (; qdcount > 0; --qdcount)
#if	defined(BIND_VER) && (BIND_VER >= 473)
		cp += dn_skipname(cp, eom) + QFIXEDSZ;
#else	/* !defined(BIND_VER) || (BIND_VER < 473) */
		cp += dn_skip(cp) + QFIXEDSZ;
#endif	/* defined(BIND_VER) && (BIND_VER >= 473) */
	first = 1;
	ok = rrtype != T_WKS;
	maxttl = 0;
	l = NULL;
	for (lhead = NULL; --ancount >= 0 && cp < eom; cp = nextcp) {
		n = dn_expand((msgdata *)&answer, eom, cp, nbuf, sizeof(nbuf));
		if (n < 0)
			break;
		if (first) {
			if (strlen((char *)nbuf) < sizeof hb)
				strcpy(hb, (char *)nbuf);
			else {
				(void) strncpy(hb, (char *)nbuf, sizeof hb);
				hb[(sizeof hb) - 1] = '\0';
			}
			first = 0;
		}
		cp += n;
		type = _getshort(cp);
 		cp += sizeof(u_short);				/* type */
 		cp += sizeof(u_short);				/* class */
		ttl = _getlong(cp);
		cp += sizeof(u_long);				/* ttl */
		n = _getshort(cp);
		cp += sizeof(u_short);				/* dlen */
		nextcp = cp + n;

		if (type != rrtype)
			continue;
		/*
		 * Assume that only one rrtype will be found.  More
		 * than one is undefined.
		 */

		if (ttl > maxttl && ttl < MAXVALIDTTL)
			maxttl = ttl;
		switch (type) {
#ifdef	T_TXT
		case T_TXT:
#endif	/* T_TXT */
#ifdef	T_UNAME
		case T_UNAME:
#endif	/* T_UNAME */
#ifdef	T_MP
		case T_MP:
#endif	/* T_MP */
#if	defined(T_TXT) || defined(T_UNAME) || defined(T_MP)
			*ttlp = maxttl;
			if ((n = (int)(*cp)) > 0 && n < sizeof hb)
				return newstring((u_char *)
						strnsave((char *)(cp+1), n));
			break;
#endif
		case T_WKS:
			/*
			 * If we have seen a WKS, it had better have SMTP,
			 * however in absence of a WKS, assume SMTP.
			 */
			if (n < sizeof (u_long) + 1)
				continue;
			ok = 0;
			cp += sizeof(u_long);		/* skip IP address */
			if (*cp++ == IPPROTO_TCP) {	/* check protocol */
				if (cp + (IPPORT_SMTP/8) < nextcp
				    && (*(cp+(IPPORT_SMTP/8))
					 & (0x80>>IPPORT_SMTP%8))) {
					*ttlp = maxttl;
					return newstring((u_char *)strnsave(hb,
							    strlen(hb)));
				}
			}
			continue;
		case T_A:
			bcopy(cp, (char *)&inaddr, sizeof inaddr);
			s = dottedquad(&inaddr);
			s = strnsave(s, strlen(s));
			if (lhead == NULL)
				lhead = l = newstring((u_char *)s);
			else {
				cdr(l) = newstring((u_char *)s);
				l = cdr(l);
			}
			*ttlp = maxttl;
			continue;
		case T_CNAME:
		case T_PTR:
			if ((n = dn_expand((msgdata *)&answer, eom, cp, nbuf,
			    sizeof(nbuf))) < 0)
				break;
			if (type == T_CNAME && !CISTREQ(nbuf, host))
				/* chase it down */
				getcrrtype((char *)nbuf, rrtype, ttlp);
			*ttlp = maxttl;
			return newstring((u_char *)
				strnsave((char *)nbuf,strlen((char *)nbuf)));
		default:
			fprintf(stderr,
				"search_res: getrrtype: unknown RR type\n");
			break;
		}
	}
	if (lhead)
		return ncons(lhead);
	if (ok) {
		*ttlp = maxttl;
		s = first ? host : hb;
		return newstring((u_char *)strnsave(s, strlen(s)));
	}
	return NULL;
}
#endif	/* TRY_AGAIN */
#endif	/* USE_RESOLV */
