/*
 * Simple trace program for the Alpha.
 * Originally written by James Bonfield. (jkb@mrc-lmb.cam.ac.uk)
 * 
 * Currently being maintained by Anthony Baxter <anthony@aaii.oz.au>.
 *
 * To use please make sure you have the /proc file system mounted by either
 * including the following line in /etc/fstab:
 * "/proc           /proc           procfs  rw              0 0"
 * or by mounting with the command:
 * "mount -t procfs /proc /proc"
 *
 * Version 1.0beta
 *
 * Last update 11/10/93
 * $Id: trace.c,v 1.23 1993/12/21 23:51:00 anthony Exp $
 */

#include "trace.h"
#include "signame.h"
#include "sysname.h"

int max_string = 40;

#define proc_cntl(a,b) \
    if (-1 == ioctl(p_fd, a, b)) \
	die(#a);

static pid_t pid = 0;
static int gotalarm = 0;
int p_fd = 0;
FILE *outfp = stderr;
static void catchalarm();

static struct prrun pr = {
    PRSTRACE,
    0, 0, 0, 0, 0, 0};

void die(char *msg) {
    if (msg)
        perror(msg);
    if (pid)
        kill(pid, SIGKILL);
    if (p_fd)
        close(p_fd);
    exit(1);
}

void pr_string(char *strp, int max_len, int fixed) {
    char c;

    fputc('"', outfp);
    while ((c = *strp++) && max_len--) {
	switch(c) {
	case '"':
	    fprintf(outfp,"\\\"");
	    break;
	case '\n':
	    fprintf(outfp,"\\n");
	    break;
	case '\t':
	    fprintf(outfp,"\\t");
	    break;
	case '\r':
	    fprintf(outfp,"\\r");
	    break;
	case '\\':
	    fprintf(outfp,"\\");
	    break;
	default:
	    if (isascii(c) && isprint(c)) {
		fputc(c,outfp);
	    } else {
		fprintf(outfp,"\\%03o", c);
	    }
	}
    }
    fputc('"', outfp);
    if (!fixed && max_len == -1 && c)
	fprintf(outfp, "...");
}

char *rd_buf(long val, size_t len) {
    char *str = (char *)malloc(len+1);

    if (!str)
	die("malloc");
    
    if (-1 == lseek(p_fd, val, SEEK_SET))
	die("lseek p_fd");
    if (-1 == read(p_fd, str, len))
	die("read p_fd");

    return str;
}

void p_arg(char c, long val, long length) {
    
    switch(c) {
    case 'd':
	fprintf(outfp,"%ld", val);
	break;
    case 'u':
	fprintf(outfp,"%uld", val);
	break;
    case 'p':
	fprintf(outfp,"0x%p", val);
	break;
    case 'o':
	fprintf(outfp,"0%o", val);
	break;
    case 's': {
	char *str = rd_buf(val, max_string+1);

	pr_string(str, max_string, 0);
	free(str);
	break;
    }
    case 'v':
	fprintf(outfp,"(void)");
	break;
    case '?': {
	char *str = rd_buf(val, max_string+1);
	int i, t = 0;
	
	for (i = 0; str[i] && i < max_string; i++) {
	    if (isascii(str[i]) && isprint(str[i]))
		t++;
	}

	if ((i-1) && (float)t/(i-1) > .8)
	    pr_string(str, max_string, 0);
	else
	    fprintf(outfp,"0x%p", val);

	free(str);
	break;
    }
    case 'b': {
	int i, t = 0, tlen = length > max_string ? max_string : length;
	char *str = rd_buf(val, tlen+1);
	
	for (i = 0; i < tlen; i++) {
	    if (isascii(str[i]) && (isprint(str[i]) || isspace(str[i])))
		t++;
	}

	if (tlen == 0 || (tlen && (float)t/tlen > .8))
	    pr_string(str, tlen, !(length > max_string));
	else
	    fprintf(outfp,"0x%p", val);

	free(str);
	break;
    }
    default:
	fprintf(outfp,"?0x%p?", val);
    }
}

/* highest possible start address for a user process */
#define MAX_U_START 0x0000020000000000

void
do_exec(argv)
char *argv[];		/* child process argument vector */
{
    char buf[1024];
    int pfd, i;
    sigset_t sigtrace;
    struct prrun prn;
    struct prpsinfo ps;
    struct sigvec vec, ovec;
    long omask;

    /* open yourself so tracing flags can be set for the child */
    sprintf(buf, "/proc/%d", getpid());
    if( ((pfd = open(buf, O_RDWR))) == -1) {
        if (errno != ENOENT)
            die("parent open");
        else {
            fprintf(stderr, "Couldn't open \"%s\"!\n", buf);
            fprintf(stderr, "You probably need to mount the /proc filesystem\n");
            fprintf(stderr, "As root, either enter\n\n");
            fprintf(stderr, "\tmount -t procfs /proc /proc\n\n");
            fprintf(stderr, "or add the following line to /etc/fstab\n\n");
            fprintf(stderr, "\t/proc /proc procfs rw 0 0\n\n");
            fprintf(stderr, "and enter\n\n");
            fprintf(stderr, "\tmount /proc\n\n");
            exit(1);
        }
    }

    /* Set trace on any exec via PIOCSSPCACT, Note: this automatically
     * sets trace on signal SIGTRAP.
     */
    i = PRFS_STOPEXEC;
    if(ioctl(pfd, PIOCSSPCACT, &i) < 0)
	die("parent PIOCSSPCACT");

    /* Set the inherit on fork flag so the child will get all our tracing
     * flags.
     */
    if(ioctl(pfd, PIOCSFORK, 0) < 0)
        die("parent PIOCSFORK");

    /* Set up signal handlers...
     */
    vec.sv_handler = catchalarm; 
    vec.sv_mask = 0;
    vec.sv_onstack = 0;
    (void) sigvec(SIGALRM, &vec, &ovec);
    omask = sigblock(sigmask(SIGALRM));
    gotalarm = 0;

    /* fork a child process; child waits for a signal from parent, child
       execs path; child should stop on exec */
    pid = fork();
    switch(pid) {
    case -1:			/* failure */
	die("fork");
    case 0:			/* child */
	while (!gotalarm)
	    sigpause(omask &~ sigmask(SIGALRM));
	/* reset our signal handlers */
	(void)sigvec(SIGALRM, &ovec, (struct sigvec *)0);
	(void)sigsetmask(omask);
	execvp(argv[0], argv);
	/* you only reach here if the execvp fails */
	die("child execvp");
    }

    /* this is all parent code from here down */

    (void) sigvec(SIGALRM, &ovec, (struct sigvec *)0);
    (void) sigsetmask(omask);

    /* open the child task */
    sprintf(buf, "/proc/%d", pid);
    fprintf(outfp, "Tracing process %s\n", buf);
    if( ((p_fd = open(buf, O_RDWR))) == -1)
        die(buf);

    /* wake the child up */
    i=SIGALRM;
    proc_cntl(PIOCKILL,&i);   

    /* wait for child process to stop at the first exec */
    proc_cntl(PIOCWSTOP, 0);

    /* Clear all tracing flags in the parent (pfork) and close its pfd */

    /* clear trace on signal */
    premptyset(&sigtrace);
    if (ioctl(pfd, PIOCSTRACE, &sigtrace) < 0)
        die("parent PIOCSTRACE");

    /* clear trace on any exec via PIOCSSPCACT */
    i = 0;
    if (ioctl(pfd, PIOCSSPCACT, &i) < 0)
        die("parent PIOCSSPCACT");
    if (close(pfd) < 0)
	die("parent close");

    proc_cntl(PIOCPSINFO, &ps);

    /* If the address is not in the range of a "normal" user process, we
     * must be in the loader, so give the process a run command to get it
     * to the actual user code.
     */
    if ((long)(ps.pr_addr) >= MAX_U_START) {
	/* The child is stopped in the user loader, run it, and wait
	 * for it to stop. This stop will be in the actual user program.
	 */
	prn.pr_flags = PRCSIG;
	proc_cntl(PIOCRUN, &prn);
	proc_cntl(PIOCWSTOP, 0);
	}
    /* now stopped at the real child process exec */

    /* Clear the inherit on fork flag in the child */
    proc_cntl(PIOCRFORK, 0);

    prn.pr_flags = PRCSIG;
    proc_cntl(PIOCRUN, &prn);
}

static void catchalarm() {
    gotalarm=1;
    return;
}

int main(int argc, char *argv[]) {
    char buf[1024];
    int i, c, print_pid=0, follow_fork=0, in_fork=0, in_execve=0, am_child=0, do_out_files=0;
    prstatus_t prstat;
    gregset_t regs;
    sysset_t ss;
    int p_opt = 0, usage = 0;
    char out_header[80];
    char output_template[MAXPATHLEN];
    char outbuf[MAXPATHLEN];
    int c_fd=0;
    pid_t child_pid, tracerpid;

    /* parse arguments */
    extern char *optarg;
    extern int optind;
    while (-1 != (c = getopt(argc, argv, "fPo:p:s:O:n"))) {
	switch (c) {
	case 'o':
	    if (NULL == (outfp = fopen(optarg, "w")))
		die(optarg);
	    break;
	case 'p':
	    pid = atoi(optarg);
	    p_opt++;
	    break;
	case 's':
	    max_string = atoi(optarg);
	    break;
	case 'P':
	    print_pid = 1; 
	    break;
	case 'f':
	    follow_fork = 1; 
	    break;
	case 'O':
	    strcpy(output_template,optarg);
	    do_out_files=1;
	    break;
	case 'n':
	    setlinebuf(stdout);
	    break;
	case '?':
	    usage++;
	}
    }

    argc -= optind;
    argv += optind;

    if (usage || (p_opt == 0 && argc == 0)) {
	fprintf(stderr,
	  "Trace for /proc - $Revision: 1.23 $ \n"
           "Usage: trace [-f] [-P] [-o outfile] [-O outfile_template] [-s string_length] command [arguments]\n"
           "       trace [-f] [-P] [-o outfile] [-O outfile_template] [-s string_length] -p pid\n");
	exit(1);
    }

    if (p_opt) {

	/* trace already running process */

	sprintf(buf, "/proc/%05d", pid);
	fprintf(outfp, "Tracing process %s\n", buf);

	if (-1 == (p_fd = open(buf, O_RDWR | O_EXCL, 0)))
	    die(buf);
    } else {

	/* start child process */

        printf("%s\n", argv[0]);

	do_exec(argv);
    }

    if(do_out_files) {
	proc_cntl(PIOCSTATUS,&prstat);
	sprintf(outbuf,"%s%05d",output_template,prstat.pr_pid);
	if(NULL == (outfp=fopen(outbuf,"w")) ) 
	    die(outbuf);
    }

    /* arrange for the run on last close flag to be set in the traced
       process; this ensures it resumes processing normally when the
       trace program exits */
    proc_cntl(PIOCSRLC, &ss);

    /* arrange for stop on entry and exit of all system calls */
    prfillset(&ss);
    prfillset(&pr.pr_trace);
    proc_cntl(PIOCSENTRY, &ss);
    proc_cntl(PIOCSEXIT, &ss);
    /* set flag to stop on termination */
    proc_cntl(PIOCGSPCACT,&i);
    i |= PRFS_STOPTERM;
    proc_cntl(PIOCSSPCACT, &i);

    /* arrange for all signals to be traced */
    proc_cntl(PIOCSTRACE, &pr.pr_trace);

    if(follow_fork)
	proc_cntl(PIOCSFORK, 0);

    /* child process will always be stopped at this point; a peer process
       may or may not be stopped; however, PIOCWSTOP doesn't seem to
       mind being called when the traced process is already stopped */

    /* now do the tracing */
    for(;;) {
	proc_cntl(PIOCWSTOP, &prstat);
	*out_header='\0';
	if(print_pid)
	    sprintf(out_header,"[%d]: ",prstat.pr_pid);

        if (prstat.pr_why == PR_SYSENTRY) {
	    proc_cntl(PIOCGREG, &regs);
	    if (prstat.pr_what == SYS_execve) {
		in_execve=2;
	    }
	    if((prstat.pr_what==SYS_fork)&&(follow_fork)) {
		in_fork=1;
		proc_cntl(PIOCSFORK, 0);
		proc_cntl(PIOCRRLC,0); /* to stop child running post-fork */
	    }
	    fputs(out_header,outfp);
	    i=(*sysent[regs.regs[0]].printargsfunc)(
		regs.regs[0],    
		&regs.regs[EF_A3],
		sysent[regs.regs[0]].numargs,
		sysent[regs.regs[0]].returns);
        } else if (prstat.pr_why == PR_SYSEXIT) {
	    proc_cntl(PIOCGREG, &regs);
	    if(in_fork) {
		in_fork=0;
		proc_cntl(PIOCSRLC,0); /* set run on last close in parent back */
		if(!regs.regs[EF_T8]) { /* did fork() succeed? */
		    print_pid=1;
		    child_pid=regs.regs[0];
		    proc_cntl(PIOCSRLC,0);
		    tracerpid=fork();
		    switch(tracerpid) {
			case -1: die("trace couldnt fork()"); break;
			case  0: am_child=1; break;
			default: am_child=0; break;
		    }
		    if(am_child) {   
			sprintf(buf, "/proc/%05d", child_pid);
			if (-1 == (c_fd = open(buf, O_RDWR | O_EXCL, 0)))
			    die(buf);
			close(p_fd);
			p_fd=c_fd;
			pid=child_pid;
			if(do_out_files) {
			    sprintf(outbuf,"%s%05d",output_template,child_pid);
			    if(NULL == (outfp=fopen(outbuf,"w")) ) 
				die(outbuf);
			}
			fprintf(outfp,"Tracing process %s\n",buf);
			prfillset(&ss);
			prfillset(&pr.pr_trace);
			proc_cntl(PIOCSENTRY, &ss);
			proc_cntl(PIOCSEXIT, &ss);
			proc_cntl(PIOCGSPCACT,&i);
			i |= PRFS_STOPTERM;
                        i ^= PRFS_STOPEXEC; /* turn off stop on exec */
			proc_cntl(PIOCSSPCACT, &i);

			/* arrange for all signals to be traced */
			proc_cntl(PIOCSTRACE, &pr.pr_trace);
			proc_cntl(PIOCSRLC, 0);
			continue;
		    }
		}  /* fork() failed - let routines below handle printing it */
	    }   /* end of fork() handling code */
/* The next bit is hairy. The system call has failed, and register 0   */
/* contains errno rather than the return value, _if_ EF_T8 is 1        */
/* Why EF_T8? Because thats how it works, as verified by disassembling */
/* syscall.o :-)                                                       */
	    i=(*sysent[prstat.pr_what].printretfunc)(
		prstat.pr_what,
		(regs.regs[EF_T8])?(long)(-1):(int)(regs.regs[0]),/* return */
		(regs.regs[EF_T8])?(regs.regs[0]):(0),   /* errno  */
		&regs.regs[EF_A3]); /* Arg ptr */

        } else if (prstat.pr_why == PR_SIGNALLED) {

	    /* normal case */
		fprintf(outfp, "%sSIGNAL [%d %s]\n", out_header, 
			(int)prstat.pr_what, sig_name[prstat.pr_what]);

	    if (prstat.pr_what == SIGTRAP) {
		/*
		 * If we're in execve then the SIGTRAP has been caused due
		 * to PIOCSSPCACT/PRFS_STOPEXEC. We should not let this
		 * signal get through to the child. We should also make sure
		 * we repeat this operation until we've done this in the
		 * actual user (rather than loader) program.
		 */
		if (in_execve) {
		    struct prrun npr = pr;
		    struct prpsinfo ps;
		    
		    proc_cntl(PIOCPSINFO, &ps);
		    if ((long)(ps.pr_addr) >= MAX_U_START) {
			/* Ideally should just be in_execve=0, but for some
			 * reason, we are above MAX_U_START even when in the
			 * loader
			 */
			in_execve--;
		    }

		    npr.pr_flags = PRCSIG;
		    proc_cntl(PIOCRUN, &npr);

		} else {
		  /* hack to prevent unfinite loops when SIGTRAP comes
		     in; needed for child process only; peer process OK */
		    for (i = 1; i <= NSIG; i++) {
			if (prismember(&prstat.pr_sigpend, i-1)) {
#ifdef _SYS_SIGINFO_H /* future OSF/1 change reqd. */
			    siginfo_t sig;
#else
			    struct siginfo sig;
#endif

			    /* signal 'i' is pending */
			    fprintf(outfp, "%sSIGNAL [%d %s]\n", 
				    out_header, i, sig_name[i]);

			    /* set this signal up as the current signal so it
			       is sent to the process being traced */
#ifdef _SYS_SIGINFO_H /* future OSF/1 change reqd. */
			    sig.si_signo = i;
#else
			    sig.fill = i;
#endif
			    proc_cntl(PIOCSSIG, &sig);
			    
			    /* remove signal from the pending list */
			    proc_cntl(PIOCUNKILL, &i);
			    
			    /* only deal with one signal at each stop */
			    break;
			}
		    }
		}
	    }
        } else if (prstat.pr_why == (short)PR_DEAD) {
            fprintf(outfp, "%sEXIT [%s=%d]\n", out_header,
		WIFEXITED(prstat.pr_what)
			? "status"
			: "signal",
		WIFEXITED(prstat.pr_what)
			? WEXITSTATUS(prstat.pr_what)
			: WTERMSIG(prstat.pr_what));
            break;

        } else {
            fprintf(outfp, "UNRECOGNISED PIOCWSTOP 0x%hx\n", prstat.pr_why);
        }

	proc_cntl(PIOCRUN, &pr);
    }
    die(0);
    /* NOTREACHED */
}
