/*
#ident	"@(#)smail/src/transports:RELEASE-3_2_0_110:tcpsmtp.c,v 1.53 2000/02/06 16:45:19 woods Exp"
 */

/*
 *    Copyright (C) 1987, 1988 Ronald S. Karr and Landon Curt Noll
 *    Copyright (C) 1992  Ronald S. Karr
 *
 * See the file COPYING, distributed with smail, for restriction
 * and warranty information.
 */

/*
 * tcpsmtp.c:
 *      Send mail using the SMTP protocol.
 *
 * See note in smtplib.c regarding logging of SMTP reply text.
 */

#define NEED_SOCKETS
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include "defs.h"
#if defined(POSIX_OS)
# include <unistd.h>
#else /* not POSIX_OS */
# if defined (HAVE_UNISTD_H)
#  include <unistd.h>
# endif
#endif	/* not POSIX_OS */

#include "../smail.h"
#include "../dys.h"
#include "../addr.h"
#include "../bindsmtpth.h"
#include "../transport.h"
#include "../route.h"
#include "../spool.h"
#include "../exitcodes.h"
#include "../smailconf.h"
#include "../parse.h"
#include "../lookup.h"
#include "tcpsmtp.h"
#include "smtplib.h"
#ifndef DEPEND
# include "../extern.h"
# include "../error.h"
# include "../debug.h"
#endif
#include "../log.h"

/* variables imported from libc */
extern int errno;

/* functions imported from libc */
extern time_t time();

/* functions local to this file */
#ifdef HAVE_BIND
static int tcpsmtp_bind_addr();
#endif
static int tcpsmtp_internal();
static char *set_short_timeout();
static char *set_long_timeout();
static char *set_timeout();
static int tcpsmtp_connect();
static struct error *connect_failure();
static int ip_address();

static struct attr_table tcpsmtp_attributes[] = {
    { "short_timeout", t_proc, NULL, (tup *)set_short_timeout, 0 },
    { "long_timeout", t_proc, NULL, (tup *)set_long_timeout, 0 },
    { "service", t_string, NULL, NULL, OFFSET(tcpsmtp_private, service) },
#ifdef HAVE_BIND
    { "use_bind", t_boolean, NULL, NULL, TCPSMTP_USE_BIND },
    BIND_ATTRIBUTES(tcpsmtp_private, bindlib_attr),
#endif
};
static struct attr_table *end_tcpsmtp_attributes = ENDTABLE(tcpsmtp_attributes);


/*
 * tpd_tcpsmtp - transport to remote machine using SMTP on top of TCP/IP.
 */
void
tpd_tcpsmtp(addr, succeed, defer, fail)
    struct addr *addr;                  /* recipient addresses for transport */
    struct addr **succeed;              /* successful deliveries */
    struct addr **defer;                /* defer until a later queue run */
    struct addr **fail;                 /* failed deliveries */
{
#ifdef HAVE_BIND
    struct transport_hints *hints;
    struct transport_hints *mx_hints;
    struct ipaddr_hint *ip_addr;
#endif
    char *service;
    char *error_text;
    struct error *lckerr = NULL;	/* lock error structure */
    struct error *connerr = NULL;	/* temporary for any connection error */
    struct tcpsmtp_private *priv;
    time_t started = 0;			/* connect start time for retry_host_unlock() */
    int defer_failure = TRUE;		/* always reset by retry_host_lock()... */
    int primary_result = SMTP_AGAIN;	/* initial result from primary MX host */

    priv = (struct tcpsmtp_private *) addr->transport->private;
    
    if (!(service = expand_string(priv->service, addr, (char *) NULL, (char *) NULL))) {
	error_text = xprintf("failed to expand service, %s", priv->service);
	insert_addr_list(addr, defer, connect_failure(addr->transport, error_text));
	return;
    }

#ifdef HAVE_BIND

    for (hints = addr->tphint_list;
	 hints && !EQ(hints->hint_name, "mx"); /* skipt past non-MX hints */
	 hints = hints->succ) {
	;
    }

    /*
     * if there were no valid MX's in the hints list, look again
     */
    if (!hints && (addr->transport->flags & TCPSMTP_USE_BIND)) {
	if (tcpsmtp_bind_addr(addr, defer, fail) != SUCCEED) {
            return;
	}

	for (hints = addr->tphint_list;
	     hints && !EQ(hints->hint_name, "mx"); /* skip initial non-MX's */
	     hints = hints->succ) {
	    ;
	}
    }

    if (hints) {			/* Try each MX in succession... */
	/*
	 * MX hint list is already sorted by preference so just walk through it
         */
	for (mx_hints = hints; mx_hints; mx_hints = mx_hints->succ) {
	    int already_locked = FALSE;	/* MX may have multiple targets -- optimise locking */
	    int tmp_result = SMTP_AGAIN;

	    if (!EQ(mx_hints->hint_name, "mx")) {
		continue;		/* ignore all but MX RRs */
	    }

	    /*
	     * Try each ip address of the target host...
	     */
# define mx_hint	((struct mx_transport_hint *) (mx_hints->private))

	    for (ip_addr = mx_hint->ipaddrs; ip_addr; ip_addr = ip_addr->succ) {
		char *lockhost;

		/* always lock the target domain name if we're using the
		 * smarthost because we don't ever want the smarthost to be
		 * blocked waiting for a retry...
		 */
		lockhost = (addr->flags & ADDR_SMARTHOST) ? addr->target : ip_addr->hostname;
		lckerr = NULL;
		if (!already_locked) {
		    if (retry_host_lock(addr->transport, lockhost, &defer_failure, &lckerr) == SUCCEED) {
			already_locked = TRUE;
		    } else {
			already_locked = FALSE;
			if (!defer_failure) {
			    /* if we didn't get the retry lock and if this
			     * message has past the retry duration, then move
			     * it to the fail list and just get out of here
			     * now...
			     */
			    lckerr->info |= ERR_NSOWNER;
			    insert_addr_list(addr, fail, lckerr);
			    return;	/* hopefully just ERR_173 "retry duration exceeded" */
			}
			continue;	/* this target is hosed for now.... */
		    }
		}
		/*
		 * if the target host was already locked, or if the above retry
		 * lock succeeded, then try to send the messsage...
		 */
		if (already_locked) {
                    (void) time(&started);
		    if ((tmp_result = tcpsmtp_internal(addr, succeed, defer, fail,
						       ip_addr->hostname, &ip_addr->addr,
						       AF_INET, service, &connerr)) == SMTP_SUCCEED) {
			/* pass a NULL error so the retry file is removed */
			retry_host_unlock(started, (struct error *) NULL);
			return;		/* good!  it's gone! */
		    }
		    DEBUG3(DBG_DRIVER_MID, "tpd_tcpsmtp(): connect to MX host %s failed (%d): %s.\n ", ip_addr->hostname, tmp_result, connerr ? connerr->message : "no error message given");
		    /*
		     * if there's another possible address to try then don't
		     * unlock it if it has the same hostname.  No sense
		     * unlocking the smarthost either as it won't change.
		     */
		    if (!(addr->flags & ADDR_SMARTHOST) &&
			(ip_addr->succ && !EQIC(ip_addr->hostname, (ip_addr->succ)->hostname))) {
			retry_host_unlock(started, connerr);
			already_locked = FALSE;
		    }
		}
	    }
	    if (mx_hints == hints) {	/* i.e. if we just did the first MX... */
		primary_result = tmp_result;
	    }
# undef mx_hint
	}
    } else	/* were no hints -- just try getting the address of the host */

#endif /* HAVE_BIND */

    {
	DEBUG1(DBG_DRIVER_MID, "tpd_tcpsmtp(): no hints -- trying host %s directly\n", addr->next_host);

	lckerr = NULL;
	if (retry_host_lock(addr->transport,
			    (addr->flags & ADDR_SMARTHOST) ? addr->target : addr->next_host,
			    &defer_failure,
			    &lckerr) == SUCCEED) {
	    struct in_addr ipaddr;
	    short family;

	    if (ip_address(addr->next_host, &ipaddr, &family, &error_text) < 0) {
		error_text = xprintf("IP address for %s not found", addr->next_host);
		connerr = connect_failure(addr->transport, error_text);
		retry_host_unlock(started, connerr);
	    } else {
		(void) time(&started);
		primary_result = tcpsmtp_internal(addr, succeed, defer, fail,
						  addr->next_host, &ipaddr,
						  family, service, &connerr);
		if (primary_result == SMTP_SUCCEED) {
		    /* pass a NULL error so the retry file is removed */
		    retry_host_unlock(started, (struct error *) NULL);
		    return;		/* good!  it's gone! */
		}
		retry_host_unlock(started, connerr);
		DEBUG2(DBG_DRIVER_LO, "tpd_tcpsmtp(): connect to host %s failed (%d).\n ", addr->next_host, primary_result);
	    }
	}
	/* note that both cases of lock failure are handled below */
    }
    /*
     * if the primary (or only) mail exchanger (host) fails the connection with
     * a 5xx error (and if none of any secondary mail exchangers accept it)
     * then we will bounce the message immediately
     */
    if (primary_result == SMTP_FAIL) {
	    defer_failure = FALSE; 
    }
    /* only use connerr if we tried a connection */
    /* NOTE: If we get here then we've encountered a problem so it is a bug for
     * there to be no error info at this point
     */
    if (started) {
	connerr->info |= ERR_NSOWNER;
    } else {
	lckerr->info |= ERR_NSOWNER;
    }
    /*
     * if we get this far then we still have the message and thus we have to
     * shuffle the address to the appropriate list for deferal or bouncing
     */
    insert_addr_list(addr, defer_failure ? defer : fail, started ? connerr : lckerr);

    return;
}

#ifdef HAVE_BIND

/*
 * tcpsmtp_bind_addr -- call bind_addr() for next_host
 *
 * return:
 *	SUCCEED if DNS lookups OK
 *	FAIL if DNS failed (addresses moved to fail or defer as appropriate)
 */
static int
tcpsmtp_bind_addr(addr, defer, fail)
    struct addr *addr;
    struct addr **defer;
    struct addr **fail;
{
    struct tcpsmtp_private *priv;
    struct rt_info rt_info;
    struct error *error;
    struct addr **notnow;		/* list to move to if problems */
    struct transport_hints **h;
    char *what;
    int result;

    priv = (struct tcpsmtp_private *)addr->transport->private;

    rt_info.next_host = NULL;
    rt_info.route = NULL;
    rt_info.transport = NULL;
    rt_info.tphint_list = NULL;

    what = xprintf("transport %s", addr->transport->name);
    result = bind_addr(addr->next_host, addr->transport->flags,
                       &priv->bindlib_attr, what, &rt_info, &error);
    xfree(what);
    if (rt_info.next_host) {
        xfree(rt_info.next_host);
    }
    if (rt_info.route) {
        xfree(rt_info.route);
    }

    notnow = NULL;
    switch (result)
    {
      case DB_SUCCEED:
        /* Found a successful match. */
        /* Append hints to address's list. */
        for (h = &addr->tphint_list; *h; h = &(*h)->succ)
            continue;
        *h = rt_info.tphint_list;
        break;

      case DB_NOMATCH:
        /* No match was found. */
        break;

      case DB_FAIL:
        /* The address should be failed, with an error of some kind. */
        notnow = fail;
        break;

      case DB_AGAIN:
        /* Routing for this address should be reattempted later. */
        notnow = defer;
	break;

      case FILE_NOMATCH:
        /* The file was not found, don't match any addresses. */
        break;

      case FILE_FAIL:
        /* Permanent router error, this is a configuration error. */
        error->info |= ERR_CONFERR;
        notnow = fail;
        break;

      case FILE_AGAIN:
        /* Temporary router database error, retry all addresses. */
        notnow = defer;
        break;
    }

    if (notnow) {
        insert_addr_list(addr, notnow, error);
        return FAIL;
    }

    return SUCCEED;
}

#endif  /* HAVE_BIND */

/*
 * tcpsmtp_internal -- the guts of sending to a given host
 *
 * return:
 *	SMTP_SUCCEED on successful message send
 *		- "addr" moved to "succeed" if sent successfully, or moved to
 *		"defer" or "failure" if error encountered during delivery.
 *	SMTP_FAIL if the connection should not be retried
 *		- "addr" is left alone for handling by the caller
 *	SMTP_AGAIN if the connection should be retried later
 *		- "addr" is left alone for handling by the caller
 */
static int
tcpsmtp_internal(addr, succeed, defer, fail, hostname, ipaddr, family, service, ep)
    struct addr *addr;                  /* recipient addresses for transport */
    struct addr **succeed;              /* successful deliveries */
    struct addr **defer;                /* defer until a later queue run */
    struct addr **fail;                 /* failed deliveries */
    char *hostname;			/* name of the host to connect to */
    struct in_addr *ipaddr;		/* IP address of the host to connect to */
    short family;			/* address family */
    char *service;			/* service to use */
    struct error **ep;			/* error structure */
{
    struct transport *tp = addr->transport;
    struct tcpsmtp_private *priv;
    int s = -1;				/* socket */
    int s2 = -1;			/* dup of s */
    struct smtp smtpbuf;		/* SMTP description buffer */
    char *error_text;
    int success;
    struct addr *ap;
    int try_ehlo;			/* allow EHLO, if supported */

    priv = (struct tcpsmtp_private *)tp->private;

    DEBUG4(DBG_DRIVER_LO, "transport %s: connect to host %s [%s]/%s...",
	   addr->transport->name, hostname, inet_ntoa(*ipaddr), service);

    /*
     * adjust the next_host for the address, so that log entries will
     * reflect the last MX host to be tried.
     */
    for (ap = addr; ap; ap = ap->succ) {
	if (ap->next_host == NULL || !EQIC(ap->next_host, hostname)) {
	    if (ap->next_host) {
		xfree(ap->next_host);
	    }
	    ap->next_host = COPY_STRING(hostname);
	}
    }
    /* Some gateways just terminate the conection when they receive an EHLO.
     * We may have to re-connect if they exhibit this bug....
     *
     * (NOTE: EHLO may not be used in smtplib.c if HAVE_EHLO is not defined.)
     */
    for (try_ehlo = 1, success = 0; !success && try_ehlo >= 0; --try_ehlo) {
	int result;

	if ((s = tcpsmtp_connect(hostname, ipaddr, family, service, &error_text)) >= 0) {
	    
	    if ((s2 = dup(s)) < 0) {
		(void) close(s);
		s = -1;
	    }
	}
	if (s < 0) {
	    *ep = connect_failure(tp, error_text);

	    return SMTP_AGAIN;
	}

	smtpbuf.in = fdopen(s, "r");	/* XXX error check? */
	smtpbuf.out = fdopen(s2, "w");	/* XXX error check? */
	smtpbuf.short_timeout = priv->short_timeout;
	smtpbuf.long_timeout = priv->long_timeout;
	smtpbuf.nl = "\r\n";
	tp->flags |= PUT_CRLF;
	smtpbuf.tp = tp;
	smtpbuf.max_size = 0;

	DEBUG(DBG_DRIVER_LO, "connected\n");

	switch ((result = smtp_startup(&smtpbuf, ep, try_ehlo))) {
	case SUCCEED:
	    success = 1;		/* we're connected! */
	    break;

	case SMTP_AGAIN:		/* server says try again */
	    (void) fclose(smtpbuf.in);
	    (void) fclose(smtpbuf.out);

	    return SMTP_AGAIN;

	case SMTP_EHLO_FAIL:		/* server hung up on us! */
	    (void) fclose(smtpbuf.in);
	    (void) fclose(smtpbuf.out);
	    /* write a message to the log file because it is interesting
	     * to know which mailers exhibit this bug....
	     */
	    write_log(WRITE_LOG_SYS, "link broken to '%s' by EHLO!", hostname);
	    break;			/* back around to try again without EHLO */

	case SMTP_FAIL:			/* remote server says "no!"... */
	    (void) fclose(smtpbuf.in);
	    (void) fclose(smtpbuf.out);

	    return SMTP_FAIL;

	default:
	    (void) fclose(smtpbuf.in);
	    (void) fclose(smtpbuf.out);
	    write_log(WRITE_LOG_PANIC, "IMPOSSIBLE result (%d) from smtp_startup() for %s", result, hostname);

	    return SMTP_AGAIN;
	}
    }

    if (dont_deliver) {
        insert_addr_list(addr, succeed, (struct error *)NULL);
        smtp_shutdown(&smtpbuf);
    } else {
	success = smtp_send(&smtpbuf, addr, succeed, defer, fail, ep);
	if (success == SUCCEED) {
	    smtp_shutdown(&smtpbuf);
	}
    }

    /* all done */
    (void) fclose(smtpbuf.in);
    (void) fclose(smtpbuf.out);

    /*
     * Always return success.  Errors in the smtp_send() phase should normally
     * only affect a single recipient and if we returned a failure code here
     * then a host retry file would be created, and we don't want that because
     * it would cause all mail to that host to be delayed.
     */
    return SMTP_SUCCEED;
}

/*
 * tpb_tcpsmtp - read the configuration file attributes
 */
char *
tpb_tcpsmtp(tp, attrs)
    struct transport *tp;               /* transport entry being defined */
    struct attribute *attrs;            /* list of per-driver attributes */
{
    char *error;
    static struct tcpsmtp_private tcpsmtp_template = {
        5 * 60,                         /* short timeout, 5 minutes */
        2 * 60 * 60,                    /* long timeout, 2 hours */
        "smtp",                         /* use the "smtp" service */
#ifdef HAVE_BIND
        BIND_TEMPLATE_ATTRIBUTES,
#endif
    };
    struct tcpsmtp_private *priv;       /* new tcpsmtp_private structure */

    /* copy the template private data */
    priv = (struct tcpsmtp_private *)xmalloc(sizeof(*priv));
    (void) memcpy((char *)priv, (char *)&tcpsmtp_template, sizeof(*priv));

    tp->private = (char *)priv;
    /* fill in the attributes of the private data */
    error = fill_attributes((char *)priv,
                            attrs,
                            &tp->flags,
                            tcpsmtp_attributes,
                            end_tcpsmtp_attributes);

    if (error) {
        return error;
    }
    return NULL;
}



/*
 * tpp_tcpsmtp - dump the configuration attributes
 */
void
tpp_tcpsmtp(f, tp)
     FILE * f;
     struct transport *tp;
{
    struct tcpsmtp_private *priv;

    (void) dump_standard_config(f,
				tp->private,
				tp->name,
				tp->flags,
				tcpsmtp_attributes,
				end_tcpsmtp_attributes);
    /* Deal with the proc config attributes */
    priv = (struct tcpsmtp_private *) tp->private;
    fprintf(f, "\tshort_timeout=%s,\n", ltoival((long) priv->short_timeout));
    fprintf(f, "\tlong_timeout=%s,\n", ltoival((long) priv->long_timeout));
}



static char *
set_short_timeout(struct_p, attr)
    char *struct_p;                     /* passed private structure */
    struct attribute *attr;             /* parsed attribute */
{
    struct tcpsmtp_private *priv = (struct tcpsmtp_private *)struct_p;

    return set_timeout(&priv->short_timeout, attr);
}

static char *
set_long_timeout(struct_p, attr)
    char *struct_p;                     /* passed private structure */
    struct attribute *attr;             /* parsed attribute */
{
    struct tcpsmtp_private *priv = (struct tcpsmtp_private *)struct_p;

    return set_timeout(&priv->long_timeout, attr);
}

static char *
set_timeout(timeout, attr)
    unsigned *timeout;                  /* set this timeout variable */
    struct attribute *attr;             /* parsed attribute */
{
    long l;

    if (attr->value == on) {
        return xprintf("%s: boolean form for non-boolean attribute",
                       attr->name);
    }
    l = ivaltol(attr->value);
    if (l < 0) {
        return xprintf("%s: %s: malformed interval",
                       attr->name, attr->value);
    }
    *timeout = (unsigned)l;
    if (*timeout != l) {
        return xprintf("%s: %s: interval too large", attr->name, attr->value);
    }
    return NULL;
}


/*
 * import h_errno; many systems don't define it in <netdb.h>
 */

#if defined(HAVE_BIND) && !defined(OBSOLETE_RESOLVER)
extern int h_errno;
#endif

/*
 * tcpsmtp_connect - return a socket connected to the remote host
 */
/* ARGSUSED */
static int
tcpsmtp_connect(remote_host, ip_addr, family, service, error)
    char *remote_host;				/* UNUSED */
    struct in_addr *ip_addr;
    short family;
    char *service;
    char **error;
{
    static int port = 0;                /* port to connect to */
    struct servent *smtp_service;       /* service entry */
    struct sockaddr_in saddrin;		/* inet socket address */
    static char *save_error = NULL;     /* keep handle to free error msgs */
    int s;                              /* socket */
    char *error_text;

    if (isdigit(*service)) {
        error_text = NULL;
        port = c_atol(service, &error_text);
        if (error_text) {
            *error = xprintf("invalid port: %s", service, error_text);
            return -1;
        }
    } else if (! port) {
        smtp_service = getservbyname(service, "tcp");
        if (! smtp_service) {
            *error = xprintf("service name %s not found", service);
            return -1;
        }
        port = smtp_service->s_port;
    }

    if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        if (save_error) {
            xfree(save_error);
        }
        *error = save_error = xprintf("socket: %s", strerror(errno));
        return -1;
    }

    if (sending_name) {
	struct hostent *hostentp;
	struct sockaddr_in lsaddrin;	/* local (source) inet socket address */

	if (!(hostentp = gethostbyname(sending_name))) {
	    if (save_error) {
		xfree(save_error);
	    }
#if defined(HAVE_BIND) && !defined(OBSOLETE_RESOLVER)
	    *error = save_error = xprintf("gethostbyname(%s): %s", sending_name, hstrerror(h_errno));
#else
	    *error = save_error = xprintf("gethostbyname(%s): failed.", sending_name);
#endif
	    return -1;
	}
	memset((char *) &lsaddrin, '\0', sizeof(lsaddrin));
	memcpy((char *) &lsaddrin.sin_addr, hostentp->h_addr_list[0], sizeof(struct in_addr));
	lsaddrin.sin_family = family;
	lsaddrin.sin_port = 0;		/* any old port will do.... */
	DEBUG2(DBG_DRIVER_MID, "\nbinding socket to local source addr (%s)[%s]... ", sending_name, inet_ntoa(lsaddrin.sin_addr));
	if (bind(s, (struct sockaddr *) &lsaddrin, sizeof(lsaddrin)) < 0) {
	    if (save_error) {
		xfree(save_error);
	    }
	    *error = save_error = xprintf("bind(%s[%s]): %s", sending_name, inet_ntoa(lsaddrin.sin_addr), strerror(errno));
	    return -1;
	}
    }

    memset((char *) &saddrin, '\0', sizeof(saddrin));
    saddrin.sin_addr = *ip_addr;
    saddrin.sin_family = family;
    saddrin.sin_port = port;

    if (connect(s, (struct sockaddr *) &saddrin, sizeof(saddrin)) < 0) {
        if (save_error) {
            xfree(save_error);
        }
        *error = save_error = xprintf("connect: %s", strerror(errno));
        (void) close(s);
        return -1;
    }

    return s;
}

static struct error *
connect_failure(tp, connect_error_text)
    struct transport *tp;
    char *connect_error_text;
{
    char *error_text;

    /*
     * ERR_148 - smtp connection failure
     *
     * DESCRIPTION
     *      We failed to connect to the smtp service of the remote
     *      host.  The reason is stored in `error'.
     *
     * ACTIONS
     *      The input addresses are deferred.
     *
     * RESOLUTION
     *      Hopefully we will connect on a retry.
     */
    error_text = xprintf("transport %s:\n%s", tp->name, connect_error_text);
    return note_error(ERR_148, error_text);
}


/*
 * ip_address - get the IP# and address family for a given host
 *
 * if the remote host name is of the form [192.2.12.3] then use an explicit
 * inet address.
 */
static int
ip_address(remote_host, ipaddr, family, errorp)
    char *remote_host;			/* hostname to look up */
    struct in_addr *ipaddr;		/* returned */
    short *family;			/* returned */
    char **errorp;			/* returned */
{
#ifdef INET_ADDR_USE_STRUCT
	struct in_addr s_inet;		/* internet address */
#endif

    if (remote_host[0] == '[') {
        /* INET addr literal address */
        char *p = strchr(remote_host, ']');
        unsigned long inet_number;

        if (p == NULL || p[1] != '\0') {
            *errorp = "Invalid host address";
            return -1;
        } else {
            *p = '\0';
#ifdef INET_ADDR_USE_STRUCT
            s_inet_number = inet_addr(remote_host + 1);
	    inet_number = s_inet_number.s_addr;
#else
            inet_number = inet_addr(remote_host + 1);
#endif
            *p = ']';
        }
        ipaddr->s_addr = inet_number;
        *family = AF_INET;
	DEBUG3(DBG_DRIVER_HI, "ip_address(%s): inet given: [0x%lx] aka %s\n",
	       (remote_host + 1), ntohl(inet_number), COPY_STRING(inet_ntoa(*ipaddr)));
    } else {
        struct hostent *hostentp;               /* addr for remote host */

        hostentp = gethostbyname(remote_host);
        if (hostentp == NULL) {
	    *errorp = hstrerror(h_errno);
            return -1;
        }
        (void) memcpy((char *) ipaddr, hostentp->h_addr, (size_t) hostentp->h_length);
        *family = hostentp->h_addrtype;
	/* XXX the following assume AF_INET */
	DEBUG3(DBG_DRIVER_HI, "ip_address(%s): gethostbyname(): [0x%lx] aka %s\n",
	       (remote_host + 1), ntohl(ipaddr->s_addr), COPY_STRING(inet_ntoa(*ipaddr)));
    }
    return 0;
}
