
/*
 * post.c -- enter messages into the mail transport system
 *
 * $Id$
 */

#include <h/mh.h>
#include <fcntl.h>
#include <h/signals.h>
#include <h/addrsbr.h>
#include <h/aliasbr.h>
#include <h/dropsbr.h>
#include <h/mhn.h>

#include <zotnet/tws/tws.h>
#include <zotnet/mts/mts.h>

#include <errno.h>
#include <setjmp.h>
#include <signal.h>

#ifdef MMDFMTS
# include <mts/mmdf/util.h>
# include <mts/mmdf/mmdf.h>
#endif

/*
 * Currently smtp and sendmail use
 * the same interface for posting.
 */
#ifdef SMTPMTS
# define SENDMTS
#endif

#ifdef SENDMTS
# include <mts/smtp/smtp.h>
#endif

#ifndef	MMDFMTS
# define uptolow(c) ((isalpha(c) && isupper (c)) ? tolower (c) : (c))
#endif

#define FCCS		10	/* max number of fccs allowed */

static struct swit switches[] = {
#define	ALIASW                    0
    { "alias aliasfile", 0 },
#define	CHKSW                     1
    { "check", -5 },			/* interface from whom */
#define	NCHKSW                    2
    { "nocheck", -7 },			/* interface from whom */
#define	DEBUGSW                   3
    { "debug", -5 },
#define	DISTSW                    4
    { "dist", -4 },			/* interface from dist */
#define	FILTSW                    5
    { "filter filterfile", 0 },
#define	NFILTSW                   6
    { "nofilter", 0 },
#define	FRMTSW                    7
    { "format", 0 },
#define	NFRMTSW                   8
    { "noformat", 0 },
#define	LIBSW                     9
    { "library directory", -7 },	/* interface from send, whom */
#define	MIMESW                   10
    { "mime", 0 },
#define	NMIMESW                  11
    { "nomime", 0 },
#define	MSGDSW                   12
    { "msgid", 0 },
#define	NMSGDSW                  13
    { "nomsgid", 0 },
#define	VERBSW                   14
    { "verbose", 0 },
#define	NVERBSW                  15
    { "noverbose", 0 },
#define	WATCSW                   16
    { "watch", 0 },
#define	NWATCSW                  17
    { "nowatch", 0 },
#define	WHOMSW                   18
    { "whom", -4 },			/* interface from whom */
#define	WIDTHSW                  19
    { "width columns", 0 },
#define VERSIONSW                20
    { "version", 0 },
#define	HELPSW                   21
    { "help", 4 },
#define	MAILSW                   22
    { "mail", -4 },			/* specify MAIL smtp mode */
#define	SAMLSW                   23
    { "saml", -4 },			/* specify SAML smtp mode */
#define	SENDSW                   24
    { "send", -4 },			/* specify SEND smtp mode */
#define	SOMLSW                   25
    { "soml", -4 },			/* specify SOML smtp mode */
#define	ANNOSW                   26
    { "idanno number", -6 },		/* interface from send    */
#define	DLVRSW                   27
    { "deliver address-list", -7 },
#define	CLIESW                   28
    { "client host", -6 },
#define	SERVSW                   29
    { "server host", -6 },		/* specify alternate SMTP server */
#define	SNOOPSW                  30
    { "snoop", -5 },			/* snoop the SMTP transaction */
#define	FILLSW                   31
    { "fill-in file", -7 },
#define	FILLUSW                  32
    { "fill-up", -7 },
#define	PARTSW                   33
    { "partno", -6 },
#define	QUEUESW                  34
    { "queued", -6 },
    { NULL, 0 }
};


struct headers {
    char *value;
    unsigned int flags;
    unsigned int set;
};

/*
 * flags for headers->flags
 */
#define	HNOP  0x0000		/* just used to keep .set around          */
#define	HBAD  0x0001		/* bad header - don't let it through      */
#define	HADR  0x0002		/* header has an address field            */
#define	HSUB  0x0004		/* Subject: header                        */
#define	HTRY  0x0008		/* try to send to addrs on header         */
#define	HBCC  0x0010		/* don't output this header               */
#define	HMNG  0x0020		/* munge this header                      */
#define	HNGR  0x0040		/* no groups allowed in this header       */
#define	HFCC  0x0080		/* FCC: type header                       */
#define	HNIL  0x0100		/* okay for this header not to have addrs */
#define	HIGN  0x0200		/* ignore this header                     */
#define	HDCC  0x0400		/* another undocumented feature           */

/*
 * flags for headers->set
 */
#define	MFRM  0x0001		/* we've seen a From:        */
#define	MDAT  0x0002		/* we've seen a Date:        */
#define	MRFM  0x0004		/* we've seen a Resent-From: */
#define	MVIS  0x0008		/* we've seen sighted addrs  */
#define	MINV  0x0010		/* we've seen blind addrs    */


static struct headers NHeaders[] = {
    { "Return-Path", HBAD,                0 },
    { "Received",    HBAD,                0 },
    { "Reply-To",    HADR|HNGR,           0 },
    { "From",        HADR|HNGR,           MFRM },
    { "Sender",      HADR|HBAD,           0 },
    { "Date",        HBAD,                0 },
    { "Subject",     HSUB,                0 },
    { "To",          HADR|HTRY,           MVIS },
    { "cc",          HADR|HTRY,           MVIS },
    { "Bcc",         HADR|HTRY|HBCC|HNIL, MINV },
    { "Dcc",         HADR|HTRY|HDCC|HNIL, MVIS },	/* sorta cc & bcc combined */
    { "Message-ID",  HBAD,                0 },
    { "Fcc",         HFCC,                0 },
    { NULL,          0,                   0 }
};

static struct headers RHeaders[] = {
    { "Resent-Reply-To",   HADR|HNGR,           0 },
    { "Resent-From",       HADR|HNGR,           MRFM },
    { "Resent-Sender",     HADR|HBAD,           0 },
    { "Resent-Date",       HBAD,                0 },
    { "Resent-Subject",    HSUB,                0 },
    { "Resent-To",         HADR|HTRY,           MVIS },
    { "Resent-cc",         HADR|HTRY,           MVIS },
    { "Resent-Bcc",        HADR|HTRY|HBCC,      MINV },
    { "Resent-Message-ID", HBAD,                0 },
    { "Resent-Fcc",        HFCC,                0 },
    { "Reply-To",          HADR,                0 },
    { "From",              HADR|HNGR,           MFRM },
#ifdef MMDFI
    { "Sender",            HADR|HNGR|HMNG,      0 },
#else
    { "Sender",            HADR|HNGR,           0 },
#endif
    { "Date",              HNOP,                MDAT },
    { "To",                HADR|HNIL,           0 },
    { "cc",                HADR|HNIL,           0 },
    { "Bcc",               HADR|HTRY|HBCC|HNIL, 0 },
    { "Fcc",               HIGN,                0 },
    { NULL,                0,                   0 }
};

static short fccind = 0;	/* index into fccfold[] */
static short outputlinelen = OUTPUTLINELEN;

static int pfd = NOTOK;		/* fd to write annotation list to        */
static uid_t myuid= -1;		/* my user id                            */
static gid_t mygid= -1;		/* my group id                           */
static int recipients = 0;	/* how many people will get a copy       */
static int unkadr = 0;		/* how many of those were unknown        */
static int badadr = 0;		/* number of bad addrs                   */
static int badmsg = 0;		/* message has bad semantics             */
static int verbose = 0;		/* spell it out                          */
static int format = 1;		/* format addresses                      */
static int mime = 0;		/* use MIME-style encapsulations for Bcc */
static int msgid = 0;		/* add msgid                             */
static int debug = 0;		/* debugging post                        */
static int watch = 0;		/* watch the delivery process            */
static int whomsw = 0;		/* we are whom not post                  */
static int checksw = 0;		/* whom -check                           */
static int linepos=0;		/* putadr()'s position on the line       */
static int nameoutput=0;	/* putadr() has output header name       */

static unsigned msgflags = 0;	/* what we've seen */

#define	NORMAL 0
#define	RESENT 1
static int msgstate = NORMAL;

static time_t tclock = 0;	/* the time we started (more or less) */

static SIGNAL_HANDLER hstat, istat, qstat, tstat;

static char tmpfil[BUFSIZ];
static char bccfil[BUFSIZ];

static char from[BUFSIZ];	/* my network address            */
static char signature[BUFSIZ];	/* my signature                  */
static char *filter = NULL;	/* the filter for BCC'ing        */
static char *subject = NULL;	/* the subject field for BCC'ing */
static char *fccfold[FCCS];	/* foldernames for FCC'ing       */

static struct headers  *hdrtab;	/* table for the message we're doing */

static struct mailname localaddrs={NULL};	/* local addrs     */
static struct mailname netaddrs={NULL};		/* network addrs   */
static struct mailname uuaddrs={NULL};		/* uucp addrs      */
static struct mailname tmpaddrs={NULL};		/* temporary queue */

#ifdef MMDFMTS
static char *submitmode = "m";		/* deliver to mailbox only    */
static char submitopts[6] = "vl";	/* initial options for submit */
#endif /* MMDFMTS */

#ifdef SENDMTS
static int snoop      = 0;
static int smtpmode   = S_MAIL;
static char *clientsw = NULL;
static char *serversw = NULL;

extern struct smtp sm_reply;
#endif /* SENDMTS */

static char prefix[] = "----- =_aaaaaaaaaa";

static int fill_up = 0;
static char *fill_in = NULL;
static char *partno = NULL;
static int queued = 0;

/*
 * static prototypes
 */
static void putfmt (char *, char *, FILE *);
static void start_headers (void);
static void finish_headers (FILE *);
static int get_header (char *, struct headers *);
static int putadr (char *, char *, struct mailname *, FILE *, unsigned int);
static void putgrp (char *, char *, FILE *, unsigned int);
static int insert (struct mailname *);
static void pl (void);
static void anno (void);
static int annoaux (struct mailname *);
static void insert_fcc (struct headers *, char *);
static void make_bcc_file (void);
static void verify_all_addresses (int);
static void chkadr (void);
static void sigon (void);
static void sigoff (void);
static void p_refile (char *);
static void fcc (char *, char *);
static void die (char *, char *, ...);
static void post (char *, int, int);
static void do_text (char *file, int fd);
static void do_an_address (struct mailname *, int);
static void do_addresses (int, int);
static int find_prefix (void);


int
main (int argc, char **argv)
{
    int state, compnum;
    char *cp, *msg = NULL, **argp = argv + 1;
    char buf[BUFSIZ], name[NAMESZ];
    FILE *in, *out;

#ifdef LOCALE
    setlocale(LC_ALL, "");
#endif
    invo_name = r1bindex (argv[0], '/');

    /* foil search of use profile/context */
    if (context_foil (NULL) == -1)
	done (1);

    mts_init (invo_name);

#if defined(MMDFMTS) && defined(MMDFII)
    mmdf_init (invo_name);
#endif /* MMDFMTS and MMDFII */

    while ((cp = *argp++)) {
	if (*cp == '-') {
	    switch (smatch (++cp, switches)) {
		case AMBIGSW: 
		    ambigsw (cp, switches);
		    done (1);
		case UNKWNSW: 
		    adios (NULL, "-%s unknown", cp);

		case HELPSW: 
		    sprintf (buf, "%s [switches] file", invo_name);
		    print_help (buf, switches, 0);
		    done (1);
		case VERSIONSW:
		    print_version(invo_name);
		    done (1);

		case LIBSW:
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    /* create a minimal context */
		    if (context_foil (cp) == -1)
			done (1);
		    continue;

		case ALIASW: 
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    if ((state = alias (cp)) != AK_OK)
			adios (NULL, "aliasing error in %s - %s",
				cp, akerror (state));
		    continue;

		case CHKSW: 
		    checksw++;
		    continue;
		case NCHKSW: 
		    checksw = 0;
		    continue;

		case DEBUGSW: 
		    debug++;
		    continue;

		case DISTSW:
		    msgstate = RESENT;
		    continue;

		case FILTSW:
		    if (!(filter = *argp++) || *filter == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    mime = 0;
		    continue;
		case NFILTSW:
		    filter = NULL;
		    continue;
		
		case FRMTSW: 
		    format++;
		    continue;
		case NFRMTSW: 
		    format = 0;
		    continue;

		case MIMESW:
		    mime++;
		    filter = NULL;
		    continue;
		case NMIMESW: 
		    mime = 0;
		    continue;

		case MSGDSW: 
		    msgid++;
		    continue;
		case NMSGDSW: 
		    msgid = 0;
		    continue;

		case VERBSW: 
		    verbose++;
		    continue;
		case NVERBSW: 
		    verbose = 0;
		    continue;

		case WATCSW: 
		    watch++;
		    continue;
		case NWATCSW: 
		    watch = 0;
		    continue;

		case WHOMSW: 
		    whomsw++;
		    continue;

		case WIDTHSW: 
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    if ((outputlinelen = atoi (cp)) < 10)
			adios (NULL, "impossible width %d", outputlinelen);
		    continue;

		case ANNOSW: 
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    if ((pfd = atoi (cp)) <= 2)
			adios (NULL, "bad argument %s %s", argp[-2], cp);
		    continue;

#ifdef MMDFMTS
		case MAILSW:
		    submitmode = "m";
		    continue;
		case SOMLSW:	/* for right now, sigh... */
		case SAMLSW:
		    submitmode = "b";
		    continue;
		case SENDSW:
		    submitmode = "y";
		    continue;
#endif /* MMDFMTS */

		case DLVRSW:
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;

#ifndef	SENDMTS
		case CLIESW:
		case SERVSW:
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;

		case SNOOPSW:
		    continue;
#else /* SENDMTS */
		case MAILSW:
		    smtpmode = S_MAIL;
		    continue;
		case SAMLSW:
		    smtpmode = S_SAML;
		    continue;
		case SOMLSW:
		    smtpmode = S_SOML;
		    continue;
		case SENDSW:
		    smtpmode = S_SEND;
		    continue;
		case CLIESW:
		    if (!(clientsw = *argp++) || *clientsw == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;
		case SERVSW:
		    if (!(serversw = *argp++) || *serversw == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;
		case SNOOPSW:
		    snoop++;
		    continue;
#endif /* SENDMTS */

		case FILLSW:
		    if (!(fill_in = *argp++) || *fill_in == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;
		case FILLUSW:
		    fill_up++;
		    continue;
		case PARTSW:
		    if (!(partno = *argp++) || *partno == '-')
			adios (NULL, "missing argument to %s", argp[-2]);
		    continue;

		case QUEUESW:
		    queued++;
		    continue;
	    }
	}
	if (msg)
	    adios (NULL, "only one message at a time!");
	else
	    msg = cp;
    }

    alias (AliasFile);

    if (!msg)
	adios (NULL, "usage: %s [switches] file", invo_name);

    if (outputlinelen < 10)
	adios (NULL, "impossible width %d", outputlinelen);

    if ((in = fopen (msg, "r")) == NULL)
	adios (msg, "unable to open");

    start_headers ();
    if (debug) {
	verbose++;
	discard (out = stdout);	/* XXX: reference discard() to help loader */
    } else {
	if (whomsw) {
	    if ((out = fopen (fill_in ? fill_in : "/dev/null", "w")) == NULL)
		adios ("/dev/null", "unable to open");
	} else {
	    strcpy (tmpfil, m_scratch ("", m_maildir (invo_name)));
	    if ((out = fopen (tmpfil, "w")) == NULL) {
		strcpy (tmpfil, m_tmpfil (invo_name));
		if ((out = fopen (tmpfil, "w")) == NULL)
		    adios (tmpfil, "unable to create");
	    }
	    chmod (tmpfil, 0600);
	}
    }

    hdrtab = msgstate == NORMAL ? NHeaders : RHeaders;

    for (compnum = 1, state = FLD;;) {
	switch (state = m_getfld (state, name, buf, sizeof(buf), in)) {
	    case FLD: 
	    case FLDEOF: 
	    case FLDPLUS: 
		compnum++;
		cp = add (buf, NULL);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    cp = add (buf, cp);
		}
		putfmt (name, cp, out);
		free (cp);
		if (state != FLDEOF)
		    continue;
		finish_headers (out);
		break;

	    case BODY: 
	    case BODYEOF: 
		finish_headers (out);
		if (whomsw && !fill_in)
		    break;
		fprintf (out, "\n%s", buf);
		while (state == BODY) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    fputs (buf, out);
		}
		break;

	    case FILEEOF: 
		finish_headers (out);
		break;

	    case LENERR: 
	    case FMTERR: 
		adios (NULL, "message format error in component #%d", compnum);

	    default: 
		adios (NULL, "getfld() returned %d", state);
	}
	break;
    }

    if (pfd != NOTOK)
	anno ();
    fclose (in);

    if (debug) {
	pl ();
	done (0);
    } else {
	fclose (out);
    }

    /* If we are doing a "whom" check */
    if (whomsw) {
	if (!fill_up)
	    verify_all_addresses (1);
	done (0);
    }

#ifdef MMDFMTS
    strcat (submitopts, submitmode);
    if (watch)
	strcat (submitopts, "nw");
#endif /* MMDFMTS */

    if (msgflags & MINV) {
	make_bcc_file ();
	if (msgflags & MVIS) {
	    verify_all_addresses (verbose);
	    post (tmpfil, 0, verbose);
	}
	post (bccfil, 1, verbose);
	unlink (bccfil);
    } else {
	post (tmpfil, 0, isatty (1));
    }

    p_refile (tmpfil);
    unlink (tmpfil);

    if (verbose)
	printf (partno ? "Partial Message #%s Processed\n" : "Message Processed\n",
		partno);
    done (0);
}


/*
 * DRAFT GENERATION
 */

static void
putfmt (char *name, char *str, FILE *out)
{
    int count, grp, i, keep;
    char *cp, *pp, *qp;
    char namep[BUFSIZ];
    struct mailname *mp, *np;
    struct headers *hdr;

    while (*str == ' ' || *str == '\t')
	str++;

    if (msgstate == NORMAL && uprf (name, "resent")) {
	advise (NULL, "illegal header line -- %s:", name);
	badmsg++;
	return;
    }

    if ((i = get_header (name, hdrtab)) == NOTOK) {
	fprintf (out, "%s: %s", name, str);
	return;
    }

    hdr = &hdrtab[i];
    if (hdr->flags & HIGN) {
	if (fill_in)
	    fprintf (out, "%s: %s", name, str);
	return;
    }
    if (hdr->flags & HBAD) {
	if (fill_in)
	    fprintf (out, "%s: %s", name, str);
	else {
	    advise (NULL, "illegal header line -- %s:", name);
	    badmsg++;
	}
	return;
    }
    msgflags |= (hdr->set & ~(MVIS | MINV));

    if (hdr->flags & HSUB)
	subject = subject ? add (str, add ("\t", subject)) : getcpy (str);
    if (hdr->flags & HFCC) {
	if (fill_in) {
	    fprintf (out, "%s: %s", name, str);
	    return;
	}

	if ((cp = strrchr(str, '\n')))
	    *cp = 0;
	for (cp = pp = str; cp = strchr(pp, ','); pp = cp) {
	    *cp++ = 0;
	    insert_fcc (hdr, pp);
	}
	insert_fcc (hdr, pp);
	return;
    }

    if (!(hdr->flags & HADR)) {
	fprintf (out, "%s: %s", name, str);
	return;
    }

    tmpaddrs.m_next = NULL;
    for (count = 0; cp = getname (str); count++)
	if ((mp = getm (cp, NULL, 0, AD_HOST, NULL))) {
	    if (tmpaddrs.m_next)
		np->m_next = mp;
	    else
		tmpaddrs.m_next = mp;
	    np = mp;
	}
	else
	    if (hdr->flags & HTRY)
		badadr++;
	    else
		badmsg++;

    if (count < 1) {
	if (hdr->flags & HNIL)
	    fprintf (out, "%s: %s", name, str);
	else {
#ifdef notdef
	    advise (NULL, "%s: field requires at least one address", name);
	    badmsg++;
#endif /* notdef */
	}
	return;
    }

    nameoutput = linepos = 0;
    sprintf (namep, "%s%s", !fill_in && (hdr->flags & HMNG) ? "Original-" : "", name);

    for (grp = 0, mp = tmpaddrs.m_next; mp; mp = np)
	if (mp->m_nohost) {	/* also used to test (hdr->flags & HTRY) */
	    pp = akvalue (mp->m_mbox);
	    qp = akvisible () ? mp->m_mbox : "";
	    np = mp;
	    if (np->m_gname)
		putgrp (namep, np->m_gname, out, hdr->flags);
	    while ((cp = getname (pp))) {
		if (!(mp = getm (cp, NULL, 0, AD_HOST, NULL))) {
		    badadr++;
		    continue;
		}
		if (hdr->flags & HBCC)
		    mp->m_bcc++;
		if (np->m_ingrp)
		    mp->m_ingrp = np->m_ingrp;
		else
		    if (mp->m_gname)
			putgrp (namep, mp->m_gname, out, hdr->flags);
		if (mp->m_ingrp)
		    grp++;
		if (putadr (namep, qp, mp, out, hdr->flags))
		    msgflags |= (hdr->set & (MVIS | MINV));
		else
		    mnfree (mp);
	    }
	    mp = np;
	    np = np->m_next;
	    mnfree (mp);
	}
	else {
	    if (hdr->flags & HBCC)
		mp->m_bcc++;
	    if (mp->m_gname)
		putgrp (namep, mp->m_gname, out, hdr->flags);
	    if (mp->m_ingrp)
		grp++;
	    keep = putadr (namep, "", mp, out, hdr->flags);
	    np = mp->m_next;
	    if (keep) {
		mp->m_next = NULL;
		msgflags |= (hdr->set & (MVIS | MINV));
	    }
	    else
		mnfree (mp);
	}

    if (grp > 0 && (hdr->flags & HNGR)) {
	advise (NULL, "%s: field does not allow groups", name);
	badmsg++;
    }
    if (linepos) {
	if (fill_in && grp > 0)
	    putc (';', out);
	putc ('\n', out);
    }
}


static void
start_headers (void)
{
    char  *cp;
    char myhost[BUFSIZ], sigbuf[BUFSIZ];
    struct mailname *mp;

    myuid = getuid ();
    mygid = getgid ();
    time (&tclock);

    strcpy (from, adrsprintf (NULL, NULL));

    strcpy (myhost, LocalName ());
    for (cp = myhost; *cp; cp++)
	*cp = uptolow (*cp);

    if ((cp = getfullname ()) && *cp) {
	strcpy (sigbuf, cp);
	sprintf (signature, "%s <%s>", sigbuf, adrsprintf (NULL, NULL));
	if ((cp = getname (signature)) == NULL)
	    adios (NULL, "getname () failed -- you lose extraordinarily big");
	if ((mp = getm (cp, NULL, 0, AD_HOST, NULL)) == NULL)
	    adios (NULL, "bad signature '%s'", sigbuf);
	mnfree (mp);
	while (getname (""))
	    continue;
    } else {
	strcpy (signature, adrsprintf (NULL, NULL));
    }
}


static void
finish_headers (FILE *out)
{
    switch (msgstate) {
	case NORMAL: 
	    if (whomsw && !fill_up)
		break;

	    fprintf (out, "Date: %s\n", dtime (&tclock, 0));
	    if (msgid)
		fprintf (out, "Message-ID: <%d.%ld@%s>\n",
			(int) getpid (), tclock, LocalName ());
	    if (msgflags & MFRM)
		fprintf (out, "Sender: %s\n", from);
	    else
		fprintf (out, "From: %s\n", signature);
	    if (whomsw)
		break;

	    if (!(msgflags & MVIS))
		fprintf (out, "Bcc: Blind Distribution List: ;\n");
	    break;

	case RESENT: 
	    if (!(msgflags & MDAT)) {
		advise (NULL, "message has no Date: header");
		badmsg++;
	    }
	    if (!(msgflags & MFRM)) {
		advise (NULL, "message has no From: header");
		badmsg++;
	    }
	    if (whomsw && !fill_up)
		break;

#ifdef MMDFI			/* sigh */
	    fprintf (out, "Sender: %s\n", from);
#endif /* MMDFI */

	    fprintf (out, "Resent-Date: %s\n", dtime (&tclock, 0));
	    if (msgid)
		fprintf (out, "Resent-Message-ID: <%d.%ld@%s>\n",
			(int) getpid (), tclock, LocalName ());
	    if (msgflags & MRFM)
		fprintf (out, "Resent-Sender: %s\n", from);
	    else
		fprintf (out, "Resent-From: %s\n", signature);
	    if (whomsw)
		break;
	    if (!(msgflags & MVIS))
		fprintf (out, "Resent-Bcc: Blind Re-Distribution List: ;\n");
	    break;
    }

    if (badmsg)
	adios (NULL, "re-format message and try again");
    if (!recipients)
	adios (NULL, "no addressees");
}


static int
get_header (char *header, struct headers *table)
{
    struct headers *h;

    for (h = table; h->value; h++)
	if (!strcasecmp (header, h->value))
	    return (h - table);

    return NOTOK;
}


static int
putadr (char *name, char *aka, struct mailname *mp, FILE *out, unsigned int flags)
{
    int len;
    char *cp;
    char buffer[BUFSIZ];

    if (mp->m_mbox == NULL || ((flags & HTRY) && !insert (mp)))
	return 0;
    if (!fill_in && (flags & (HBCC | HDCC)) || mp->m_ingrp)
	return 1;

    if (!nameoutput) {
	fprintf (out, "%s: ", name);
	linepos += (nameoutput = strlen (name) + 2);
    }

    if (*aka && mp->m_type != UUCPHOST && !mp->m_pers)
	mp->m_pers = getcpy (aka);
    if (format) {
	if (mp->m_gname && !fill_in)
	    sprintf (cp = buffer, "%s;", mp->m_gname);
	else
	    cp = adrformat (mp);
    }
    else
	cp = mp->m_text;
    len = strlen (cp);

    if (linepos != nameoutput)
	if (len + linepos + 2 > outputlinelen)
	    fprintf (out, ",\n%*s", linepos = nameoutput, "");
	else {
	    fputs (", ", out);
	    linepos += 2;
	}

    fputs (cp, out);
    linepos += len;

    return (flags & HTRY);
}


static void
putgrp (char *name, char *group, FILE *out, unsigned int flags)
{
    int len;
    char *cp;

    if (!fill_in && (flags & HBCC))
	return;

    if (!nameoutput) {
	fprintf (out, "%s: ", name);
	linepos += (nameoutput = strlen (name) + 2);
	if (fill_in)
	    linepos -= strlen (group);
    }

    cp = fill_in ? group : concat (group, ";", NULL);
    len = strlen (cp);

    if (linepos > nameoutput)
	if (len + linepos + 2 > outputlinelen) {
	    fprintf (out, ",\n%*s", nameoutput, "");
	    linepos = nameoutput;
	}
	else {
	    fputs (", ", out);
	    linepos += 2;
	}

    fputs (cp, out);
    linepos += len;
}


static int
insert (struct mailname *np)
{
    struct mailname *mp;

    if (np->m_mbox == NULL)
	return 0;

    for (mp = np->m_type == LOCALHOST ? &localaddrs
	    : np->m_type == UUCPHOST ? &uuaddrs
	    : &netaddrs;
	    mp->m_next;
	    mp = mp->m_next)
	if (!strcasecmp (np->m_host, mp->m_next->m_host)
		&& !strcasecmp (np->m_mbox, mp->m_next->m_mbox)
		&& np->m_bcc == mp->m_next->m_bcc)
	    return 0;

    mp->m_next = np;
    recipients++;
    return 1;
}


static void
pl (void)
{
    int i;
    struct mailname *mp;

    printf ("-------\n\t-- Addresses --\nlocal:\t");
    for (mp = localaddrs.m_next; mp; mp = mp->m_next)
	printf ("%s%s%s", mp->m_mbox,
		mp->m_bcc ? "[BCC]" : "",
		mp->m_next ? ",\n\t" : "");

    printf ("\nnet:\t");
    for (mp = netaddrs.m_next; mp; mp = mp->m_next)
	printf ("%s%s@%s%s%s", mp->m_path ? mp->m_path : "",
		mp->m_mbox, mp->m_host,
		mp->m_bcc ? "[BCC]" : "",
		mp->m_next ? ",\n\t" : "");

    printf ("\nuucp:\t");
    for (mp = uuaddrs.m_next; mp; mp = mp->m_next)
	printf ("%s!%s%s%s", mp->m_host, mp->m_mbox,
		mp->m_bcc ? "[BCC]" : "",
		mp->m_next ? ",\n\t" : "");

    printf ("\n\t-- Folder Copies --\nfcc:\t");
    for (i = 0; i < fccind; i++)
	printf ("%s%s", fccfold[i], i + 1 < fccind ? ",\n\t" : "");
    printf ("\n");
}


static void
anno (void)
{
    struct mailname *mp;

    for (mp = localaddrs.m_next; mp; mp = mp->m_next)
	if (annoaux (mp) == NOTOK)
	    goto oops;

    for (mp = netaddrs.m_next; mp; mp = mp->m_next)
	if (annoaux (mp) == NOTOK)
	    goto oops;

    for (mp = uuaddrs.m_next; mp; mp = mp->m_next)
	if (annoaux (mp) == NOTOK)
	    break;

oops: ;
    close (pfd);
    pfd = NOTOK;
}


static int
annoaux (struct mailname *mp)
{
    int i;
    char buffer[BUFSIZ];

    sprintf (buffer, "%s\n", adrformat (mp));
    i = strlen (buffer);

    return (write (pfd, buffer, i) == i ? OK : NOTOK);
}


static void
insert_fcc (struct headers *hdr, char *pp)
{
    char *cp;

    for (cp = pp; isspace (*cp); cp++)
	continue;
    for (pp += strlen (pp) - 1; pp > cp && isspace (*pp); pp--)
	continue;
    if (pp >= cp)
	*++pp = 0;
    if (*cp == 0)
	return;

    if (fccind >= FCCS)
	adios (NULL, "too many %ss", hdr->value);
    fccfold[fccind++] = getcpy (cp);
}

/*
 * BCC GENERATION
 */

static void
make_bcc_file (void)
{
    int fd, i;
    pid_t child_id;
    char *vec[6];
    FILE *out;

    strcpy (bccfil, m_tmpfil ("bccs"));
    if ((out = fopen (bccfil, "w")) == NULL)
	adios (bccfil, "unable to create");
    chmod (bccfil, 0600);

    fprintf (out, "Date: %s\n", dtime (&tclock, 0));
    if (msgid)
	fprintf (out, "Message-ID: <%d.%ld@%s>\n",
		(int) getpid (), tclock, LocalName ());
    fprintf (out, "From: %s\n", signature);
    if (subject)
	fprintf (out, "Subject: %s", subject);
    fprintf (out, "BCC:\n");

    /*
     * Use MIME encapsulation for Bcc messages
     */
    if (mime) {
	char *cp;

	if ((cp = strchr(prefix, 'a')) == NULL)
	    adios (NULL, "lost prefix start");
	while (find_prefix () == NOTOK)
	    if (*cp < 'z')
		(*cp)++;
	    else
		if (*++cp == 0)
		    adios (NULL, "can't find a unique delimiter string");
		else
		    (*cp)++;

	fprintf (out, "%s: %s\n%s: multipart/digest; boundary=\"",
		 VRSN_FIELD, VRSN_VALUE, TYPE_FIELD);
	fprintf (out, "%s\"\n\n--%s\n\n", prefix, prefix);
    } else {
	fprintf (out, "\n------- Blind-Carbon-Copy\n\n");
    }

    fflush (out);

    /*
     * Do mhl filtering of Bcc messages instead
     * of MIME encapsulation.
     */
    if (filter != NULL) {
	vec[0] = r1bindex (mhlproc, '/');

	for (i = 0; (child_id = fork()) == NOTOK && i < 5; i++)
	    sleep (5);
	switch (child_id) {
	    case NOTOK: 
		adios ("fork", "unable to");

	    case OK: 
		dup2 (fileno (out), 1);

		i = 1;
		vec[i++] = "-forward";
		vec[i++] = "-form";
		vec[i++] = filter;
		vec[i++] = tmpfil;
		vec[i] = NULL;

		execvp (mhlproc, vec);
		fprintf (stderr, "unable to exec ");
		perror (mhlproc);
		_exit (-1);

	    default: 
		pidXwait (child_id, mhlproc);
		break;
	}
    } else {
	if ((fd = open (tmpfil, O_RDONLY)) == NOTOK)
	    adios (tmpfil, "unable to re-open");
	if (mime)
	    cpydata (fd, fileno (out), tmpfil, bccfil);
	else
	    cpydgst (fd, fileno (out), tmpfil, bccfil);
	close (fd);
    }

    fseek (out, 0L, SEEK_END);
    if (mime)
	fprintf (out, "\n--%s--\n", prefix);
    else
	fprintf (out, "\n------- End of Blind-Carbon-Copy\n");
    fclose (out);
}


static int
find_prefix (void)
{
    int	len, result;
    char buffer[BUFSIZ];
    FILE *in;

    if ((in = fopen (tmpfil, "r")) == NULL)
	adios (tmpfil, "unable to re-open");

    len = strlen (prefix);

    result = OK;
    while (fgets (buffer, sizeof(buffer) - 1, in))
	if (buffer[0] == '-' && buffer[1] == '-') {
	    char *cp;

	    for (cp = buffer + strlen (buffer) - 1; cp >= buffer; cp--)
		if (!isspace (*cp))
		    break;
	    *++cp = '\0';
	    if (strcmp (buffer + 2, prefix) == 0) {
		result = NOTOK;
		break;
	    }
	}

    fclose (in);
	
    return result;
}


#define	plural(x) (x == 1 ? "" : "s")

static void
chkadr (void)
{
    if (badadr && unkadr)
	die (NULL, "%d address%s unparsable, %d addressee%s undeliverable",
		badadr, plural (badadr), unkadr, plural (badadr));
    if (badadr)
	die (NULL, "%d address%s unparsable", badadr, plural (badadr));
    if (unkadr)
	die (NULL, "%d addressee%s undeliverable", unkadr, plural (unkadr));
}


static void
do_addresses (int bccque, int talk)
{
    int retval;
    int	state;
    struct mailname *lp;

    state = 0;
    for (lp = localaddrs.m_next; lp; lp = lp->m_next)
	if (lp->m_bcc ? bccque : !bccque) {
	    if (talk && !state)
		printf ("  -- Local Recipients --\n");
	    do_an_address (lp, talk);
	    state++;
	}

    state = 0;
    for (lp = uuaddrs.m_next; lp; lp = lp->m_next)
	if (lp->m_bcc ? bccque : !bccque) {
	    if (talk && !state)
		printf ("  -- UUCP Recipients --\n");
	    do_an_address (lp, talk);
	    state++;
	}

    state = 0;
    for (lp = netaddrs.m_next; lp; lp = lp->m_next)
	if (lp->m_bcc ? bccque : !bccque) {
	    if (talk && !state)
		printf ("  -- Network Recipients --\n");
	    do_an_address (lp, talk);
	    state++;
	}

    chkadr ();

#ifdef MMDFMTS
    if (rp_isbad (retval = mm_waend ()))
	die (NULL, "problem ending addresses [%s]\n", rp_valstr (retval));
#endif /* MMDFMTS */

#ifdef SENDMTS
    if (rp_isbad (retval = sm_waend ()))
	die (NULL, "problem ending addresses; %s", rp_string (retval));
#endif /* SENDMTS */
}


/*
 * MTS-SPECIFIC INTERACTION
 */


/*
 * SENDMAIL/SMTP routines
 */

#ifdef SENDMTS

static void
post (char *file, int bccque, int talk)
{
    int fd, onex;
    int	retval;

    onex = !(msgflags & MINV) || bccque;
    if (verbose) {
	if (msgflags & MINV)
	    printf (" -- Posting for %s Recipients --\n",
		    bccque ? "Blind" : "Sighted");
	else
	    printf (" -- Posting for All Recipients --\n");
    }

    sigon ();

    if (rp_isbad (retval = sm_init (clientsw, serversw, watch, verbose,
				    snoop, onex, queued))
	    || rp_isbad (retval = sm_winit (smtpmode, from)))
	die (NULL, "problem initializing server; %s", rp_string (retval));

    do_addresses (bccque, talk && verbose);
    if ((fd = open (file, O_RDONLY)) == NOTOK)
	die (file, "unable to re-open");
    do_text (file, fd);
    close (fd);
    fflush (stdout);

    sm_end (onex ? OK : DONE);
    sigoff ();

    if (verbose) {
	if (msgflags & MINV)
	    printf (" -- %s Recipient Copies Posted --\n",
		    bccque ? "Blind" : "Sighted");
	else
	    printf (" -- Recipient Copies Posted --\n");
    }

    fflush (stdout);
}


/* Address Verification */

static void
verify_all_addresses (int talk)
{
    int retval;
    struct mailname *lp;

    sigon ();

    if (!whomsw || checksw)
	if (rp_isbad (retval = sm_init (clientsw, serversw, 0, 0, snoop, 0, 0))
		|| rp_isbad (retval = sm_winit (smtpmode, from)))
	    die (NULL, "problem initializing server; %s", rp_string (retval));

    if (talk && !whomsw)
	printf (" -- Address Verification --\n");
    if (talk && localaddrs.m_next)
	printf ("  -- Local Recipients --\n");
    for (lp = localaddrs.m_next; lp; lp = lp->m_next)
	do_an_address (lp, talk);

    if (talk && uuaddrs.m_next)
	printf ("  -- UUCP Recipients --\n");
    for (lp = uuaddrs.m_next; lp; lp = lp->m_next)
	do_an_address (lp, talk);

    if (talk && netaddrs.m_next)
	printf ("  -- Network Recipients --\n");
    for (lp = netaddrs.m_next; lp; lp = lp->m_next)
	do_an_address (lp, talk);

    chkadr ();
    if (talk && !whomsw)
	printf (" -- Address Verification Successful --\n");

    if (!whomsw || checksw)
	sm_end (DONE);

    fflush (stdout);
    sigoff ();
}


static void
do_an_address (struct mailname *lp, int talk)
{
    int retval;
    char *mbox, *host;
    char addr[BUFSIZ];

    switch (lp->m_type) {
	case LOCALHOST: 
	    mbox = lp->m_mbox;
	    host = lp->m_host;
	    strcpy (addr, mbox);
	    break;

	case UUCPHOST: 
	    mbox = auxformat (lp, 0);
	    host = NULL;
	    sprintf (addr, "%s!%s", lp->m_host, lp->m_mbox);
	    break;

	default: 		/* let SendMail decide if the host is bad  */
	    mbox = lp->m_mbox;
	    host = lp->m_host;
	    sprintf (addr, "%s at %s", mbox, host);
	    break;
    }

    if (talk)
	printf ("  %s%s", addr, whomsw && lp->m_bcc ? "[BCC]" : "");

    if (whomsw && !checksw) {
	putchar ('\n');
	return;
    }
    if (talk)
	printf (": ");
    fflush (stdout);

    switch (retval = sm_wadr (mbox, host,
			 lp->m_type != UUCPHOST ? lp->m_path : NULL)) {
	case RP_OK: 
	    if (talk)
		printf ("address ok\n");
	    break;

	case RP_NO: 
	case RP_USER: 
	    if (!talk)
		fprintf (stderr, "  %s: ", addr);
	    fprintf (talk ? stdout : stderr, "loses; %s\n",
			rp_string (retval));
	    unkadr++;
	    break;

	default: 
	    if (!talk)
		fprintf (stderr, "  %s: ", addr);
	    die (NULL, "unexpected response; %s", rp_string (retval));
    }

    fflush (stdout);
}


static void
do_text (char *file, int fd)
{
    int retval, state;
    char buf[BUFSIZ];

    lseek (fd, (off_t) 0, SEEK_SET);

    while ((state = read (fd, buf, sizeof(buf))) > 0) {
	if (rp_isbad (retval = sm_wtxt (buf, state)))
	    die (NULL, "problem writing text; %s\n", rp_string (retval));
    }

    if (state == NOTOK)
	die (file, "problem reading from");

    switch (retval = sm_wtend ()) {
	case RP_OK: 
	    break;

	case RP_NO: 
	case RP_NDEL: 
	    die (NULL, "posting failed; %s", rp_string (retval));

	default: 
	    die (NULL, "unexpected response; %s", rp_string (retval));
    }
}

#endif /* SENDMTS */

/*
 * MMDF routines
 */

#ifdef MMDFMTS

static void
post (char *file, int bccque, int talk)
{
    int fd, onex;
    int	retval;
#ifdef RP_NS
    int	len;
    struct rp_bufstruct reply;
#endif /* RP_NS */

    onex = !(msgflags & MINV) || bccque;
    if (verbose) {
	if (msgflags & MINV)
	    printf (" -- Posting for %s Recipients --\n",
		    bccque ? "Blind" : "Sighted");
	else
	    printf (" -- Posting for All Recipients --\n");
    }

    sigon ();

    if (rp_isbad (retval = mm_init ())
	    || rp_isbad (retval = mm_sbinit ())
	    || rp_isbad (retval = mm_winit (NULL, submitopts, from)))
	die (NULL, "problem initializing MMDF system [%s]",
		rp_valstr (retval));
#ifdef RP_NS
	if (rp_isbad (retval = mm_rrply (&reply, &len)))
	    die (NULL, "problem with sender address [%s]",
		    rp_valstr (retval));
#endif /* RP_NS */

    do_addresses (bccque, talk && verbose);
    if ((fd = open (file, O_RDONLY)) == NOTOK)
	die (file, "unable to re-open");
    do_text (file, fd);
    close (fd);
    fflush (stdout);

    mm_sbend ();
    mm_end (OK);
    sigoff ();

    if (verbose)
	if (msgflags & MINV)
	    printf (" -- %s Recipient Copies Posted --\n",
		    bccque ? "Blind" : "Sighted");
	else
	    printf (" -- Recipient Copies Posted --\n");
    fflush (stdout);
}


/* Address Verification */

static void
verify_all_addresses (int talk)
{
    int retval;
    struct mailname *lp;

#ifdef RP_NS
    int	len;
    struct rp_bufstruct reply;
#endif /* RP_NS */

    sigon ();

    if (!whomsw || checksw) {
	if (rp_isbad (retval = mm_init ())
		|| rp_isbad (retval = mm_sbinit ())
		|| rp_isbad (retval = mm_winit (NULL, submitopts, from)))
	    die (NULL, "problem initializing MMDF system [%s]",
		    rp_valstr (retval));
#ifdef RP_NS
	if (rp_isbad (retval = mm_rrply (&reply, &len)))
	    die (NULL, "problem with sender address [%s]", rp_valstr (retval));
#endif /* RP_NS */
    }

    if (talk && !whomsw)
	printf (" -- Address Verification --\n");
    if (talk && localaddrs.m_next)
	printf ("  -- Local Recipients --\n");
    for (lp = localaddrs.m_next; lp; lp = lp->m_next)
	do_an_address (lp, talk);

    if (talk && uuaddrs.m_next)
	printf ("  -- UUCP Recipients --\n");
    for (lp = uuaddrs.m_next; lp; lp = lp->m_next)
	do_an_address (lp, talk);

    if (talk && netaddrs.m_next)
	printf ("  -- Network Recipients --\n");
    for (lp = netaddrs.m_next; lp; lp = lp->m_next)
	do_an_address (lp, talk);

    chkadr ();
    if (talk && !whomsw)
	printf (" -- Address Verification Successful --\n");

    if (!whomsw || checksw)
	mm_end (NOTOK);

    fflush (stdout);
    sigoff ();
}


static void
do_an_address (struct mailname *lp, int talk)
{
    int len, retval;
    char *mbox, *host, *text, *path;
    char addr[BUFSIZ];
    struct rp_bufstruct reply;

    switch (lp->m_type) {
	case LOCALHOST: 
	    mbox = lp->m_mbox;
	    host = LocalName ();
	    strcpy (addr, mbox);
	    break;

	case UUCPHOST: 
	    fprintf (talk ? stdout : stderr, "  %s!%s: %s\n",
		lp->m_host, lp->m_mbox, "not supported; UUCP address");
	    unkadr++;
	    fflush (stdout);
	    return;

	default: 		/* let MMDF decide if the host is bad */
	    mbox = lp->m_mbox;
	    host = lp->m_host;
	    sprintf (addr, "%s at %s", mbox, host);
	    break;
    }

    if (talk)
	printf ("  %s%s", addr, whomsw && lp->m_bcc ? "[BCC]" : "");

    if (whomsw && !checksw) {
	putchar ('\n');
	return;
    }
    if (talk)
	printf (": ");
    fflush (stdout);

#ifdef MMDFII
    if (lp->m_path)
	path = concat (lp->m_path, mbox, "@", host, NULL);
    else
#endif /* MMDFII */
	path = NULL;
    if (rp_isbad (retval = mm_wadr (path ? NULL : host, path ? path : mbox))
	    || rp_isbad (retval = mm_rrply (&reply, &len)))
	die (NULL, "problem submitting address [%s]", rp_valstr (retval));

    switch (rp_gval (reply.rp_val)) {
	case RP_AOK: 
	    if (talk)
		printf ("address ok\n");
	    fflush (stdout);
	    return;

#ifdef RP_DOK
	case RP_DOK: 
	    if (talk)
		printf ("nameserver timeout - queued for checking\n");
	    fflush (stdout);
	    return;
#endif /* RP_DOK */

	case RP_NO: 
	    text = "you lose";
	    break;

#ifdef RP_NS
	case RP_NS: 
	    text = "temporary nameserver failure";
	    break;

#endif /* RP_NS */

	case RP_USER: 
	case RP_NDEL: 
	    text = "not deliverable";
	    break;

	case RP_AGN: 
	    text = "try again later";
	    break;

	case RP_NOOP: 
	    text = "nothing done";
	    break;

	default: 
	    if (!talk)
		fprintf (stderr, "  %s: ", addr);
	    text = "unexpected response";
	    die (NULL, "%s;\n    [%s] -- %s", text,
		    rp_valstr (reply.rp_val), reply.rp_line);
    }

    if (!talk)
	fprintf (stderr, "  %s: ", addr);
    fprintf (talk ? stdout : stderr, "%s;\n    %s\n", text, reply.rp_line);
    unkadr++;

    fflush (stdout);
}


static void
do_text (char *file, int fd)
{
    int retval, state;
    char buf[BUFSIZ];
    struct rp_bufstruct reply;

    lseek (fd, (off_t) 0, SEEK_SET);

    while ((state = read (fd, buf, sizeof(buf))) > 0) {
	if (rp_isbad (mm_wtxt (buf, state)))
	    die (NULL, "problem writing text [%s]\n", rp_valstr (retval));
    }

    if (state == NOTOK)
	die (file, "problem reading from");

    if (rp_isbad (retval = mm_wtend ()))
	die (NULL, "problem ending text [%s]\n", rp_valstr (retval));

    if (rp_isbad (retval = mm_rrply (&reply, &state)))
	die (NULL, "problem getting submission status [%s]\n",
		rp_valstr (retval));

    switch (rp_gval (reply.rp_val)) {
	case RP_OK: 
	case RP_MOK: 
	    break;

	case RP_NO: 
	    die (NULL, "you lose; %s", reply.rp_line);

	case RP_NDEL: 
	    die (NULL, "no delivery occurred; %s", reply.rp_line);

	case RP_AGN: 
	    die (NULL, "try again later; %s", reply.rp_line);

	case RP_NOOP: 
	    die (NULL, "nothing done; %s", reply.rp_line);

	default: 
	    die (NULL, "unexpected response;\n\t[%s] -- %s",
		    rp_valstr (reply.rp_val), reply.rp_line);
    }
}

#endif /* MMDFMTS */


/*
 * SIGNAL HANDLING
 */

static RETSIGTYPE
sigser (int i)
{
#ifndef RELIABLE_SIGNALS
    SIGNAL (i, SIG_IGN);
#endif

    unlink (tmpfil);
    if (msgflags & MINV)
	unlink (bccfil);

#ifdef MMDFMTS
    if (!whomsw || checksw)
	mm_end (NOTOK);
#endif /* MMDFMTS */

#ifdef SENDMTS
    if (!whomsw || checksw)
	sm_end (NOTOK);
#endif /* SENDMTS */

    done (1);
}


static void
sigon (void)
{
    if (debug)
	return;

    hstat = SIGNAL2 (SIGHUP, sigser);
    istat = SIGNAL2 (SIGINT, sigser);
    qstat = SIGNAL2 (SIGQUIT, sigser);
    tstat = SIGNAL2 (SIGTERM, sigser);
}


static void
sigoff (void)
{
    if (debug)
	return;

    SIGNAL (SIGHUP, hstat);
    SIGNAL (SIGINT, istat);
    SIGNAL (SIGQUIT, qstat);
    SIGNAL (SIGTERM, tstat);
}

/*
 * FCC INTERACTION
 */

static void
p_refile (char *file)
{
    int i;

    if (fccind == 0)
	return;

    if (verbose)
	printf (" -- Filing Folder Copies --\n");
    for (i = 0; i < fccind; i++)
	fcc (file, fccfold[i]);
    if (verbose)
	printf (" -- Folder Copies Filed --\n");
}


/*
 * Call the `fileproc' to add the file to the folder.
 */

static void
fcc (char *file, char *folder)
{
    pid_t child_id;
    int i, status;
    char fold[BUFSIZ];

    if (verbose)
	printf ("  %sFcc %s: ", msgstate == RESENT ? "Resent-" : "", folder);
    fflush (stdout);

    for (i = 0; (child_id = fork ()) == NOTOK && i < 5; i++)
	sleep (5);

    switch (child_id) {
	case NOTOK: 
	    if (!verbose)
		fprintf (stderr, "  %sFcc %s: ",
			msgstate == RESENT ? "Resent-" : "", folder);
	    fprintf (verbose ? stdout : stderr, "no forks, so not ok\n");
	    break;

	case OK: 
	    /* see if we need to add `+' */
	    sprintf (fold, "%s%s",
		    *folder == '+' || *folder == '@' ? "" : "+", folder);

	    /* now exec the fileproc */
	    execlp (fileproc, r1bindex (fileproc, '/'),
		    "-link", "-file", file, fold, NULL);
	    _exit (-1);

	default: 
	    if ((status = pidwait (child_id, OK))) {
		if (!verbose)
		    fprintf (stderr, "  %sFcc %s: ",
			    msgstate == RESENT ? "Resent-" : "", folder);
		pidstatus (status, verbose ? stdout : stderr, NULL);
	    } else {
		if (verbose)
		    printf ("folder ok\n");
	    }
    }

    fflush (stdout);
}

/*
 * TERMINATION
 */

static void
die (char *what, char *fmt, ...)
{
    va_list ap;

    unlink (tmpfil);
    if (msgflags & MINV)
	unlink (bccfil);

#ifdef MMDFMTS
    if (!whomsw || checksw)
	mm_end (NOTOK);
#endif /* MMDFMTS */

#ifdef SENDMTS
    if (!whomsw || checksw)
	sm_end (NOTOK);
#endif /* SENDMTS */

    va_start(ap, fmt);
    advertise (what, NULL, fmt, ap);
    va_end(ap);
    done (1);
}


#ifdef MMDFMTS
/* 
 * err_abrt() is used by the mm_ routines
 *    	 do not, under *ANY* circumstances, remove it from post,
 *	 or you will lose *BIG*
 */

void
err_abrt (int code, char *fmt, ...)
{
    char buffer[BUFSIZ];
    va_list ap;

    sprintf (buffer, "[%s]", rp_valstr (code));

    va_start(ap, fmt);
    advertise (buffer, NULL, fmt, ap);
    va_end(ap);

    done (1);
}
#endif /* MMDFMTS */
