
/*
 * whatnowsbr.c -- the WhatNow shell
 *
 * $Id$
 */

#include <h/mh.h>
#include <fcntl.h>
#include <signal.h>

#ifdef MIME
#include <h/mhn.h>
#endif

#ifndef MIME
# define MIMEminc(a) (a)
#else
# define MIMEminc(a)  0
#endif

static struct swit whatnowswitches[] = {
#define	DFOLDSW                 0
    { "draftfolder +folder", 0 },
#define	DMSGSW                  1
    { "draftmessage msg", 0 },
#define	NDFLDSW                 2
    { "nodraftfolder", 0 },
#define	EDITRSW                 3
    { "editor editor", 0 },
#define	NEDITSW                 4
    { "noedit", 0 },
#define	PRMPTSW                 5
    { "prompt string", 4 },
#define VERSIONSW               6
    { "version", 0 },
#define	HELPSW                  7
    { "help", 4 },
    { NULL, 0 }
};

/*
 * Options at the "whatnow" prompt
 */
static struct swit aleqs[] = {
#define	DISPSW                         0
    { "display [<switches>]", 0 },
#define BUILDMIMESW                    1
    { "mime [<switches>]", 0 },
#define	EDITSW                         2
    { "edit [<editor> <switches>]", 0 },
#define	LISTSW                         3
    { "list [<switches>]", 0 },
#define	PUSHSW                         4
    { "push [<switches>]", 0 },
#define	QUITSW                         5
    { "quit [-delete]", 0 },
#define	REFILEOPT                      6
    { "refile [<switches>] +folder", 0 },
#define	SENDSW                         7
    { "send [<switches>]", 0 },
#define	WHOMSW                         8
    { "whom [<switches>]", 0 },
    { NULL, 0 }
};

static char *myprompt = "\nWhat now? ";

/*
 * static prototypes
 */
static int editfile (char **, char **, char *, int, struct msgs *, char *, char *);
static int sendfile (char **, char *, int);
static void sendit (char *, char **, char *, int);
static int build_mime (char **, char *);
static int whomfile (char **, char *);

#ifdef HAVE_LSTAT
static int copyf (char *, char *);
#endif

#ifdef MIME
static int mhnfile (char *);
#endif


int
WhatNow (int argc, char **argv)
{
    int isdf = 0, nedit = 0, use = 0;
    char *cp, *dfolder = NULL, *dmsg = NULL;
    char *ed = NULL, *drft = NULL, *msgnam = NULL;
    char buf[100], prompt[BUFSIZ];
    char **ap, **argp, *arguments[MAXARGS];
    struct stat st;

    invo_name = r1bindex (argv[0], '/');
    if ((cp = m_find (invo_name)) != NULL) {
	ap = brkstring (cp = getcpy (cp), " ", "\n");
	ap = copyip (ap, arguments);
    } else {
	ap = arguments;
    }
    copyip (argv + 1, ap);
    argp = arguments;


    while ((cp = *argp++)) {
	if (*cp == '-')
	    switch (smatch (++cp, whatnowswitches)) {
	    case AMBIGSW: 
		ambigsw (cp, whatnowswitches);
		done (1);
	    case UNKWNSW: 
		adios (NULL, "-%s unknown", cp);

	    case HELPSW: 
		sprintf (buf, "%s [switches] [file]", invo_name);
		print_help (buf, whatnowswitches);
		done (1);
	    case VERSIONSW:
		print_version(invo_name);
		done (1);

	    case DFOLDSW: 
		if (dfolder)
		    adios (NULL, "only one draft folder at a time!");
		if (!(cp = *argp++) || *cp == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		dfolder = path (*cp == '+' || *cp == '@' ? cp + 1 : cp,
				*cp != '@' ? TFOLDER : TSUBCWF);
		continue;
	    case DMSGSW: 
		if (dmsg)
		    adios (NULL, "only one draft message at a time!");
		if (!(dmsg = *argp++) || *dmsg == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    case NDFLDSW: 
		dfolder = NULL;
		isdf = NOTOK;
		continue;

	    case EDITRSW: 
		if (!(ed = *argp++) || *ed == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		nedit = 0;
		continue;
	    case NEDITSW: 
		nedit++;
		continue;

	    case PRMPTSW:
		if (!(myprompt = *argp++) || *myprompt == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    }
	if (drft)
	    adios (NULL, "only one draft at a time!");
	else
	    drft = cp;
    }

    if ((drft == NULL && (drft = getenv ("mhdraft")) == NULL) || *drft == 0)
	drft = getcpy (m_draft (dfolder, dmsg, 1, &isdf));
    msgnam = (cp = getenv ("mhaltmsg")) && *cp ? getcpy (cp) : NULL;
    if (ed == NULL && ((ed = getenv ("mheditor")) == NULL || *ed == 0))
	ed = NULL, nedit++;
    if ((cp = getenv ("mhuse")) && *cp)
	use = atoi (cp);
    if (!nedit && editfile (&ed, NULL, drft, use, NULL, msgnam, NULL) < 0)
	done (1);

    sprintf (prompt, myprompt, invo_name);
    for (;;) {
	if (!(argp = getans (prompt, aleqs))) {
	    unlink (LINK);
	    done (1);
	}
	switch (smatch (*argp, aleqs)) {
	case DISPSW:
	    /* display the message being replied to, or distributed */
	    if (msgnam)
		showfile (++argp, msgnam);
	    else
		advise (NULL, "no alternate message to display");
	    break;

	case BUILDMIMESW:
	    /* Translate MIME composition file */
	    ++argp;
	    if (build_mime(argp, drft) == NOTOK)
		done (1);
	    break;

	case EDITSW:
	    /* Call an editor on the draft file */
	    if (*++argp)
		ed = *argp++;
	    if (editfile (&ed, argp, drft, NOUSE, NULL, msgnam, NULL) == NOTOK)
		done (1);
	    break;

	case LISTSW: 
	    /* display the draft file */
	    showfile (++argp, drft);
	    break;

	case WHOMSW:
	    /* Check to whom the draft would be sent */
	    whomfile (++argp, drft);
	    break;

	case QUITSW:
	    if (*++argp && (*argp[0] == 'd' ||
			    ((*argp)[0] == '-' && (*argp)[1] == 'd'))) {
		if (unlink (drft) == NOTOK)
		    adios (drft, "unable to unlink");
	    } else
		if (stat (drft, &st) != NOTOK)
		    advise (NULL, "draft left on %s", drft);
	    done (1);

	case PUSHSW:
	    /* Send draft in background */
	    if (sendfile (++argp, drft, 1))
		done (1);
	    break;

	case SENDSW: 
	    /* Send draft */
	    sendfile (++argp, drft, 0);
	    break;

	case REFILEOPT: 
	    /* Refile the draft */
	    if (refile (++argp, drft) == 0)
		done (0);
	    break;

	default: 
	    advise (NULL, "say what?");
	    break;
	}
    }
    /*NOTREACHED*/
}

/*
 * EDIT
 */

static int  reedit = 0;
static char *edsave = NULL;


static int
editfile (char **ed, char **arg, char *file, int use,
          struct msgs *mp, char *altmsg, char *cwd)
{
    int pid, status;
    int vecp;
    char *cp;
    char altpath[BUFSIZ], linkpath[BUFSIZ];
    char *vec[MAXARGS];
    struct stat st;

#ifdef HAVE_LSTAT
    int	slinked;
#if 0
    int oumask;	/* PJS: for setting permissions on symlinks. */
#endif
#endif /* HAVE_LSTAT */

    /* set initial editor */
    if (!reedit) {
	if (*ed == NULL && (*ed = m_find ("editor")) == NULL)
	    *ed = sysed;
    } else {
	if (!*ed) {		/* no explicit editor */
	    *ed = edsave;
	    if ((cp = r1bindex (*ed, '/')) == NULL)
		cp = *ed;
	    cp = concat (cp, "-next", NULL);
	    if ((cp = m_find (cp)) != NULL)
		*ed = cp;
	}
    }

    if (altmsg) {
	if (mp == NULL || *altmsg == '/' || cwd == NULL)
	    strcpy (altpath, altmsg);
	else
	    sprintf (altpath, "%s/%s", mp->foldpath, altmsg);
	if (cwd == NULL)
	    strcpy (linkpath, LINK);
	else
	    sprintf (linkpath, "%s/%s", cwd, LINK);
    }

    if (altmsg) {
	unlink (linkpath);
#ifdef HAVE_LSTAT
	if (link (altpath, linkpath) == NOTOK) {
#if 0
	    /* I don't think permission on symlinks matters /JLR */
	    oumask = umask(0044);	/* PJS: else symlinks are world 'r' */
#endif
	    symlink (altpath, linkpath);
#if 0
	    umask(oumask);		/* PJS: else symlinks are world 'r' */
#endif
	    slinked = 1;
	} else {
	    slinked = 0;
	}
#else /* not HAVE_LSTAT */
	link (altpath, linkpath);
#endif /* not HAVE_LSTAT */
    }

    m_update ();	/* update the context file */
    fflush (stdout);

    switch (pid = vfork ()) {
	case NOTOK: 
	    advise ("fork", "unable to");
	    status = NOTOK;
	    break;

	case OK: 
	    if (cwd)
		chdir (cwd);
	    if (altmsg) {
		if (mp)
		    m_putenv ("mhfolder", mp->foldpath);
		m_putenv ("editalt", altpath);
	    }

	    vecp = 0;
	    vec[vecp++] = r1bindex (*ed, '/');
	    if (arg)
		while (*arg)
		    vec[vecp++] = *arg++;
	    vec[vecp++] = file;
	    vec[vecp] = NULL;

	    execvp (*ed, vec);
	    fprintf (stderr, "unable to exec ");
	    perror (*ed);
	    _exit (-1);

	default: 
	    if ((status = pidwait (pid, NOTOK))) {
#ifdef ATTVIBUG
		if ((cp = r1bindex (*ed, '/'))
			&& strcmp (cp, "vi") == 0
			&& (status & 0x00ff) == 0)
		    status = 0;
		else {
#endif
		if (((status & 0xff00) != 0xff00)
			&& (!reedit || (status & 0x00ff)))
		    if (!use && (status & 0xff00) &&
			    (rename (file, cp = m_backup (file)) != NOTOK)) {
			advise (NULL, "problems with edit--draft left in %s",
				cp);
		    }
		    else
			advise (NULL, "problems with edit--%s preserved",
				file);
		status = -2;	/* maybe "reedit ? -2 : -1"? */
		break;
#ifdef ATTVIBUG
		}
#endif
	    }

	    reedit++;
#ifdef HAVE_LSTAT
	    if (altmsg
		    && mp
		    && (!mp->msgflags & READONLY)
		    && (slinked
		           ? lstat (linkpath, &st) != NOTOK
				&& S_ISREG(st.st_mode)
				&& copyf (linkpath, altpath) == NOTOK
		           : stat (linkpath, &st) != NOTOK
				&& st.st_nlink == 1
				&& (unlink (altpath) == NOTOK
					|| link (linkpath, altpath) == NOTOK)))
		advise (linkpath, "unable to update %s from", altmsg);
#else /* HAVE_LSTAT */
	    if (altmsg
		    && mp
		    && (!mp->msgflags & READONLY)
		    && stat (linkpath, &st) != NOTOK
		    && st.st_nlink == 1
		    && (unlink (altpath) == NOTOK
			|| link (linkpath, altpath) == NOTOK))
		advise (linkpath, "unable to update %s from", altmsg);
#endif /* HAVE_LSTAT */
    }

    edsave = getcpy (*ed);
    *ed = NULL;
    if (altmsg)
	unlink (linkpath);

    return status;
}


#ifdef HAVE_LSTAT
static int
copyf (char *ifile, char *ofile)
{
    int i, in, out;
    char buffer[BUFSIZ];

    if ((in = open (ifile, O_RDONLY)) == NOTOK)
	return NOTOK;
    if ((out = open (ofile, O_WRONLY | O_TRUNC)) == NOTOK) {
	admonish (ofile, "unable to open and truncate");
	close (in);
	return NOTOK;
    }

    while ((i = read (in, buffer, sizeof(buffer))) > OK)
	if (write (out, buffer, i) != i) {
	    advise (ofile, "may have damaged");
	    i = NOTOK;
	    break;
	}

    close (in);
    close (out);
    return i;
}
#endif /* HAVE_LSTAT */


/*
 * SEND
 */

static int
sendfile (char **arg, char *file, int pushsw)
{
    pid_t child_id;
    int i, vecp;
    char *cp, *sp, *vec[MAXARGS];

#ifdef MIME
    /* Translate MIME composition file, if necessary */
    if ((cp = m_find ("automimeproc"))
	    && (uleq (cp, "1"))
	    && !getenv ("NOMHNPROC")
	    && mhnfile (file)
	    && (build_mime (NULL, file) == NOTOK))
	return 0;

    /* For backwards compatibility */
    if ((cp = m_find ("automhnproc"))
	    && !getenv ("NOMHNPROC")
	    && mhnfile (file)
	    && (i = editfile (&cp, NULL, file, NOUSE, NULL, NULL, NULL)))
	return 0;
#endif

    /*
     * If the sendproc is the nmh command `send', then we call
     * those routines directly rather than exec'ing the command.
     */
    if (strcmp (sp = r1bindex (sendproc, '/'), "send") == 0) {
	cp = invo_name;
	sendit (invo_name = sp, arg, file, pushsw);
	invo_name = cp;
	return 1;
    }

    m_update ();	/* update the context file */
    fflush (stdout);

    for (i = 0; (child_id = vfork()) == NOTOK && i < 5; i++)
	sleep (5);
    switch (child_id) {
	case NOTOK: 
	    advise (NULL, "unable to fork, so sending directly...");
	case OK: 
	    vecp = 0;
	    vec[vecp++] = invo_name;
	    if (pushsw)
		vec[vecp++] = "-push";
	    if (arg)
		while (*arg)
		    vec[vecp++] = *arg++;
	    vec[vecp++] = file;
	    vec[vecp] = NULL;

	    execvp (sendproc, vec);
	    fprintf (stderr, "unable to exec ");
	    perror (sendproc);
	    _exit (-1);

	default: 
	    if (pidwait(child_id, OK) == 0)
		done (0);
	    return 1;
    }
}


/*
 * Translate MIME composition file
 */

static int
build_mime (char **argp, char *file)
{
    int i;
    char **args, *ed;

    ed = buildmimeproc;

    /* allocate space for arguments */
    i = 0;
    if (argp) {
	while (argp[i])
	    i++;
    }
    if ((args = (char **) malloc((i + 2) * sizeof(char *))) == NULL)
	adios (NULL, "unable to malloc memory");

    /* need to add -build if using mhn as buildmimeproc */
    i = 0;
    if (strcmp (r1bindex (ed, '/'), "mhn") == 0)
	args[i++] = "-build";

    /* copy any other arguments */
    while (argp && *argp)
	args[i++] = *argp++;
    args[i] = NULL;

    i = editfile (&ed, args, file, NOUSE, NULL, NULL, NULL);
    free (args);

    if (i)
	return NOTOK;
    else
	return OK;
}

#ifdef MIME

/*
 *  Check if draft is a mhn composition file
 */

static int
mhnfile (char *msgnam)
{
    int	state;
    char buf[BUFSIZ], name[NAMESZ];
    FILE *fp;

    if ((fp = fopen (msgnam, "r")) == NULL)
	return 0;
    for (state = FLD;;)
	switch (state = m_getfld (state, name, buf, sizeof(buf), fp)) {
	    case FLD:
	    case FLDPLUS:
	    case FLDEOF:
		/*
		 * If draft already contains any of the
		 * Content-XXX fields, then assume it already
		 * been converted.
		 */
	        if (uprf (name, XXX_FIELD_PRF)) {
		    fclose (fp);
		    return 0;
		}
		while (state == FLDPLUS)
		    state = m_getfld (state, name, buf, sizeof(buf), fp);
		break;

	    case BODY:
	        do {
		    char *bp;

		    for (bp = buf; *bp; bp++)
			if (*bp != ' ' && *bp != '\t' && *bp != '\n') {
			    fclose (fp);
			    return 1;
			}

		    state = m_getfld (state, name, buf, sizeof(buf), fp);
		} while (state == BODY);
		/* and fall... */

	    default:
		fclose (fp);
		return 0;
	}
}

#endif /* MIME */


static struct swit  sendswitches[] = {
#define	ALIASW            0
    { "alias aliasfile", 0 },
#define	DEBUGSW           1
    { "debug", -5 },

#define	ENCRSW            2
#ifndef	TMA
    { "encrypt", -7 },
#else
    { "encrypt", 0 },
#endif

#define	NENCRSW           3
#ifndef	TMA
    { "noencrypt", -9 },
#else
    { "noencrypt", 0 },
#endif

#define	FILTSW            4
    { "filter filterfile", 0 },
#define	NFILTSW           5
    { "nofilter", 0 },
#define	FRMTSW            6
    { "format", 0 },
#define	NFRMTSW           7
    { "noformat", 0 },
#define	FORWSW            8
    { "forward", 0 },
#define	NFORWSW           9
    { "noforward", 0 },
#define	MIMESW           10
    { "mime", MIMEminc(-4) },
#define	NMIMESW          11
    { "nomime", MIMEminc(-6) },
#define	MSGDSW           12
    { "msgid", 0 },
#define	NMSGDSW          13
    { "nomsgid", 0 },
#define	SPSHSW           14
    { "push", 0 },
#define	NSPSHSW          15
    { "nopush", 0 },
#define	SPLITSW          16
    { "split seconds", MIMEminc(-5) },
#define	UNIQSW           17
    { "unique", -6 },
#define	NUNIQSW          18
    { "nounique", -8 },
#define	VERBSW           19
    { "verbose", 0 },
#define	NVERBSW          20
    { "noverbose", 0 },
#define	WATCSW           21
    { "watch", 0 },
#define	NWATCSW          22
    { "nowatch", 0 },
#define	WIDTHSW          23
    { "width columns", 0 },
#define SVERSIONSW       24
    { "version", 0 },
#define	SHELPSW          25
    { "help", 4 },
#define	MAILSW           26
    { "mail", -4 },
#define	SAMLSW           27
    { "saml", -4 },
#define	SSNDSW           28
    { "send", -4 },
#define	SOMLSW           29
    { "soml", -4 },
#define	CLIESW           30
    { "client host", -6 },
#define	SERVSW           31
    { "server host", -6 },
#define	SNOOPSW          32
    { "snoop", -5 },
#define SDRFSW           33
    { "draftfolder +folder", -6 },
#define SDRMSW           34
    { "draftmessage msg", -6 },
#define SNDRFSW          35
    { "nodraftfolder", -3 },
    { NULL, 0 }
};


extern int debugsw;		/* from sendsbr.c */
extern int forwsw;
extern int inplace;
extern int pushsw;
extern int splitsw;
extern int unique;
extern int verbsw;

extern char *altmsg;		/*  .. */
extern char *annotext;
extern char *distfile;


static void
sendit (char *sp, char **arg, char *file, int pushed)
{
    int	vecp = 1;
    char *cp, buf[100], **ap, **argp;
    char *arguments[MAXARGS], *vec[MAXARGS];
    struct stat st;

#ifndef	lint
    int	distsw = 0;
#endif
#ifdef UCI
    FILE *fp;
#endif

    if (arg)
	copyip (arg, vec);
    if ((cp = m_find (sp)) != NULL) {
	ap = brkstring (cp = getcpy (cp), " ", "\n");
	ap = copyip (ap, arguments);
    }
    else
	ap = arguments;
    if (arg)
	copyip (vec, ap);
    argp = arguments;

    debugsw = 0, forwsw = 1, inplace = 0, unique = 0;
    altmsg = annotext = distfile = NULL;
    vec[vecp++] = "-library";
    vec[vecp++] = getcpy (m_maildir (""));

    while ((cp = *argp++)) {
	if (*cp == '-')
	    switch (smatch (++cp, sendswitches)) {
		case AMBIGSW: 
		    ambigsw (cp, sendswitches);
		    return;
		case UNKWNSW: 
		    advise (NULL, "-%s unknown\n", cp);
		    return;

		case SHELPSW: 
		    sprintf (buf, "%s [switches]", sp);
		    print_help (buf, sendswitches);
		    return;
		case SVERSIONSW:
		    print_version (invo_name);
		    return;

		case SPSHSW: 
		    pushed++;
		    continue;
		case NSPSHSW: 
		    pushed = 0;
		    continue;

		case SPLITSW: 
		    if (!(cp = *argp++) || sscanf (cp, "%d", &splitsw) != 1) {
			advise (NULL, "missing argument to %s", argp[-2]);
			return;
		    }
		    continue;

		case UNIQSW: 
		    unique++;
		    continue;
		case NUNIQSW: 
		    unique = 0;
		    continue;
		case FORWSW: 
		    forwsw++;
		    continue;
		case NFORWSW: 
		    forwsw = 0;
		    continue;

		case VERBSW: 
		    verbsw++;
		    vec[vecp++] = --cp;
		    continue;
		case NVERBSW:
		    verbsw = 0;
		    vec[vecp++] = --cp;
		    continue;

		case DEBUGSW: 
		    debugsw++;	/* fall */
		case NFILTSW: 
		case FRMTSW: 
		case NFRMTSW: 
		case MIMESW: 
		case NMIMESW: 
		case MSGDSW: 
		case NMSGDSW: 
		case WATCSW: 
		case NWATCSW: 
		case MAILSW: 
		case SAMLSW: 
		case SSNDSW: 
		case SOMLSW: 
		case ENCRSW: 
		case NENCRSW: 
		case SNOOPSW: 
		    vec[vecp++] = --cp;
		    continue;

		case ALIASW: 
		case FILTSW: 
		case WIDTHSW: 
		case CLIESW: 
		case SERVSW: 
		    vec[vecp++] = --cp;
		    if (!(cp = *argp++) || *cp == '-') {
			advise (NULL, "missing argument to %s", argp[-2]);
			return;
		    }
		    vec[vecp++] = cp;
		    continue;

		case SDRFSW: 
		case SDRMSW: 
		    if (!(cp = *argp++) || *cp == '-') {
			advise (NULL, "missing argument to %s", argp[-2]);
			return;
		    }
		case SNDRFSW: 
		    continue;
	    }
	advise (NULL, "usage: %s [switches]", sp);
	return;
    }
    if ((cp = m_find ("Aliasfile"))) {	/* allow Aliasfile: profile entry */
	char *dp = NULL;

	for (ap = brkstring(dp = getcpy(cp), " ", "\n"); ap && *ap; ap++) {
	    vec[vecp++] = "-alias";
	    vec[vecp++] = *ap;
	}
    }

#ifdef TMA
    if ((cp = getenv ("KDS")) == NULL || *cp == NULL)
	if ((cp = m_find ("kdsproc")) && *cp)
	    m_putenv ("KDS", cp);
    if ((cp = getenv ("TMADB")) == NULL || *cp == NULL)
	if ((cp = m_find ("tmadb")) && *cp)
	    m_putenv ("TMADB", m_maildir (cp));
#endif /* TMA */

    if ((cp = getenv ("SIGNATURE")) == NULL || *cp == 0)
	if ((cp = m_find ("signature")) && *cp)
	    m_putenv ("SIGNATURE", cp);
#ifdef UCI
	else {
	    sprintf (buf, "%s/.signature", mypath);
	    if ((fp = fopen (buf, "r")) != NULL
		&& fgets (buf, sizeof(buf), fp) != NULL) {
		    fclose (fp);
		    if (cp = strchr (buf, '\n'))
			*cp = 0;
		    m_putenv ("SIGNATURE", buf);
	    }
	}
#endif /* UCI */

    if ((annotext = getenv ("mhannotate")) == NULL || *annotext == 0)
	annotext = NULL;
    if ((altmsg = getenv ("mhaltmsg")) == NULL || *altmsg == 0)
	altmsg = NULL;
    if (annotext && ((cp = getenv ("mhinplace")) != NULL && *cp != 0))
	inplace = atoi (cp);

    if ((cp = getenv ("mhdist"))
	    && *cp
#ifndef lint
	    && (distsw = atoi (cp))
#endif /* not lint */
	    && altmsg) {
	vec[vecp++] = "-dist";
	distfile = getcpy (m_scratch (altmsg, invo_name));
	if (link (altmsg, distfile) == NOTOK)
	    adios (distfile, "unable to link %s to", altmsg);
    }
    else
	distfile = NULL;

    if (altmsg == NULL || stat (altmsg, &st) == NOTOK)
	st.st_mtime = 0, st.st_dev = 0, st.st_ino = 0;
    if ((pushsw = pushed))
	push ();

    vec[0] = r1bindex (postproc, '/');
    closefds (3);

    if (sendsbr (vec, vecp, file, &st) == OK)
	done (0);
}

/*
 * WHOM
 */

static int
whomfile (char **arg, char *file)
{
    pid_t pid;
    int vecp;
    char *vec[MAXARGS];

    m_update ();	/* update the context file */
    fflush (stdout);

    switch (pid = vfork ()) {
	case NOTOK: 
	    advise ("fork", "unable to");
	    return 1;

	case OK: 
	    vecp = 0;
	    vec[vecp++] = r1bindex (whomproc, '/');
	    vec[vecp++] = file;
	    if (arg)
		while (*arg)
		    vec[vecp++] = *arg++;
	    vec[vecp] = NULL;

	    execvp (whomproc, vec);
	    fprintf (stderr, "unable to exec ");
	    perror (whomproc);
	    _exit (-1);		/* NOTREACHED */

	default: 
	    return (pidwait (pid, NOTOK) & 0377 ? 1 : 0);
    }
}
