/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1992 Electrotechnical Laboratry (ETL)

Permission to use, copy, modify, and distribute this material 
for any purpose and without fee is hereby granted, provided 
that the above copyright notice and this permission notice 
appear in all copies, and that the name of ETL not be 
used in advertising or publicity pertaining to this 
material without the specific, prior written permission 
of an authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY 
OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", 
WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type: program/C; charset=US-ASCII
Program:      mimehead.c (MIME header encoder/decoder)
Author:       Yutaka Sato <ysato@etl.go.jp>
Description:
	RFC1522(1342) encoder/decoder for multibyte ISO-2022 charsets

History:
        920515	extracted from mmsclient.c
        920516	added MIME header encoder/decoder for ISO-2022-JP
	920930	added switch to ASCII at the end of splitted encoded-text
	930826	added DecodeOverwrite()
	930915	added LWSP_CHAR -> liner-white-space translation in EN_FPUTC()
	930925	moved unfolding operation from DE_FGETC() to DE_FPUTC()
	930925	removed LWSP BETWEEN encoded word rather than AFTER
	930925	added folding in ascii text "noencode_word"
	930925	added output flush on FORMFEED code
	930927	removed generation of empty encoded-text in encode-word
	930927	removed folding in noencode_word except preceeded encode-word
	931007	fixed the DE_FPUTC bug which eats the spaces at the top of body
	931022	fixed the tmpHeaderEncode which skips message body
	931029	infinite loop on folding to avoid encode_word()<MIN_ENCODEDLEN
	931103	decoded E-W come always terminate in ASCII at the end of line
	931105	finally fixed the "infinite loop on folding.." in encode_word()
	931116	extracted from mmsencode.c
	931227	added parameter decl. for DEC-Alpha(Thanks to <ko@soum.co.jp>)
	931228	changed type ch->int to detect EOF at disp_word on RS6000/AIX
Bugs:
	Any linear-white-space between encoded-words should be ignored
	Should support any charsets & encodings in ISO-2022 or ISO-8859
///////////////////////////////////////////////////////////////////////*/

#include <stdio.h>
#include "str_stdio.h"
FILE *str_fopen();
char *index(),*getenv();
#define MAX_LNSIZE	1024
typedef char MsgLine[MAX_LNSIZE];
typedef char SwSeq[8];

/*//////////////////////////////////////////////////////////////////////*/
#define MAXCOL			72
#define DISPCOLS		80

#define SWCODE_LENG		4 /*length(encoded(charset sw ESC seq))*/
#define MIN_ENCODEDLEN		4 /* minimum length of encoded text(base64)*/

#define ENCODE_BEGIN		"=?"
#define CHARSET_DONE		'?'
#define ENCODING_DONE		'?'
#define ENCODE_DONE		"?="

#define NL			'\n'
#define CR			'\r'
#define TAB			'\t'
#define SPACE			' '
#define FORMFEED		'\f'
#define LWSP_CHAR(ch)		(ch == TAB || ch == SPACE)
#define FOLD_CHAR		SPACE
#define SPECIALS		"()<>@,;:\\\".[]"
#define DELIMITER(ch)		(ch==NL || LWSP_CHAR(ch) || index(SPECIALS,ch))

#define NLNL			0x80000001
#define XC_FOLD			0x80000002	/* FOLD COMMAND */
#define XC_UNFOLD		0x80000003
#define XC_CATENATE		0x80000004
#define XC_AFTER_EWORD		0x80000005
#define XC_FORMFEED		0x80000006
#define XC_OFLUSH		0x80000007

#define ESC			033
#define GOTO1BCODE		'('
#define GOTO2BCODE		'$'
#define GOTO_ASCII_SEQ		"\033(B"
#define GOTO_ASCII_SEQ_LEN	(sizeof(GOTO_ASCII_SEQ)-1)

char US_ASCII[]		=	"US-ASCII";
char ISO_8859_8[]	=	"ISO-8859-8";
char ISO_2022_JP[]	=	"ISO-2022-JP";

#define ENCODE_NONE		 0
#define ENCODE_BASE64		"B"
#define	ENCODE_QP		"Q"

typedef struct {
	int	local;
	char	codesw;
	char   *charset;
	char   *encoding;
} CodeSwitch;

static CodeSwitch Codes1[16] = {
	{1,	'B',	US_ASCII,	ENCODE_NONE	},
	{1,	'J',	US_ASCII,	ENCODE_NONE	},/* ASCII FAMILY */
	0
};

static CodeSwitch Codes2[16] = {
	{1,	'@',	ISO_2022_JP,	ENCODE_BASE64	},
	{1,	'B',	ISO_2022_JP,	ENCODE_BASE64	},
	0
};
MIME_localCharset(charset)
	char *charset;
{	int csi;
	char *cs;

	for(csi = 0; cs = Codes1[csi].charset; csi++)
		if( strcasecmp(cs,charset) == 0 )
			return Codes1[csi].local;

	for(csi = 0; cs = Codes2[csi].charset; csi++)
		if( strcasecmp(cs,charset) == 0 )
			return Codes2[csi].local;
	return 0;
}


typedef struct {
	FILE	*in_file;
	char	*in_charset;
	char	*in_encoding;
	SwSeq	 in_codesw_seq;
	int	 in_prevch;

	FILE	*out_file;
	char	*out_prev_charset;
	SwSeq	 out_codesw_seq;
	SwSeq	 out_prev_codesw_seq;
	int	 out_column;
	int	 out_prevch;
	int	 out_pending[4];

	union {
		int	 all;
		struct {
			int	 folding:1,
				 unfolding:1,
				 canbe_folded:1,
				 ignore_SPACE:1;
		} mode;
	} out_modes;
} INOUT;
#define FOLDING		out_modes.mode.folding
#define UNFOLDING	out_modes.mode.unfolding
#define AFTER_EWORD	out_modes.mode.ignore_SPACE
#define AFTER_EWORD_EOL	out_modes.mode.canbe_folded

#define PENDING0	out_pending[0]
#define CLEAR_PENDING(io)	(io->out_pending[0] = 0)

static INOUT_init(io,in,out)
	INOUT *io;
	FILE *in,*out;
{
	io->in_file = in;
	io->in_charset = US_ASCII;
	io->in_encoding = ENCODE_NONE;
	io->in_codesw_seq[0] = 0;
	io->in_prevch = EOF;

	io->out_file = out;
	io->out_prev_charset = US_ASCII;
	io->out_codesw_seq[0] = 0;
	io->out_prev_codesw_seq[0] = 0;
	io->out_column = 0;
	io->out_prevch = 0;
	io->out_pending[0] = 0;
	io->out_modes.all = 0;
}

static NLfgetc(in)
	FILE *in;
{	int ch;

	ch = fgetc(in);
	if( ch == CR ){
		ch = fgetc(in);
		if( ch != NL ){
			if( ch != EOF )
				ungetc(ch,in);
			ch = CR;
		}
	}
	return ch;
}


static EN_FGETC(io)
	INOUT *io;
{	int ch;
	CodeSwitch *csw;
	int ci;
	char *charset;
	char *sw;
	SwSeq swseq;
	FILE *infile = io->in_file;

GET1:
	ch = NLfgetc(infile);
GOT1:
	if( ch != ESC )
		goto exit;

	/* got ESC character */
	if( (ch = fgetc(infile)) == EOF )
		goto exit;

	if( io->in_prevch == NL )
		if( LWSP_CHAR(ch) )
			goto GET1;

	sw = swseq;
	*sw++ = ESC;
	*sw++ = ch;

	switch( ch ){
		default:	goto exit;
		case GOTO1BCODE: csw = Codes1; break;
		case GOTO2BCODE: csw = Codes2; break;
	}
	if( (ch = fgetc(infile)) == EOF )
		goto exit;

	*sw++ = ch;
	*sw++ = 0;

	for( ci = 0; charset = csw[ci].charset; ci++ )
		if( ch == csw[ci].codesw ){
			io->in_charset = charset;
			io->in_encoding = csw[ci].encoding;
			strcpy(io->in_codesw_seq,swseq);
		}

	ch = NLfgetc(infile);
	if( ch == ESC )
		goto GOT1;
exit:
	io->in_prevch = ch;
	if( ch == NL || ch == NLNL )
		io->AFTER_EWORD_EOL = 0;
	return ch;
}

static ew_overhead(charset,encoding)
	char *charset,*encoding;
{	MsgLine overhead;

	sprintf(overhead,"=?%s?%s??= ",charset,encoding);
	return strlen(overhead);
}

static EN_FPUTC0(ch,io)
	INOUT *io;
{
	if( ch == NL )
		io->out_column = 0;
	else	io->out_column += 1;
	fputc(ch,io->out_file);
}

/*
 *	extra folding before a lengthy encoded-word
 *	put =?charset?encoding? at the beginning of non-ASCII
 *	put SPACE before it if the previous char is not DELIMITER
 *
 *	put ?= at the end of non-ASCII
 *	put SPACE after it if the next char is not DELIMITER
 */
static EN_FPUTC1(ch,io,charset,encoding)
	INOUT *io;
	char *charset,*encoding;
{	char *cp;
	MsgLine line;

	if( charset != io->out_prev_charset ){

		/* AT THE END OF A ENCODED WORD */
		if( io->out_prev_charset != US_ASCII ){
			for( cp = ENCODE_DONE; *cp; cp++ )
				EN_FPUTC0(*cp,io);

			if( !DELIMITER(ch) )
				EN_FPUTC0(SPACE,io);
		}

		/* AT THE BEGINNING OF A ENCODED WORD */
		if( charset != US_ASCII ){
			int reqlen,remlen;

			if( !DELIMITER(io->out_prevch) )
				EN_FPUTC0(SPACE,io);

			reqlen = ew_overhead(charset,encoding);
			remlen = MAXCOL - (io->out_column + reqlen);

			if( (remlen-SWCODE_LENG) < MIN_ENCODEDLEN ){
				EN_FPUTC0(NL,io);
				EN_FPUTC0(FOLD_CHAR,io);
			}
			sprintf(line,"=?%s?%s?",charset,encoding);
			for( cp = line; *cp; cp++ )
				EN_FPUTC0(*cp,io);
			io->AFTER_EWORD_EOL = 1;
		}
		io->out_prev_charset = charset;
	}

	if( ch != EOF ){
		if( ch != NL ){
			/* split at LWSP_CHAR ... */
			if( !encoding )
			if( io->AFTER_EWORD_EOL )
			if( MAXCOL <= io->out_column )
			if( LWSP_CHAR(ch) )
				EN_FPUTC0(NL,io);
		}
		EN_FPUTC0(ch,io);
		io->out_prevch = ch;
	}
}
static EN_FPUTC(ch,io,charset,encoding)
	INOUT *io;
	char *charset,*encoding;
{	int lwsp;

	if( ch == XC_FOLD ){
		if( lwsp = io->PENDING0 )
			io->PENDING0 = 0;
		else	lwsp = SPACE;
		EN_FPUTC1(NL,io,US_ASCII,ENCODE_NONE);
		EN_FPUTC1(lwsp,io,US_ASCII,ENCODE_NONE);
	}else{
		if( lwsp = io->PENDING0 ){
			EN_FPUTC1(lwsp,io,US_ASCII,ENCODE_NONE);
			io->PENDING0 = 0;
		}
		if(LWSP_CHAR(ch) && charset==US_ASCII && encoding==ENCODE_NONE)
			io->PENDING0 = ch;
		else	EN_FPUTC1(ch,io,charset,encoding);
	}
}

MIME_headerEncode0(in,out)
	FILE *in,*out;
{	char *ip,*op;
	INOUT iob,*io = &iob;
	int ch;

	INOUT_init(io,in,out);
	for(;;){
		ch = EN_FGETC(io);
		ungetc(ch,io->in_file);
		if( io->in_charset == US_ASCII ){
			ch = noencode_word(io);
			if( ch == EOF )
				break;
			if( ch == NLNL )
				break;
		}else{
			for(;;){
				ch = encode_word(io);
				if( io->in_charset == US_ASCII )
					break;
			}
			if( ch == EOF )
				break;
		}
	}
	if( ch == EOF )
		EN_FPUTC(ch,io,US_ASCII,ENCODE_NONE);
	return ch;
}
MIME_headerEncode(in,out)
	FILE *in,*out;
{	int ch;

	ch = MIME_headerEncode0(in,out);
	if( ch != EOF ){
		fputc(NL,out);
		while( (ch = NLfgetc(in)) != EOF )
			fputc(ch,out);
	}
}

/*
 *	PASS THROUGH AN ASCII WORD
 */
static noencode_word(io)
	INOUT *io;
{	int ch,nch,inx;
	int canbe_folded;
	MsgLine line;

	canbe_folded = io->AFTER_EWORD_EOL;
	for(inx = 0; inx <= MAXCOL; inx++){
		ch = EN_FGETC(io);
		if( io->in_charset != US_ASCII ){
			ungetc(ch,io->in_file);
			break;
		}
		if( ch == EOF )
			break;
		if( ch == NL ){
			line[inx++] = NL;
			switch( nch = EN_FGETC(io) ){
			    case NL:  ch = NLNL; break;
			    case EOF: break;
			    default:  ungetc(nch,io->in_file); break;
			}
			break;
		}
/*
if( canbe_folded )
if( DELIMITER(ch) )
*/
/* might be harmful for tools don't treat unfolding properly */
		if( LWSP_CHAR(ch) )
		{
			line[inx++] = ch;
			break;
		}
		line[inx] = ch;
	}
	line[inx] = 0;

	if( line[0] != NL )
	if( canbe_folded )/* safety for non-MIMEr like inews/Cnews */
	if( MAXCOL+2 < io->out_column+inx )
		EN_FPUTC(XC_FOLD,io,US_ASCII,ENCODE_NONE);

	{	int ch,ci;
		for( ci = 0; ch = line[ci]; ci++ )
			EN_FPUTC(ch,io,US_ASCII,ENCODE_NONE);
	}
	return ch;
}

static encode_one(encoding,ins,ilen,outs,osize)
	char *encoding,*ins,*outs;
{	int len;

	if( strcasecmp(encoding,ENCODE_QP) == 0 )
		len = str_toqp(ins,ilen,outs,osize);
	else
	if( strcasecmp(encoding,ENCODE_BASE64) == 0 )
		len = str_to64(ins,ilen,outs,osize);
	else{
		strncpy(outs,ins,ilen);
		len = ilen;
	}
	outs[len] = 0;
	return len;
}

static encode_word(io)
	INOUT *io;
{	char *charset;	/* charset of this encoded-word */
	char *encoding;	/* encoding of this encoded-word */
	MsgLine ins,outs;
	int inx,outx,prefold;
	int char_bytes,nchar,reqlen,remlen,outlen;
	char ch,encoded_ch;

	charset = io->in_charset;
	char_bytes = 2;
	encoding = io->in_encoding;
	reqlen = ew_overhead(charset,encoding);

	/*
	 *	firstly, add the code switch sequence in a encoded format
	 */
	strcpy(ins,io->in_codesw_seq);
	inx = strlen(ins);
	outlen = encode_one(encoding,ins,inx,outs,sizeof(outs));

	/*
	 *	if remaining length is not enough, fold the line
	 */
	remlen = MAXCOL - (io->out_column + reqlen);
	if( (remlen-outlen) <= MIN_ENCODEDLEN ){
		remlen = MAXCOL - (1 + reqlen);
		prefold = 1;
	}else	prefold = 0;

	/*
	 *	scan a word to be encoded expanding byte by byte.
	 *	every encoded-texts end with the switch to US_ASCII
	 */
	for(nchar = 0; ;nchar++){
		strcpy(&ins[inx],GOTO_ASCII_SEQ);
		outlen = encode_one(encoding,ins,inx+GOTO_ASCII_SEQ_LEN,
			 outs,sizeof(outs));

		if( nchar % char_bytes == 0 && remlen <= outlen )
			break;

		ch = EN_FGETC(io);
		if( ch == EOF )
			break;

		if( io->in_charset != charset ){
			if( io->in_charset == US_ASCII )
				/* possibly an ascii family like JIS-Roman */
				strcpy(&ins[inx],io->in_codesw_seq);
			ungetc(ch,io->in_file);
			break;
		}
		ins[inx++] = ch;
		ins[inx] = 0;
	}
	inx += strlen(&ins[inx]);

	if( nchar == 0 )
		return ch;
	if( prefold )
		EN_FPUTC(XC_FOLD,io,US_ASCII,ENCODE_NONE);
	/*
	 *	output the scanned word
	 */
	outlen = encode_one(encoding,ins,inx,outs,sizeof(outs));
	for(outx = 0; outx < outlen; outx++){
		encoded_ch = outs[outx];
		if( encoded_ch == NL )
			continue;
		EN_FPUTC(encoded_ch,io,charset,encoding);
	}
	return ch;
}
/* it may be desirable to fold before an encoded-word, which length is
 *  shorter than MAXCOL, but will be splitted in current line.  */

static decode_word(io)
	INOUT *io;
{	MsgLine reads,charset,encoding,itext,dtext;
	int ilen,dsize,len,pad,dlen;
	int eow;

	*charset = *encoding = *itext = 0;
	eow = scan_encoded_word(io->in_file,reads,charset,encoding,itext);

	if( eow == NL || eow == EOF ){
		fprintf(io->out_file,"=?%s",reads);
		if( eow != EOF )
			ungetc(eow,io->in_file);
		return eow;
	}

	if( !MIME_localCharset(charset) ){
		fprintf(io->out_file,"=?%s?%s?%s?=",charset,encoding,itext);
		if( eow )
			fprintf(io->out_file,"%c",eow);
		return 0;
	}

	ilen = strlen(itext);
	dsize = sizeof(dtext);
	if( strcasecmp(encoding,ENCODE_QP) == 0 )
		len = str_fromqp(itext,ilen,dtext,dsize);
	else
	if( strcasecmp(encoding,ENCODE_BASE64) == 0 )
		len = str_from64(itext,ilen,dtext,dsize);
	else{
		strcpy(dtext,itext);
		len = ilen;
	}
	dlen = disp_word(0,dtext,len);
	if( io->out_column + dlen < DISPCOLS )
		DE_FPUTC(XC_CATENATE,io);
	else	DE_FPUTC(XC_OFLUSH,io);
	disp_word(io,dtext,len);
	return 0;
}

static scan_encoded_word(in,reads,charset,encoding,text)
	FILE *in;
	char *reads,*charset,*encoding,*text;
{	int i,cs;

	for(i = 0; ;i++){
		cs = NLfgetc(in);
		if(cs==NL || cs==EOF) goto error;
		*reads++ = cs;
		if(cs==CHARSET_DONE) break;
		charset[i] = cs;
		charset[i+1] = 0;
	}
	for(i = 0; ;i++){
		cs = NLfgetc(in);
		if(cs==NL || cs==EOF) goto error;
		*reads++ = cs;
		if(cs==ENCODING_DONE) break;
		encoding[i] = cs;
		encoding[i+1] = 0;
	}
	for(i = 0; i < 80; i++ ){
		cs = NLfgetc(in);
		if(cs==NL || cs==EOF) goto error;
		*reads++ = cs;
		if(cs == ENCODE_DONE[0]){
			cs = NLfgetc(in);
			if(cs==NL || cs==EOF) goto error;
			*reads++ = cs;
			if( cs == ENCODE_DONE[1] ){
				text[i] = 0;
				break;
			}
			ungetc(cs,in);
			cs = ENCODE_DONE[0];
		}
		text[i] = cs;
		text[i+1] = 0;
	}
	return 0;
error:
	*reads = 0;
	return cs;
}

static disp_word(io,dtext,len)
	INOUT *io;
	char *dtext;
{	FILE *DecodedText;
	INOUT tmpInb,*tmpIn = &tmpInb;
	char *charset1;
	int dch;
	int dlen;

	if( len <= 0 )
		return 0;

	DecodedText = str_fopen(dtext,len);
	INOUT_init(tmpIn,DecodedText,NULL);
	charset1 = US_ASCII;

	dlen = 0;
	while((dch = EN_FGETC(tmpIn)) != EOF ){
		if( tmpIn->in_charset != charset1 ){
			if(io) set_outcodesw_seq(io,tmpIn->in_codesw_seq);
			charset1 = tmpIn->in_charset;
		}
		if(io) DE_FPUTC(dch,io);
		dlen++;
	}
	str_fclose(DecodedText);

	if(io){
		DE_FPUTC(XC_AFTER_EWORD,io);
		io->AFTER_EWORD_EOL = 1;
		if( charset1 != US_ASCII )
			if( tmpIn->in_charset == US_ASCII )
				set_outcodesw_seq(io,tmpIn->in_codesw_seq);
			else	set_outcodesw_seq(io,GOTO_ASCII_SEQ);
	}
	return dlen;
}

static nodecode_word(io,ch)
	INOUT *io;
{
	if( io->AFTER_EWORD_EOL ){
		/* if the next noencoded-word ends before DISPCOLS ...*/
		if( io->out_column < MAXCOL )
			DE_FPUTC(XC_UNFOLD,io);
		else{
			/* the following is experimental */
			if( LWSP_CHAR(ch) ){
				DE_FPUTC(XC_OFLUSH,io);
				if( MAXCOL <= io->out_column )
					DE_FPUTC(NL,io);
			}
		}
	}
	DE_FPUTC(ch,io);
}

static DE_FPUTC(ch,io)
	INOUT *io;
{	char pch;

	if( ch == XC_AFTER_EWORD ){
		io->AFTER_EWORD = 1;
		return;
	}
	if( ch == XC_CATENATE ){
		/* REMOVE PENDING SPACE IF EXISTS */
		if( ! io->AFTER_EWORD ){
			if( (pch = io->out_pending[0]) == NL ){
				/* discard the NEWLINE */
				pch = io->out_pending[1];
			}
			if( pch )
				DE_FPUTC1(pch,io);
		}
		CLEAR_PENDING(io);
		return;
	}
	if( ch == XC_UNFOLD ){
		if( io->out_pending[0] == NL )
		if( pch = io->out_pending[1] ){
			DE_FPUTC1(pch,io);
			CLEAR_PENDING(io);
		}
		return;
	}
	if( ch == XC_FORMFEED ){
		if( pch = io->out_pending[0] ){
			DE_FPUTC1(pch,io);
			if( pch = io->out_pending[1] )
				DE_FPUTC1(pch,io);
			CLEAR_PENDING(io);
		}
		DE_FPUTC1(FORMFEED,io);
		DE_FPUTC1(NL,io);
		fflush(io->out_file);
		return;
	}

	if( io->out_pending[0] == NL ){
		if( LWSP_CHAR(ch) ){ /* linear-white-space */
			io->out_pending[1] = ch;
			io->out_pending[2] = 0;
			return;
		}

		DE_FPUTC1(NL,io);
		if( pch = io->out_pending[1] )
			DE_FPUTC1(pch,io);
		CLEAR_PENDING(io);
	}else
	if( pch = io->out_pending[0] ){
		DE_FPUTC1(pch,io);
		CLEAR_PENDING(io);
	}

	if( io->UNFOLDING ){
		if( ch == NL || io->AFTER_EWORD && LWSP_CHAR(ch)){
			io->out_pending[0] = ch;
			io->out_pending[1] = 0;
			return;
		}
	}
	DE_FPUTC1(ch,io);
}

static DE_FPUTC1(ch,io)
	INOUT *io;
{	FILE *out;
	SwSeq seq;

	out = io->out_file;
	/* FLUSH PENDING CODE SWITCH SEQUENCE IF EXISTS */
	if( get_outcodesw_seq(io,seq) )
		fputs(seq,out);

	/* FLUSH PENDING OUTPUT (necessary ?) */
	if( ch == EOF ){
		if( io->out_pending[0] ){
			fputs(io->out_pending,out);
			io->out_pending[0] = 0;
		}
		return;
	}

	io->AFTER_EWORD = 0;
	if( ch == XC_OFLUSH )
		return;

	if( ch == NL )
		io->out_column = 0;
	else	io->out_column++;
	fputc(ch,out);
}
static set_outcodesw_seq(io,seq)
	INOUT *io;
	char *seq;
{
	strcpy(io->out_codesw_seq,seq);
}
static get_outcodesw_seq(io,nseq)
	INOUT *io;
	char *nseq;
{	char *cseq = io->out_codesw_seq;
	char *pseq = io->out_prev_codesw_seq;

	if( cseq[0] ){
		if( strcmp(cseq,pseq) != 0 ){
			strcpy(nseq,cseq);
			strcpy(pseq,cseq);
			cseq[0] = 0;
			return 1;
		}
		cseq[0] = 0;
	}
	return 0;
}

static DE_FGETC(io)
	INOUT *io;
{	FILE *in;
	int ch;

	in = io->in_file;
	ch = NLfgetc(in);
	if( ch == FORMFEED ){
		ch = NLfgetc(in);
		if( ch == NL )
			ch = XC_FORMFEED;
		else{
			if( ch != EOF )
				ungetc(ch,in);
			ch = FORMFEED;
		}
		io->AFTER_EWORD_EOL = 0;
	}else
	if( ch == NL ){
		ch = NLfgetc(in);
		if( !LWSP_CHAR(ch) )/* at the top of a filed */
			io->AFTER_EWORD_EOL = 0;

		if( ch == NL )
			ch = NLNL;
		else{
			if( ch != EOF )
				ungetc(ch,in);
			ch = NL;
		}
	}
	return ch;
}
MIME_headerDecode(in,out,bodytoo)
	FILE *in,*out;
{	int ch,next_ch;
	INOUT iob,*io = &iob;

	INOUT_init(io,in,out);
	io->UNFOLDING = 1;

	for(;;){
		ch = DE_FGETC(io);
		if( ch == EOF )
			break;

		if( ch == ENCODE_BEGIN[0] ){
			ch = NLfgetc(in);
			if( ch == EOF )
				break;
			if( ch == ENCODE_BEGIN[1] ){
				if( decode_word(io) == EOF )
					break;
			}else{
				DE_FPUTC(ENCODE_BEGIN[0],io);
				ungetc(ch,in);
			}
		}else{
			if( ch == NLNL ){
				io->UNFOLDING = 0;
				DE_FPUTC(NL,io);
				DE_FPUTC(NL,io);
				break;
			}
			nodecode_word(io,ch);
		}
	}
	io->UNFOLDING = 0;
	if( ch != EOF && bodytoo )
		while( (ch = NLfgetc(in)) != EOF )
			DE_FPUTC(ch,io);
	DE_FPUTC(EOF,io);
}

MIME_strHeaderDecode(ins,outs,osize)
	char *ins,*outs;
{	FILE *In,*Out;
	int oi;

	In = str_fopen(ins,strlen(ins));
	Out = str_fopen(outs,osize);
	MIME_headerDecode(In,Out,1);
	fflush(Out);
	for(oi = 0; outs[oi]; oi++)
		if((outs[oi] & 0xFF) == 0xFF)
			strcpy(&outs[oi],&outs[oi+1]);
	str_fclose(In);
	str_fclose(Out);
}
MIME_strHeaderEncode(ins,outs,osize)
	char *ins,*outs;
{	FILE *In,*Out;

	In = str_fopen(ins,strlen(ins));
	Out = str_fopen(outs,osize);
	MIME_headerEncode(In,Out);
	fflush(Out);
	str_fclose(In);
	str_fclose(Out);
}

is_MIME_header(fp)
	FILE *fp;
{	MsgLine line;
	int off;

	off = ftell(fp);
	while( fgets(line,sizeof(line),fp) != NULL ){
		if( *line == NL )
			break;
		if( *line == CR && line[1] == NL )
			break;

		if( strstr(line,ENCODE_BEGIN) ){
			fseek(fp,off,0);
			return 1;
		}
	}
	fseek(fp,off,0);
	return 0;
}

FILE *
MIME_tmpHeaderDecode(fp,bodytoo)
	FILE *fp;
{	FILE *tfp;

	if( fp == NULL )
		return NULL;

	if( fseek(fp,0,1) == 0 ){
		if( !is_MIME_header(fp) )
			return NULL;
	}

	tfp = tmpfile();
	MIME_headerDecode(fp,tfp,bodytoo);
	fflush(tfp);
	fseek(tfp,0,0);
	return tfp;
}
FILE *
MIME_tmpHeaderEncode(fp,savFILE)
	FILE *fp,savFILE;
{	FILE *tin,*tfp;
	MsgLine line;
	int ch;

	if( fp == NULL )
		return;
	tin = tmpfile();
	while( fgets(line,sizeof(line),fp) != NULL ){
		fputs(line,tin);
		if(strcmp(line,".\n")==0 || strcmp(line,".\r\n")==0)
			break;
	}
	fflush(tin);
	fseek(tin,0,0);

	tfp = tmpfile();
	ch = MIME_headerEncode0(tin,tfp);
	if( ch == NLNL ){
		fputs("\r\n",tfp);
		while( fgets(line,sizeof(line),tin) != NULL )
			fputs(line,tfp);
	}
	fputs(".\r\n",tfp);
	fflush(tfp);
	fseek(tfp,0,0);

	fclose(tin);
	return tfp;
}

/*//////////////////////////////////////////////////////////////////////*/
MIME_localStrColumns(str)
	char *str;
{	INOUT iob,*io = &iob;
	FILE *sfp;
	int len;

	sfp = str_fopen(str,strlen(str));
	INOUT_init(io,sfp,NULL);

	len = 0;
	while( EN_FGETC(io) != EOF )
		len++;

	str_fclose(sfp);
	return len;
}
