/*
 *	Program Name:	imap server (see rfc1064)
 *	Module:         imapdate.c
 *
 *	Author(s):	William J. Yeager
 *			Symbolic Systems Resources Group
 *			Knowledge Systems Laboratory
 *			Departments of Computer Science and Medicine
 *			Stanford University
 *			Stanford, CA  94305
 *
 *	Date:		December 29th, MCMLXXXVIII
 *
 *	Sponsorship:	This work was supported by the Biomedical Research
 *			Technology Program of the National Institutes of Health
 *			under grant number RR-00785.
 *
 *      Copyright (c) 1988 by The Leland Stanford Junior University.
 *
 *  "This program may be distributed without restriction for non-commercial 
 *   use. Any sale or use of this program or adaptations thereof for commercial
 *   purposes is prohibited except under license from the Stanford Office of
 *   Technology Licensing."
 *
 */


/* #define DEBUG */
#include <strings.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include "sin.h"
#include "requests.h"
/*
 *  internaldate(cp,tzm,dlt)		char *cp;	string time
 *					int *tzm;	returned minutes
 *							west of GMT if
 *							non-zero.
 *					int *dlt;	returned true if
 *							the date during
 *							day light time.
 *  Take among other formats, a RFC822 (more or less) date, and turn into
 *  unix time(UT or GMT seconds since Jan 1, 1970). 
 *  Allows a variety of illegal formats commonly in use.
 *
 *	By example we will take the following formats:
 *	Note: Alpha months can be full or abbreviated,
 *	      and years can be 2 digit or 4 digit.
 *
 *	"Nov 23, 1968 02:18:01-PST"	ALPHAMONTHFIRST
 *	"23 Nov 1968 02:18:01-PST"	DIGITDAYFIRST
 *	"23-Nov-68 02:18:01-PST"	DIGITDAYFIRST (smtp format)
 *      In any of the above "November" can replace "Nov"
 *	"11/23/68  02:18:01-PST"	SLASHLIKEFORMAT
 *	"11-23-68  02:18:01-PST"	SLASHLIKEFORMAT
 *	"11 23 68  02:18:01-PST"	SLASHLIKEFORMAT
 *	"Sat Nov 23 02:18:01 PST 1968"  CTIMEFORMAT 
 *
 *	The time/time-zone may be absent in all cases.
 *	In this case we use the zone provided by gettimeofday(),
 *	and default hours = minutes = seconds = 0;
 */

#define nil 0
typedef struct match_tab {
	char *str;
	int value;
} MATCHTAB;

/*
 *  Month tab matches month names to number.
 */
MATCHTAB 	day_tab[]= {
	"Sun",1,"Mon",2,"Tue",3,"Wed",4,"Thu",5,"Fri",6,"Sat",7,nil,0
};

MATCHTAB	month_tab[]  =  {
	"january", 1,	"february", 2,	"march", 3,	"april", 4,
	"may", 5,	"june", 6,	"july", 7,	"august", 8,
	"september", 9,	"october", 10,	"november", 11,	"december", 12,
	nil, 0
};

#define	H(x)	((x)*60*60)
MATCHTAB	zone_tab[] = {
	"ut", 0,	"gmt", 0,	"est", H(-5),	"edt", H(-4),
	"cst", H(-6),	"cdt", H(-5),	"mst", H(-7),	"mdt", H(-6),
	"pst", H(-8),	"pdt", H(-7),	"z", 0,
	"a", H(-1),	"b", H(-2),	"c", H(-3),	"d", H(-4),
	"e", H(-5),	"f", H(-6),	"g", H(-7),	"h", H(-8),
	"i", H(-9),	"k", H(-10),	"l", H(-11),	"m", H(-12),
	"n", H(1),	"o", H(2),	"p", H(3),	"q", H(4),
	"r", H(5),	"s", H(6),	"t", H(7),	"u", H(8),
	"v", H(9),	"w", H(10),	"x", H(11),	"y", H(12),
	nil, -1
};

#define	dysize(A) (((A)%4) ? 365: 366)
static int dmsize[12] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
extern  int timezonesecs, daylight;
/*
 *	Tables for daylight savings time calculations.
 */

/*
 * The following table is used for 1974 and 1975 and
 * gives the day number of the first day after the Sunday of the
 * change.
 */
struct dstab {
	int	dayyr;
	int	daylb;
	int	dayle;
};

static struct dstab usdaytab[] = {
	1974,	5,	333,	/* 1974: Jan 6 - last Sun. in Nov */
	1975,	58,	303,	/* 1975: Last Sun. in Feb - last Sun in Oct */
	1987,   96,     303,    /* 1987: First Sun in Apr - last Sun in Oct */
	1988,   96,     303,    /* 1988: First Sun in Apr - last Sun in Oct */
	1989,   96,     303,    /* 1989: First Sun in Apr - last Sun in Oct */
	0,	119,	303,	/* all other years: end Apr - end Oct */
};
static struct dstab ausdaytab[] = {
	1970,	400,	0,	/* 1970: no daylight saving at all */
	1971,	303,	0,	/* 1971: daylight saving from Oct 31 */
	1972,	303,	58,	/* 1972: Jan 1 -> Feb 27 & Oct 31 -> dec 31 */
	0,	303,	65,	/* others: -> Mar 7, Oct 31 -> */
};

/*
 * The European tables ... based on hearsay
 * Believed correct for:
 *	WE:	Great Britain, Portugal?
 *	ME:	Belgium, Luxembourg, Netherlands, Denmark, Norway,
 *		Austria, Poland, Czechoslovakia, Sweden, Switzerland,
 *		DDR, DBR, France, Spain, Hungary, Italy, Jugoslavia
 *		Finland (EE timezone, but ME dst rules)
 * Eastern European dst is unknown, we'll make it ME until someone speaks up.
 *	EE:	Bulgaria, Greece, Rumania, Turkey, Western Russia
 *
 * Ireland is unpredictable.  (Years when Easter Sunday just happens ...)
 * Years before 1983 are suspect.
 */
static struct dstab wedaytab[] = {
	1983,	89,	296,	/* 1983: end March - end Oct */
	0,	89,	303,	/* others: end March - end Oct */
};

static struct dstab medaytab[] = {
	1983,	89,	296,	/* 1983: end March - end Oct */
	0,	89,	272,	/* others: end March - end Sep */
};

/*
 * Canada, same as the US, except no early 70's fluctuations.
 * Can this really be right ??
 */
static struct dstab candaytab[] = {
	0,	119,	303,	/* all years: end Apr - end Oct */
};

static struct dayrules {
	int		dst_type;	/* number obtained from system */
	int		dst_hrs;	/* hours to add when dst on */
	struct	dstab *	dst_rules;	/* one of the above */
	enum {STH,NTH}	dst_hemi;	/* southern, northern hemisphere */
} dayrules [] = {
	DST_USA,	1,	usdaytab,	NTH,
	DST_AUST,	1,	ausdaytab,	STH,
	DST_WET,	1,	wedaytab,	NTH,
	DST_MET,	1,	medaytab,	NTH,
	DST_EET,	1,	medaytab,	NTH,	/* XXX */
#ifdef DST_CAN
	DST_CAN,	1,	candaytab,	NTH,
#endif
	-1,
};
/*
 *   Returned from Unix gettimeofday():
 *
          0    NONE: Daylight Savings Time not observed
          1    USA: United States DST
          2    AUST: Australian DST
          3    WET: Western European DST
          4    MET: Middle European DST
          5    EET: Eastern European DST
          6    CAN: Canadian DST
 *
 */


#define ALPHAMONTHFIRST 1
#define DIGITDAYFIRST 2
#define SLASHLIKEFORMAT 3
#define CTIMEFORMAT 4
long internaldate(cp,tzm,dlt)
	char	*cp;
	int	*tzm;
	int	*dlt;
{
	int	day, month, year, hours, minutes, seconds;
	int	dayno;
	long	tval;
	int	i;
	char	*dp,*ep;
	int	zone;
        int	format;
	int	needtime;
	int 	isdlt;

	cp = (char *)clearwhitespace(cp);
	dp = cp;

	format = guessFormat(cp);		/* choose the format(gulp!) */
#ifdef DEBUG
	printFormat(format);
#endif
        if (!format) return(-1);		/* do not know it */
	switch (format) {
		case ALPHAMONTHFIRST: /* "Nov 23, 1968 02:18:01-PST" */
		    if (!fetchAlphamonth(&cp,&month)) return(-1);
		    if (!fetchday(&cp,&day)) return(-1);
		    if (!fetchyear(&cp,&year)) return(-1);
		    needtime = 1;
		    break;
		case DIGITDAYFIRST: /* 23-Nov-68 02:18:01-PST" */
		    if (!fetchday(&cp,&day)) return(-1);
		    if (!fetchAlphamonth(&cp,&month)) return(-1);
		    if (!fetchyear(&cp,&year)) return(-1);
		    needtime = 1;
		    break;
		case SLASHLIKEFORMAT: /* 11/23/88 ... */
		    month = atoi(cp);
		    while (*cp && isdigit(*cp)) ++cp;	/* /23/88 .. */
		    if (*cp == 0) return(-1);
		    if (!fetchday(&cp,&day)) return(-1);
		    if (!fetchyear(&cp,&year)) return(-1);
		    needtime = 1;
		    break;
		case CTIMEFORMAT: /* Sat Nov 23 02:18:00 PST 1968 */
		    needtime = 0;
		    if (!ctimeformat(dp,&day,&month,
			&year,&hours,&minutes,&seconds,&zone)) return(-1);
		    break;
	}
 	if (needtime)
	    if (!fetchtime(cp,&hours,&minutes,&seconds,&zone)) return(-1);
	if(year < 1900) year += 1900;
	if(month < 1 || month > 12)  return(-1);
	if(day < 1 || day > 31) return(-1);
	if(hours == 24)  {
		hours = 0;
		day++;
	}
	if(hours < 0 || hours > 23) return(-1);
	if(minutes < 0 || minutes > 59) return(-1);
	if(seconds < 0 || seconds > 59) return(-1);
#ifdef DEBUG
	printf("\n Year %d Month %d Day %d %02d:%02d:%02d",year,
		month,day,hours,minutes,seconds);
#endif

	tval = 0;
	dayno = 0;
	for(i = 1970; i < year; i++)
		tval += dysize(i);
	if(dysize(year) == 366  && month >= 3)
		dayno += 1;
	while(--month) dayno += dmsize[month-1];
#ifdef DEBUG
	printf(" Day Number %d.",dayno + day - 1);
#endif
	tval += dayno;
	tval += day-1;
	tval *= 24;
	tval += hours;
	tval *= 60;
	tval += minutes;
	tval *= 60;
	tval += seconds;
	tod();				/* sets time zone secs/daylight*/
	if (daylight) 
	    isdlt = checkfordlt(dayno+day-1,year,hours,daylight);
	else
	    isdlt = 0;
	if (dlt) *dlt = isdlt;
	if(zone == -1)  {/* zone not present */
		zone = timezonesecs;		
		if (daylight && isdlt) zone -= 60*60;
	}
#ifdef DEBUG
	printf(" Zone %d secs", zone);
#endif
	tval += zone;			/* Add to get GMT */
	if (tzm) *tzm = timezonesecs/60;
	return(tval);
}
#ifdef DEBUG
static char *weekdays[]= {"Sun","Mon","Tue","Wed","Thur","Fri","Sat"};
#endif
static int sunday(d, year)
int 	d,
	year;
{
	int i;
	int	pdays= 0,
		dow;

	for(i = 1970; i < year; i++)
		pdays += dysize(i);
	if(d >= 58 && dysize(year) == 366) d++;
	/*
	 * Get day of week of target day. Note that 
	 * Jan 1, 1970 was a Thursday.
	 */
	dow = (pdays + d - 3) % 7;
#ifdef DEBUG
	printf("\n Day %d is on a %s",d,weekdays[dow]);
#endif
	return(d - dow);
}
match(table,str,len)
  register MATCHTAB *table;
  register char *str;
  register len;
  {
    while (table->str != nil) {
	if (partialMatch(str,table->str,len)) return(table->value);
	++table;
    }
    return(table->value);		/* end-of-table */
  }
/*
 *	source must match known for "len of source" chars.
 *	The source must be <= known in length.
 */
partialMatch(src,known,len)
  register char *known,*src;
  register len;
  {
    register char k,s;
    while (len-- > 0) {
	k = *known++;
	s = *src++;
	if (k == 0) return(0);			/* known ran out */
  	if (CVU(k) != CVU(s)) return(0);	/* mismatch */
    }
    return(1);
  }
/*
 *  guessFormat(cp)		
 *	We look for one of the following (by example):
 *
 *	"Nov 23, 1968 02:18:01-PST"	ALPHAMONTHFIRST
 *	"23 Nov 1968 02:18:01-PST"	DIGITDAYFIRST
 *	"23-Nov-68 02:18:01-PST"	DIGITDAYFIRST (smtp format)
 *	"11/23/68  02:18:01-PST"	SLASHLIKEFORMAT
 *	"11-23-68  02:18:01-PST"	SLASHLIKEFORMAT
 *	"Sat Nov 23 02:18:01 PST 1968"  CTIMEFORMAT 
 */
guessFormat(cp)
  register char *cp;
  {
    if (match(day_tab,cp,3)) return(CTIMEFORMAT);
    if (isdigit(*cp)) {
	while (isdigit(*cp)) ++cp;	/* skip to char after digits */
	++cp;				/* Now skip to next char */
	if (isdigit(*cp)) return(SLASHLIKEFORMAT);
	else
	    return(DIGITDAYFIRST);
    } else
	  return(ALPHAMONTHFIRST);
  }
#ifdef DEBUG
printFormat(i)
 int i;
 {
    printf("\n{Guessed format like: ");
    switch (i) {
	case ALPHAMONTHFIRST:
	    printf("Nov 23, 1968 02:18:01-PST}"); break;
	case DIGITDAYFIRST:
	    printf("23 Nov 1968 02:18:01-PST}"); break;
	case SLASHLIKEFORMAT:
	    printf("11/23/68  02:18:01-PST}"); break;
	case CTIMEFORMAT :
	    printf("Sat Nov 23 02:18:01 PST 1968(unix ctime)}" ); break;
	default: printf("\n Unknown!"); break;
    }
  }
#endif
ctimeformat(dp,day,month,year,hours,minutes,seconds,zone)
  char *dp;
  int   *day,
	*month,
	*year,
	*hours,
	*minutes,
	*seconds,
	*zone;
  {
	/*  
	 *	Hack here, look for ctime format. 
	 *	EG: "Sun Dec 25 17:50:01 1945" or
	 *	    "Sun Dec 25 17:50:01 PST 1945"
	 *	     0123456789012345678901234
	 *		       |         |
         *		       10        20
	 */
    int zonep;
    char *fp= &dp[8];

    if (*fp == SP) ++fp;
    *day = atoi(fp);
    *month = match(month_tab, &dp[4], 3);
    if (month == 0) return(0); 
    fp = &dp[20];
    if (isalpha(*fp)) {
	fp += 4;		/* have zone */
	zonep = 1;
    } else
	  zonep = 0;
    *year = atoi(fp);
    *hours = atoi(&dp[11]);
    *minutes = atoi(&dp[14]);
    *seconds = atoi(&dp[17]);
    if (zonep) *zone = -match(zone_tab, &dp[20], 3);
    else
        *zone = -1;
    return(1);
  }

fetchAlphamonth(cp,month)
  char **cp;
  int *month;
  {
    register char *p1,*p2;

    p1 = *cp;
    while (*p1 && !isalpha(*p1)) ++p1;
    if (!*p1) return(0);
    p2 = p1;
    while (isalpha(*p2)) ++p2;			/* first char after "Mon" */
    *month = match(month_tab,p1,p2-p1);
    if (month == 0) return(0);
    *cp = p2;
    return(1);
  }

fetchyear(cp,year)
  char **cp;
  int *year;
  {
    char *p= *cp;

    while (*p && !isdigit(*p)) ++p;
    if (!*p) return(0);
    *year = atoi(p);
    while(isdigit(*p)) ++p;
    *cp = p;
    return(1);
  }

fetchday(cp,day)
  char **cp;
  int *day;
  {
    char *p= *cp;

    while (*p && !isdigit(*p)) ++p;
    if (!*p) return(0);
    *day = atoi(p);
    while(isdigit(*p)) ++p;
    *cp = p;
    return(1);
  }
fetchtime(cp,hours,minutes,seconds,zone)
  char *cp;
  int	*hours,
	*minutes,
	*seconds,
	*zone;
  {/*
    *	hh:mm:ss-ZONE
    */
    char *dp;

    while(*cp && !isdigit(*cp)) cp++;
    if (!*cp) {/* Not there, so ... */
	*hours = *minutes = *seconds = 0;
	*zone = -1;
        return(1);	
    }
    *hours = atoi(cp);
    if (*hours > 60)  {/* HHMM */
    	*minutes = *hours %100;
	*hours = *hours/100;
    } else {
	cp = (char *)index(cp, ':');
	if (cp == nil) return(0);
	*minutes = atoi(++cp);
    }
    dp = index(cp, ':');
    if (dp) {
	*seconds = atoi(++dp);
	cp = dp;
    } else
	  *seconds = 0;

    while (*cp && isdigit(*cp)) cp++;
    cp = (char *)clearwhitespace(cp);
    if (!*cp) {
	*zone = -1;
	return(1);
    }
   if ((*cp == '+' || *cp == '-')  &&
	isdigit(cp[1]) &&
	isdigit(cp[2]) &&
	isdigit(cp[3]) &&
	isdigit(cp[4])) {/* HHMM */

	*zone = atoi(cp+3) * 60;	/* mins to secs */
	cp[3] = 0;
	*zone += (atoi(cp+1) * 3600);	/* + hours to secs */
        if (*cp == '+') *zone = -*zone;
    } else {/* something like PST */
	++cp;				/* skip delimeter */
	dp = cp;
	while (isalpha(*cp)) cp++;
	*cp = 0;
	*zone = -match(zone_tab, dp, cp - dp);
    }
    return(1);
  }
/*
 *	These return start/end of daylight time as a function of
 *	the year.
 */
getdlstart()
  {
    return(98);
  }
getdlend()
  {
    return(298);
  }
/* #define DLDEBUG */
checkfordlt(dayno,year,hours,dst_type)
  int   dayno,
	hours,
	year,
	dst_type;
  {
	int beg, end;
	register struct dayrules *dr;
	register struct dstab *ds;
	int endhour;
	
	/*
	 *	Find the rules for the  daylight savings type
	 */
	for (dr = dayrules; dr->dst_type >= 0; dr++)
		if (dr->dst_type == dst_type)
			break;
	if (dr->dst_type >= 0) {/* find start, end for year */
	    for (ds = dr->dst_rules; ds->dayyr; ds++)
		if (ds->dayyr == year) 	break;
	    beg = ds->daylb;	/* Target on or after starting Sunday */
	    end = ds->dayle;	/* Target on or after ending Sunday */
	    beg = sunday(beg, year);	/* Find the Sunday */
	    end = sunday(end, year);
#ifdef DLDEBUG
	printf("\nFor B(%d)= %d, E(%d)= %d: Is Day %d for hour %d DLT?",
		ds->daylb,beg,ds->dayle,end,dayno,hours);
#endif      
	    if (dr->dst_hemi == NTH) endhour = 1;
	    else
		endhour = 2;
	    if (((dayno>beg) || ((dayno == beg) && (hours >= 2))) &&
	        ((dayno<end) || ((dayno == end) && (hours < endhour)))) {
#ifdef DLDEBUG
		    printf(" Yes\n");
#endif
		    return(1);
	    }
	} 
#ifdef DLDEBUG
	printf(" No\n");
#endif
	return(0);
  }
monthmatch(sp,len)
 {
   return(match(month_tab,sp,len));
 }
