/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Miscellaneous string-handling functions */


#include "exim.h"





/*************************************************
*       Convert a number to base 62 format       *
*************************************************/

/* Convert a long integer into an ASCII base 62 string.
Always return exactly 6 characters plus zero, in a static area. */

char *string_base62(unsigned long int value)
{
static char yield[7];
static char base62_chars[] =
    "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
char *p = yield + sizeof(yield) - 1;
*p = 0;
while (p > yield)
  {
  *(--p) = base62_chars[value % 62];
  value /= 62;
  }
return yield;
}



/*************************************************
*          Ensure string is printable            *
*************************************************/

/* This function is called for critical strings. It checks for any
non-printing characters, and if any are found, it makes a new copy
of the string with suitable escape sequences. The old string is
freed if the flag is true. */

char *string_printing(char *s, BOOL free_old)
{
int nonprintcount = 0;
int length = 0;
unsigned char *t = (unsigned char *)s;
char *ss, *tt;

while (*t != 0)
  {
  length++;
  if (!isprint(*t++)) nonprintcount++;
  }

if (nonprintcount == 0) return s;

/* Get a new block of store guaranteed big enough to hold the
expanded string. */

ss = store_malloc(length + nonprintcount * 4 + 1);

/* Copy everying, escaping non printers. The unsigned char thing is
for systems that have signed chars by default. */

t = (unsigned char *)s;
tt = ss;

while (*t != 0)
  {
  if (isprint(*t)) *tt++ = *t++; else
    {
    *tt++ = '\\';
    switch (*t)
      {
      case '\n': *tt++ = 'n'; break;
      case '\r': *tt++ = 'r'; break;
      case '\b': *tt++ = 'b'; break;
      case '\t': *tt++ = 't'; break;
      case '\v': *tt++ = 'v'; break;
      case '\f': *tt++ = 'f'; break;
      default: sprintf(tt, "%03o", *t); tt += 3; break;
      }
    t++;
    }
  }
*tt = 0;

/* Free the old if requested, and return the new. */

if (free_old) store_free(s);
return ss;
}




/*************************************************
*            Copy and save string                *
*************************************************/

char *string_copy(char *s)
{
char *ss = (char *)store_malloc((int)strlen(s) + 1);
strcpy(ss, s);
return ss;
}



/*************************************************
*       Copy, lowercase and save string          *
*************************************************/

char *string_copylc(char *s)
{
char *ss = (char *)store_malloc((int)strlen(s) + 1);
char *p = ss;
while (*s != 0) *p++ = tolower(*s++);
*p = 0;
return ss;
}



/*************************************************
*       Copy and save string, given length       *
*************************************************/

/* It is assumed the data contains no zeros. A zero is added
onto the end. */

char *string_copyn(char *s, int n)
{
char *ss = (char *)store_malloc(n + 1);
strncpy(ss, s, n);
ss[n] = 0;
return ss;
}


/*************************************************
* Copy, lowercase, and save string, given length *
*************************************************/

/* It is assumed the data contains no zeros. A zero is added
onto the end. */

char *string_copynlc(char *s, int n)
{
char *ss = (char *)store_malloc(n + 1);
char *p = ss;
while (n-- > 0) *p++ = tolower(*s++);
*p = 0;
return ss;
}



/*************************************************
*          Format a string and save it           *
*************************************************/

char *string_sprintf(char *format, ...)
{
char buffer[8192];
va_list ap;
va_start(ap, format);
vsprintf(buffer, format, ap);
va_end(ap);
return string_copy(buffer);
}



/*************************************************
*         Case-independent strncmp() function    *
*************************************************/

int strncmpic(char *s, char *t, int n)
{
while (n--)
  {
  int c = tolower(*s++) - tolower(*t++);
  if (c) return c;
  }
return 0;
}


/*************************************************
*         Case-independent strcmp() function     *
*************************************************/

int strcmpic(char *s, char *t)
{
while (*s != 0)
  {
  int c = tolower(*s++) - tolower(*t++);
  if (c) return c;
  }
return *t;
}


/*************************************************
*         Case-independent strstr() function     *
*************************************************/

/* The third argument specifies whether whitespace is required
to follow the matched string. */

char *strstric(char *s, char *t, int space_follows)
{
char *p = t;
char *yield = NULL;
int cl = tolower(*p);
int cu = toupper(*p);

while (*s)
  {
  if (*s == cl || *s == cu)
    {
    if (yield == NULL) yield = s;
    if (*(++p) == 0)
      {
      if (!space_follows || s[1] == ' ' || s[1] == '\n' ) return yield;
      yield = NULL;
      p = t;
      }
    cl = tolower(*p);
    cu = toupper(*p);
    }
  else if (yield != NULL)
    {
    yield = NULL;
    p = t;
    cl = tolower(*p);
    cu = toupper(*p);
    }
  s++;
  }
return NULL;
}




/*************************************************
*       Get next string from separated list      *
*************************************************/

static unsigned char *listptr = NULL;
static char returnstring[1024];

char *string_nextinlist(char *list, int separator)
{
int p = 0;
unsigned char *s;

if (list != NULL) listptr = (unsigned char *)list;
if (listptr == NULL || listptr[0] == 0) return NULL;

for (s = listptr; *s != 0 && *s != separator; s++)
  {
  if (p < 1023) returnstring[p++] = *s;
  }
returnstring[p] = 0;
if (*s == separator) s++;

listptr = s;
return returnstring;
}




/*************************************************
*             Add chars to string                *
*************************************************/

/* This function is used when building up strings of unknown length.
The first argument points to the string, with the second giving its
actual size in bytes. The third is the offset at which len characters
pointed to by s are to be added. There must always be room left for
a terminating zero byte. The yield is a pointer to the string, possibly
changed if a copy to a larger bit of store happened. */

char *string_cat(char *string, int *size, int *ptr, char *s, int len)
{
int p = *ptr;
if (p + len >= *size)
  {
  char *newstring;
  while (*size <= p + len) *size += 50;
  newstring = (char *)store_malloc(*size);
  memcpy(newstring, string, p);
  store_free(string);
  string = newstring;
  }
strncpy(string + p, s, len);
*ptr = p + len;
return string;
}

/* End of string.c */
