/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions for handling an incoming SMTP call. */


#include "exim.h"


#define cmd_buffer_size 512      /* Ref. RFC 821 */


/* Structure for SMTP command list */

typedef struct {
  char *name;
  int len;
  int cmd;
} smtp_cmd_list;

/* Codes for identifying commands */

enum { HELO_CMD, EHLO_CMD, MAIL_CMD, RCPT_CMD, DATA_CMD, VRFY_CMD,
  EXPN_CMD, QUIT_CMD, RSET_CMD, NOOP_CMD, DEBUG_CMD, HELP_CMD,
  EOF_CMD, OTHER_CMD };

/* Local variables for remembering whether a calling host is permitted to
use unqualified sender or recipient addresses. */

static BOOL allow_unqualified_sender = FALSE;
static BOOL allow_unqualified_recipient = FALSE;



/*************************************************
*                Local static variables          *
*************************************************/

static char *smtp_data;
static FILE *smtp_in;
static FILE *smtp_out;

static char cmd_buffer[cmd_buffer_size + 1];

static smtp_cmd_list cmd_list[] = {
  "helo",       sizeof("helo")-1,       HELO_CMD,
  "ehlo",       sizeof("ehlo")-1,       EHLO_CMD,
  "mail from:", sizeof("mail from:")-1, MAIL_CMD,
  "rcpt to:",   sizeof("rcpt to:")-1,   RCPT_CMD,
  "data",       sizeof("data")-1,       DATA_CMD,
  "vrfy",       sizeof("vrfy")-1,       VRFY_CMD,
  "expn",       sizeof("expn")-1,       EXPN_CMD,
  "quit",       sizeof("quit")-1,       QUIT_CMD,
  "rset",       sizeof("rset")-1,       RSET_CMD,
  "noop",       sizeof("noop")-1,       NOOP_CMD,
  "debug",      sizeof("debug")-1,      DEBUG_CMD,
  "help",       sizeof("help")-1,       HELP_CMD };

static smtp_cmd_list *cmd_list_end =
  cmd_list + sizeof(cmd_list)/sizeof(smtp_cmd_list);



/*************************************************
*          SMTP command read timeout             *
*************************************************/

static void command_timeout_handler(int sig)
{
if (!smtp_batched_input)
  {
  DEBUG(3) debug_printf("421 %s: SMTP command timeout - closing connection\n",
    primary_hostname);
  fprintf(smtp_out, "421 %s: SMTP command timeout - closing connection\r\n",
    primary_hostname);
  fflush(smtp_out);
  }
log_write(LOG_MAIN, "SMTP command timeout%s%s",
  (sender_fullhost != NULL)? " while connected to " : "",
  (sender_fullhost != NULL)? sender_fullhost : "");
exit(EXIT_FAILURE);
}



/*************************************************
*               SIGTERM received                 *
*************************************************/

static void command_sigterm_handler(int sig)
{
if (!smtp_batched_input)
  {
  DEBUG(3) debug_printf("421 %s: Service not available - closing connection\n",
    primary_hostname);
  fprintf(smtp_out, "421 %s: Service not available - closing connection\r\n",
    primary_hostname);
  }
log_write(LOG_MAIN, "SMTP connection closed after SIGTERM%s%s",
  sender_fullhost? " while connected to " : "",
  sender_fullhost? sender_fullhost : "");
exit(EXIT_FAILURE);
}



/*************************************************
*           Read one command line                *
*************************************************/

/* The line ends up with [CR]LF removed from its end. If
we get an overlong line, treat as an unknown command. Carry on
in the face of interrupts. (There appears to be no standard way
of disabling them.) */

static int smtp_read_command(void)
{
int c;
int ptr = 0;
smtp_cmd_list *p;

alarm(smtp_receive_timeout);

while ((c = getc(smtp_in)) != '\n' && c != EOF)
  {
  if (ptr >= cmd_buffer_size) { alarm(0); return OTHER_CMD; }
  cmd_buffer[ptr++] = c;
  }
alarm(0);

/* If hit end of file, return pseudo EOF command. Whether we have a
part-line already read doesn't matter, since this is an error state. */

if (c == EOF) return EOF_CMD;

/* Remove any CR at the end of the line, and terminate the string. */

if (ptr > 0 && cmd_buffer[ptr-1] == '\r') ptr--;
cmd_buffer[ptr] = 0;

DEBUG(3) debug_printf("%s\n", cmd_buffer);

/* Scan command list and return identity, having set the data pointer
to the start of the actual data characters. */

for (p = cmd_list; p < cmd_list_end; p++)
  {
  if (strncmpic(cmd_buffer, p->name, p->len) == 0)
    {
    smtp_data = cmd_buffer + p->len;
    while (isspace(*smtp_data)) smtp_data++;
    return p->cmd;
    }
  }

return OTHER_CMD;
}





/*************************************************
*  Initialize for incoming batched SMTP message  *
*************************************************/

/* This function is called from smtp_setup_msg() in the case when
smtp_batched_input is true. This happens when -bS is used to pass a whole batch
of messages in one file with SMTP commands between them. All errors must be
reported by sending a message, and only MAIL FROM, RCPT TO, and DATA are
relevant. */

static int smtp_setup_batch_msg(FILE *fin)
{
int done = 0;

if (feof(fin)) return 0;   /* Treat EOF as QUIT */

received_protocol = "bsmtp";

/* Deal with SMTP commands. The reading routine sets up a timeout
for each one. If the timeout happens, or we get SIGTERM, exim just
gives up and dies. */

smtp_in  = fin;
signal(SIGALRM, command_timeout_handler);
signal(SIGTERM, command_sigterm_handler);

/* This loop is exited by setting done to a POSITIVE value. The values
are 2 larger than the required yield of the function. */

while (done <= 0)
  {
  char *errmess;
  char *receiver = NULL;
  char *orig_sender = NULL;
  char *orig_receiver = NULL;
  int errcode, start, end, domain;

  switch(smtp_read_command())
    {
    /* The HELO/EHLO commands are simply ignored, except that they do
    a reset of the state. */

    case HELO_CMD:
    case EHLO_CMD:
    case RSET_CMD:
    accept_free_recipients();
    sender_address = NULL;
    break;


    /* The MAIL FROM command requires an address as an operand. All we
    do here is to parse it for syntactic correctness. The form "<>" is
    a special case which converts into an empty string. The start/end
    pointers in the original are not used further for this address, as
    it is the canonical extracted address which is all that is kept. */

    case MAIL_CMD:
    if (sender_address != NULL)
      {
      moan_smtp_batch("503 Sender already given");
      break;
      }

    if (smtp_data[0] == 0)
      {
      moan_smtp_batch("501 MAIL FROM must have an address operand");
      break;
      }

    /* The TRUE flag allows "<>" as a sender address */

    orig_sender =
      parse_extract_address(smtp_data, &errmess, &start, &end, &domain, TRUE);
    if (orig_sender == NULL)
      {
      moan_smtp_batch("501 %s: %s", smtp_data, errmess);
      break;
      }
    sender_address = string_copy(orig_sender);

    /* Qualify unqualified sender addresses. There doesn't seem much point
    in causing trouble here. */

    if (domain == 0 && sender_address[0] != 0 && sender_address[0] != '@')
      {
      sender_address = rewrite_address_qualify(sender_address, FALSE);
      DEBUG(9) debug_printf("unqualified address %s accepted\n",
        orig_sender);
      }

    /* If configured to check sender addresses, do the preliminary check
    now, unless the sender is local (in which case whatever is given here
    is ignored anyway). The check will fail if the message is to be refused at
    this stage. Another check function is called after the message has been
    received, to do more checking when the headers are available. */

    errmess = NULL;
    if (sender_verify || sender_try_verify)
      {
      if (!sender_local && !verify_sender_preliminary(&errcode, &errmess))
        {
        moan_smtp_batch("%d rejected: %s %s\n", errcode, errmess,
          orig_sender);
        log_write(LOG_MAIN|LOG_REJECT, "Rejected in SMTP batch: %s %s",
          errmess, orig_sender);
        store_free(sender_address);
        sender_address = NULL;
        break;
        }
      }

    /* RFC 821 says MAIL FROM resets state at start of message */
    accept_free_recipients();
    break;


    /* The RCPT TO command requires an address as an operand. All we do
    here is to parse it for syntactic correctness. There may be any number
    of RCPT TO commands, specifying multiple senders. We build them all into
    a data structure that is in argc/argv format. The start/end values
    given by parse_extract_address are not used, as we keep only the
    extracted address. */

    case RCPT_CMD:
    if (sender_address == NULL)
      {
      moan_smtp_batch("503 No sender yet given");
      break;
      }

    if (smtp_data[0] == 0)
      {
      moan_smtp_batch("501 RCPT TO must have an address operand");
      break;
      }

    /* Don't allow "<>" as a recipient address */

    orig_receiver =
      parse_extract_address(smtp_data, &errmess, &start, &end, &domain, FALSE);
    if (orig_receiver == NULL)
      {
      moan_smtp_batch("501 %s: %s", smtp_data, errmess);
      break;
      }
    receiver = string_copy(orig_receiver);

    /* If the receiver address is unqualified, qualify it. There doesn't seem
    much point in causing trouble here. */

    if (domain == 0 && receiver[0] != '@')
      {
      DEBUG(9) debug_printf("unqualified address %s accepted\n",
        receiver);
      receiver = rewrite_address_qualify(receiver, TRUE);
      }

    /* Add to the list of receivers, and set value to NULL to prevent
    freeing. */

    accept_add_recipient(receiver);
    receiver = NULL;
    break;


    /* The DATA command is legal only if it follows successful MAIL FROM
    and RCPT TO commands. This function is complete when a valid DATA
    command is encountered. */

    case DATA_CMD:
    if (sender_address == NULL)
      {
      moan_smtp_batch("503 MAIL FROM command must precede DATA");
      break;
      }
    if (recipients_count <= 0)
      {
      moan_smtp_batch("503 Valid RCPT TO <recipient> must precede DATA");
      break;
      }
    done = 3;
    break;


    /* The VRFY, EXPN, HELP, and DEBUG commands are ignored. */

    case VRFY_CMD:
    case EXPN_CMD:
    case HELP_CMD:
    case DEBUG_CMD:
    case NOOP_CMD:
    break;


    case QUIT_CMD:
    case EOF_CMD:
    accept_free_recipients();
    done = 2;
    break;


    default:
    moan_smtp_batch("500 Command unrecognized: %s", cmd_buffer);
    break;
    }

  /* Free temporary store. */

  if (orig_sender != NULL) store_free(orig_sender);
  if (orig_receiver != NULL) store_free(orig_receiver);
  if (receiver != NULL) store_free(receiver);
  }

/* Reset the signal handlers used in this function, and if no
message is in progress, ensure the store is cleaned up. */

signal(SIGALRM, SIG_DFL);
signal(SIGTERM, SIG_DFL);

if (done < 3) accept_free_recipients();
return done - 2;  /* Convert yield values */
}




/*************************************************
*          Build host+ident message              *
*************************************************/

/* Used when logging rejections below. */

static char *host_and_ident(void)
{
if (sender_ident == NULL)
  sprintf(big_buffer, "%s", sender_fullhost);
else
  sprintf(big_buffer, "%s (%s)", sender_fullhost, sender_ident);
return big_buffer;      
}





/*************************************************
*       Initialize for SMTP incoming message     *
*************************************************/

/* This function conducts the initial dialogue at the start of an incoming SMTP
message, and builds a list of recipients. However, if the incoming message
is part of a batch (-bS option) a separate function is called since it would
be messy having tests splattered about all over this function. This function
therefore handles the case where interaction is occurring.

If the first flag is true, it is prepared to handle HELO or EHLO at the start
of the dialogue, causing a setting of the sender_host* variables. The first
flag also resets memory of permission to handle unqualified senders and
recipients.

The global recipients_list is set to point to a vector of string pointers,
whose number is given by recipients_count. The global variable sender_address
is set to the sender's address. The yield is +1 if a message has been
successfully started, 0 if a QUIT command was encountered or the connection was
refused from the particular host, or -1 if the connection was lost.

If there are host accept/reject configuration settings, this is where
they are checked. */

int smtp_setup_msg(FILE *fin, FILE *fout, BOOL first)
{
int done = 0;

accept_free_recipients();
sender_address = NULL;

if (smtp_batched_input) return smtp_setup_batch_msg(fin);

/* Default, changed to esmtp later if we get EHLO. */

if (first) received_protocol = "smtp";

/* Check for forbidden hosts and reject the call if it is one of them, if this
is the first call to this function for a given connection. This checking
doesn't apply when messages are input locally via the -bs option, in which case
sender_host_unknown is set. When -bs is used from inetd, this flag is not set,
causing the sending host to be checked. Batched input, using the -bS option,
calls a different function.

(1) If sender_{host,net}_accept is set, accept only from those hosts/nets,
otherwise accept from any host that is not in sender_{host,net}_reject. There
can be ident values associated with any host.

(2) If smtp_accept_max and smtp_accept_reserve are set, keep some connections
in reserve for certain hosts and/or networks. */

if (first && !sender_host_unknown)
  {
  if (sender_host_accept != NULL || sender_net_accept != NULL)
    {
    if (!verify_check_host(sender_host_accept, &sender_host_accept_hosts) &&
        !verify_check_net(sender_net_accept, &sender_net_accept_nets))
      {
      log_write(LOG_MAIN, "rejecting connection from %s: host not in accept "
        "list", host_and_ident());
      DEBUG(3) debug_printf("554 SMTP service not available\n");
      fprintf(fout, "554 SMTP service not available\r\n");
      return 0;
      }
    }

  if (sender_host_reject != NULL || sender_net_reject != NULL)
    {
    if (verify_check_host(sender_host_reject, &sender_host_reject_hosts) ||
        verify_check_net(sender_net_reject, &sender_net_reject_nets))
      {
      log_write(LOG_MAIN, "rejecting connection from %s: host in reject list",
        host_and_ident());
      DEBUG(3) debug_printf("554 SMTP service not available\n");
      fprintf(fout, "554 SMTP service not available\r\n");
      return 0;
      }
    }

  /* Check for reserved slots. Note that the count value doesn't include
  this process, as it gets upped in the parent process. */

  if (smtp_accept_max > 0 &&
      smtp_accept_count + 1 > smtp_accept_max - smtp_accept_reserve)
    {
    if (!verify_check_host(smtp_reserve_hosts, &smtp_reserve_hostlist) &&
        !verify_check_net(smtp_reserve_nets, &smtp_reserve_netlist))
      {
      log_write(LOG_MAIN, "rejecting connection from %s: not in reserve list: "
        "connected=%d max=%d reserve=%d", host_and_ident(),
        smtp_accept_count, smtp_accept_max, smtp_accept_reserve);
      DEBUG(3) debug_printf("421 %s: Too many concurrent SMTP connections; "
        "please try again later\n", primary_hostname);
      fprintf(fout, "421 %s: Too many concurrent SMTP connections; "
        "please try again later\r\n", primary_hostname);
      return 0;
      }
    }
  }

/* Output the initial message for an SMTP connection. It may contain
newlines, which then cause a multi-line response to be given. Also, reset
the unqualified permission flags. */

if (first)
  {
  char *s = expand_string(smtp_banner);
  char *p = s;

  if (s == NULL)
    log_write(LOG_PANIC_DIE, "Expansion of \"%s\" (smtp_banner) failed",
      smtp_banner);

  while (*p != 0)
    {
    int c;
    DEBUG(3) debug_printf("220%c", (strchr(p, '\n') == NULL)? ' ' : '-');
    fprintf(fout, "220%c", (strchr(p, '\n') == NULL)? ' ' : '-');
    while ((c = *p) != 0)
      {
      p++; /* NB can't include in while because used in previous while */
      if (c == '\n') break;
      DEBUG(3) debug_printf("%c", c);
      fputc(c, fout);
      }
    DEBUG(3) debug_printf("\n");
    putc('\r', fout);
    putc('\n', fout);
    }

  store_free(s);
  fflush(fout);

  allow_unqualified_sender = allow_unqualified_recipient = FALSE;
  }

/* Now deal with SMTP commands. The reading routine sets up a timeout
for each one. If the timeout happens, or we get SIGTERM, exim just
gives up and dies. */

smtp_in  = fin;
smtp_out = fout;

signal(SIGALRM, command_timeout_handler);
signal(SIGTERM, command_sigterm_handler);

/* This loop is exited by setting done to a POSITIVE value. The values
are 2 larger than the required yield of the function. */

while (done <= 0)
  {
  char *errmess;
  char *receiver = NULL;
  char *orig_sender = NULL;
  char *orig_receiver = NULL;
  char *hello = NULL;
  int multiline = ' ';
  int errcode, start, end, domain;

  switch(smtp_read_command())
    {
    /* The HELO/EHLO commands are permitted to appear in the middle of
    a session as well as at the beginning. They have the effect of a
    reset in addition to their other functions. Their absence at the
    start cannot be taken to be an error. */

    case HELO_CMD:
    hello = "HELO";
    received_protocol = "smtp";      /* could be resetting in principle */
    multiline = ' ';
    /* fall through with hello != NULL */

    case EHLO_CMD:
    if (hello == NULL)
      {
      hello = "EHLO";
      received_protocol = "esmtp";
      multiline = '-';
      }

    /* Ensure an argument is supplied */

    if (smtp_data[0] == 0)
      {
      DEBUG(3) debug_printf("501 %s must have a domain name operand\n", hello);
      fprintf(fout, "501 %s must have a domain name operand\r\n", hello);
      break;
      }

    /* Generate an OK reply, including the ident if present, and also
    the IP address if present. Reflecting back the ident is intended
    as a deterrent to mail forgers. */

    fprintf(fout, "250%c%s: Hello %s%s%s", multiline, primary_hostname,
      (sender_ident == NULL)?  "" : sender_ident,
      (sender_ident == NULL)?  "" : " at ",
      smtp_data);

    DEBUG(3) debug_printf("250%c%s: Hello %s%s%s", multiline, primary_hostname,
      (sender_ident == NULL)?  "" : sender_ident,
      (sender_ident == NULL)?  "" : " at ",
      smtp_data);

    if (sender_host_address != NULL)
      {
      fprintf(fout, " [%s]", sender_host_address);
      DEBUG(3) debug_printf(" [%s]", sender_host_address);
      }

    fprintf(fout, "\r\n");
    DEBUG(3) debug_printf("\n");

    /* If we received EHLO, we have started a multiline response. Finish it
    off with the functions supported. Currently not much! */

    if (multiline == '-')
      {
      fprintf(fout, "250-Supported functions are:\r\n");
      DEBUG(3) debug_printf("250-Supported functions are:\n");

      /* I'm not entirely happy with this, as an MTA is supposed to check
      that it has enough room to accept a message of maximum size before
      it sends this. However, there seems little point in not sending it. */

      if (message_size_limit > 0)
        {
        fprintf(fout, "250-SIZE %d\r\n", message_size_limit);
        DEBUG(3) debug_printf("250-SIZE %d\n", message_size_limit);
        }

      fprintf(fout, "250 HELP\r\n");
      DEBUG(3) debug_printf("250 HELP\n");
      }

    fflush(fout);

    /* If sender_host_unknown is true, we have got here via the -bs interface,
    not called from inetd. In this case, HELO/EHLO should *not* set the host
    name. Otherwise, we are running an IP connection and the host address will
    be set. */

    if (!sender_host_unknown)
      {
      if (sender_host_name != NULL) store_free(sender_host_name);
      sender_host_name = store_malloc((int)strlen(smtp_data) + 1);
      strcpy(sender_host_name, smtp_data);

      if (sender_fullhost != NULL) store_free(sender_fullhost);
      sender_fullhost =
        store_malloc((int)strlen(sender_host_name) +
        (int)strlen(sender_host_address) + 4);
      sprintf(sender_fullhost, "%s [%s]", sender_host_name,
        sender_host_address);
      }

    /* Ensure we are in the reset state */

    accept_free_recipients();
    sender_address = NULL;
    break;


    /* The MAIL FROM command requires an address as an operand. All we
    do here is to parse it for syntactic correctness. The form "<>" is
    a special case which converts into an empty string. The start/end
    pointers in the original are not used further for this address, as
    it is the canonical extracted address which is all that is kept. */

    case MAIL_CMD:
    if (sender_address != NULL)
      {
      DEBUG(3) debug_printf("503 Sender already given\n");
      fprintf(fout, "503 Sender already given\r\n");
      break;
      }

    if (smtp_data[0] == 0)
      {
      DEBUG(3) debug_printf("501 MAIL FROM must have an address operand\n");
      fprintf(fout, "501 MAIL FROM must have an address operand\r\n");
      break;
      }
      
    /* If this session was initiated with EHLO and message_size_limit is
    non-zero, Exim will have indicated that it supports the SIZE option.
    The remote is then permitted to add "SIZE=n" on the end of the MAIL
    FROM command. Just parse this out by ad hoc code for the moment. We
    know the command line starts "MAIL FROM", so the backwards searches
    will always terminate without the need for explicit stops. */
    
    if (message_size_limit > 0 && strcmp(received_protocol, "esmtp") == 0)
      {
      char *p = smtp_data + (int)strlen(smtp_data);
      while (isspace(p[-1])) *(--p) = 0; 
      while (isdigit(p[-1])) p--;
      if (*p != 0 && strncmpic(p-5, "SIZE=", 5) == 0)
        {
        int size = atoi(p); 
        p[-5] = 0; 
        if (size > message_size_limit)
          {
          DEBUG(3) debug_printf("552 Message size exceeds maximum permitted\n"); 
          fprintf(fout, "552 Message size exceeds maximum permitted\r\n");
          break; 
          }     
        }  
      }  

    /* Now extract the address. The TRUE flag allows "<>" as a sender 
    address. */

    orig_sender =
      parse_extract_address(smtp_data, &errmess, &start, &end, &domain, TRUE);
    if (orig_sender == NULL)
      {
      DEBUG(3) debug_printf("501 %s: %s\n", smtp_data, errmess);
      fprintf(fout, "501 %s: %s\r\n", smtp_data, errmess);
      break;
      }
    sender_address = string_copy(orig_sender);

    /* If sender_address is unqualified, reject it, unless this is a
    locally generated message, in which case it will be ignored anyway.
    However, if the sending host or net is listed as permitted to send
    unqualified addresses - typically local machines behaving as MUAs -
    then just qualify the address. Run the check only for the first message
    in a connection. */

    if (!sender_local && domain == 0 && sender_address[0] != 0 &&
        sender_address[0] != '@')
      {
      if (!allow_unqualified_sender && first) allow_unqualified_sender =
        verify_check_host(sender_unqualified_hosts,
          &sender_unqualified_hostlist) ||
        verify_check_net(sender_unqualified_nets,
          &sender_unqualified_netlist);

      if (allow_unqualified_sender)
        {
        domain = (int)strlen(sender_address) + 1;
        sender_address = rewrite_address_qualify(sender_address, FALSE);
        DEBUG(9) debug_printf("unqualified address %s accepted\n",
          orig_sender);
        }
      else
        {
        DEBUG(3) debug_printf("501 %s: sender address must contain a domain\n",
          smtp_data);
        fprintf(fout, "501 %s: sender address must contain a domain\r\n",
          smtp_data);
        store_free(sender_address);
        sender_address = NULL;
        break;
        }
      }

    /* If configured to reject any senders explicitly (spam filtering),
    check now. This is independent of sender verification, and does not
    happen for local senders. */

    if (sender_reject != NULL && !sender_local)
      {
      int llen = 0;
      re_block **chain_ad = &re_sender_reject;
      BOOL match = TRUE;
      char *spammer;
      char *localpart = sender_address + domain - 1;

      /* Get a pointer to the local part, and its length, by moving
      backwards from the domain pointer. This gets the final local part
      and domain from source-routed addresses. */

      while (localpart > sender_address && localpart[-1] != ':')
        {
        localpart--;
        llen++;
        }

      /* Loop through all the entries in the reject list. */

      for (spammer = string_nextinlist(sender_reject, ':');
           spammer != NULL;
           spammer = string_nextinlist(NULL, ':'))
        {
        /* Handle a regular expression, which must match the entire
        incoming address. */

        if (spammer[0] == '^')
          {
          match = match_string(sender_address, spammer, chain_ad);
          chain_ad = &((*chain_ad)->next);
          }

        /* If not a regular expression, either part may begin with an
        asterisk, and both parts must match. If there's no '@' in the
        pattern, then it is just a domain and treated as if it had
        *@ on the front. */

        else
          {
          int sllen;
          char *slocalpart = spammer;
          char *sdomain = strchr(spammer, '@');

          /* No @ => assume user matches */

          if (sdomain == NULL)
            {
            sdomain = spammer;
            sllen = 0;
            match = TRUE;
            }
          else
            {
            sllen = sdomain - spammer;
            sdomain += 1;
            }

          /* Check the local part if one is given in the list */

          if (sllen > 0)
            {
            if (slocalpart[0] == '*')
              {
              int cllen = sllen - 1;
              match = llen >= cllen &&
                strncmpic(localpart + llen - cllen, slocalpart + 1, cllen) == 0;
              }
            else match =
              llen == sllen && strncmpic(localpart, slocalpart, llen) == 0;
            }

          /* If the local part matched, check the domain using the generalized
          function, which supports file lookups. */

          if (match)
            match = match_string(sender_address + domain, sdomain, NULL);
          }

        /* If sender_address matched a prohibited sender, reject. */

        if (match)
          {
          DEBUG(3) debug_printf("501 rejected: administrative prohibition\n");
          fprintf(fout, "501 rejected: administrative prohibition\r\n");
          log_write(LOG_MAIN|LOG_REJECT, "Rejected%s%s: administrative "
            "prohibition: %s",
            (sender_fullhost != NULL)? " from " : "",
            (sender_fullhost != NULL)? sender_fullhost : "",
            orig_sender);
          store_free(sender_address);
          sender_address = NULL;
          break;          /* Ends the string list loop */
          }
        }
      if (match) break;   /* Ends the case statement */
      }

    /* If configured to check sender addresses, do the preliminary check
    now, unless the sender is local (in which case whatever is given here
    is ignored anyway). The check will fail if the message is to be refused at
    this stage. Another check function is called after the message has been
    received, to do more checking when the headers are available. */

    errmess = NULL;
    if (sender_verify || sender_try_verify)
      {
      if (!sender_local && !verify_sender_preliminary(&errcode, &errmess))
        {
        DEBUG(3) debug_printf("%d rejected: %s %s\n", errcode, errmess,
          orig_sender);
        fprintf(fout, "%d rejected: %s %s\r\n", errcode, errmess,
          orig_sender);
        log_write(LOG_MAIN|LOG_REJECT, "Rejected%s%s: %s <%s>",
          (sender_fullhost != NULL)? " from " : "",
          (sender_fullhost != NULL)? sender_fullhost : "",
          errmess,
          orig_sender);
        store_free(sender_address);
        sender_address = NULL;
        break;
        }
      }

    /* The sender address is acceptable. For now. If verification is running
    in warning mode, errmess is set non-NULL if there is a warning to be
    given. If the address has been verified, a new sender address is always
    produced on success. However, we reflect the original one to the outside
    world. */

    if (errmess != NULL)
      {
      DEBUG(3) debug_printf("%d %s %s\n", errcode, errmess, orig_sender);
      fprintf(fout, "%d %s %s\r\n", errcode, errmess, orig_sender);
      log_write(LOG_MAIN|LOG_REJECT, "%s %s%s%s", errmess, orig_sender,
        (sender_fullhost != NULL)? " received from " : "",
        (sender_fullhost != NULL)? sender_fullhost : "");
      }
    else
      {
      DEBUG(3) debug_printf("250 <%s> is syntactically correct\n",
        orig_sender);
      fprintf(fout, "250 <%s> is syntactically correct\r\n", orig_sender);
      }

    /* RFC 821 says MAIL FROM resets state at start of message */
    accept_free_recipients();
    break;


    /* The RCPT TO command requires an address as an operand. All we do
    here is to parse it for syntactic correctness. There may be any number
    of RCPT TO commands, specifying multiple senders. We build them all into
    a data structure that is in argc/argv format. The start/end values
    given by parse_extract_address are not used, as we keep only the
    extracted address. */

    case RCPT_CMD:
    if (sender_address == NULL)
      {
      DEBUG(3) debug_printf("503 No sender yet given\n");
      fprintf(fout, "503 No sender yet given\r\n");
      break;
      }

    if (smtp_data[0] == 0)
      {
      DEBUG(3) debug_printf("501 RCPT TO must have an address operand\n");
      fprintf(fout, "501 RCPT TO must have an address operand\r\n");
      break;
      }

    /* Don't allow "<>" as a recipient address */

    orig_receiver =
      parse_extract_address(smtp_data, &errmess, &start, &end, &domain, FALSE);
    if (orig_receiver == NULL)
      {
      DEBUG(3) debug_printf("501 %s: %s\n", smtp_data, errmess);
      fprintf(fout, "501 %s: %s\r\n", smtp_data, errmess);
      break;
      }
    receiver = string_copy(orig_receiver);

    /* If the receiver address is unqualified, reject it, unless this is a
    locally generated message. However, unqualified addresses are permitted
    from a configured list of hosts and nets - typically when behaving as
    MUAs rather than MTAs. Sad that SMTP is used for both types of traffic,
    really. As a message may have many recipients, and indeed an SMTP call
    may have many messages, we remember that a host is on the permitted list
    to avoid unnecessary double checks. */

    if (!sender_local && domain == 0 && receiver[0] != '@')
      {
      if (!allow_unqualified_recipient && first) allow_unqualified_recipient =
        verify_check_host(receiver_unqualified_hosts,
          &receiver_unqualified_hostlist) ||
        verify_check_net(receiver_unqualified_nets,
          &receiver_unqualified_netlist);

      if (allow_unqualified_recipient)
        {
        DEBUG(9) debug_printf("unqualified address %s accepted\n",
          receiver);
        receiver = rewrite_address_qualify(receiver, TRUE);
        }
      else
        {
        DEBUG(3)
          debug_printf("501 %s: recipient address must contain a domain\n",
            smtp_data);
        fprintf(fout, "501 %s: recipient address must contain a domain\r\n",
          smtp_data);
        break;
        }
      }

    /* If configured to check the receiver address now, do so. */

    if (receiver_verify || receiver_try_verify)
      {
      BOOL receiver_local;
      int rc = verify_address(receiver, TRUE, FALSE, NULL, &receiver_local,
        NULL, FALSE, FALSE);

      /* Failure causes a hard error */

      if (rc == FAIL)
        {
        if (receiver_local)
          {
          DEBUG(3) debug_printf("550 Unknown local part in <%s>\n",
            orig_receiver);
          fprintf(fout, "550 Unknown local part in <%s>\r\n", orig_receiver);
          }
        else
          {
          DEBUG(3) debug_printf("550 Cannot route to <%s>\n", orig_receiver);
          fprintf(fout, "550 Cannot route to <%s>\r\n", orig_receiver);
          }
        log_write(LOG_MAIN, "verify failed for SMTP recipient %s%s%s",
          orig_receiver,
          (sender_fullhost == NULL)? "" : " from ",
          (sender_fullhost == NULL)? "" : sender_fullhost);
        break;     /* End of handling the RCPT TO command */
        }

      /* If verification can't be done now, give a temporary error unless
      receiver_try_verify is set, in which case accept the address, but say 
      it's unverified. */

      if (rc == DEFER)
        {
        if (!receiver_try_verify)
          {
          DEBUG(3) debug_printf("451 Cannot check <%s> at this time - "
            "please try later\n", orig_receiver);
          fprintf(fout, "451 Cannot check <%s> at this time - "
            "please try later\r\n", orig_receiver);
          break;   /* End of handling the RCPT TO command */
          }

        DEBUG(3) debug_printf("250 Cannot check <%s> at this time - "
          "accepted unverified\n", orig_receiver);
        fprintf(fout, "250 Cannot check <%s> at this time - "
          "accepted unverified\r\n", orig_receiver);
        }

      /* Verification succeeded */

      else
        {
        DEBUG(3) debug_printf("250 <%s> verified\n", orig_receiver);
        fprintf(fout, "250 <%s> verified\r\n", orig_receiver);
        }
      }

    /* Otherwise the receiver address is only known to be syntactically
    acceptable. Any delivery errors will happen later. */

    else
      {
      DEBUG(3) debug_printf("250 <%s> is syntactically correct\n",
        orig_receiver);
      fprintf(fout, "250 <%s> is syntactically correct\r\n", orig_receiver);
      }

    /* Add to the list of receivers, and set value to NULL to prevent
    freeing. */

    accept_add_recipient(receiver);
    receiver = NULL;
    break;


    /* The DATA command is legal only if it follows successful MAIL FROM
    and RCPT TO commands. This function is complete when a valid DATA
    command is encountered. */

    case DATA_CMD:
    if (sender_address == NULL)
      {
      DEBUG(3) debug_printf("503 MAIL FROM command must precede DATA\n");
      fprintf(fout, "503 MAIL FROM command must precede DATA\r\n");
      break;
      }
    if (recipients_count <= 0)
      {
      DEBUG(3) debug_printf("503 Valid RCPT TO <recipient> must precede DATA\n");
      fprintf(fout, "503 Valid RCPT TO <recipient> must precede DATA\r\n");
      break;
      }
    DEBUG(3) debug_printf("354 Enter message, ending with \".\" on a line by itself\n");
    fprintf(fout, "354 Enter message, ending with \".\" on a line by itself\r\n");
    done = 3;
    break;


    /* The VRFY command is enabled by a configuration option. Despite RFC1123
    it defaults disabled. */

    case VRFY_CMD:
    if (!smtp_verify)
      {
      DEBUG(3) debug_printf("502 VRFY command not available\n");
      fprintf(fout, "502 VRFY command not available\r\n");
      }

    /* When VRFY is enabled, it verifies only addresses that contain no domain
    or one of the local domains. However, we have to let the verify function
    and the routers and directors decide what is local. */

    else
      {
      BOOL is_local_domain;
      int rc;
      char *address = parse_extract_address(smtp_data, &errmess, &start, &end,
        &domain, FALSE);

      if (address == NULL)
        {
        DEBUG(3) debug_printf("501 %s\n", errmess);
        fprintf(fout, "501 %s\r\n", errmess);
        break;
        }

      rc = verify_address(address, TRUE, TRUE, NULL, &is_local_domain, NULL,
        FALSE, FALSE);

      if (!is_local_domain)
        {
        DEBUG(3) debug_printf("551 Not a local domain\n");
        fprintf(fout, "551 Not a local domain\r\n");
        }

      else switch (rc)
        {
        case OK:
        DEBUG(3) debug_printf("250 Deliverable\n");
        fprintf(fout, "250 Deliverable\r\n");
        break;

        case DEFER:
        DEBUG(3) debug_printf("450 Cannot resolve at this time\n");
        fprintf(fout, "450 Cannot resolve at this time\r\n");
        break;

        case FAIL:
        DEBUG(3) debug_printf("550 Not deliverable\n");
        fprintf(fout, "550 Not deliverable\r\n");
        break;
        }
      }
    break;


    case EXPN_CMD:
    DEBUG(3) debug_printf("502 EXPN command not available\n");
    fprintf(fout, "502 EXPN command not available\r\n");
    break;


    case QUIT_CMD:
    DEBUG(3) debug_printf("221 %s closing connection\n", primary_hostname);
    fprintf(fout, "221 %s closing connection\r\n", primary_hostname);
    accept_free_recipients();
    done = 2;
    break;


    case RSET_CMD:
    accept_free_recipients();
    sender_address = NULL;
    DEBUG(3) debug_printf("250 Reset OK\n");
    fprintf(fout, "250 Reset OK\r\n");
    break;


    case NOOP_CMD:
    DEBUG(3) debug_printf("250 OK\n");
    fprintf(fout, "250 OK\r\n");
    break;


    case DEBUG_CMD:
    DEBUG(3) debug_printf("500 No way!\n");
    fprintf(fout, "500 No way!\r\n");
    break;


    case HELP_CMD:
    DEBUG(3)
      {
      debug_printf("214-Commands supported:\n");
      debug_printf("214-    HELO EHLO MAIL RCPT DATA\n");
      debug_printf("214     NOOP QUIT RSET HELP %s\n",
        smtp_verify? "VRFY" : "");
      }
    fprintf(fout, "214-Commands supported:\r\n");
    fprintf(fout, "214-    HELO EHLO MAIL RCPT DATA\r\n");
    fprintf(fout, "214     NOOP QUIT RSET HELP %s\r\n",
      smtp_verify? "VRFY" : "");
    break;


    case EOF_CMD:
    DEBUG(3) debug_printf("421 %s lost input connection\n", primary_hostname);
    fprintf(fout, "421 %s lost input connection\r\n", primary_hostname);
    done = 1;
    break;


    default:
    DEBUG(3) debug_printf("500 Command unrecognized\n");
    fprintf(fout, "500 Command unrecognized\r\n");
    break;
    }

  /* Ensure output gets sent, and free temporary store. */

  fflush(fout);

  if (orig_sender != NULL) store_free(orig_sender);
  if (orig_receiver != NULL) store_free(orig_receiver);
  if (receiver != NULL) store_free(receiver);
  }

/* Reset the signal handlers used in this function, and if no
message is in progress, ensure the store is cleaned up. */

signal(SIGALRM, SIG_DFL);
signal(SIGTERM, SIG_DFL);

if (done < 3) accept_free_recipients();
return done - 2;  /* Convert yield values */
}

/* End of smtp_in.c */
