/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* A set of functions to search databases in various formats. An open database
is represented by a void * value. This set of functions has no private state. */

#include "exim.h"

#if HAVE_NIS
#include <rpcsvc/ypclnt.h>
#endif


/* Returns from the file credential check function. */

enum { check_ok, check_mode, check_user, check_group, check_fstat };


/* Place to cache lookups for efficiency. */

static char cached_filename[256] = "";
static char cached_keystring[256] = "";
static int  cached_type = -1;
static char *cached_data = NULL;



/*************************************************
*         Check a file's credentials             *
*************************************************/

/* fstat can normally be expected to work on an open file, but there are some
NFS states where it may not. */

static int check_file(int fd, int modemask, int *owners, int *owngroups)
{
struct stat statbuf;

if (fstat(fd, &statbuf) != 0) return check_fstat;

if ((statbuf.st_mode & S_IFMT) != S_IFREG ||
    (statbuf.st_mode & modemask) != 0) return check_mode;

if (owners != NULL)
  {
  int p = 0;
  BOOL uid_ok = FALSE;
  while (owners[p] >= 0)
    if (owners[p++] == statbuf.st_uid) { uid_ok = TRUE; break; }
  if (!uid_ok) return check_user;
  }

if (owngroups != NULL)
  {
  int p = 0;
  BOOL gid_ok = FALSE;
  while (owngroups[p] >= 0)
    if (owngroups[p++] == statbuf.st_gid) { gid_ok = TRUE; break; }
  if (!gid_ok) return check_group;
  }

return check_ok;
}



/*************************************************
*               Release cached resources         *
*************************************************/

/* When search_open is called, it can be handed the anchor of a tree in which
to keep any files that get opened for database lookups. These are closed only
when this tidyup routine is called, typically at the end of sections of code
where a number of lookups might occur.

To reduce the number of mallocs and frees, the store allocation of each tree
node and its attached data is done in one call, with the attached data first,
because its length is fixed (the tree node contains the key at the end).
Therefore, freeing the attached block frees the node as well. */

/* First, there is an internal, recursive subroutine. */

static tidyup_subtree(tree_node *t)
{
search_openfile *oo = (search_openfile *)(t->data.ptr);
if (t->left != NULL) tidyup_subtree(t->left);
if (t->right != NULL) tidyup_subtree(t->right);
search_close(oo->handle, oo->type);
store_free(oo);
}

/* The external entry point */

void search_tidyup(tree_node **tree)
{
if (*tree != NULL)
  {
  tidyup_subtree(*tree);
  *tree = NULL;
  }
}




/*************************************************
*             Open search database               *
*************************************************/

/* A mode, and lists of owners and groups are passed over for checking in
the cases where the database is one or more files. Return NULL, with a message
pointed to by message, in cases of error.

For search types that use a file or files, check up on the mode after opening.
It is tempting to do a stat before opening the file, and use it as an existence
check. However, doing that opens a small security loophole in that the status
could be changed before the file is opened. Can't quite see what problems this
might lead to, but you can't be too careful where security is concerned.
Fstat() on an open file can normally be expected to succeed, but there are some
NFS states where it does not.

If a pointer to an tree root is given, scan the tree to see if this file is
already open for the correct search type. If so, return the saved handle. If
not, put the handle in the tree for possible subsequent use. See search_tidyup
above for closing the cached files. */

void *search_open(char *filename, int search_type,
    int modemask, int *owners, int *owngroups, char **message, tree_node **tree)
{
int rc;
FILE *f;
EXIM_DB *d;
char *nis_domain;
tree_node *t;
void *handle;

/* See if we already have this file open for this type of search, and if so,
pass back the previous handle. */

if (tree != NULL && (t = tree_search(*tree, filename)) != NULL &&
   ((search_openfile *)(t->data.ptr))->type == search_type)
  {
  DEBUG(9) debug_printf("search_open for %s found it cached\n", filename);
  return ((search_openfile *)(t->data.ptr))->handle;
  }

/* Otherwise, open the file - each search type has its own code. */

switch (search_type)
  {
  /* Linear search */

  case stype_lsearch:
  f = fopen(filename, "r");
  if (f == NULL)
    {
    *message = string_sprintf("failed to open %s for linear search: %s",
      filename, strerror(errno));
    return NULL;
    }

  if ((rc = check_file(fileno(f), modemask, owners, owngroups)) != check_ok)
    {
    *message = (rc == check_fstat)?
      string_sprintf("%s: failed to fstat open file", filename) :
      string_sprintf("%s (linear search): wrong %s", filename,
        (rc == check_mode)? "mode" :
        (rc == check_user)? "owner" : "group");
    return NULL;
    }
  handle = f;
  break;


  /* DBM search */

  case stype_dbm:
  d = EXIM_DBOPEN(filename, O_RDONLY, 0);
  if (d == NULL)
    {
    *message = string_sprintf("failed to open %s as a %s file: %s", filename,
      EXIM_DBTYPE, strerror(errno));
    return NULL;
    }

  /* This needs to know more about the underlying files than is good for it! */
   
  #ifdef USE_DB
  if ((rc = check_file(EXIM_DBFD(d), modemask, owners, owngroups)) != check_ok)
  #else
  if (((rc = check_file(dbm_pagfno(d), modemask, owners, owngroups)) != check_ok)
      &&
      ((rc = check_file(dbm_dirfno(d), modemask, owners, owngroups)) != check_ok))
  #endif
    {
    *message = (rc == check_fstat)?
      string_sprintf("%s: failed to fstat open file", filename) :
      string_sprintf("%s (dbm search): wrong %s", filename,
        (rc == check_mode)? "mode" :
        (rc == check_user)? "user" : "group");
    return NULL;
    }
  handle = d;
  break;

  /* NIS search */

  #if HAVE_NIS
  case stype_nis:
  case stype_nis0: 
  if (yp_get_default_domain(&nis_domain) != 0)
    {
    *message = string_sprintf("failed to get default NIS domain");
    return NULL;
    }
  handle = nis_domain;
  break;
  #endif

  /* Oops */

  default:
  *message = string_sprintf("unknown search type in search_open for file %s",
    filename);
  return NULL;
  break;
  }

/* Get here only if the file has been successfully opened. If there is
a tree pointer, enter the file into the tree. */

if (tree != NULL)
  {
  search_openfile *oo = store_malloc(sizeof(search_openfile) +
    sizeof(tree_node) + (int)strlen(filename));
  t = (tree_node *)(oo + 1);
  strcpy(t->name, filename);
  t->data.ptr = oo;
  oo->type = search_type;
  oo->handle = handle;
  tree_insertnode(tree, t);
  }

return handle;
}





/*************************************************
*              Find item in database             *
*************************************************/

/*The answer is always put into dynamic store. If the key contains a colon,
then it is treated as a double key: the first part is the key for the record in
the file, and the remainder is a subkey that is used to extract a subfield from
the main data. Subfields are specified as subkey=value in the records.

The last lookup is cached by file name and key - using the handle is no good as
it isn't unique enough. */

char *search_find(void *handle, char *filename, char *keystring, int type)
{
int length;
int nis_data_length;
FILE *f;
EXIM_DB *d;
EXIM_DATUM key, data;
BOOL free_orig = FALSE;
char *subkey;
char *colon = strchr(keystring, ':');
char *nis_data;
char buffer[256];

/* If there's a colon, temporarily terminate the main key, and set up
the subkey. */

if (colon == NULL) subkey = NULL; else
  {
  subkey = colon + 1;
  *colon = 0;
  }

/* If there is no cached record of the right type, or if this lookup is for a
new key, or in a different file, we must search the file and set up new cached
data. */

if (type != cached_type ||
    strcmp(keystring, cached_keystring) != 0 ||
    strcmp(filename, cached_filename) != 0)
  {
  DEBUG(9) debug_printf("file lookup required for %s%s%s in %s\n",
    keystring,
    (subkey == NULL)? "" : ":",
    (subkey == NULL)? "" : subkey,
    filename);

  if (cached_data != NULL)
    {
    store_free(cached_data);    /* free previous */
    cached_data = NULL;
    }

  /* Length of key to match */

  length = (int)strlen(keystring);

  switch(type)
    {
    /* Linear search */

    case stype_lsearch:
    f = (FILE *)handle;
    rewind(f);

    while (fgets(buffer, sizeof(buffer), f) != NULL)
      {
      int ptr, size;
      int p = (int)strlen(buffer);
      char *s = buffer;

      while (p > 0 && isspace(buffer[p-1])) p--;
      buffer[p] = 0;
      if (buffer[0] == 0 || buffer[0] == '#' || isspace(buffer[0])) continue;
      while (*s != 0 && *s != ':' && !isspace(*s)) s++;
      if (s-buffer != length || strncmpic(buffer, keystring, length) != 0)
        continue;

      if (*s == ':') s++;
      while (isspace(*s)) s++;

      size = 100;
      ptr = 0;
      cached_data = store_malloc(size);
      if (*s != 0)
        cached_data = string_cat(cached_data, &size, &ptr, s, (int)strlen(s));

      while (fgets(buffer, sizeof(buffer), f) != NULL)
        {
        p = (int)strlen(buffer);
        while (p > 0 && isspace(buffer[p-1])) p--;
        buffer[p] = 0;
        if (buffer[0] == 0 || buffer[0] == '#') continue;
        if (!isspace(buffer[0])) break;
        cached_data = string_cat(cached_data, &size, &ptr, buffer, (int)strlen(buffer));
        }

      cached_data[ptr] = 0;
      break;
      }
    break;

    /* DBM search. */

    case stype_dbm:
    d = (EXIM_DB *)handle;
    EXIM_DATUM_DATA(key) = keystring;
    EXIM_DATUM_SIZE(key) = length + 1;
    if (EXIM_DBGET(d, key, data))
      cached_data = string_copy(EXIM_DATUM_DATA(data));
    break;

    /* NIS search */

    #if HAVE_NIS
    case stype_nis:
    case stype_nis0: 
    if (yp_match((char *)handle, filename, keystring, 
        length + ((type == stype_nis)? 0 : 1),
        &nis_data, &nis_data_length) == 0)
      {
      cached_data = string_copy(nis_data);
      cached_data[nis_data_length] = 0;    /* remove final '\n' */
      }
    break;
    #endif
    }

  /* A record that has been found is now in cached_data, which is either NULL
  or points to a bit of dynamic store. Remember the file name, main key, and
  lookup type, but only if the file name and main key are < 256 characters
  long (the size of the cache slots). Longer keys are presumably exceedingly
  rare... */

  if ((int)strlen(filename) < 256 && length < 256)
    {
    strcpy(cached_filename, filename);
    strcpy(cached_keystring, keystring);
    cached_type = type;
    }
  else cached_type = -1;    /* Force lookup next time */
  }

else DEBUG(9) debug_printf("cached data used for lookup of %s%s%s in %s\n",
  keystring,
  (subkey == NULL)? "" : ":",
  (subkey == NULL)? "" : subkey,
  filename);

/* Put back the colon if it was overwritten */

if (colon != NULL) *colon = ':';

/* If we have found data, pick out the subfield if required. Otherwise
make a fresh copy of the whole cached string. */

if (subkey != NULL && cached_data != NULL)
  return expand_getkeyed(subkey, cached_data);
    else if (cached_data != NULL) return string_copy(cached_data);
      else return NULL;
}




/*************************************************
*              Close an open database            *
*************************************************/

void search_close(void *handle, int type)
{
switch(type)
  {
  case stype_lsearch:
  fclose((FILE *)handle);
  break;

  case stype_dbm:
  EXIM_DBCLOSE((EXIM_DB *)handle);
  break;

  #if HAVE_NIS
  case stype_nis:
  case stype_nis0: 
  break;
  #endif
  }
}

/* End of search.c */
