/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */


#include "../exim.h"
#include "localuser.h"



/* Options specific to the localuser director. */

optionlist localuser_director_options[] = {
  { "directory", opt_stringptr,
      (void *)(offsetof(localuser_director_options_block, pw_dir)) }
};

/* Size of the options list. An extern variable has to be used so that its
address can appear in the tables drtables.c. */

int localuser_director_options_count =
  sizeof(localuser_director_options)/sizeof(optionlist);

/* Default private options block for the localuser director. */

localuser_director_options_block localuser_director_option_defaults = { 
  NULL,        /* pw_dir */
  NULL         /* re_pw_dir */ 
};



/*************************************************
*          Initialization entry point            *
*************************************************/

/* Called for each instance, after its options have been read, to
enable consistency checks to be done, or anything else that needs
to be set up. */

void localuser_director_init(director_instance *dblock)
{
localuser_director_options_block *ob = 
  (localuser_director_options_block *)(dblock->options_block);
if (dblock->transport == NULL)
  log_write(LOG_PANIC_DIE, "Exim configuration error for %s director:\n  "
  "a transport is required by the localuser director driver", dblock->name); 
}



/*************************************************
*              Main entry point                  *
*************************************************/

/* See local README for interface description. */

int localuser_director_entry(
  director_instance *dblock,      /* data for this instantiation */
  address_item *addr,             /* address we are working on */
  address_item **addr_local,      /* add it to this if it's local */
  address_item **addr_remote,     /* add it to this if it's remote */
  address_item **addr_new,        /* put new addresses on here */
  BOOL verify)                    /* TRUE when verifying */ 
{
localuser_director_options_block *ob = 
  (localuser_director_options_block *)(dblock->options_block);
int rc;   
  
/* The second argument to direct_finduser() must be NULL here, to prevent a
digit string being taken as a numeric uid. */
 
struct passwd *pw = direct_finduser(addr->local_part, NULL);

if (pw == NULL) 
  {
  DEBUG(2) debug_printf("%s director failed for %s: no such user\n", 
    dblock->name, addr->local_part);  
  return FAIL;
  } 
  
/* If there is a setting for pw_dir, check for a match. Expand the string
if necessary. */

if (ob->pw_dir != NULL)
  {
  BOOL match; 
  char *dir = ob->pw_dir;
  if (strchr(dir, '$') != NULL) 
    {
    dir = expand_string(dir);
    if (dir == NULL)
      log_write(LOG_MAIN|LOG_PANIC_DIE, "%s director: failed to expand "
        "directory string \"%s\"", dblock->name, ob->pw_dir); 
    } 
  match = match_string(pw->pw_dir, dir, &(ob->re_pw_dir));
  if (dir != ob->pw_dir) store_free(dir);
  if (!match)
    {
    DEBUG(2) debug_printf("%s director failed to match pw_dir with \"%s\" "
      "for %s\n", dblock->name, ob->pw_dir, addr->local_part);
    return FAIL;
    }       
  } 

/* Do file existence tests, allowing for home directory expansion (deliver_
home gets reset at calling level). */

deliver_home = pw->pw_dir;
if (!match_exists(dblock->require_files))
  {
  DEBUG(9) debug_printf("%s director failed: file existence failure\n",
    dblock->name);
  return FAIL;   
  }

/* This local user is OK; accept the address */

addr->transport = dblock->transport;
addr->uid = pw->pw_uid;
addr->gid = pw->pw_gid;
addr->home_dir = string_copy(pw->pw_dir);

if (addr->transport->info->local)
  {
  addr->next = *addr_local;
  *addr_local = addr;
  }
else
  {
  addr->next = *addr_remote;
  *addr_remote = addr;
  }

return OK;
}

/* End of director/localuser.c */
