/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Copyright (c) University of Cambridge 1995 - 1996 */
/* See the file NOTICE for conditions of use and distribution. */

/* Functions concerned with running Exim as a daemon */


#include "exim.h"



/*************************************************
*               Local static variables           *
*************************************************/

static BOOL sigalrm_seen;
static BOOL sigchld_seen;
static BOOL sighup_seen;

static int  accept_retry_count = 0;

static int  queue_run_count = 0;
static int  queue_pid_slotcount;
static int *queue_pid_slots;

static int  smtp_pid_slotcount;
static int *smtp_pid_slots;




/*************************************************
*             SIGHALRM Handler                   *
*************************************************/

static void sigalrm_handler(int sig)
{
sigalrm_seen = TRUE;
signal(SIGALRM, sigalrm_handler);
}



/*************************************************
*             SIGHUP Handler                     *
*************************************************/

static void sighup_handler(int sig)
{
sighup_seen = TRUE;
signal(SIGHUP, sighup_handler);
}



/*************************************************
*             SIGCHLD Handler                    *
*************************************************/

/* Don't re-enable the handler here, since we aren't doing the
waiting here. If the signal is re-enabled, there will just be an 
infinite sequence of calls to this handler. The SIGCHLD signal is
used just as a means of waking up the daemon so that it notices
terminated subprocesses as soon as possible. */

static void sigchld_handler(int sig)
{
sigchld_seen = TRUE;
signal(SIGCHLD, SIG_DFL);
}




/*************************************************
*            Handle a connected SMTP call        *
*************************************************/

/* This function is called when an SMTP connection has been accepted.
If there are too many, give an error message and close down. Otherwise
spin off a sub-process to handle the call. */

static void handle_smtp_call(int listen_socket, int accept_socket,
  struct sockaddr_in *accepted)
{
FILE *in;
FILE *out = fdopen(accept_socket, "w");
int pid;
int dup_accept_socket = dup(accept_socket);

DEBUG(1) debug_printf("Connection request from [%s]\n",
  inet_ntoa(accepted->sin_addr));

/* Check that duplication of the socket worked, and if so, make
an input file. */

if (dup_accept_socket < 0)
  {
  DEBUG(1) debug_printf("Couldn't dup socket descriptor\n");
  DEBUG(1) debug_printf("421 %s: Connection refused: %s\n", primary_hostname,
    strerror(errno));
  fprintf(out, "421 %s: Connection refused: %s\r\n", primary_hostname,
    strerror(errno));
  fclose(out);
  return;
  }

in = fdopen(dup_accept_socket, "r");

/* Check maximum number of connections. We do not check for reserved
connections or unacceptable hosts here. That is done in the subprocess because
it might take some time. */

if (smtp_accept_max > 0 && smtp_accept_count >= smtp_accept_max)
  {
  DEBUG(1) debug_printf("rejecting SMTP connection: count=%d max=%d\n",
    smtp_accept_count, smtp_accept_max);
  DEBUG(1) debug_printf("421 %s: Too many concurrent SMTP connections; "
    "please try again later.\n", primary_hostname);
  fprintf(out, "421 %s: Too many concurrent SMTP connections; "
    "please try again later.\r\n", primary_hostname);
  fclose(out);
  fclose(in);
  return;
  }


/* Now fork the accepting process */

pid = fork();

/* Handle the child process */

if (pid == 0)
  {
  BOOL local_queue_only = queue_only;
  BOOL smtp_first = TRUE;

  close(listen_socket);
  signal(SIGCHLD, SIG_DFL);

  /* Make the address available in ASCII representation, and also
  set it up as a default sender_fullhost, for cases when HELO is
  not present. */

  sender_host_address = string_copy(inet_ntoa(accepted->sin_addr));
  sender_fullhost = string_sprintf("[%s]", sender_host_address);
  set_process_info("handling incoming connection from [%s]",
    sender_host_address);

  /* Attempt to get an id from the sending machine via the RFC 1413
  protocol. We do this in the sub-process in order not to hold up the
  main process if there is any delay. */

  verify_get_ident(accept_socket);

  DEBUG(2)
    {
    debug_printf("Process %d is handling incoming connection from [%s]",
      getpid(), sender_host_address);
    if (sender_ident != NULL) debug_printf(" (%s)\n", sender_ident);
      else debug_printf("\n");
    }

  /* If there are too many child processes for immediate delivery,
  set the local_queue_only flag, which is initialized from the
  configured value and may therefore already be TRUE. */

  if (smtp_accept_queue > 0 && smtp_accept_count >= smtp_accept_queue)
    {
    DEBUG(1) debug_printf(
      "Too many connections (%d) for immediate delivery (max %d)\n",
        smtp_accept_count, smtp_accept_queue);

    local_queue_only = TRUE;
    }

  /* Now loop, accepting incoming messages from the SMTP connection. The
  end will come at the QUIT command, when accept_msg() returns FALSE.
  A break in the connection causes the process to die (see accept.c). */

  sender_local = sender_host_unknown = FALSE;
  smtp_input = TRUE;

  for (;;)
    {
    int rc;
    message_id[0] = 0;   /* Clear out any previous message_id */

    /* Smtp_setup_msg() returns 0 on QUIT or if the call is from an
    unacceptable host, -1 on connection lost, and +1 on validly reaching
    DATA. Accept_msg() always returns TRUE when smtp_input is true; just
    retry if no message was accepted (can happen for invalid message
    parameters). */

    if ((rc = smtp_setup_msg(in, out, smtp_first)) > 0)
      {
      smtp_first = FALSE;
      (void) accept_msg(in, out, FALSE);
      if (message_id[0] == 0) continue;
      }
    else exit((rc == 0)? EXIT_SUCCESS : EXIT_FAILURE);

    /* Show the recipients when debugging */

    DEBUG(2)
      {
      int i;
      if (sender_address != NULL)
        debug_printf("Sender: %s\n", sender_address);
      if (recipients_list != NULL)
        {
        debug_printf("Recipients:\n");
        for (i = 0; i < recipients_count; i++)
          debug_printf("  %s\n", recipients_list[i]);
        }
      }

    /* A message has been accepted. Unless there are too many
    delivery processes in existence, spin off a new one to handle
    it, after freeing the store we have used. If we are not root,
    we have to re-exec exim unless root can be regained by the use
    of seteuid. */

    store_free(sender_address);
    accept_free_recipients();

    if (!local_queue_only && fork() == 0)
      {
      if (security_level >= 2)
        {
        int i = 0;
        char *argv[6];
        argv[i++] = exim_path;
        if (debug_level > 0)
          argv[i++] = string_sprintf("-d%d", debug_level);  
        if (dont_deliver) argv[i++] = "-N";   
        argv[i++] = "-Mc";
        argv[i++] = message_id;
        argv[i++] = (char *)0;
        execv(argv[0], argv);
        log_write(LOG_PANIC_DIE, "exec of exim -Mc failed");
        }
      /* No need to re-exec */
      (void)deliver_message(message_id, FALSE, FALSE);
      _exit(EXIT_SUCCESS);
      }
    }
  }


/* Carrying on in the parent daemon process... Can't do much if the fork
failed. Otherwise, keep count of the number of accepting processes and
remember the pid for ticking off when the child completes. */

if (pid < 0)
  {
  DEBUG(1) debug_printf("fork of SMTP accept process failed\n");
  fprintf(out, "421: %s Connection refused: %s\r\n", primary_hostname,
    strerror(errno));
  }
else
  {
  int i;
  for (i = 0; i < smtp_pid_slotcount; ++i)
    {
    if (smtp_pid_slots[i] <= 0)
      {
      smtp_pid_slots[i] = pid;
      smtp_accept_count++;
      break;
      }
    }
  DEBUG(2) debug_printf("%d SMTP accept process%s running\n",
    smtp_accept_count, (smtp_accept_count == 1)? "" : "es");
  }

 /* Close the streams associated with the socket which will also
 close the socket fds in the parent process. */

fclose(in);
fclose(out);
}




/*************************************************
*              Exim Daemon Mainline              *
*************************************************/

/* The daemon can do two jobs, either of which is optional:

   (1) Listens for incoming SMTP calls and spawn off a sub-process to handle
   each one. This is requested by the -bd option, with -oX specifying the SMTP
   port on which to listen (for testing).

   (2) Spawns a queue-running process every so often. This is controlled by the
   -q option with a an interval time. (If no time is given, a single queue run
   is done from the main function, and control doesn't get here.)

Root privilege is required in order to attach to port 25. Some systems require
it when calling socket() rather than bind(). To cope with all cases, we run as
root for both socket() and bind(). When a listener is required, this function
is entered as root for security levels 0 or 2; otherwise seteuid to exim.

Once the socket is bound, root privilege is given up if there is an exim uid,
either by seteuid for security level 1, or by setuid for higher levels. In the
latter case, a re-exec is subsequently required to cause a received message to
be delivered.

When listening is not required - i.e. the daemon is simply a means of
periodically starting a queue runner - this function is entered with the
appropriate uid already set. */


void daemon_go(void)
{
int listen_socket;
u_short net_port;
FILE *f;
char buff[256];

/* Create the name of the file holding the daemon's pid. This is written just
to make it easier to find the daemon process. There's nothing to stop multiple
daemons running, as long as no more than one listens on a given TCP/IP port. We
put a non-standard port number in the file name to distinguish running/testing
versions, but if someone runs two similar daemons simultaneously, then only the
last started will get its pid written. */

if (smtp_port < 0)
  sprintf(buff, "%s/exim-daemon.pid", spool_directory);
else
  sprintf(buff, "%s/exim-daemon.%d.pid", spool_directory, smtp_port);

/* Close all open file descriptors and disconnect from the controlling
terminal, if we are not debugging. Most modern Unixes seem to have setsid() for
getting rid of the controlling terminal. For any OS that doesn't, setsid() can
be #defined as a no-op, or as something else. */

if (debug_level == 0 && !debug_trace_memory)
  {
  int pid, fd;
   
  for (fd = mac_maxfd; fd >= 0; fd--) close(fd);

  /* Fork, in case current process is a process group leader (see 'man
  setsid' for an explanation). */

  pid = fork();
  if (pid < 0) log_write(LOG_PANIC_DIE, "fork() failed: %s", strerror(errno));
  if (pid > 0) exit(EXIT_SUCCESS);      /* in parent process, just exit */

  /* Release controlling terminal */

  (void)setsid();
  }

/* If SMTP listening is requested, set up a socket on the SMTP port or
a given port, and compile the verification acceptance data, if any, so
that it is available to all spawned processes. */

if (daemon_listen)
  {
  int i;
  int on = 1;
  struct sockaddr_in sin;

  /* If there are lists of nets and hosts + idents for various checks,
  pre-process them now so that this work is not repeated for each message. For
  a sender_reject list, just do a match that won't work so as to get any
  regular expressions compilied. */

  if (sender_reject != NULL)
    (void)match_isinlist(":@:", sender_reject, &re_sender_reject);

  if (sender_verify_except_hosts != NULL)
    verify_setup_hostlist(sender_verify_except_hosts,
      &sender_verify_except_hostlist);

  if (sender_verify_except_nets != NULL)
    verify_setup_netlist(sender_verify_except_nets,
      &sender_verify_except_netlist);

  if (rfc1413_except_hosts != NULL)
    verify_setup_hostlist(rfc1413_except_hosts, &rfc1413_except_hostlist);

  if (rfc1413_except_nets != NULL)
    verify_setup_netlist(rfc1413_except_nets, &rfc1413_except_netlist);

  if (sender_host_accept != NULL)
    verify_setup_hostlist(sender_host_accept, &sender_host_accept_hosts);

  if (sender_net_accept != NULL)
    verify_setup_netlist(sender_net_accept, &sender_net_accept_nets);

  if (sender_host_reject != NULL)
    verify_setup_hostlist(sender_host_reject, &sender_host_reject_hosts);

  if (sender_net_reject != NULL)
    verify_setup_netlist(sender_net_reject, &sender_net_reject_nets);

  if (smtp_reserve_hosts != NULL)
    verify_setup_hostlist(smtp_reserve_hosts, &smtp_reserve_hostlist);

  if (smtp_reserve_nets != NULL)
    verify_setup_netlist(smtp_reserve_nets, &smtp_reserve_netlist);

  if (receiver_unqualified_hosts != NULL)
    verify_setup_hostlist(receiver_unqualified_hosts,
      &receiver_unqualified_hostlist);

  if (receiver_unqualified_nets != NULL)
    verify_setup_netlist(receiver_unqualified_nets,
      &receiver_unqualified_netlist);

  if (sender_unqualified_hosts != NULL)
    verify_setup_hostlist(sender_unqualified_hosts,
      &sender_unqualified_hostlist);

  if (sender_unqualified_nets != NULL)
    verify_setup_netlist(sender_unqualified_nets,
      &sender_unqualified_netlist);

  /* Find the standard SMTP port if no port number given; otherwise
  convert the given port to network order. */

  if (smtp_port < 0)
    {
    struct servent *smtp_service;
    if ((smtp_service = getservbyname("smtp", "tcp")) == NULL)
      log_write(LOG_PANIC_DIE, "cannot find smtp/tcp service");
    net_port = smtp_service->s_port;
    smtp_port = ntohs(net_port);
    }
  else net_port = htons(smtp_port);

  /* Ensure root privilege. It will only not exist at this stage if seteuid
  can be used to regain it. */

  if (geteuid() != root_uid) mac_seteuid(root_uid);

  /* Create a socket */

  listen_socket = socket(AF_INET, SOCK_STREAM, 0);
  if (listen_socket < 0)
    log_write(LOG_PANIC_DIE, "socket creation failed: %s", strerror(errno));

  /* Set SO_REUSEADDR so that the daemon can be restarted while a connection
  is being handled.  Without this, a connection will prevent reuse of the
  smtp port for listening. */

  if (setsockopt(listen_socket, SOL_SOCKET, SO_REUSEADDR, (char *)(&on),
    sizeof(on)) < 0)
      log_write(LOG_PANIC_DIE, "setting SO_REUSEADDR on socket failed: %s",
        strerror(errno));

  /* Now bind the socket to the required port; if Exim is being restarted
  it may now always be possible to bind immediately, even with SO_REUSEADDR
  set, so try 10 times, waiting 30 seconds between each try. */

  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = INADDR_ANY;
  sin.sin_port = net_port;

  for (i = 9; i >= 0; i--)
    {
    if (bind(listen_socket, (struct sockaddr *)&sin, sizeof(sin)) < 0)
      {
      if (i == 0)
        {
        char *msg = strerror(errno); 
        log_write(LOG_MAIN, "socket bind() to port %d failed: %s: "
          "daemon abandoned", smtp_port, msg); 
        log_write(LOG_PANIC_DIE, "socket bind() to port %d failed: %s",
          smtp_port, msg);
        }
      log_write(LOG_MAIN, "socket bind() to port %d failed: %s: waiting "
        "before trying again", smtp_port, strerror(errno));
      sleep(30);
      }
    else break;
    }

  /* If exim_uid is set, give up root privilege at this point, using setuid or
  seteuid as appopriate. The macros expand to -1 on systems that don't have
  the sete{g,u}id functions, but the security level cannot be set to values
  implying the use of these functions on such systems. */

  if (exim_uid > 0)
    {
    if (security_level >= 2)
      {
      setgid(exim_gid);
      setuid(exim_uid);
      }
    else
      {
      mac_setegid(exim_gid);
      mac_seteuid(exim_uid);
      }
    }

  /* Establish the maximum backlog of connections that is allowed. */

  listen(listen_socket, smtp_connect_backlog);

  /* Do a sanity check on the max connects value; zero means no limit,
  in which case we don't need to keep a list of them. */

  if (smtp_accept_max < 0 || smtp_accept_max > 4095) smtp_accept_max = 0;

  /* Do a sanity check on the "max connects until we queue only" (no delivery)
  value. Again, zero means no limit, and there's no point setting it unless it
  is less than the max connects limit. */

  if (smtp_accept_queue < 0 || smtp_accept_queue > 4095) smtp_accept_queue = 0;
  if (smtp_accept_max > 0 && smtp_accept_queue > smtp_accept_max)
    smtp_accept_queue = 0;

  /* Get somewhere to keep the list of SMTP accepting pids if we are keeping
  track of them, either for total number, or for queue-only handling. */

  smtp_pid_slotcount = (smtp_accept_queue > smtp_accept_max)?
    smtp_accept_queue : smtp_accept_max;

  if (smtp_pid_slotcount > 0)
    {
    int i;
    smtp_pid_slots = store_malloc(smtp_pid_slotcount * sizeof(int));
    for (i = 0; i < smtp_pid_slotcount; i++) smtp_pid_slots[i] = 0;
    }
  }


/* Get somewhere to keep the list of queue-runner pids if we are keeping track
of them (and also if we are doing queue runs). */

if (queue_interval > 0)
  {
  queue_pid_slotcount = queue_run_max;
  if (queue_pid_slotcount > 0)
    {
    int i;
    queue_pid_slots = store_malloc(queue_pid_slotcount * sizeof(int));
    for (i = 0; i < queue_pid_slotcount; i++) queue_pid_slots[i] = 0;
    }
  }

/* Set up the handler for termination of child processes. */

sigchld_seen = FALSE;
signal(SIGCHLD, sigchld_handler);

/* Set up the handler for SIGHUP, which causes a restart of the daemon. */

sighup_seen = FALSE;
signal(SIGHUP, sighup_handler);

/* If we are to run the queue periodically, pretend the alarm has just
gone off. This will cause the first queue-runner to get kicked off straight
away, and the alarm to be reset. */

sigalrm_seen = (queue_interval > 0);


/* Log the start up of a daemon. */

if (queue_interval > 0)
  {
  if (daemon_listen)
    {
    log_write(LOG_MAIN,
      "exim daemon started: pid=%d, -q%s, listening for SMTP on port %d", getpid(),
      readconf_printtime(queue_interval), smtp_port);
    set_process_info("daemon: -q%s, listening on port %d",
      readconf_printtime(queue_interval), smtp_port);
    }
  else
    {
    log_write(LOG_MAIN,
      "exim daemon started: pid=%d, -q%s, not listening for SMTP", getpid(),
      readconf_printtime(queue_interval));
    set_process_info("daemon: -q%s, not listening",
      readconf_printtime(queue_interval));
    }
  }
else
  {
  log_write(LOG_MAIN,
    "exim daemon started: pid=%d, no queue runs, listening for SMTP on port %d",
     getpid(), smtp_port);
  set_process_info("daemon: no queue runs, port %d", smtp_port);
  }

/* Write the pid to a known file for assistance in identification. */

f = fopen(buff, "w");
if (f != NULL)
  {
  fprintf(f, "%d\n", getpid());
  fclose(f);
  }

/* Close the log so it can be renamed and moved. This process doesn't write
to the log again, unless it is about to die or exec and in the latter case
it closes the log first. */

log_close();

DEBUG(2) debug_printf("daemon running with uid=%d gid=%d euid=%d egid=%d\n",
  getuid(), getgid(), geteuid(), getegid());

/* Enter the never-ending loop... */

for (;;)
  {
  struct sockaddr_in accepted;
  int len = sizeof(accepted);
  int accept_socket;
  int pid; 

  /* This code is placed first in the loop, so that it gets obeyed at the
  start, before the first wait. This causes the first queue-runner to be
  started immediately. */

  if (sigalrm_seen)
    {
    DEBUG(9) debug_printf("SIGALRM received\n");

    /* Do a full queue run in a child process, if required, unless we already
    have enough queue runners on the go. If we are not running as root, a
    re-exec is required. */

    if (queue_interval > 0 &&
       (queue_run_max <= 0 || queue_run_count < queue_run_max))
      {
      int pid;
      if ((pid = fork()) == 0)
        {
        DEBUG(1) debug_printf("Starting queue-runner: pid %d\n", getpid());
        close(listen_socket);
        signal(SIGALRM, SIG_DFL);      /* Reset signals in the child */
        signal(SIGHUP,  SIG_DFL);
        signal(SIGCHLD, SIG_DFL);
        if (geteuid() != root_uid)
          {
          int i = 0;
          char *argv[3];
          argv[i++] = exim_path;
          argv[i++] = "-q";
          argv[i++] = (char *)0;
          execv(argv[0], argv);
          log_write(LOG_PANIC_DIE, "exec of exim -q failed");
          }
        /* No need to re-exec */
        queue_run(NULL, NULL);
        _exit(EXIT_SUCCESS);
        }

      if (pid < 0)
        {
        log_write(LOG_PANIC, "fork of queue-runner process failed");
        }
      else
        {
        int i;
        for (i = 0; i < queue_pid_slotcount; ++i)
          {
          if (queue_pid_slots[i] <= 0)
            {
            queue_pid_slots[i] = pid;
            queue_run_count++;
            break;
            }
          }
        DEBUG(2) debug_printf("%d queue-runner process%s running\n",
          queue_run_count, (queue_run_count == 1)? "" : "es");
        }
      }

    /* Reset the alarm time */

    sigalrm_seen = FALSE;
    signal(SIGALRM, sigalrm_handler);
    alarm(queue_interval);
    }


  /* Sleep till a connection happens if listening, and handle the connection if
  that is why we woke up. */

  if (daemon_listen)
    {
    DEBUG(2) debug_printf("listening on port %d...\n", smtp_port);
    accept_socket = accept(listen_socket, (struct sockaddr *)&accepted, &len);

    /* If accept has failed and this was not caused by an interruption,
    wait for a bit and then try again, but do not continue doing this
    for ever. */

    if (accept_socket < 0 && errno != EINTR)
      {
      log_write(LOG_PANIC, "accept() failed: %s", strerror(errno));
      sleep(5);
      if (++accept_retry_count > 10)
        log_write(LOG_PANIC_DIE, "too many accept() errors: giving up");
      continue;
      }

    /* Either accept was interrupted, or it succeeded. If it succeeded,
    deal with it. */

    accept_retry_count = 0;
    if (accept_socket >= 0)
      handle_smtp_call(listen_socket, accept_socket, &accepted);
    }

  /* If not listening, then just sleep for the queue interval. */

  else sleep(queue_interval);
  
  /* Handle the termination of a child process. Theoretically, this need
  be done only when sigchld_seen is TRUE, but rumour has it that some systems
  lose SIGCHLD signals at busy times, so to be on the safe side, just
  do it each time round. It shouldn't be too expensive. */

  while ((pid = waitpid(-1, NULL, WNOHANG)) > 0)
    {
    int i;
    DEBUG(2) debug_printf("child %d ended\n", pid);
  
    /* Deal with an accepting process. */
  
    for (i = 0; i < smtp_pid_slotcount; ++i)
      {
      if (smtp_pid_slots[i] == pid)
        {
        smtp_pid_slots[i] = 0;
        if (--smtp_accept_count < 0) smtp_accept_count = 0;
        DEBUG(2) debug_printf("%d SMTP accept process%s now running\n",
          smtp_accept_count, (smtp_accept_count == 1)? "" : "es");
        break;
        }
      }
  
    /* Deal with a queue-runner process. */
  
    for (i = 0; i < queue_pid_slotcount; ++i)
      {
      if (queue_pid_slots[i] == pid)
        {
        queue_pid_slots[i] = 0;
        if (--queue_run_count < 0) queue_run_count = 0;
        DEBUG(2) debug_printf("%d queue-runner process%s now running\n",
          queue_run_count, (queue_run_count == 1)? "" : "es");
        break;
        }
      }
    }
    
  /* Re-enable the SIGCHLD handler if it has been run. It can't do it
  for itself, because it isn't doing the waiting itself. */
  
  if (sigchld_seen)
    {
    sigchld_seen = FALSE;
    signal(SIGCHLD, sigchld_handler);
    } 

  /* Handle being woken by SIGHUP. We know at this point that the result
  of accept() has been dealt with, so we can re-exec exim safely, first
  closing the listening socket so that it can be reused. */

  if (sighup_seen)
    {
    log_write(LOG_MAIN, "pid %d: SIGHUP received: re-exec daemon", getpid());
    log_close();
    close(listen_socket);
    sighup_argv[0] = exim_path;
    execv(exim_path, sighup_argv);
    log_write(LOG_PANIC_DIE, "pid %d: exec of %s failed", getpid());
    }

  }   /* End of main loop */

/* Control never reaches here */
}

/* End of exim_daemon.c */
