;; ========================================================================
;; hyperB-x.el -- Useful hyperbutton handlers
;; Author          : Mike Williams <mike-w@cs.aukuni.ac.nz>
;; Created On      : Wed Aug 21 15:00:13 1991
;; Last Modified By: Mike Williams
;; Last Modified On: Mon Aug 26 11:16:39 1991
;; RCS Info        : $Revision: 1.6 $ $Locker:  $
;; ========================================================================
;; [[ CheckMeOut ]] [[ CheckMeIn ]]
;; 
;; NOTE: this file must be recompiled if changed.
;;
;; Copyright (C) Mike Williams <mike-w@cs.aukuni.ac.nz> 1991
;;
;; This file is not part of GNU Emacs, but is made available under the
;; same conditions.
;;
;; GNU Emacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
;; to anyone for the consequences of using it or for whether it serves
;; any particular purpose or works at all, unless he says so in writing.
;; Refer to the GNU Emacs General Public License for full details.
;;
;; Everyone is granted permission to copy, modify and redistribute GNU
;; Emacs, but only under the conditions described in the GNU Emacs
;; General Public License.  A copy of this license is supposed to have
;; been given to you along with GNU Emacs so you can know your rights and
;; responsibilities.  It should be in a file named COPYING.  Among other
;; things, the copyright notice and this notice must be preserved on all
;; copies.

(require 'hyperB)

(provide 'hyperB-x)

;;=== Usage ===============================================================
;;
;; The easiest way to install this package is in your hyperB-load-hooks.
;; Simply place the following in your .emacs (before loading hyperB):
;;
;; (setq hyperB-load-hooks 
;;       '(lambda () 
;;          (require 'hyperB-x)
;;          (install-all-hyperB-x-handlers)))
;;

;; You'll need my lib-complete package as well if you want to use
;; hyperB:find-header-file and/or hyperB:find-elisp-library.  Check out
;; lib-complete.el for details.

;;=== Hook variable modification ==========================================

;; This a general function for adding hooks, and not limited to 
;; hyperB.  It's primarily included for use in the
;; install-all-hyperB-x-handlers function, but you could use it to 
;; make adding selected handlers easier.

(defun hyperB:add-hook (hook-var &rest hook-forms)
  "Modify HOOK-VAR by adding hooks.  Following arguments are forms to add
to the hook, if they are not already present.  Each hook form may be a symbol
with a function definition, a lambda abstraction or an arbitrary elisp form.  
In the latter case, the form is wrapped with (lambda () ...) before being 
added to the hook variable."
  (let ((hook-value (if (boundp hook-var) (symbol-value hook-var))))
    (while hook-forms
      (let ((hook-form (car hook-forms)))
	;; Wrap function applications in a lambda
	(if (and (consp hook-form) (not (eq (car hook-form) 'lambda)))
	    (setq hook-form
		  (list 'lambda nil hook-form)))
	(cond
	 ((and (listp hook-value) (not (eq (car hook-value) 'lambda)))
	  (if (equal-member hook-form hook-value) nil
	    (setq hook-value (cons hook-form hook-value))))
	 (t (if (equal hook-form hook-value) nil
	      (setq hook-value (list hook-form hook-value))))))
      (setq hook-forms (cdr hook-forms)))
    (set hook-var hook-value)))

;; Need this function for membership testing using 'equal.  Emacs-19 
;; will provide a subr for doing this ('member).

(if (fboundp 'equal-member) nil
  (defun equal-member (x list)
    "Returns non-nil if ELT is an element of LIST.  Comparison done with EQUAL.
The value is actually the tail of LIST whose car is ELT."
    (while (and list (not (equal x (car list))))
      (setq list (cdr list)))
    list)
  )
  
;;=== C header files ======================================================
;;
;; Installation:
;;
;;   In your c-mode-hook
;;
;;      (hyperB:add-local-handler 'hyperB:find-header-file)
;;   

(defvar include-file-path '("/usr/include")
  "List of directories to search for C header files")

(defun hyperB:find-header-file (INCLUDE &optional VIEW)
  "HyperB handler to find header file referenced in an #include directive. 

Clicking on `#include <stdio.h>' will search for stdio.h in the 
directories listed in include-file-path.  Clicking on `#include \"stdio.h\" 
will just search in the current directory."
  (interactive (list (thing-at-point 'line)))
  ;; Requires locate-file, from the lib-complete.el package
  (if (not (featurep 'lib-complete)) 
      (error "hyperB:find-header-file requires the lib-complete package"))
  (let (header
	(search-path include-file-path))
    (cond 
     ((string-match "#include\\s +<\\(.+\\)>\\s *" INCLUDE)
      (setq header (substring INCLUDE (match-beginning 1) (match-end 1))))
     ((string-match "#include\\s +\"\\(.+\\)\"\\s *" INCLUDE)
      (setq header (substring INCLUDE (match-beginning 1) (match-end 1)))
      (setq search-path '(nil))))
    (if header
	(let ((path (locate-file header search-path nil)))
	  (if path (` (find-file (, path))) ; was more-file ??
	    (error "Can't find %s" header))))))

;;=== Elisp library files =================================================
;;
;; Installation:
;;
;;   In your emacs-lisp-mode-hook
;;
;;      (hyperB:add-local-handler 'hyperB:find-elisp-library)
;;

(defun hyperB:find-elisp-library (FORM &optional VIEW)
  "HyperB handler to find emacs lisp source file referenced in a require, 
load or load-library call.  Files are searched for on the load-path."
  (interactive (list (thing-at-point 'list)))
  ;; Requires locate-file, from the lib-complete.el package
  (if (not (featurep 'lib-complete))
      (error "hyperB:find-header-file requires the lib-complete package"))
  (if (stringp FORM)
      (setq FORM (read-from-whole-string FORM)))
  (if (not (listp FORM)) nil
      (let ((library
	     (cond 
	      ((memq (car FORM) '(load load-library)) (nth 1 FORM))
	      ((eq (car FORM) 'require) 
	       (or (nth 2 FORM) (symbol-name (eval (nth 1 FORM))))))))
	(if library
	    (let ((path (locate-file library load-path '(nil ".el" ".el.Z"))))
	      (if path (` (find-file (, path))))))))) ; Was: more-file ??


;;=== Support for "HyperCards" ============================================
;;
;; Installation:
;;
;;   In your .emacs
;;
;;    (define-global-hyperB 'hyperB 
;;      "^\\s *Card:\\s *\\(\\S +\\)\\s *$"
;;      '(hyperB:goto-card (substring hyperB-name 
;;                                    (match-beginning 1) (match-end 1))))
;;
;; NOTE: This isn't installed by install-all-hyperB-x-handlers.

(defvar hyperB:hypercard-path '(nil "~/.HyperCard"))

(defvar hyperB:hypercard-view-mode t)

(defun hyperB:goto-card (CARD)
  "View a file somewhere on the hyperB:hypercard-path."
  (interactive (list (thing-at-point 'filename)))
  ;; Requires locate-file, from the lib-complete.el package
  (if (not (featurep 'lib-complete))
      (error "hyperB:goto-card requires the lib-complete package"))
  (let ((path (and CARD 
		   (locate-file CARD hyperB:hypercard-path))))
    (if path
	(if hyperB:hypercard-view-mode 
	    (` (view-file (, path)))
	  (` (find-file (, path)))))))


;;=== Info-mode browsing ==================================================
;;
;; Based on info-mouse.el by Bob Weiner <weiner@bbn.com>
;; 
;; Installation:
;;   In your Info-mode-hook:
;; 
;; (mapcar 'hyperB:add-local-handler
;; 	   '(hyperB:Info-header
;; 	     hyperB:Info-menu
;; 	     hyperB:Info-note))
;;
;; NOTE: The standard version 18 info.el doesn't recognise Info-mode-hook.
;;       This means you have to either:
;;
;;   (a) Run this hook by hand after loading info.el.  (Yuk!!)
;;
;;   (b) Alter these functions to be conditional on being in the "*info*"
;;       buffer, and add them to hyperB:global-handlers.  (Gross!!)
;;   
;;   (c) Use a enhanced version of info.el, such as the one posted by Dave
;;       Gillespie <daveg@csvax.cs.caltech.edu>.  (Yeah, go Dave!!)
;;
;;   (d) Apply the following patch to info.el (after removing the ";; |"
;;       prefix):
;;
;; +---- CUT HERE ---------------------------------------------------------
;; |*** /users/staff/mike-w/Emacs/ElispSrc/info.el  Thu Jul 20 18:24:18 1989
;; |--- info.el     Sat Jun 29 14:29:38 1991
;; |***************
;; |*** 648,651 ****
;; |--- 648,652 ----
;; |    (make-local-variable 'Info-tag-table-marker)
;; |    (make-local-variable 'Info-history)
;; |+   (run-hooks 'Info-mode-hook)
;; |    (Info-set-mode-line))
;; |
;; +---- AND HERE ---------------------------------------------------------
;;

(defun hyperB:Info-header ()
  (interactive)
  (let* ((first-line (1+ (count-lines 1 (point-min))))
	 (current-line (count-lines 1 (1+ (point)))))
    (if (not (equal current-line first-line))
	nil
      (let ((nodename "Top") (filep nil))
	(save-excursion
	  (if (and
	       (re-search-forward "[:, \t\n]" nil t)
	       (re-search-backward
		"\\(File\\|Node\\|Up\\|Prev\\|Previous\\|Next\\)[: \t]" nil t))
	      (progn
		(setq filep (string-equal
			     "file"
			     (downcase (buffer-substring
					(match-beginning 1)
					(match-end 1)))))
		(if (re-search-forward (concat ":[ \n]\\([^,.\t\n"
					       (if filep " ")
					       "]*\\)") nil t)
		    (setq nodename (buffer-substring
				    (match-beginning 1)
				    (match-end 1)))))
	    (error "Node header not found.")))
	(if filep (setq nodename (concat "(" nodename ")" "Top")))
	(` (Info-goto-node (, nodename)))))))

(defun hyperB:Info-note ()
  (interactive)
  (let ((note-name nil) (bol nil))
    (save-excursion
      (if (re-search-forward "[:.\n]" nil t)
	  (progn
	    (save-excursion
	      (beginning-of-line)
	      (setq bol (point)))
	    (if (re-search-backward 
		 "\*\\(Note\\|Ref\\)[ \n]+\\([^:]*\\):" bol t)
		(setq note-name (buffer-substring
				 (match-beginning 2)
				 (match-end 2)))))))
    (if (not note-name)
	nil
      (` (Info-follow-reference (, note-name))))))

(defun hyperB:Info-menu ()
  (interactive)
  (let ((in-menu nil) (curr-point (point)))
    (save-excursion
      (goto-char (point-min))
      (setq in-menu 
	    (and (re-search-forward "^\* Menu:" nil t)
		 (< (point) curr-point))))
    (if (not in-menu)
	nil
      (forward-char) ; Pass '*' char if point is in front of
      (if (re-search-backward "^\*" nil t)
	  (progn (forward-char 2)
		 (` (Info-goto-node (, (Info-extract-menu-node-name)))))))))

;;=== Find tags ===========================================================
;;
;; Installation:
;;
;;   In your c-mode-hook, emacs-lisp-mode-hook, gdb-mode-hook, etc.
;;
;;      (hyperB:add-local-handler 'hyperB:find-tag)
;;
;; NOTE: This handler does nothing if tags-file-name is not set, so ensure
;;       that you've used visit-tags-table first.
;;

(defun hyperB:find-tag () 
  (interactive)
  (require 'tags)
  (let* ((tag (find-tag-default))
	 tag-file start-pos line-prefix)
    ;; Search for exact tag match
    (if (not tags-file-name) nil
      (save-excursion
	(visit-tags-table-buffer) (goto-char (point-min))
	(cond 
	 ((re-search-forward (concat "[ \t*]" (regexp-quote tag) 
				     "[ \t;({].*\177")
			     nil t)
	  (setq line-prefix
		(buffer-substring 
		 (1- (point))
		 (save-excursion (beginning-of-line) (point))))
	  (setq tag-file 
		(expand-file-name 
		 (file-of-tag) (file-name-directory tags-file-name)))
	  (search-forward ",")
	  (setq start-pos (read (current-buffer)))
	  (or (file-exists-p tag-file) (error "Can't find %s" tag-file))
	  (find-file tag-file)
	  ;; Find tag in file
	  (let ((start-pos (or start-pos (point-min)))
		(search-pat (concat "^" (regexp-quote line-prefix)))
		(offset 500) found)
	    (while (and (not found)
			(goto-char (- start-pos offset))
			(not (bobp)))
	      (setq found (re-search-forward search-pat 
					     (+ start-pos offset) t))
	      (setq offset (* 3 offset)))
	    (or found
		(re-search-forward search-pat nil t)
		(error "%s not found in %s" tag tag-file))
	    (setq last-tag tag)
	    (setq tags-loop-form '(find-tag nil t))
	    (beginning-of-line)
	    (` (progn
		 (switch-to-buffer (, (current-buffer))) 
		 (push-mark) (goto-char (, (point)))
		 (run-hooks 'find-tag-hook)))
	    )))))))

;;=== Compilation errors ==================================================
;;
;; Installation:
;;
;;   In your .emacs, or hyperB-load-hooks:
;;
;;      (hyperB:add-global-handler 'hyperB:goto-error)
;;
;;  NOTE: Because the *compilation* buffer has no startup hook, this
;;        handler has to be global
;;

(defun hyperB:goto-error ()
  (interactive)
  (if (not (equal (buffer-name) "*compilation*")) nil
    ;; Make sure we have a parsed error-list
    (if (eq compilation-error-list t)
	(progn (compilation-forget-errors)
	       (setq compilation-parsing-end 1)))
    (if (not compilation-error-list)
	(save-excursion
	  (set-buffer-modified-p nil)
	  (compilation-parse-errors)))
    ;; Search error list for current position
    (let ((error-point (save-excursion (beginning-of-line) (point)))
	  (error-list compilation-error-list)
	  target)
      (while (and error-list (not target))
	(if (eq (marker-position (car (car error-list))) error-point)
	    (setq target (car (cdr (car error-list))))
	  (setq error-list (cdr error-list))))
      (cond
       ((not target) (error "No error on that line"))
       ((not (marker-buffer target)) 
	(error "Associated buffer no longer exists"))
       (t ;; Return code to bring up error text
	(` (progn
	     (switch-to-buffer (, (marker-buffer target)))
	     (goto-char (, (marker-position target)))))))
      )))

;;=========================================================================
;;=== Easy installation ===================================================

(defun install-all-hyperB-x-handlers ()
  "Install all the handlers defined in hyperB-x.el.
You may find calling this function easier than modifying your
hooks manually."
  (hyperB:add-hook 'Info-mode-hook
		   '(mapcar 'hyperB:add-local-handler
			    '(hyperB:Info-header
			      hyperB:Info-menu
			      hyperB:Info-note)))
  (hyperB:add-hook 'emacs-lisp-mode-hook
		   '(mapcar 'hyperB:add-local-handler
			    '(hyperB:find-tag
			      hyperB:find-elisp-library)))
  (hyperB:add-hook 'c-mode-hook
		   '(mapcar 'hyperB:add-local-handler
			    '(hyperB:find-tag
			      hyperB:find-header-file)))
  (hyperB:add-hook 'gdb-mode-hook 
		   '(mapcar 'hyperB:add-local-handler
			    '(hyperB:find-tag)))
  (hyperB:add-global-handler 'hyperB:goto-error))
			
;;=== END of hyperB-x.el ==================================================
