/*
 * rmnt.c -- readmnt() function for lsof library
 */


/*
 * Copyright 1997 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */


#include "../machine.h"

#if	defined(USE_LIB_READMNT)

# if	!defined(lint)
static char copyright[] =
"@(#) Copyright 1997 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: rmnt.c,v 1.2 97/09/23 04:08:35 abe Exp $";
# endif	/* !defined(lint) */

#include "../lsof.h"



/*
 * The caller may define an RMNT_EXPDEV macro to expand (ala EP/IX)
 * device numbers.
 *
 * EP/IX, for example, uses:
 *
 *	#define RMNT_EXPDEV(n) expdev(n)
 *
 * The caller may define a custom macro for making decisions to
 * skip entries -- e.g., ones whose mnt_type is MNTTYPE_IGNORE.
 */

#if	!defined(RMNT_EXPDEV)
#define	RMNT_EXPDEV(n)	n
#endif	/* !defined(RMNT_EXPDEV) */


/*
 * readmnt() - read mount table
 */

int
readmnt()
{
	char *dn = NULL;
	int err = 0;
	char *ln;
	FILE *mfp;
	struct mntent *mp;
	struct mounts *mtp;
	char *opt, *opte;
	struct stat sb;
/*
 * Open access to the mount table.
 */
	if ((mfp = setmntent(MOUNTED, "r")) == NULL) {
		(void) fprintf(stderr, "%s: can't access %s\n", Pn, MOUNTED);
		return(0);
	}
/*
 * Read mount table entries.
 */
	while ((mp = getmntent(mfp)) != NULL) {

#if	defined(MNTSKIP)
	/*
	 * Specfy in the MNTSKIP macro the decisions needed to determine
	 * that this entry should be skipped.
	 *
	 * Typically entries whose mnt_type is MNTTYPE_IGNORE are skipped.
	 *
	 * The MNTSKIP macro allows the caller to use other tests.
	 */
		MNTSKIP
#endif	/* MNTSKIP */

	/*
	 * Interpolate a possible symbolic directory link.
	 */
		if (dn)
			(void) free((FREE_P *)dn);
		if ((dn = (char *)malloc((MALLOC_S)(strlen(mp->mnt_dir) + 1)))
		== NULL) {
			err = 1;
			break;
		}
		(void) strcpy(dn, mp->mnt_dir);
		if ((ln = Readlink(dn)) == NULL) {
		    if (!Fwarn){
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
		    }
		    err = 2;
		    continue;
		}
		if (ln != dn) {
			(void) free((FREE_P *)dn);
			dn = ln;
		}
	/*
	 * Stat() the directory.
	 */
		if (statsafely(dn, &sb)) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "%s: WARNING: can't stat() %s file system %s\n",
			    Pn, mp->mnt_type, mp->mnt_dir);
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
		    }
		    err = 2;
		    if ((opt = strstr(mp->mnt_opts, "dev=")) != NULL) {
			(void) zeromem(&sb, sizeof(sb));
			if ((opte = x2dev(opt + 4, (dev_t *)&sb.st_dev))) {
			    sb.st_mode = S_IFDIR | 0777;

#if	defined(HASFSTYPE)
			    (void) strncpy(sb.st_fstype, mp->mnt_type,
				sizeof(sb.st_fstype));
			    sb.st_fstype[sizeof(sb.st_fstype) - 1 ] = '\0';
#endif	/* defined(HASFSTYPE */

			    if (!Fwarn)
				(void) fprintf(stderr,
				    "      assuming \"%.*s\" from %s\n",
				    (opte - opt), opt, MOUNTED);
			} else
			    opt = NULL;
		    }
		    if (opt == NULL)
			continue;
		}
	/*
	 * Allocate and fill a local mount structure.
	 */
		if ((mtp = (struct mounts *)malloc(sizeof(struct mounts)))
		== NULL) {
			err = 1;
			break;
		}
		if ((mtp->fsname =
			(char *)malloc((MALLOC_S)(strlen(mp->mnt_fsname)+1)))
		== NULL) {
			err = 1;
			break;
		}
		(void) strcpy(mtp->fsname, mp->mnt_fsname);
		mtp->dir = dn;
		dn = NULL;
		mtp->next = Mtab;
		mtp->dev = RMNT_EXPDEV(sb.st_dev);
		mtp->rdev = RMNT_EXPDEV(sb.st_rdev);
		mtp->inode = sb.st_ino;
		mtp->mode = sb.st_mode;
		Mtab = mtp;
	}
	(void) endmntent(mfp);
	if (dn)
		(void) free((FREE_P *)dn);
/*
 * Handle errors.
 */
	switch (err) {
	case 1:
		(void) fprintf(stderr, "%s: no space for mount at %s (%s)\n",
			Pn, mp->mnt_fsname, mp->mnt_dir);
		return(0);
	case 2:
		return(1);
	}
	return(1);
}
#else	/* !defined(USE_LIB_READMNT) */
static char d1[] = "d"; static char *d2 = d1;
#endif	/* defined(USE_LIB_READMNT) */
