#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/errno.h>
#include <stdio.h>
#include <netdb.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <pwd.h>
#include <syslog.h>
#if (defined(sun) && !defined(SOLARIS)) || defined(sgi)
#include <strings.h>
#else
#include <string.h>
#endif
#include "socks.h"
#ifdef LINUX
#include <linux/time.h>
#endif

/* >>> K. Shackelford */
#if defined(hpux) || defined(ultrix) || defined (__NetBSD__) || defined(AIX)
extern int h_errno;
#endif
/* <<< K.Shackelford */

#define NAMELEN	128
char socks_dst_name[NAMELEN], socks_dst_serv[NAMELEN];
char socks_src_name[NAMELEN], socks_src_user[NAMELEN], socks_real_user[NAMELEN];
char *socks_def_server;
char *socks_server;
char *socks_serverlist;

char socks_cmd[] = "connect";

extern int 	errno;
extern char	*getenv();
extern char	*getlogin();
static struct sockaddr_in	cursin;
unsigned long		SocksHost;
static unsigned short socks_port;
static unsigned short	socks_conn_port = 0;

struct sockaddr_in socks_nsin;
static struct sockaddr_in	me;
static struct passwd		*pw;
static int	direct = 0;

extern	char	*porttoserv();
extern	char	*saddrtoname();


int	check_result(code)
char code;
{
	switch (code) {
		case SOCKS_FAIL:
/*
			errno = ETIMEDOUT;
*/
			errno = ECONNREFUSED;
			return -1;
		case SOCKS_NO_IDENTD:
			errno = ECONNREFUSED;
			fprintf(stderr, "Error: SOCKS proxy server cannot connect to identd on your machine.\n");
			return -1;
		case SOCKS_BAD_ID:
			errno = ECONNREFUSED;
			fprintf(stderr, "Error: user-id does not agree with the one reported by identd on your machine.\n");
			return -1;
		default:
			return 0;
	}
}

/*
	SOCKSinit() must be called once in the application program.
 */

SOCKSinit(Progname)
char *Progname; /* name of the calling program, "rfinger", "rftp", etc. */
{
#ifdef SOCKS_DEFAULT_NS
	static char	defaultNS[] = SOCKS_DEFAULT_NS;
#endif
	static char	defaultSERVER[] = SOCKS_DEFAULT_SERVER;
	char		*cp, *ns = (char *)0;
	struct hostent	*hp;
	struct servent	*sp;
	int		v,uid;

	socks_port = htons(SOCKS_DEF_PORT);

	/* skip the path if included in Progname */
	if( (cp = rindex(Progname, '/')) == NULL)
		cp = Progname;
	else
		cp++;

#ifndef LOG_DAEMON
	(void) openlog(cp, LOG_PID);
#else
	(void) openlog(cp, LOG_PID, LOG_DAEMON);
#endif

	gethostname(socks_src_name, sizeof(socks_src_name));
	if ( (hp = gethostbyname(socks_src_name)) == NULL ) {
		fprintf (stderr, "gethostbyname(%s): error #%d\n",
			socks_src_name, h_errno);
		return (1);
	}
	bcopy(hp->h_addr_list[0], &me.sin_addr.s_addr, hp->h_length);

	res_init();
#ifdef sgi
	sethostresorder("local:nis:bind");
#endif

	if ((cp = getenv("SOCKS_NS")) != NULL) {
		ns = cp;
#ifdef SOCKS_DEFAULT_NS
	} else {
		ns = defaultNS;
#endif
	}

	if (ns != NULL) {
#ifdef ultrix
		_res.ns_list[0].addr.sin_addr.s_addr = inet_addr(ns);
#else
		_res.nsaddr_list[0].sin_addr.s_addr = inet_addr(ns);
#endif
		_res.nscount = 1;
	}

/* >>> jon r. luini <jonl@hal.com> */
#ifdef	SOCKS_DEFAULT_DNAME
	bzero (_res.defdname, sizeof (_res.defdname));
	strncpy (_res.defdname, SOCKS_DEFAULT_DNAME, sizeof (_res.defdname)-1);
#endif
/* <<< jon r. luini <jonl@hal.com> */


	if ((socks_def_server = getenv("SOCKS_SERVER")) == NULL)
		socks_def_server = defaultSERVER;
	fprintf(stderr, "CSTC version %s SOCKS client. Default SOCKS server: %s\n",
		CSTC_RELEASE, socks_def_server);
	socks_server = socks_def_server;

	if ((hp = gethostbyname(socks_server)) == NULL) {
		SocksHost = inet_addr(socks_server);
	} else {
		bcopy(hp->h_addr_list[0], &SocksHost, hp->h_length);
	}

	if ((sp = getservbyname("socks", "tcp")) != NULL)
		socks_port = sp->s_port;

	if ((cp = getlogin()) == NULL) {
		if ((pw = getpwuid(uid=getuid())) == NULL) {
			fprintf(stderr, "Unknown user-id %d\n",uid);
			return (1);
		}
		cp = pw->pw_name;
	}
	strncpy(socks_real_user, cp, sizeof(socks_real_user));

	if ((pw = getpwuid(uid=geteuid())) == NULL) {
		fprintf(stderr, "Unknown user-id %d\n",uid);
		return (1);
	}
	strncpy(socks_src_user, pw->pw_name, sizeof(socks_src_user));

	socks_nsin.sin_family = AF_INET;
	socks_nsin.sin_port = socks_port;
	socks_nsin.sin_addr.s_addr = SocksHost;
	
}


int connect_sockd(sock)
int	sock;
/* returns 0 if successfully connected to a SOCKS server,
   returns -1 otherwise
 */
{
#ifndef VERSATILE_CLIENTS
	if (connect(sock, &socks_nsin, sizeof(struct sockaddr_in)) == 0)
		return 0;
	else {
		syslog(LOG_NOTICE, "Failed to connect to sockd at %s: %m",
			socks_server);
		return -1;
	}
#else /* Now the version when VERSATILE_CLIENTS is defined */
	int	last = 0;
	int	new_sock;
	struct hostent	*hp;

/*
	while (1) {
		if ((socks_server = socks_serverlist) != NULL) {
			if (socks_serverlist = index(socks_serverlist, ','))
				*socks_serverlist++ = '\0';
			if ((hp = gethostbyname(socks_server)) == NULL)
				socks_nsin.sin_addr.s_addr = inet_addr(socks_server);
			else
				bcopy(hp->h_addr_list[0], &socks_nsin.sin_addr, hp->h_length);
		} else {
			socks_server = socks_def_server;
			socks_nsin.sin_addr.s_addr = SocksHost;
			last = 1;
		}
	
		if (connect(sock, (struct sockaddr *)&socks_nsin, sizeof(struct sockaddr_in)) == 0)
			return 0;
		else {
			syslog(LOG_NOTICE, "Failed to connect to sockd at %s: %m",
				socks_server);
			if (last) {
				return -1;
			}
			if (errno == ECONNREFUSED) {
				if ((new_sock = socket(PF_INET, SOCK_STREAM, 0)) < 0)
					return -1;
				if (dup2(new_sock, sock) < 0)
					return -1;
				close(new_sock);
			}
		}
	}
*/
	while (socks_server = socks_serverlist) {
		if (socks_serverlist = index(socks_serverlist, ','))
			*socks_serverlist++ = '\0';
		if ((hp = gethostbyname(socks_server)) == NULL)
			socks_nsin.sin_addr.s_addr = inet_addr(socks_server);
		else
			bcopy(hp->h_addr_list[0], &socks_nsin.sin_addr, hp->h_length);
		if (connect(sock, (struct sockaddr *)&socks_nsin, sizeof(struct sockaddr_in)) == 0)
			return 0;
		else {
#ifdef SVR4
			if ((errno == EISCONN) || (errno == EINPROGRESS) || (errno == EAGAIN))
#else
			if ((errno == EISCONN) || (errno == EINPROGRESS))
#endif
				return -1;
			syslog(LOG_NOTICE, "Failed to connect to sockd at %s: %m",
				socks_server);
			if (!(socks_serverlist)) {
				return -1;
			}
			if (errno == ECONNREFUSED) {
				if ((new_sock = socket(PF_INET, SOCK_STREAM, 0)) < 0)
					return -1;
				if (dup2(new_sock, sock) < 0)
					return -1;
				close(new_sock);
			}
		}
	}
#endif /* #ifndef VERSATILE_CLIENTS */
	
}


Rconnect(sock, sin, size)
int			sock;
struct sockaddr_in	*sin;
int			size;
{
	Socks_t		dst;
	int	i;
	int	res_ret, con_ret, con_errno;

/*
	if ((size != sizeof(struct sockaddr_in))||(sin->sin_family != AF_INET)){
		errno = EAFNOSUPPORT;
		return(-1);
	}
*/
	strcpy(socks_cmd, "connect");
	saddrtoname(&sin->sin_addr, socks_dst_name, sizeof(socks_dst_name));
	porttoserv(sin->sin_port, socks_dst_serv, sizeof(socks_dst_serv));
#ifdef DEBUG
	fprintf(stderr, "Rconnect(%d, %s, %s)\n", sock, socks_dst_name, socks_dst_serv);
#endif

#ifdef VERSATILE_CLIENTS
	direct = check_cconf(&me, sin);
#ifdef DEBUG
	fprintf(stderr, "Rconnect(): direct = %d\n", direct);
#endif
	if (direct < 0) {
		syslog(LOG_NOTICE, "refused -- connect() from %s(%s) to %s (%s)",
			socks_src_user, socks_real_user, socks_dst_name, socks_dst_serv);
		errno = ECONNREFUSED;
		return(-1);
	}

	if (direct == 1) {
		syslog(LOG_NOTICE, "connect() directly from %s(%s) to %s (%s)",
			socks_src_user, socks_real_user, socks_dst_name, socks_dst_serv);
		return(connect(sock, (struct sockaddr *)sin, size));
	}
#endif /* #ifdef VERSATILE_CLIENTS */
	
	if(((con_ret = connect_sockd(sock)) < 0) && (errno != EISCONN))
		return -1;
	con_errno = errno;
	syslog(LOG_NOTICE, "connect() from %s(%s) to %s (%s) using sockd at %s",
			socks_src_user, socks_real_user, socks_dst_name, socks_dst_serv, socks_server);

	dst.version = SOCKS_VERSION;
	dst.cmd = SOCKS_CONNECT;
	dst.port = sin->sin_port;
	dst.host = sin->sin_addr.s_addr;

	SendDst(sock, &dst);
	write(sock, socks_src_user, strlen(socks_src_user) + 1);

	GetDst(sock, &dst);

	if (dst.cmd == SOCKS_RESULT)
		socks_conn_port = sin->sin_port;
	if(res_ret = check_result(dst.cmd))
		return(res_ret);
	else {
		errno = con_errno;
		return(con_ret);
	}
}

/*
**  Set up a bind for a remote host, add fill 'cursin' in with the
**   remote server information.
*/
Rbind(sock, sin, size, remhost)
int			sock;
struct sockaddr_in	*sin;
int			size;
unsigned long		remhost; /* as in sin_addr.s_addr */
{
	Socks_t			dst;
	struct sockaddr_in	psin;
	int	i;

	strcpy(socks_cmd, "bind");
	saddrtoname(&remhost, socks_dst_name, sizeof(socks_dst_name));
	porttoserv(socks_conn_port, socks_dst_serv, sizeof(socks_dst_serv));
#ifdef DEBUG
	fprintf(stderr, "Rbind(%d, %s, %s)\n", sock, socks_dst_name, socks_dst_serv);
#endif

	psin.sin_addr.s_addr = remhost;
	psin.sin_port = socks_conn_port;
#ifdef VERSATILE_CLIENTS
	direct = check_cconf(&me, &psin);
#ifdef DEBUG
	fprintf(stderr, "Rbind(): direct = %d\n", direct);
#endif
	if (direct < 0) {
		syslog(LOG_NOTICE, "Refused -- bind() from %s(%s) for %s (%s)",
			socks_src_user, socks_real_user, socks_dst_name, socks_dst_serv);
		errno = ECONNREFUSED;
		return -1;
	}

	if (direct == 1) {
		syslog(LOG_NOTICE, "bind() directly from %s(%s) for %s (%s)",
			socks_src_user, socks_real_user, socks_dst_name, socks_dst_serv);
		return (bind(sock, (struct sockaddr *)sin, size));
	}
#endif /* #ifdef VERSATILE_CLIENTS */
	
	if (connect_sockd(sock) < 0 )
		return -1;
	syslog(LOG_NOTICE, "bind() from %s(%s) for %s (%s) using sockd at %s",
		socks_src_user, socks_real_user, socks_dst_name, socks_dst_serv, socks_server);

	dst.version = SOCKS_VERSION;
	dst.cmd     = SOCKS_BIND;
	dst.port    = socks_conn_port;
	dst.host    = remhost;

	SendDst(sock, &dst);
	write(sock, socks_src_user, strlen(socks_src_user) + 1);

	GetDst(sock, &dst);
	cursin.sin_family = AF_INET;
	cursin.sin_port = dst.port;
	if (ntohl(dst.host) == INADDR_ANY)
		cursin.sin_addr.s_addr = socks_nsin.sin_addr.s_addr;
	else
		cursin.sin_addr.s_addr = dst.host;

	return (check_result(dst.cmd));
}

/*
**  Stub routine since the listen will have alread succeded on the
**   server.
*/
Rlisten(s, n)
int	s, n;
{
#ifdef DEBUG
	fprintf(stderr, "direct=%d, Rlisten(%d, %d)\n", direct, s, n);
#endif
#ifdef VERSATILE_CLIENTS
	if (direct)
		return (listen(s, n));
#endif /* #ifdef VERSATILE_CLIENTS */

	return 0;
}

/*
**  Well we know where we got a connection from.
*/
Rgetsockname(sock, sin, size)
int			sock;
struct sockaddr_in	*sin;
int			*size;
{
#ifdef DEBUG
	saddrtoname(&sin->sin_addr, socks_dst_name, sizeof(socks_dst_name));
	porttoserv(sin->sin_port, socks_dst_serv, sizeof(socks_dst_serv));
	fprintf(stderr, "direct= %d, Rgetsockname(%d, %s, %s)\n",
		direct, sock, socks_dst_name, socks_dst_serv);
#endif
#ifdef VERSATILE_CLIENTS
	if (direct)
		return (getsockname(sock, (struct sockaddr *)sin, size));
#endif /* #ifdef VERSATILE_CLIENTS */

	*size = sizeof(struct sockaddr_in);
	*sin = cursin;

	return 0;
}

/*
**  Do an accept, which is really a select for some data on
**    the present socket.
*/
Raccept(sock, sin, size)
int			sock;
struct sockaddr_in	*sin;
int			*size;
{
	fd_set		fds;
	Socks_t		dst;
	int		fdsbits = sock + 1;

#ifdef DEBUG
	saddrtoname(&sin->sin_addr, socks_dst_name, sizeof(socks_dst_name));
	porttoserv(sin->sin_port, socks_dst_serv, sizeof(socks_dst_serv));
	fprintf(stderr, "direct= %d, Raccept(%d, %s, %s)\n",
		direct, sock, socks_dst_name, socks_dst_serv);
#endif

#ifdef VERSATILE_CLIENTS
	if (direct)
		return(accept(sock, (struct sockaddr *)sin, size));
#endif /* #ifdef VERSATILE_CLIENTS */

	FD_ZERO(&fds);
	FD_SET(sock, &fds);

	if (select(fdsbits, &fds, NULL, NULL, NULL) > 0)
		if (FD_ISSET(sock, &fds)) {
			GetDst(sock, &dst);
			sin->sin_family = AF_INET;
			sin->sin_port = dst.port;
			sin->sin_addr.s_addr = dst.host;

			return dup(sock);
		}
	return -1;
}

/*===========================================================*/
