/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1993.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XFTP shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XFTP for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include <sys/param.h>
#include <Xm/Xm.h>
#include <Xm/SelectioB.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/PushBG.h>
#include <Xm/List.h>
#include "xftp.h"
#include "xfer.h"

struct {
	int host;
	struct entry_link *head;
} move_ctrl;

static struct {
	Widget w_dialog;
	Widget w_form;
	Widget w_instruction;
	Widget w_dirNameForm;
	Widget w_dirNameMenu;
	Widget w_dirNameMenuItem[MAXLINKS];
	Widget w_dirName;
	Widget w_dirList;
} move;

static char *move_help[] = {
	"This dialog enables you to move entries from the associated",
	"host's current directory to another directory on the same",
	"host.  This is similar to the UNIX \"mv\" command, but",
	"without rename capability.  (Select RENAME SELECTED ENTRY(S)",
	"in the host's OPS menu to rename entries.)\n",
	"\n",
	"Display the contents of the directory you wish to move the",
	"selected items to.  Traverse down the directory structure by",
	"double-clicking on a subdirectory name in the directory list.  ",
	"Traverse up the directory structure by selecting an item in",
	"the directory name menu (the directory name menu contains",
	"each link of the path of the displayed directory).\n",
	"\n",
	"Click on the MOVE button to perform the move and remove the",
	"dialog.  Click on the CANCEL button to remove the dialog",
	"without performing the move.",
	NULL
};


extern Widget w_toplev;
extern Widget w_dirList[];
extern struct st_host_info hinfo[];
extern int inquire_on_move;
extern XtAppContext app;
extern int beep_when_ops_done;
extern int diagnostics;
extern int symbol_mode;

char **parse_path_to_links();
void cb_move_ok();
void cb_move_help();
void cb_move_set_dir();
void cb_move_double_click();
int cb_move_entries();


/*
 * cb_move_selected_entries - Callback to move selected entries.
 */
void
cb_move_selected_entries(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	int host = (int)client_data;
	int retval;

    /* Clear error flag */
    raise_okflag();

	/* Ask the user if okay to move */
	if (inquire_on_move) {
		if (!verify_selection(host, "Do you really want to move these items?"))
			return;
	}

    create_move_dialog();

	/* Remember the host */
	XtVaSetValues(move.w_dialog, XmNuserData, host, NULL);

	/* Try to initialize destination directory to last one used */
	if (hinfo[host].move_wd != NULL) {
	    use_busy_cursor();
		retval = change_move_directory(host, hinfo[host].move_wd);
		restore_prev_cursor();
		switch (retval) {
		case 0:
			XtManageChild(move.w_dialog);
			add_dialog_to_list(move.w_dialog);
			return;
		case -3:
        	lost_connection(host);
			return;
		}
	}

	/* Initialize destination directory to current working directory */
    use_busy_cursor();
	retval = change_move_directory(host, hinfo[host].wd);
	restore_prev_cursor();
	if (retval < 0) {
		lost_connection(host);
		return;
	}
	XtManageChild(move.w_dialog);
	add_dialog_to_list(move.w_dialog);
}


/*
 * create_move_dialog - Create dialog that prompts for destination
 *                      directory for move operation.
 */
create_move_dialog()
{
	static int initialized = False;
    XmString label;
    int i;
    Arg args[1];
	Widget widget;

    /* Create dialog only once */
    if (initialized)
        return;
    initialized = True;

    /* Create prompt dialog to get destination directory */
    i = 0;
    XtSetArg(args[i], XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL); i++;
    move.w_dialog = XmCreatePromptDialog(w_toplev, "move", args, i);
    XtAddCallback(move.w_dialog, XmNokCallback, cb_move_ok, NULL);
	XtAddCallback(move.w_dialog, XmNhelpCallback, cb_move_help, NULL);

    /* Don't show prompt dialog's selection label and text */
    widget = XmSelectionBoxGetChild(move.w_dialog, XmDIALOG_SELECTION_LABEL);
    XtUnmanageChild(widget);
    widget = XmSelectionBoxGetChild(move.w_dialog, XmDIALOG_TEXT);
    XtUnmanageChild(widget);

	/* Monitor dialog to control help dialog modality */
	register_as_modal(move.w_dialog);

    /* Kludge to add title bar under OLWM */
    AddOLWMDialogFrame(move.w_dialog);

    /* Create form for control area */
    move.w_form = XtVaCreateWidget(
        "form",
        xmFormWidgetClass,
        move.w_dialog,
        NULL
    );

	/* Create label that gives instruction */
	move.w_instruction = XtVaCreateManagedWidget(
		"instruction",
		xmLabelWidgetClass,
		move.w_form,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		XmNtopOffset,		10,
		NULL
	);

    /* Create form for directory name option menu (this kluge forces resize) */
    move.w_dirNameForm = XtVaCreateManagedWidget(
        "dirNameForm",
        xmFormWidgetClass,
        move.w_form,
        XmNtopAttachment,   XmATTACH_WIDGET,
        XmNtopWidget,       move.w_instruction,
        XmNleftAttachment,  XmATTACH_FORM,
        XmNrightAttachment, XmATTACH_FORM,
        XmNtopOffset,       14,
        NULL
    );

    /* Create the directory name pulldown menu */
    move.w_dirNameMenu = XmCreatePulldownMenu(move.w_dirNameForm,
        "dirNameMenu", NULL, 0);
    label = XmStringCreateSimple("Dummy");
    for (i=0; i<MAXLINKS; i++) {
        move.w_dirNameMenuItem[i] = XtVaCreateWidget(
            "",
            xmPushButtonGadgetClass,
            move.w_dirNameMenu,
            XmNlabelString, label,
            NULL
        );
        XtAddCallback(move.w_dirNameMenuItem[i], XmNactivateCallback,
            cb_move_set_dir, NULL);
    }
    XmStringFree(label);

    /* Create directory name option menu */
    i = 0;
    XtSetArg(args[i], XmNsubMenuId, move.w_dirNameMenu); i++;
    move.w_dirName = XmCreateOptionMenu(move.w_dirNameForm, "dirName", args, i);
    XtManageChild(move.w_dirName);

    /* Create directory list  */
    i = 0;
    XtSetArg(args[i], XmNlistSizePolicy, XmCONSTANT); i++;
    move.w_dirList = XmCreateScrolledList(move.w_form, "dirList", args, i);
    XtVaSetValues(XtParent(move.w_dirList),
        XmNleftAttachment,      XmATTACH_FORM,
        XmNrightAttachment,     XmATTACH_FORM,
        XmNbottomAttachment,    XmATTACH_FORM,
        XmNleftOffset,          0,
        XmNrightOffset,         0,
        XmNtopAttachment,       XmATTACH_WIDGET,
        XmNtopWidget,           move.w_dirNameForm,
        NULL
    );
    XtManageChild(move.w_dirList);
    fix_list_color(move.w_dirList);

    /* Add double-click callback to directory list */
    XtAddCallback(
        move.w_dirList,
        XmNdefaultActionCallback,
        cb_move_double_click,
        (XtPointer)NULL
    );

    XtManageChild(move.w_dirList);
	XtManageChild(move.w_form);
}


/*
 * cb_move_set_dir - Callback to change the destination directory for the
 *                   move command in response to the user selection a
 *                   directory link.
 */
void
cb_move_set_dir(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    int index;
    char **wd_links;
    int len = 0;
    char *new_wd;
    int i;
	int retval;
	int host;

    /* Clear error flag */
    raise_okflag();

	/* Get host */
	XtVaGetValues(move.w_dialog, XmNuserData, &host, NULL);

    /* Get position of selected link in path */
    XtVaGetValues(widget, XmNuserData, &index, NULL);

    /* Parse working directory path */
    if ((wd_links = parse_path_to_links(hinfo[host].move_wd)) == NULL)
        fatal_error("Unable to parse working directory");

    /* Determine length of path name */
    for (i=0; i<index; i++)
        len += strlen(wd_links[i])+1;

    /* Build path name for new working directory */
    new_wd = XtMalloc(len+1);
    strcpy(new_wd, "");
    for (i=0; i<index; i++) {
        strcat(new_wd, wd_links[i]);
        if (new_wd[strlen(new_wd)-1] != '/')
            strcat(new_wd, "/");
    };

    /* Free up memory returned by parse_path_to_links */
    release_path_links(wd_links);

    /* Try to cd to specified directory */
    use_busy_cursor();
	retval = change_move_directory(host, new_wd);
	restore_prev_cursor();
	XtFree(new_wd);
	switch (retval) {
	case -1:
        warning_error("Unable to change directory");
		break;
	case -3:
		XtUnmanageChild(move.w_dialog);
        lost_connection(host);
	}

	return;
}


/*
 * cb_move_double_click - Callback to change the destination directory for the
 *                        move command in response to the user double- clicking
 *                        on a directory entry.
 */
void
cb_move_double_click(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    XmListCallbackStruct *cbs = (XmListCallbackStruct *)call_data;
    char *entry;
    char *new_wd;
	int retval;
	int host;

    /* Clear error flag */
    raise_okflag();

	/* Get host */
	XtVaGetValues(move.w_dialog, XmNuserData, &host, NULL);

    /* Get entry that was double-clicked */
    if (XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &entry) == False)
        fatal_error("Trouble in cb_move_double_click()");
    strip_off_symbol(entry);

    /* Form full path name */
    new_wd = XtMalloc(strlen(hinfo[host].move_wd)+strlen(entry)+2);
    strcpy(new_wd, hinfo[host].move_wd);
    if (new_wd[strlen(new_wd)-1] != '/')
        strcat(new_wd, "/");
    strcat(new_wd, entry);
    XtFree(entry);

    /* Try to cd to specified directory */
    use_busy_cursor();
	retval = change_move_directory(host, new_wd);
	restore_prev_cursor();
	XtFree(new_wd);
	switch (retval) {
	case -1:
        warning_error("Unable to change directory");
		break;
	case -3:
		XtUnmanageChild(move.w_dialog);
        lost_connection(host);
	}

	return;
}


/*
 * cb_move_ok - Callback to actually initiate the move command.
 */
void
cb_move_ok(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	XmStringTable selected_items;
	int nselected_items;
	int i;
	int host;
	struct entry_link *ptr;
	struct entry_link *head;

	/* Get host */
	XtVaGetValues(move.w_dialog, XmNuserData, &host, NULL);

	/* Check for identical source and sink directories */
	if (strcmp(hinfo[host].move_wd, hinfo[host].wd) == 0) {
		warning_error("Source and destination directories are the same");
		return;
	}

	/* This might take some time */
	use_busy_cursor();

	/* Pop up multipurpose monitor */
	show_mp_monitor("Now Moving:");
	clear_mp_monitor();

	/* Disable controls */
	enable_controls(False);

	/* Form linked list of selected entries */
    XtVaGetValues(
        w_dirList[host],
        XmNselectedItemCount, &nselected_items,
        XmNselectedItems,     &selected_items,
        NULL
    );
	head = NULL;
	for (i=nselected_items-1; i>=0; i--) {
		ptr = XtNew(struct entry_link);
        if (XmStringGetLtoR(selected_items[i], XmSTRING_DEFAULT_CHARSET,
            &ptr->entry) == False)
            fatal_error("Trouble in cb_move_ok()");
        strip_off_symbol(ptr->entry);
        ptr->next = head;
        head = ptr;
	}

	/* Set up move control block */
	move_ctrl.host = host;
	move_ctrl.head = head;

	/* Perform move */
	XtAppAddWorkProc(app, (XtWorkProc)cb_move_entries, NULL);
}


/*
 * update_move_dirname_menu - Update the destination directory name menu
 *                            for the move command dialog for the specified
 *                            host.
 */
update_move_dirname_menu(host)
int host;
{
    XmString    label;
    char        **wd_links;
    int         nlinks = 0;
    int         i;

    /* Parse working directory path. */
    if ((wd_links = parse_path_to_links(hinfo[host].move_wd)) == NULL)
        fatal_error("Unable to parse working directory");

    /* Count number of links in path name */
    while (wd_links[nlinks])
        nlinks++;

    /* Add the path links to menu in reverse order */
    for (i=0; i<nlinks; i++) {
        label = XmStringCreateSimple(wd_links[nlinks-1-i]);
        XtVaSetValues(move.w_dirNameMenuItem[i],
            XmNlabelString, label,
            XmNuserData,    nlinks-i,
            NULL
        );
        XmStringFree(label);
        XtManageChild(move.w_dirNameMenuItem[i]);
    }

    /* Unmanage unused menu items */
    for (i=nlinks; i<MAXLINKS; i++)
        XtUnmanageChild(move.w_dirNameMenuItem[i]);

    /* Free up memory returned by parse_path_to_links */
    release_path_links(wd_links);

    /* Make first menu item current */
    XtVaSetValues(
        move.w_dirName,
        XmNmenuHistory, move.w_dirNameMenuItem[0],
        NULL
    );
}


/*
 * update_move_dirlist - Update the list widget displaying the contents
 *                       of the destination directory for the move operation
 *                       on "host".  Returns 0 for success, -3 for broken
 *                       connection and -1 for other errors.
 */
update_move_dirlist(host)
int host;
{
    struct sl_struct *dlist;
    int retval;
    int i;
    XmString string;

    /* This might take some time */
    use_busy_cursor();

    /* Get short directory list */
    switch (hinfo[host].type) {
    case NEITHER:
		fatal_error("Bug in update_move_dirlist()");
    case LOCAL:
        if (local_ls(hinfo[host].move_wd, &dlist, symbol_mode) != 0)
            fatal_error("Unable to get local directory list");
        break;
    case REMOTE:
        if ((retval = remote_ls(host, hinfo[host].move_wd, &dlist,
				symbol_mode))) {
            restore_prev_cursor();
            return retval;
        }
    }

    /* Delete scrollinglist items */
    reset_list(move.w_dirList);

    /* Place directory entries into short list */
    for (i=0; i<dlist->nentries; i++) {
        string = XmStringCreateSimple(dlist->entries[i]);
        XmListAddItem(move.w_dirList, string, 0);
        XmStringFree(string);
    }

    /* Free directory list */
    release_string_list(dlist);

    restore_prev_cursor();
    return 0;
}


/*
 * change_move_directory - Change the move destination directory to "new_wd"
 *                         on "host".  Returns 0 if successful, -3 for lost
 *                         connection, and -1 for other errors.
 */
change_move_directory(host, new_wd)
int host;
char *new_wd;
{
    char *wd;
	int retval;

    /* cd into new directory */
    if (hinfo[host].type == LOCAL) {
        if (local_cd(host, new_wd, True) < 0)
			return -1;
        XtFree(hinfo[host].move_wd);
        if (local_pwd(&(hinfo[host].move_wd)) != 0)
            fatal_error("Unable to get local working directory");
    } else {
		if ((retval = remote_cd(host, new_wd, False, True)) < 0)
			return retval;
        if (remote_pwd(host, &wd) < 0)
            return -3;
        XtFree(hinfo[host].move_wd);
        hinfo[host].move_wd = wd;
    }

    /* Update the display */
    if (update_move_dirlist(host) < 0)
        return -3;
	update_move_dirname_menu(host);

	return 0;
}


/*
 * clean_up_move_ctrl - Free up memory used by the "Move" control block
 *                      data structure.
 */
clean_up_move_ctrl()
{
	struct entry_link *ptr;

	while (move_ctrl.head) {
		XtFree(move_ctrl.head->entry);
		ptr = move_ctrl.head;
		move_ctrl.head = move_ctrl.head->next;
		XtFree((char *)ptr);
	}
}


/*
 * cb_move_entries - Work proc to actually move directory entries.
 */
cb_move_entries()
{
    struct entry_link *ptr;
	char *src_path;
	char *snk_path;
	int retval;
	char msg[MAXPATHLEN+40];

    /* Are we done? */
    if (move_ctrl.head == NULL)
        goto done;

	/* Did user push abort button? */
	if (mp_abort_requested()) {
		hide_abort_dialog();
	    warning_error("Move(s) aborted");
		goto done;
	}

	/* Let user know where we are */
	update_mp_monitor(move_ctrl.head->entry);

	/* Create source and sink paths */
	src_path = XtMalloc(strlen(hinfo[move_ctrl.host].wd)
		+strlen(move_ctrl.head->entry)+2);
	strcpy(src_path, hinfo[move_ctrl.host].wd);
	strcat(src_path, "/");
	strcat(src_path, move_ctrl.head->entry);
	snk_path = XtMalloc(strlen(hinfo[move_ctrl.host].move_wd)
		+strlen(move_ctrl.head->entry)+2);
	strcpy(snk_path, hinfo[move_ctrl.host].move_wd);
	strcat(snk_path, "/");
	strcat(snk_path, move_ctrl.head->entry);

	/* Move entry */
	if (hinfo[move_ctrl.host].type == LOCAL)
		retval = local_rename(move_ctrl.host, src_path, snk_path);
	else
		retval = remote_rename(move_ctrl.host, src_path, snk_path);
	XtFree(src_path);
	XtFree(snk_path);
	switch (retval) {
	case -3:
		goto lost;
	case -1:
		sprintf(msg, "Unable to move %s", move_ctrl.head->entry);
		warning_error(msg);
		break;
	case 0:
		if (diagnostics >= NORMAL) {
			sprintf(msg, "*** Successfully moved:  %s\n",move_ctrl.head->entry);
			write_log(msg);
		}
	}
	ptr = move_ctrl.head;
	move_ctrl.head = move_ctrl.head->next;
	XtFree(ptr->entry);
	XtFree((char *)ptr);
	return False;

done:

	clean_up_move_ctrl();
	hide_mp_monitor();
	if (mp_abort_requested())
		hide_abort_dialog();
	enable_controls(True);
    if (update_dir_displays(move_ctrl.host, False) < 0)
        goto lost;
    update_xfer_controls();
    update_host_controls(move_ctrl.host);
    if (beep_when_ops_done)
        beep();
    restore_prev_cursor();
    return True;

lost:

    clean_up_move_ctrl();
    hide_mp_monitor();
	if (mp_abort_requested())
		hide_abort_dialog();
    enable_controls(True);
    clear_selected_entries(move_ctrl.host);
    restore_prev_cursor();
    lost_connection(move_ctrl.host);
    if (beep_when_ops_done)
        beep();
    return True;
}


/*
 * cb_move_help - Callback to display Move dialog' help package.
 */
void
cb_move_help(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    help_dialog("Move", move_help);
}


