/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1993.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XFTP shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XFTP for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <Xm/Xm.h>
#include "xftp.h"

#ifndef S_ISLNK
#define S_ISLNK(mode)   (((mode)&0170000) == 0120000) /* Kludge for nonPOSIX */
#endif

#ifndef S_ISSOCK
#define S_ISSOCK(mode)  (((mode)&0170000) == 0140000) /* Kludge for nonPOSIX */
#endif

#define MAXDIRLINE  200

extern int max_ftp_retries;
extern int diagnostics;

struct sl_struct *create_dlist();
int strcmp();

 
/* 
 * local_ls - Sets "dlist" to a pointer to a structure that contains a
 *            sorted list of directory entries in "directory" on the
 *            local host.  Set sym_mode to True if type symbol is to be
 *            appended to directory entries.  Returns 0 if successful,
 *            else returns -1.  If unable to read the directory because
 *            of lack of permission, returns 0 and a single entry named
 *            ".unreadable".
 */ 
local_ls(directory, dlist, sym_mode)
char *directory;
struct sl_struct **dlist;
int sym_mode;
{ 
	struct entry_link *head = NULL;
	DIR *dirp;
	struct dirent *dp;
	char entry[MAXPATHLEN];
	struct stat status;
	int retval;

	if ((dirp = opendir(directory)) == NULL) {
		if (errno == EACCES) {
			add_to_temp_dlist(&head, ".unreadable");
			goto done;
		} else
			return -1;
	}

	if (sym_mode)
		retval = chdir(directory);

	for (dp=readdir(dirp); dp!=NULL; dp=readdir(dirp))
		if (strcmp(dp->d_name, ".") != 0 && strcmp(dp->d_name, "..") != 0) {
			if (sym_mode && !retval && !lstat(dp->d_name, &status)) {
				strcpy(entry, dp->d_name);
				if (S_ISREG(status.st_mode)) {
					if ((S_IXUSR | S_IXGRP | S_IXOTH) & status.st_mode)
						strcat(entry, "*");
				} else if (S_ISDIR(status.st_mode))
					strcat(entry, "/");
				else if (S_ISLNK(status.st_mode))
					strcat(entry, "@");
				else if (S_ISSOCK(status.st_mode))
					strcat(entry, "=");
				else if ((S_IXUSR | S_IXGRP | S_IXOTH) & status.st_mode)
					strcat(entry, "*");
				add_to_temp_dlist(&head, entry);
			} else
	       		add_to_temp_dlist(&head, dp->d_name);
		}
	
	closedir(dirp);

done:
 
    /* Convert linked list into array */
    *dlist = create_dlist(&head);
 
    /* Sort entries */
    quicksort((*dlist)->entries, (*dlist)->nentries, strcmp);
	release_temp_dlist(&head);

    return 0;
}


/* 
 * local_dir - Sets "dlist" to a pointer to a structure that contains a
 *             sorted list of directory entries in "directory" on the
 *             local host.  Returns 0 if successful, else returns -1.
 */ 
local_dir(directory, dlist)    
char *directory;
struct sl_struct **dlist;
{
    struct entry_link *head = NULL;
    char dirline[MAXDIRLINE+1];
    FILE *fp;
    int len;
    int dummy;
	char cmd[MAXPATHLEN+14];

	sprintf(cmd, "/bin/ls -gal %s", directory);
    if ((fp = popen(cmd, "r")) == NULL)
		return -1;

    while (fgets(dirline, MAXDIRLINE, fp) != NULL) {
        len = strlen(dirline);
        if (dirline[len-1] == '\n')
            dirline[--len] = '\0';
		if (head != NULL)
			add_to_temp_dlist(&head, dirline);
		else if (sscanf(dirline, "total %d", &dummy) == 1)
			continue;
		else if (len >= 2 && strcmp(&(dirline[len-2]), " .") == 0)
			continue;
		else if (len >= 3 && strcmp(&(dirline[len-3]), " ..") == 0)
			continue;
		else
			add_to_temp_dlist(&head, dirline);
    }

    if (ferror(fp)) {
		release_temp_dlist(&head);
		return -1;
	}

    pclose(fp);

    /* Convert linked list into array */
    *dlist = create_dlist(&head);

	return 0;
}


/*
 * remote_dirlist - Set "dlist" to a pointer to a structure that
 *                  contains a list of entries in "directory" on the
 *                  remote host.  Set "request" to "NLST" for a short
 *                  list (e.g., like "ls" gives) or to "LIST" for a
 *                  long list (i.e., like "ls -lag" gives).  "host" is
 *                  the host id.  Set sym_mode to True if type symbol
 *                  is to be appended to directory entries.  Returns 0
 *                  if successful, -3 for broken connection, and -1 for
 *                  other errors.
 */
remote_dirlist(host, directory, request, dlist, sym_mode)
int host;
char *directory;
char *request;
struct sl_struct **dlist;
int sym_mode;
{
    char ftpline[MAXFTPLINE+1];
    struct entry_link *head;
    int data_fd;
    char ch;
    int nbytes;
    int code;
    char cmd[MAXPATHLEN+9];
    char reply[MAXFTPREPLY];
    int sock;
    int linelen;
    char tcpbuf[MAXTCPBUF];
    int len_tcpbuf;
    int index_tcpbuf;
    int dummy;
    int retval;
	int retries = 0;
	int scode;

    /* Verify request is valid */
    if (strcmp(request, "NLST") != 0 && strcmp(request, "LIST") != 0)
        return -1;

retry:

	/* Initialize linked list of directory entries */
	head = NULL;

    /* "cd" to directory */
    if ((retval = remote_cd(host, directory, False, True)) < 0)
        return retval;

    /* Initialize data connection */
    if ((sock = init_data_conn(host)) < 0)
        return sock;

    /* Send command to FTP server */
    sprintf(cmd, "%s", request);
	if (sym_mode && strcmp(request, "NLST") == 0)
		strcat(cmd, " -F");
    if ((retval = send_ftp_cmd(host, cmd)) < 0) {
        close(sock);
        return retval;
    }

    /* Get initial response from FTP server */
	scode = get_ftp_reply(host, reply, MAXFTPREPLY, &code, MAXT, False);
    switch (scode) {
    case 1:
        break;
    case 4:
		close(sock);
        if (code >= 450 && code < 460 && retries++ < max_ftp_retries) {
            if (diagnostics >= VERBOSE)
                write_log("*** Directory list failed.  Will retry.\n");
            goto retry;
        }
		if (diagnostics < VERBOSE)
			print_ftp_reply(host, reply);
        return -1;
	case 5:
		close(sock);
		if (code == 550)     /* Kludge to handle empty UNICOS directories */
			goto convert;
		else if (code == 501)
			goto convert;    /* Kludge to handle empty MacOS directories */
		if (diagnostics < VERBOSE)
			print_ftp_reply(host, reply);
		return -1;
    case -3:
        close(sock);
        return -3;
	case -2:
		report_client_timeout(host);
    default:
		if (scode > 0 && diagnostics < VERBOSE)
			print_ftp_reply(host, reply);
        close(sock);
        return -1;
    }

    /* Establish data connection */
    data_fd = accept_data_conn(host, sock);
    close(sock);
    if (data_fd < 0)
        return -1;

    /* Get directory list from FTP server */
    len_tcpbuf = 0;
    index_tcpbuf = 0;
    linelen = 0;
    while (1) {
        if (index_tcpbuf >= len_tcpbuf) {
            if ((nbytes = read_tcp(host, data_fd, tcpbuf, MAXTCPBUF, MAXT,
				False)) == -2) {
                release_temp_dlist(&head);
                goto abort;
            } else if (nbytes == -1) {
                release_temp_dlist(&head);
                close(data_fd);
                return -1;
            } else if (nbytes == 0)
                break;
            index_tcpbuf = 0;
            len_tcpbuf = nbytes;
        }
        ch = tcpbuf[index_tcpbuf++];
        if (ch == '\n') {
            ftpline[linelen] = '\0';
            if (diagnostics == DEBUG) {
                write_log(ftpline);
                write_log("\n");
            }
			/* Kludge to correct for Mac's NCSA Telnet */
			if (!sym_mode && linelen > 0 && ftpline[linelen-1] == '/')
				ftpline[linelen-1] = '\0';
            if (head != NULL || strcmp(request, "LIST") != 0
                             || sscanf(ftpline, "total %d", &dummy) != 1)
                if (strcmp(ftpline, ".") != 0 && strcmp(ftpline, "..") != 0)
                    add_to_temp_dlist(&head, ftpline);
            linelen = 0;
        } else if (ch != '\r') {
            if (linelen >= MAXFTPLINE) {
                release_temp_dlist(&head);
                close(data_fd);
                return -1;
            }
            ftpline[linelen++] = ch;
        }
    }

    /* Get final response from FTP server */
	scode = get_ftp_reply(host, reply, MAXFTPREPLY, &code, MAXT, False);
    switch (scode) {
    case 2:
        close(data_fd);
        break;
    case -3:
		release_temp_dlist(&head);
        close(data_fd);
        return -3;
	case -2:
		report_client_timeout(host);
    default:
		if (scode == 4 && code/10 == 45 && retries++ < max_ftp_retries) {
            if (diagnostics >= VERBOSE)
                write_log("*** Directory list failed.  Will retry.\n");
			release_temp_dlist(&head);
			close(data_fd);
            goto retry;
        }
		if (scode > 0 && diagnostics < VERBOSE)
			print_ftp_reply(host, reply);
		release_temp_dlist(&head);
        close(data_fd);
        return -1;
    }

convert:

    /* Convert linked list into array */
    *dlist = create_dlist(&head);

    return 0;

abort:

    /* Send urgent abort to server */
    urgent_abort(host);

    /* Clear out all incoming data */
    while (read_tcp(host, data_fd, tcpbuf, MAXTCPBUF, 10, False) > 0);
    close(data_fd);

    /* Get responses from server */
    retval = get_ftp_reply(host, reply, MAXFTPREPLY, &code, 15, False);
    if (retval > 0 && code != 421)
        retval = get_ftp_reply(host, reply, MAXFTPREPLY, &code, 5, False);

    if (retval == -3 || code == 421)
        return -3;
    else if (retval == -2)
		report_client_timeout(host);

    return -1;
}


/*
 * remote_ls - Sets "dlist" to a pointer to a structure that contains
 *             a sorted list of directory entries in "directory" on the
 *             remote host.  "host" is the host id.  Set sym_mode to
 *             True if type symbol is to be appended to directory
 *             entries.  Returns 0 if successful, -3 for broken
 *             connection, and -1 for other errors.
 */
remote_ls(host, directory, dlist, sym_mode)
int host;
char *directory;
struct sl_struct **dlist;
int sym_mode;
{
    int retval;

    /* Get short list */
    if ((retval = remote_dirlist(host, directory, "NLST", dlist, sym_mode)) < 0)
        return retval;

    /* Sort entries */
    quicksort((*dlist)->entries, (*dlist)->nentries, strcmp);

    return 0;
}


/*
 * remote_dir - Sets "dlist" to a pointer to a structure that contains 
 *              a long list of directory entries in "directory" on the
 *              remote host.  "host" is the host id.  Returns 0 if
 *              successful, -3 for broken connection, and -1 for other
 *				errors.
 */
remote_dir(host, directory, dlist)
int host;
char *directory;
struct sl_struct **dlist;
{
    int retval;

    /* Get extended list */
    if ((retval = remote_dirlist(host, directory, "LIST", dlist, False)) <0)
        return retval;

    return 0;
}


/*
 * add_to_temp_dlist - Add a link containing the specified entry name into
 *                     the temporary linked list containing the directory 
 *                     entries.  Returns 0 if successful, else -1;
 */
add_to_temp_dlist(head, name)
struct entry_link **head;
char *name;
{
	struct entry_link *ptr;

	ptr = XtNew(struct entry_link);
	ptr->entry = XtNewString(name);
	ptr->next = *head;
	*head = ptr;
}


/*
 * release_temp_dlist - Free all links of the temporary linked list 
 *                      containing the directory entries.  Sets head
 *                      to NULL.
 */
release_temp_dlist(head)
struct entry_link **head;
 {
	struct entry_link *ptr;

	while (*head) {
		ptr = *head;
		*head = (*head)->next;
		XtFree(ptr->entry);
		XtFree((char *)ptr);
	}
 }


/*
 * create_dlist - Create an array of pointers to directory entries from the
 *                temporary linked list of directory entries.  Returns a
 *                pointer to a struct.  This function releases the temporary
 *                linked list.
 */
struct sl_struct *
create_dlist(head)
struct entry_link **head;
{
	struct sl_struct *list;
	struct entry_link *ptr;
	int nentries = 0;
	int index = 0;

	/* Allocate structure */
	list = XtNew(struct sl_struct);

	/* Count the number of entries */
	ptr = *head;
	while (ptr) {
		nentries++;
		ptr = ptr->next;
	}
    list->nentries = nentries;

    /* Allocate array of pointers. */
    list->entries = (char **)XtMalloc(sizeof(char *)*nentries);

    /* Move entries names into arrays from linked list */
    index = nentries-1;
    while (*head) {
        list->entries[index--] = (*head)->entry;
        ptr = *head;
        *head = (*head)->next;
        XtFree((char *)ptr);
    }

	return list;
}

