/*
**  Copyright (c) 1991 Bolt Beranek and Newman, Inc.
**  All rights reserved.
**
**  Redistribution and use in source and binary forms are permitted
**  provided that: (1) source distributions retain this entire copyright
**  notice and comment, and (2) distributions including binaries display
**  the following acknowledgement:  ``This product includes software
**  developed by Bolt Beranek and Newman, Inc. and CREN/CSNET'' in the
**  documentation or other materials provided with the distribution and in
**  all advertising materials mentioning features or use of this software.
**  Neither the name of Bolt Beranek and Newman nor CREN/CSNET may be used
**  to endorse or promote products derived from this software without
**  specific prior written permission.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
**  WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/
/*
 * Copyright (c) 1992 Purdue University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Purdue University.  The name of the University may not be used
 * to endorse or promote products derived * from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Note: this copyright applies to portions of this software developed
 * at Purdue beyond the software covered by the original copyright.
 */
#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <setjmp.h>
#include <sgtty.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stream.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <net/if.h>
#include "dp_str.h"
#include "dpd.h"
#include "dp.h"

#ifdef	sun
#define GETDATAVAL(ifr)		(*(int *)ifr.ifr_data)
#define SETDATAVAL(ifr, i)	(*(int *)(ifr).ifr_data = (i))
#else
#define GETDATAVAL(ifr)		((int)ifr.ifr_data)
#define SETDATAVAL(ifr, i)	((ifr).ifr_data = (caddr_t)(i))
#endif	/* sun */

/*
** Local Variables
*/
static char	device[12];	/* Device name				*/
static char	tty[12];	/* Modem in use				*/
static int	mypid;
static int	dialfout;	/* Modem descriptor			*/
static long	startipkts;	/* Starting input packet count		*/
static long	startopkts;	/* Starting output packet count		*/
static struct ifreq ifr;
static jmp_buf	opentimeout;	/* Used for open timeouts		*/
static int	sock;		/* The global socket for use everywhere	*/
static time_t	starttime;	/* Time call started			*/
static time_t	stoptime;	/* Time call stopped			*/
static long	duration;	/* Duration of call in seconds		*/
static char	sitename[30];	/* Name of site called			*/


extern char	*strerror();
extern void	d_vlog(),
		writelog();


#include <varargs.h>

/*
**  Log that a call failed.
*/
void
failcall(va_alist)
va_dcl
{
    char		*mesg;
    char		buff[256], *b = buff;
    static char		WHERE[] = "failcall";
    int			s;
    struct ifreq	ifr;
    va_list		ap;
    REMOTE		*rp;
    MODEM		*mp;

    va_start(ap);
    rp = va_arg(ap, REMOTE *);
    mp = va_arg(ap, MODEM *);
    mesg = va_arg(ap, char *);

    if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	d_log(DLOG_GENERAL, WHERE, "Can't create socket for \"%s\", %m",
	    rp->Device);
	d_log(DLOG_GENERAL, WHERE, "%s", mesg);
	return;
    }

    (void)strcpy(ifr.ifr_name, rp->Device);
    ifr.ifr_data[0] = 0;
    if (ioctl(s, SIOCICALLSTAT, (caddr_t)&ifr) < 0) {
	d_log(DLOG_GENERAL, WHERE, "Can't SIOCICALLSTAT on \"%s\", %m",
	    rp->Device);
	d_log(DLOG_GENERAL, WHERE, "%s", mesg);
	return;
    }
    (void)close(s);

    (void)sprintf(b, "%s (%s)", rp->Device, rp->Sitename);
    b += strlen(b);
    if (mp) {
	(void)sprintf(b, " via %s", mp->Line);
	b += strlen(b);
    }
    (void)sprintf(b, ": %s", mesg);
    d_vlog(DLOG_GENERAL, WHERE, buff, ap);
    va_end(ap);
}

/*
**  Get the packet counts from the interface.
*/
static void
getpacketcounts(ipkts, opkts)
    long	*ipkts;
    long	*opkts;
{
    static char	WHERE[] = "getpacketcounts";

    /* Zero out the counts */
    *ipkts = *opkts = 0;

#if	0
    if (ioctl(sock, SIOCGIPKTS, (caddr_t)&ifr) < 0) {
	d_log(DLOG_GENERAL, WHERE, "Can't SIOCGIPKTS for \"%s\", %m", device);
	unlock_pid();
	exit(1);
    }
    *ipkts = (long)GETDATAVAL(ifr);

    if (ioctl(sock, SIOCGOPKTS, (caddr_t)&ifr) < 0) {
	d_log(DLOG_GENERAL, WHERE, "Can't SIOCGOPKTS for \"%s\", %m", device);
	unlock_pid();
	exit(1);
    }
    *opkts = (long)GETDATAVAL(ifr);
#endif
}

void		collect_ppp();
/*
**  Catch signal and exit.
*/
void
hangup(sig)
    int			sig;
{
    static char	WHERE[] = "hangup";
    long		totalpkts;
    long		endipkts;
    long		endopkts;
    int			status;

    /* Now ignore the signals since we're shutting down. */
    (void)signal(SIGHUP, SIG_IGN);
    (void)signal(SIGTERM, SIG_IGN);
    (void)signal(SIGCHLD, SIG_IGN);
    if (sig)
	d_log(DLOG_ALL, WHERE, "Process %d got signal %d for \"%s\"",
	    mypid, sig, device);

    /* Let go of the lock file */
    (void)uu_unlock(tty);
    unlock_pid();

    /* Log the statistics */
    (void)time(&stoptime);
    duration = stoptime - starttime;
    writelog(sitename, tty, starttime, stoptime, duration);

    getpacketcounts(&endipkts, &endopkts);
    totalpkts = (endipkts - startipkts) + (endopkts - startopkts);
    d_log(DLOG_INFO, WHERE, "Outbound interface \"%s\" up %ld seconds",
	device, duration);
    d_log(DLOG_DIAG, WHERE,
	"Inpackets = %ld, Outpackets = %ld, Average = %.2f packets/sec",
	endipkts - startipkts, endopkts - startopkts,
	totalpkts ? ((float)totalpkts) / ((float)duration) : 0);

    status = 0;

    /*
     * Terminate the Point to Point Protocol, and make sure that
     * the PPP program has exitted.
     */
    kill_ppp(sig);
    collect_ppp(0);

    /* Drop DTR. */
    if (ioctl(dialfout, TIOCCDTR, (caddr_t)0) < 0) {
#if	0
	/*
	 * This call seems to fail a lot.
	 * It's not clear why.
	 * At any rate, DTR is getting dropped anyway.
	 */
	d_log(DLOG_GENERAL, WHERE, "Can't TIOCCDTR \"%s\", %m", device);
#endif
	status = 1;
    }

    /* All done, report that we made it. */
    d_log(DLOG_GENERAL, WHERE, "Disconnected \"%s\"", device);
    exit(status);
}


/*
 * Get a modem from the list of usable ones for this system and return
 * the first one that isn't already in use.  NOTE:  We must respect locks
 * made by uucp, tip, cmdf et al
 * Also, if the open fails with EBUSY, we assume that the modem is in use.
 * We return
 *	file descriptor	- success
 *	-1		- no available modems
 */

find_modem(re, rop, mop)
REMOTE *re;
MODEM **rop, **mop;
{
    register int i, j;
    MODEM *m, **mp, *ro;
    int md;

    for (i = re->NModems, mp = re->Modems ; i-- ; mp++) {
	ro = *mp;
	if (ro->NModems) {
	    ro->LastUsed = get_rot_mod(ro);
	    for (j = ro->NModems ; j-- ; ) {
		ro->LastUsed = (ro->LastUsed + 1) % ro->NModems;
		m = ro->Modems[ro->LastUsed];
		if ((md = open_modem(m)) >= 0) {
		    *rop = ro;
		    *mop = m;
		    set_rot_mod(ro, ro->LastUsed);
		    return md;
		}
	    }
	}
	else {
	    m = ro;
	    if ((md = open_modem(m)) >= 0) {
		*rop = (MODEM *)0;
		*mop = m;
		return md;
	    }
	}
    }
    *rop = *mop = (MODEM *)0;
    return -1;
}

/*
 * Lock and open a modem.  Return open file descriptor on success.
 * On failure, make sure we don't leave a lock, and return -1;
 */
open_modem(m)
MODEM *m;
{
    char devtty[MAXNAMLEN];
    int md;
    static char		WHERE[] = "open_modem";

    if (uu_lock(m->Line))
	return -1;

    (void)sprintf(devtty, "/dev/%s", m->Line);

    if ((md = open(devtty, O_RDWR|O_NDELAY)) < 0) {
	(void)uu_unlock(m->Line);
	if (errno != EBUSY)
	    d_log(DLOG_GENERAL, WHERE, "open failed on \"%s\", %m", m->Line);
    }
    return md;
}

/*
 * Get or set the last used modem in a given rotary.
 */
#define	MAXMODEMDIGS	4

get_rot_mod(ro)
MODEM *ro;
{
    int m, mn = ro->NModems - 1;
    char mnum[MAXMODEMDIGS+2];
    char rfile[sizeof(LOG_DIR) + MAXNAMLEN];

    (void)sprintf(rfile, "%s/ROT.%s", LOG_DIR, ro->Name);

    if ((m = open(rfile, O_RDONLY)) < 0)
	return mn;
    if (read(m, mnum, sizeof(mnum)) != sizeof(mnum)) {
	(void)close(m);
	return mn;
    }
    (void)close(m);
    mn = atoi(mnum);
    return (mn >= 0 && mn < ro->NModems) ? mn : ro->NModems - 1;
}

set_rot_mod(ro, mn)
MODEM *ro;
int mn;
{
    int m;
    char mnum[MAXMODEMDIGS+2];
    char rfile[sizeof(LOG_DIR) + MAXNAMLEN];

    (void)sprintf(rfile, "%s/ROT.%s", LOG_DIR, ro->Name);
    (void)sprintf(mnum, "%*d\n", MAXMODEMDIGS, mn);

    if ((m = open(rfile, O_WRONLY|O_CREAT, 0644)) >= 0) {
	(void)write(m, mnum, sizeof(mnum));
	(void)close(m);
    }
}

/*
**  Catch an alarm while waiting for the open to succeed.
*/
static void
open_timeout()
{
    longjmp(opentimeout, 1);
    /* NOTREACHED */
}

#define	SPEED(rotary, modem)	\
	((modem)->Speed ? (modem)->Speed :	\
			  ((rotary) ? (rotary)->Speed : 0))
#define	DIALSCRIPT(rotary, modem)	\
	((modem)->DialScript ? (modem)->DialScript :	\
			       ((rotary) ? (rotary)->DialScript : (char *)0))

static int
makecall_real(rp)
    REMOTE		*rp;
{
    static char		WHERE[] = "makecall";
    char		devtty[30];
    FILE		*fp;
    struct sgttyb	sgtty;
    char		*tname;
    int			i;
    MODEM		*rotp, *mp;

#if	0
    /* Open our own path to the terminal and close the old connections */
    if ((i = open("/dev/tty", O_RDONLY)) >= 0) {
	(void)ioctl(i, TIOCNOTTY, (caddr_t)0);
	(void)close(i);
    }
#else
    (void)setsid();
#endif
    (void)fclose(stdin);
    (void)fclose(stdout);

    (void)signal(SIGHUP, SIG_IGN);
    (void)signal(SIGTERM, SIG_IGN);
    (void)signal(SIGCHLD, collect_ppp);

    /*
     * The find_modem function will attempt to find a modem that is not in
     * use. On success, it will be "uu_lock"ed.
     */
    if ((dialfout = find_modem(rp, &rotp, &mp)) < 0) {
	failcall(rp, mp, "No modem -- check lock files");
	return CALL_NOMODEM;
    }

    (void)strcpy(tty, mp->Line);
    (void)strcpy(sitename, rp->Sitename);
    (void)strcpy(device, rp->Device);
    (void)sprintf(devtty, "/dev/%s", tty);
    (void)record_pid(rp->Device);

    /* Get the TTY modes. */
    if (ioctl(dialfout, TIOCGETP, (caddr_t)&sgtty) < 0) {
	failcall(rp, mp, "TIOCGETP failed, %s", strerror(errno));
	(void)uu_unlock(tty);
	return CALL_IOCTL_ERR;
    }

    /* Set the new bits. */
    if (SPEED(rotp, mp) > 0)
	sgtty.sg_ispeed = sgtty.sg_ospeed = mapspeed(SPEED(rotp, mp));
    sgtty.sg_flags |= CBREAK;
    sgtty.sg_flags &= ~ECHO;

    /* Set the new mode. */
    if (ioctl(dialfout, TIOCSETP, (caddr_t)&sgtty) < 0) {
	failcall(rp, mp, "TIOCSETP failed, %s", strerror(errno));
	(void)uu_unlock(tty);
	return CALL_IOCTL_ERR;
    }

    /* Change process group, set hangup on close. */
    if (ioctl(dialfout, TIOCHPCL, (caddr_t)&sgtty) < 0) {
	failcall(rp, mp, "TIOCHPCL failed, %s", strerror(errno));
	(void)uu_unlock(tty);
	return CALL_IOCTL_ERR;
    }

    /* Open up the modem port. */
    if ((fp = fdopen(dialfout, "r+")) == NULL) {
	failcall(rp, mp, "fdopen failed, %s", strerror(errno));
	(void)uu_unlock(tty);
	return CALL_PERMISSION;
    }

    /* Set up the name of the transaction file. */
    tname = rp->Transcript[0] ? rp->Transcript : NULL;

    /* Now run the script. */
    d_log(DLOG_GENERAL, WHERE, "Dialing %s (%s) via %s",
	rp->Device, rp->Sitename, tty);
    if ((i = runscript(rp, fp, DIALSCRIPT(rotp, mp), tname) < 0)) {
	failcall(rp, mp, "script %s failed",
		 (i == -1) ? mp->DialScript : rp->Script);
	(void)uu_unlock(tty);
	return CALL_FAIL;
    }

#if	0
    /* Connection set up, set the line discipline. */
    if ((stat = duconnect(rp->Device)) != CALL_SUCCESS) {
	(void)uu_unlock(tty);
	return stat;
    }
#endif

    /* Connection made - get things going */
    d_log(DLOG_GENERAL, WHERE, "Connected to \"%s\" via \"%s\"",
	sitename, device);

    if (start_ppp(rp, devtty, SPEED(rotp, mp), 0) < 0) {
	d_log(DLOG_GENERAL, WHERE, "Failed to start PPP to \"%s\"", sitename);
	return CALL_FAIL;
    }

    d_log(DLOG_GENERAL, WHERE, "PPP started to \"%s\"", sitename);

    return CALL_SUCCESS;
}

int
makecall(rp)
    REMOTE		*rp;
{
    static char		WHERE[] = "makecall";
    int			uptime;
    int			old_if;

    /* Record the PID */
    mypid = getpid();

    (void)if_dtimeouts(rp);

    if (makecall_real(rp) != CALL_SUCCESS)
	return;


    /* Create a socket for gathering info. */
    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	d_log(DLOG_GENERAL, WHERE, "Can't create socket for \"%s\", %m",
	    device);
	unlock_pid();
	exit(1);
    }

    /* Set up info, collect starting stats. */
    (void)strcpy(ifr.ifr_name, device);
    (void)time(&starttime);
    getpacketcounts(&startipkts, &startopkts);
    d_log(DLOG_DIAG, WHERE,
	"Outbound interface \"%s\" starting ipkts = %ld, opkts = %ld",
	device, startipkts, startopkts);

    /* Now keep tabs on the interface */
    for (old_if = 0, uptime = 0; ; ) {
	/* See if the IF flags changed. */
	if (ioctl(sock, SIOCGIFFLAGS, (caddr_t)&ifr) < 0)
	    d_log(DLOG_GENERAL, WHERE, "Can't SIOCGIFFLAGS, %m");
	else if (ifr.ifr_flags != old_if) {
	    d_log(DLOG_ALL, WHERE,
		"\"%s\" IF flags changed from 0x%04x to 0x%04x",
		device, old_if, ifr.ifr_flags);
	    old_if = ifr.ifr_flags;
	}

#if	0
	/* See if the soft flags changed. */
	if (ioctl(sock, SIOCGSOFTFLAGS, (caddr_t)&ifr) < 0)
	    d_log(DLOG_GENERAL, WHERE, "Can't SIOCGSOFTFLAGS, %m");
	else if (GETDATAVAL(ifr) != old_soft) {
	    d_log(DLOG_ALL, WHERE,
		"\"%s\" SOFT flags changed from 0x%04x to 0x%04x",
		device, old_soft, GETDATAVAL(ifr));
	    old_soft = GETDATAVAL(ifr);
	}
#endif

	(void)sleep(UPTIME_INTERVAL * 60);
	uptime += UPTIME_INTERVAL;
	d_log(DLOG_INFO, WHERE, "Outbound interface \"%s\" up %d minutes",
	    device, uptime);
    }
}

mapspeed(speed)
int speed;
{
    switch (speed) {
     case 1200:		return B1200;
     case 2400:		return B2400;
     case 4800:		return B4800;
     case 9600:		return B9600;
     case 19200:	return B19200;
     case 38400:	return B38400;
     default:		return -1;
    }
}
