/*
 * Copyright (c) 1989, 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 */

#include <time.h>
#include <stdio.h>
#include <signal.h>
#include <sys/param.h>
#include <string.h>
#include "CommonDefs.h"
#include "pathnames.h"
#include "problem.h"
#include "hostfile.h"
#include "ctools.h"
#include "netquery.h"

#define NET_BROKE_DELAY  20    	/* delay if things are broken */
#define NET_OK_DELAY 	120  	/* delay if they are ok */

char *progname;

static char *DefaultPingkyDir = DEFAULT_PINGKY_DIR;

static char Checking_File[MAXPATHLEN];	/* What node are we Checking_File? */
static char cycletime[MAXPATHLEN];      /* How long did pingky take to cycle */
static char logfile[MAXPATHLEN];        /* Where do we log things */
static char Problem_File[MAXPATHLEN];   /* Where do we log things */
static char hostfile[MAXPATHLEN];       /* List of nodes we are Testing */
static char hostname[MAXHOSTNAMELEN];	/* Hostname on which we are running */

#define SLEEPING_LONG_NAP 	1
#define SLEEPING_SHORT_NAP 	2
#define POLLING 		3

#define CHECKINGINTERVAL 	60

#define ENTRY(i, j, k)	( Node[i].Test[j].argv[k] )	/* for readability */

static char CollectorState = 0;
static char *Name;
static char msg_buf[BUFSIZ];		/* for error messages */

SIG_FN EXIT()
{
	extern void exit();
	extern int In_Critical_Section;

	sprintf(msg_buf, "%s: interrupt.......\n", progname);
	fprintf(stderr, msg_buf);
	Log(msg_buf, logfile);
	if (In_Critical_Section) {
		Log("progname: Interrupt during Fileio", logfile);
		return;
	}
	exit(0);
}

#ifdef ALRM

AlarmClock()
{
	switch(CollectorState) {

	case SLEEPING_LONG_NAP:
		strcpy( msg_buf, "SLN - ");
		strcat( msg_buf, hostname );
		strcat( msg_buf, " Sleeping a Long Nap");
		break;

	case SLEEPING_SHORT_NAP:
		strcpy( msg_buf, "SSN - ");
		strcat( msg_buf, hostname );
		strcat( msg_buf, " taking a Short Nap");
		break;

	case POLLING:
		strcpy( msg_buf, "POL ");
		strcat( msg_buf, hostname );
		strcat( msg_buf, " > ");
		strcat( msg_buf, Name );
		break;

	default:
		strcpy( msg_buf,"ERR ");
		break;
	}

	LogChecking( msg_buf, Checking_File );
	signal(SIGALRM, AlarmClock);
	alarm(CHECKINGINTERVAL);
}

#endif

main(argc,argv)
int argc;
char *argv[];
{
	FILE *fp;
	register int i, j;
	extern int NumNodes;
	extern char Version[];
	struct HostFileNodeType *Node;
	char *service, *uid, *pingkydir, *getenv(), *user, Alert[100];
	time_t StartClock, ProblemFileLastChanged, elapsed;
	extern SIG_FN EXIT();
	extern void init_netquery();
	extern time_t time();

	progname = argv[0];
	if (argc != 1) {
		fprintf(stderr, "usage: %s\n" , argv[0]);
		exit(1);
	}

	if ((user = getenv("USER")) == NULL) {
		fprintf(stderr, "Unknown user - $USER not set\n");
		exit(1);
	}
	if (getenv("TEST") == NULL) {
		sprintf(msg_buf,"ps aux | grep -v dbx | grep %s | grep %s | grep -v grep | grep -v %d", user, argv[0], getpid());
		if (!system(msg_buf)) {
			fprintf(stderr, "%s ERROR: %s %s already running\n",
				argv[0], user, argv[0]);
			exit(1);
		}
	}

	/*
	 *  Lots of initialization here.
	 */
	if ((pingkydir = getenv("PINGKYDIR")) == NULL) 
		pingkydir = DefaultPingkyDir;
	sprintf(Checking_File, "%s/%s", pingkydir, CHECK_FILE);
	sprintf(cycletime, "%s/%s", pingkydir, ROVERD_CYCLE);
	sprintf(logfile, "%s/%s", pingkydir, PROBLEM_LOG);
	sprintf(hostfile, "%s/%s", pingkydir, HOST_FILE);
	sprintf(Problem_File, "%s/%s", pingkydir, PROBLEM_FILE);
	gethostname(hostname, sizeof(hostname));
	(void) strtok(hostname, ". \t\n");
	sprintf(msg_buf, "Starting %s %s\n", hostname, Version);
	printf(msg_buf);
	Log(msg_buf, logfile);
	signal(SIGINT, EXIT);
	signal(SIGHUP, EXIT);
#ifdef ALRM
	signal(SIGALRM, AlarmClock);
	alarm(10);
#endif
	if ((fp = fopen(cycletime,"w")) == NULL)
		syserr("fopen inetroverd.cycle");
	fprintf(fp, "--:--");
	if (fclose(fp) < 0) 
		syserr("fclose inetroverd.cycle");
	init_netquery();

	/*
	 *  Now start the work.
	 */
	for (;;) {
		CollectorState = POLLING;
		sprintf(msg_buf, "POL - %s %s patrolling ...", 
						hostname, progname);
		/*LogChecking(msg_buf, Checking_File);*/
		Node = ReadNodeFile(hostfile);
		StartClock = time((time_t *) NULL);

		/*
		 *  ping each entry in the host file and inform 
		 *  the Manager of the results.
		 */
		for (i = 0; Node[i].NodeName != NULL; i++) {

			if ((fp = fopen(cycletime, "w")) == NULL)
				syserr("fopen inetroverd.cycle");
			fprintf(fp, "top of loop %s\n",
				Node[i].NodeName);
			if (fclose(fp) < 0) 
				syserr("fclose inetroverd.cycle");


			Name = Node[i].NodeName;
			uid = Node[i].UniqueID;
			service = ENTRY(i, 0, 0);


			/* present different alerts based on parameter */
			strcpy( Alert, service );
			if ( Node[i].Test[0].argv[1] ) {
				strcat( Alert, "(" );
				strcat( Alert, Node[i].Test[0].argv[1] ); /* just param #1 */
				strcat( Alert, ")" );
			}

			/*
			 *  If the node doesn't respond, mark it as such,
			 *  and move on to the next entry in the list.
			 */
			if (NetQuery(uid, &Node[i].Test[0]) != ITWORKS ) {
				/*Problem_Manager(ADD_PRIMARY_PROBLEM, service, 
						Name, uid, (char *) NULL);*/
				Problem_Manager(ADD_PRIMARY_PROBLEM, Alert, 
						Name, uid, (char *) NULL);
				continue;
			}

			/*
			 *  It responded.  Delete this as a problem, and
			 *  perform further checks.
			 */
			/*Problem_Manager(DELETE_PROBLEM, service, Name, uid, 
								(char *) NULL);*/
			Problem_Manager(DELETE_PROBLEM, Alert, Name, uid, 
								(char *) NULL);
			for(j = 1; ENTRY(i, j, 0) != NULL; j++) {
				int action;
				service = ENTRY(i, j, 0);
				/* present different alerts based on parameter */
				strcpy( Alert, service );
				if ( Node[i].Test[j].argv[1] ) {
					strcat( Alert, "(" );
					strcat( Alert, Node[i].Test[j].argv[1] ); /* just param #1 */
					strcat( Alert, ")" );
				}
				if (NetQuery(uid, &Node[i].Test[j]) == ITWORKS)
					action = DELETE_PROBLEM;
				else 
					action = ADD_PROBLEM;
				/*Problem_Manager(action, service, Name, uid, 
								(char *) NULL);*/
				Problem_Manager(action, Alert, Name, uid, 
								(char *) NULL);
				if ((fp = fopen(cycletime, "w")) == NULL)
					syserr("fopen inetroverd.cycle");
				fprintf(fp, "testing %s:%s\n",Node[i].NodeName,&Node[i].Test[j]);
				if (fclose(fp) < 0) 
					syserr("fclose inetroverd.cycle");
			}
		}

		/*
		 *  Record cycle time.
		 */
		elapsed = time((time_t *) NULL) - StartClock;
		if ((fp = fopen(cycletime, "w")) == NULL)
			syserr("fopen inetroverd.cycle");
		fprintf(fp, "%2.2d:%2.2d", elapsed / 60, elapsed % 60);
		if (fclose(fp) < 0) 
			syserr("fclose inetroverd.cycle");

#ifdef DYNAMICSLEEP
		if (FileChanged(Problem_File, &ProblemFileLastChanged)) {
			CollectorState = SLEEPING_SHORT_NAP;
			sprintf(msg_buf, "SSN - %s %s Napping zzz ;-)",
						hostname, progname);
			LogChecking(msg_buf, Checking_File);
			sleep(NET_BROKE_DELAY);
		} 
		else {
			CollectorState = SLEEPING_LONG_NAP;
			sprintf(msg_buf, "SLN - %s %s snoozing zzzz |-O",
						hostname, progname);
			LogChecking(msg_buf, Checking_File);
			sleep(NET_OK_DELAY);
		}
#else
		sleep( 120 );
#endif
	}
}
