/*
 * Copyright (c) 1989, 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 */

#include <curses.h>
#include <string.h>
#include <sys/param.h>
#include <signal.h>
#include <errno.h>
#include "CommonDefs.h"
#include "pathnames.h"
#include "problem.h"
#include "ctools.h"
#include "display.h"

char hostfile[MAXPATHLEN];		/* List of nodes we are checking */
char logfile[MAXPATHLEN];		/* Where do we log things */
char Filter_File[MAXPATHLEN];		/* Filter out/in Nodes based on regexp*/

char *title = DEFAULT_TITLE;		/* display title */
char *Defaulttz = DEFAULT_TIME_ZONE;	/* default time zone */
char *progname;				/* libraries expect this to be here */

char DisplayType[40];			/* Where do we log things */
char Updateable;			/* is the display now updateable? */

int PID, PPID;				/* Process and Parent Process ID */
extern int In_Critical_Section;		/* Don't interupt code while writing */

int Updating;
int CurrentLine;
int TimeStampFormat = HUMANTIME;
extern int FilterFuture;
int CollectorDiedInterval=5;
char *pingkydir;

char *tz;			/* time zone */

main(argc,argv)
int argc;
char *argv[];
{
	int c;
	char *getenv();
	char buffer[BUFSIZ];
	char ProblemFile[MAXPATHLEN];
	char hostname[MAXHOSTNAMELEN];
	extern int EmptyIfFileDoesntExist;
	extern char Version[];
	extern char *optarg;
	extern char *getcwd();
	extern void exit();

	/* don't let the Display programs create files */
	EmptyIfFileDoesntExist = 0;

	/* do args */
	progname = argv[0];
	while ((c = getopt(argc, argv, "T:t:C:c:")) != -1)
		switch (c) {
		case 'T' :
		case 't' :
			title = optarg;
			break;
		case 'C' :
		case 'c' :
			CollectorDiedInterval = atoi(optarg);
			if ( CollectorDiedInterval <= 0 ) {
				fprintf(stderr,"Collector Died Interval MUST be greater than 0\n");
				exit(1);
			}
			break;
		default :
			printf("\n\nUsage: %s [ -t Title ] [ -c CollectorDiedInterval ] \n\
where: \n\
-t Title appears in the top center of the screen\n\
-c CollectorDiedInterval is in minutes\n\
		CollectorDiedInterval default is %d\n\
-A requires operator to hit a key to acknowledge new problems (0=ON, 1=OFF)\n\
",argv[0], CollectorDiedInterval );
			exit(1);
		}

	/* do some initialization */
	if ((pingkydir = getenv("PINGKYDIR")) == NULL) 
		pingkydir = DEFAULT_PINGKY_DIR;
	if ((tz = getenv("TZ")) == NULL) 
		tz = Defaulttz;

	sprintf(logfile, "%s/%s", pingkydir, PROBLEM_LOG);
	sprintf(hostfile, "%s/%s", pingkydir, HOST_FILE);
	sprintf(ProblemFile, "%s/%s", pingkydir, PROBLEM_FILE);
	sprintf(Filter_File, "%s/%s", pingkydir, FILTER_FILE);

	if (getcwd(buffer, 100) == NULL) 
		strcpy(DisplayType, "Unknown");
	else 
		strcpy(DisplayType, strrchr(buffer,'/') + 1);
	strcat(DisplayType, " "); 
	strncat(DisplayType, Version, 13);

	/* log that we are starting up */
	gethostname(hostname, sizeof(hostname));
	strtok(hostname, " .\t\n");
	sprintf(buffer, "Starting %s %s %s\n", progname, hostname, Version);
	Log(buffer, logfile);
	printf("%s\n", buffer);
    
	/* now just run */
	SetupFilters();
	SetUpdateable( ProblemFile );
	initscr();
	Loop( );
	endwin();
	exit(0);
	/*NOTREACHED*/
}

/****************************************************************
 * Loop:  main loop of program.  				*
 *		Set an alarm to go off in 30 seconds		*
 *			and provide address of alarm routine	*
 *		Read the Problem List, Checking file		*
 *		Update the screen and display the current time	*
 *		Forever						*
 *			Get a key stroke and handle user cmd.	*	
 ****************************************************************/

Loop()
{
	extern SIG_FN RefreshScreen();
	extern SIG_FN sigcatcher();
	extern int NumProblems;
	extern char Version[];
	extern void ClearMessage(), Update(), Message();
	int ch;

	signal(SIGALRM,RefreshScreen);	/*  Invoke RefreshScreen on alarm  */
	signal(SIGHUP,sigcatcher);		/*  Catch all signals */
	signal(SIGINT,sigcatcher);		/*  Catch all signals */
	signal(SIGQUIT,sigcatcher);		/*  Catch all signals */
	signal(SIGTERM,sigcatcher);		/*  Catch all signals */
	signal(SIGTSTP,sigcatcher);		/*  Catch all signals */
	alarm(UPDATEINTERVAL);		/* Alarm in UPDATEINTERVAL seconds */
	DrawFrame();				/* Start out with Framework */
	for (;;) {
		/*
		 *  Draw the changing part of the screen.  Turn off the
		 *  alarm; DrawDynamics will restart the alarm for us.
		 */
		DrawDynamics();
		ch=getch();
		alarm(0);
		switch(ch) {
		case 'u':
		case 'U': 		/* Update a problem */
			if ( Updateable ) {
				Update( 0 );
			}
			else { Message("Permission Denied"); sleep(5); }
			break;
		case 'c':		/* Claim a problem */
		case 'C': 
			if ( Updateable ) {
				Update( 1 );
			}
			else { Message("Permission Denied"); sleep(5); }
			break;
		case 'q':
		case 'Q':
			addstr("uit");
			move(LINES-1,0);
			clrtoeol();
			refresh();
			return;
			/*NOTREACHED*/
		case 'T':
			if ( Updateable ) {
				TraceRoute();		
				DrawFrame();	/* Screen got trashed */
                        }
                        else { Message("Permission Denied"); sleep(5); }
			break;
			/*NOTREACHED*/
		case 'P':
			if ( Updateable ) {
				Ping();		
				DrawFrame();	/* Screen got trashed */
                        }
                        else { Message("Permission Denied"); sleep(5); }
			break;
			/*NOTREACHED*/
		case 'h':
		case 'H':
			if ( Updateable ) {
				GetHelp();		
				DrawFrame();	/* Screen got trashed */
                        }
                        else { Message("Permission Denied"); sleep(5); }
			break;
		case 'd':
		case 'D':
			addstr("elete Problem # ");
                        if ( Updateable ) {
				DeleteProblem();
                        }
                        else { Message("Permission Denied"); sleep(5); }
			break;
		case 'F':
		case 'f':
			addstr("ilter ");
                        if ( Updateable ) {
				GetFilter();
				DrawFrame();	/* Frame will get corrupted */
                        }
                        else { Message("Permission Denied"); sleep(5); }
			break; 
		case 'a':
		case 'A':
			FilterFuture^=1;
			break;
		case '/':
			AdvanceClock();
			break;
		case '>':	/* ^DOWN */
			if ( (CurrentLine + LINES-6 )> NumProblems )
				CurrentLine=NumProblems-LINES-6;
			else CurrentLine=CurrentLine + LINES-6-1;
			break;
		case '<':	/* ^UP */
			if ( (CurrentLine - LINES-6 ) <=0 )
				CurrentLine=0;
			else CurrentLine=CurrentLine - LINES-6;
			break;
		case 't':
			addstr("rouble ticket # ");
                        if ( Updateable ) {
				GetTroubleTicket();
				DrawFrame();	/* Frame will get corrupted */
                        }
                        else { Message("Permission Denied"); sleep(5); }
			break; 
		case 'V':
		case 'v':
			addstr("ersion: ");
			Message(Version);
			sleep( 5 );
			break; 
		case '~':	
			if ( ++TimeStampFormat > 2 ) TimeStampFormat=1;
			DrawFrame();	/* Frame will be different */
			break;
		case '?':
			Message("   Cmds: (U)pdate (H)elp (D)elete (Q)uit (t)t (V)ersion (A)larms-Future ~<>  (T)raceroute (P)ing");
			sleep(5);
			ClearMessage();
			break;
		case ' ':
			ClearMessage();
			DrawFrame();
			break;
		default:  
			if ( ( ch == -1 ) && ( errno != EINTR ) && 
			     ( ! In_Critical_Section ) ) {
				/******************************************/
				/* Unexpected return from getch() 	  */
				/* On the RS6K, we don't get a HUP signal */
				/* when a terminal disconnects, so deal	  */
				/* with it by trapping here 		  */
				kill(PPID,1);	/* Kill parent process 	  */
				kill(PID,1);	/* Kill myself 		  */
				/* 			  		  */
				/******************************************/
				/*NOTREACHED*/
			}
			refresh();
            		break;
	}	/* switch */ 
   }	/* While () */
}


SIG_FN sigcatcher()
{
	extern int In_Critical_Section;
	extern void exit();

	printf("Interrupt. ERR! Don't do that!\n");
	if ( In_Critical_Section ) {
		Log("Display: Interrupt during Fileio",logfile); 
		printf("Display: Interrupt during Fileio - try again",logfile); 
		return;
	}
	exit( 0 );
}

beep()
{
/*printf("beep()");*/
	printf("");
	sleep(1);
}
flash()
{
/*printf("flash()");*/
	Message(" - - - - - - - Something changed - - - - - - - ");
	refresh();
	sleep(1);
	ClearMessage();
	refresh();
}

/****************************************************************
 * DrawDynamics:  Redraw what does change			*
 *			time of day				*
 *			PROBLEM.FILE				*
 *			CHECKING				*
 *			Goto the command line 			*
 ****************************************************************/
DrawDynamics()
{
static int virgin=1;

	signal(SIGALRM,RefreshScreen);	/*  Invoke RefreshScreen on alarm  */
	alarm( 0 );		/* Don't interrupt us with alarms ! */

	Print_Time();	/* Update the time of day */
	/*PrintCycle_Time();/* Update the Cycle Time portion */
	ReadProblems();
	ReadChecking();	/* Update the checking portion */
	GotoCmdLine();	/* Go to the command line and */
	refresh();	/* refresh the screen */
	virgin=0;
	signal(SIGALRM,RefreshScreen);	/*  Invoke RefreshScreen on alarm  */
	alarm( UPDATEINTERVAL );
}

/****************************************************************
 * RefreshScreen:  Force Update of the user's screen		*
 *		   DO NOT CALL THIS ROUTINE!!!!			*
 *		   The user callable version is DrawDynamic	*
 ****************************************************************/
SIG_FN RefreshScreen()
{
   if (!Updating) 
	DrawDynamics();	
}
