/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 * StatusFileio.c - All programs that need to access the Network
 *		    Status File should use these routines.
 *
 * See StatusFileio.h and CommonDefs.h for structure definitions.
 *
 */

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <memory.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "CommonDefs.h"

/*
 *
 *	MOVED_LINK_WAIT_TIME - How long do we wait before deleting
 *	MERIT INPLINKS.  PCP-SCP ether links are dynamic, and can move.
 *	
 *	We let the LINK stay in the MOVED state for this many seconds.
 */
#define MOVED_LINK_WAIT_TIME 600
int ResolveNames = 0;

static int CheckForLinkInconsistencies = 0;

/************************************************************************
 *	AddNode - Add/Modify Node in Network 				*
 ************************************************************************/
void AddNode( Network, NodeName, Type, Queryable, State, TimeStamp, IPAddress )
struct NetworkType *Network;
char *NodeName;
int Type, Queryable, State;
time_t TimeStamp;
char *IPAddress;
{
	register struct NodeType *Node;
	extern struct NodeType *FindNode();
	extern void AppendNode();

	/*if ((NodeName==NULL) && (strcmp(IPAddress,"Unknown")==0)) return;*/
	if ((Node = FindNode(Network, IPAddress)) == NULL) {
		AppendNode(Network, NodeName, Type, Queryable, State, 
							TimeStamp, IPAddress );
		return;
	}
	if (!((State == BUSY ) && (Node->State == UP))) /* No UP->BUSY*/ 
		if ( Node->State != State ) {
			Node->TimeStamp = TimeStamp;
			Node->State = State;
		}
}

/*************************************************************************
 * SetCheckForLinkClash() - The AddLink/AddNode routines should complain *
 *			    when nodes/links change state.		 *
 *************************************************************************/
void SetCheckForLinkClash()
{
        CheckForLinkInconsistencies = 1;
}

/************************************************************************
 * ClearCheckForLinkClash() - The AddLink/AddNode routines shouldn't	*
 *			    complain when nodes/links change state.	*
 ************************************************************************/
void ClearCheckForLinkClash()
{
        CheckForLinkInconsistencies = 0;
}

/************************************************************************
 *	AddLink-Append/Modify Link in Network				*
 ************************************************************************/
void AddLink( Network, Node1, Node2, State, TimeStamp , Type )
struct NetworkType *Network;
char *Node1, *Node2;
int  State;
time_t TimeStamp;
int Type;
{
	register struct LinkType *Link;
	extern struct LinkType *FindLink();
	extern char *Error_Log;
	extern void AppendLink();
	static char Buf[BUFSIZ];

	if ((Link = FindLink( Network, Node1, Node2, Type )) == NULL) {
		AppendLink( Network, Node1, Node2, State, TimeStamp, Type );
		return;
	}
	if (Link->State == State)
		return;

	if (CheckForLinkInconsistencies &&
			   ((( Link->State == DOWN ) && ( State == UP )) ||
			   ((  Link->State == UP )   && ( State == DOWN)))) {
		/*sprintf(Buf, "AddLink(): Inconsistent Link State Detected on link %s-%s\n", Node1, Node2);
                Log(Buf,Error_Log);*/
		Link->TimeStamp = TimeStamp;
		Link->State = DOWN;
	}
	else { 
		Link->TimeStamp = TimeStamp;
		Link->State = State;
	}
}

/************************************************************************
 *	AppendNode-Append a Node to this Network Node Linked List	*
 ************************************************************************/
static void AppendNode( Network, NodeName, Type, Queryable, State, TimeStamp,
								 IPAddress )
struct NetworkType *Network;
char *NodeName;
int State,Queryable;
long TimeStamp;
char *IPAddress;
{
	register struct NodeType *Node;
	struct hostent *hp;
	struct in_addr ipaddr;
	extern struct NodeType * AllocNode();

	if ((Node = AllocNode()) == NULL) 
		panic("AllocNode Failed\n");;
	if (Network->NodeHead != NULL)
		(Network->NodeTail)->Next = Node;
	else
		Network->NodeHead = Node;
	Node->Next = (struct NodeType *) NULL;
	Network->NodeTail = Node;
	if ( NodeName != NULL ) strcpy(Node->Name, NodeName);
	else Node->Name[0]='\0';
	Node->Type = Type;
	Node->Queryable = Queryable;
	Node->State = State;
	Node->TimeStamp = TimeStamp;
	if (IPAddress == NULL) {
		if ((hp = gethostbyname(Node->Name)) == NULL)
			strcpy(Node->IPAddress, "NotKnown");
		else {
			memcpy((char *) &ipaddr, hp->h_addr_list[0],
					sizeof(struct in_addr));
			sprintf(Node->IPAddress, "%s", 
				inet_ntoa(ipaddr));
		}
	}
	else
		strcpy(Node->IPAddress, IPAddress);
}

/**** To Do: 	Put links in in alphabetical order instead of appending */
	/* This will make searching faster to */
/************************************************************************
 *      AppendLink-Append a Link to this Network Link Linked List       *
 ************************************************************************/
static void AppendLink( Network, Node1, Node2, State, TimeStamp , Type)
struct NetworkType *Network;
char *Node1, *Node2;
int  State;
long TimeStamp;
int Type;
{
	register struct LinkType *Link;
	extern struct LinkType * AllocLink();

	if ((Link = AllocLink()) == NULL) 
		panic("AllocLink Failed\n");
	if (Network->LinkHead != NULL)
		Network->LinkTail->Next = Link;
	else
		Network->LinkHead = Link;
	Network->LinkTail = Link;
	strcpy( Link->Node1, Node1 );
	strcpy( Link->Node2, Node2 );
	Link->State = State;
	Link->Type = Type;
	Link->TimeStamp = TimeStamp;
	Link->Next = (struct LinkType *) NULL;
}

/************************************************************************
 * FindNode-Find this Node in this Networks Node Linked List		*
 ************************************************************************/
struct NodeType *FindNode( Network, IPAddress)
struct NetworkType *Network;
char *IPAddress;
{
	register struct NodeType *NPtr;

	for (NPtr = Network->NodeHead; NPtr != NULL; NPtr = NPtr->Next)
		if (strcmp(NPtr->IPAddress, IPAddress) == 0) 
			return(NPtr);
	return((struct NodeType *) NULL);
}

/************************************************************************
 * FindLink-Find this Link in this Networks Link Linked List            *
 ************************************************************************/
struct LinkType *FindLink( Network, Node1, Node2 , Type )
struct NetworkType *Network;
char *Node1, *Node2;
int Type;
{
	register struct LinkType *LPtr;


	for (LPtr = Network->LinkHead; LPtr != NULL; LPtr = LPtr->Next ) {
#ifdef ANYORDER
		if ((( strcmp( LPtr->Node1, Node1 ) == 0 ) && 
		     ( strcmp( LPtr->Node2, Node2 ) == 0) ||
		    (( strcmp( LPtr->Node1, Node2 ) == 0 ) && 
		     ( strcmp( LPtr->Node2, Node1 ) == 0))) &&
#else
		if (( strcmp( LPtr->Node1, Node1 ) == 0 ) && 
		    ( strcmp( LPtr->Node2, Node2 ) == 0) &&
#endif
			(LPtr->Type == Type)) 
				return(LPtr);
	}
	return((struct LinkType *) NULL);
}

void FreeConfig( Net )
struct NetworkType *Net;
{
	extern void FreeNodeChain(), FreeLinkChain();

	FreeNodeChain(Net->NodeHead); 
	Net->NodeHead = Net->NodeTail = NULL;
	FreeLinkChain(Net->LinkHead); 
	Net->LinkHead = Net->LinkTail = NULL;
}

/************************************************************************
 *	ReadConfig- Read the config file for this Network into this	*
 *		    Network Structure					*
 ************************************************************************/
struct NetworkType *ReadConfig(BaseNet, NetworkType, filename)
struct NetworkType *BaseNet;
char NetworkType;
char *filename;
{
	FILE *stream;
	extern void InitNetwork();
	extern time_t time();
	extern char *Error_Log;
	char inbuff[BUFSIZ], msg[BUFSIZ];
	char *TimeStamp, *PLineType, *Name, *Address, *PType, *PState;
	int Line = 0, LineType, Type, State, Valid;
	struct stat buf;
	time_t t_TimeStamp,TimeNow = time(&TimeNow);
#ifdef CLEARB4ALLOC
	static int virgin = TRUE;
#endif

	lock(filename);

#ifdef ADVISORY
	for (;;) {
		if (stat(filename,&buf) != 0) {
			perror("Can't open statefile: ");
			break;
        	}
		if ((TimeNow - buf.st_mtime) > 1 ) 
			break;
		fprintf(stderr, "%s being written TimeNow=%lu mtime=%lu...waiting a sec\n",filename,TimeNow,buf.st_mtime);
		sleep(1);
		TimeNow = time(&TimeNow);
	}
#endif

#ifdef CLEARB4ALLOC
	/*
	 * We free old data structures on subsequent
	 * reads of the network status.
	 */

	if (!virgin) {
		FreeConfig(BaseNet);
		virgin = FALSE;
	}
#endif

	InitNetwork(BaseNet, NetworkType, filename);
	if (stat(filename, &buf ) < 0) {
		perror("Status File stat(): ");
		BaseNet->TimeStamp = 0;
   	}
	else
		BaseNet->TimeStamp = buf.st_mtime;
	
	if ((stream = fopen(filename, "r")) == NULL) {
		printf("\n\tWarning %s -  file Unavailable\n", filename);
		perror("fopen");
		unlock(filename);
		return((struct NetworkType *) NULL);
	}

	/*
	 *  Read lines.  If a line is syntactically incorrect, jump down
	 *  to the bottom of the loop, and start again.
	 */
	while (fgets(inbuff, sizeof(inbuff), stream) != NULL) {
		Valid = FALSE;
		Line++;
		if (inbuff[0] == '#') 
			continue;
		/*
		 *  The syntax of a line is like this:
		 *
		 *	timestamp [ NODE | LINK ] type name uid state
		 *
		 *  uid is often an IP address 
		 */
                if ((TimeStamp = strtok(inbuff, DELIMITERS)) == NULL) {
			sprintf(msg, "Missing timestamp"); 
			goto end;
		}
                if ((PLineType = strtok((char *) NULL, DELIMITERS)) == NULL) {
			sprintf(msg, "Missing line type"); 
			goto end;
		}
		if ((LineType = Str_To_LineType(PLineType)) == 0) {
                        sprintf(msg, "Unknown line type: %s", PLineType); 
			goto end;
		}
                if ((PType = strtok((char *) NULL, DELIMITERS)) == NULL) {
                        sprintf(msg, "Missing type"); 
			goto end;
		}
		if (LineType == NODE) {
			if ((Type = Str_To_NodeType(PType)) == 0)  {
                        	sprintf(msg, "Unknown %s Type: %s",
							PLineType, PType); 
				goto end;
			}
		}
		else if (LineType == LINK) {
			if ((Type = Str_To_LinkType(PType)) == 0) {
                                sprintf(msg, "Unknown %s Type: %s",
                                        		PLineType, PType); 
				goto end;
			}
		}
                if ((Name = strtok((char *) NULL, DELIMITERS)) == NULL) {
                        sprintf(msg, "Missing name"); 
			goto end;
		}
                if ((Address = strtok((char *) NULL, DELIMITERS)) == NULL) {
                        sprintf(msg, "Missing address"); 
			goto end;
		}
                if ((PState = strtok((char *) NULL, DELIMITERS)) == NULL) {
                        sprintf(msg, "Missing state"); 
			goto end;
		}
		if (LineType == NODE) {
			if ((State = Str_To_NodeState(PState)) == 0) {
                        	sprintf(msg,"Unknown %s state: %s",
							PLineType, PState); 
				goto end; 
			}
		}
		else if (LineType == LINK) {
			if ((State = Str_To_LinkState(PState)) == 0) {
                        	sprintf(msg, "Unknown %s state: %s",
                                			PLineType, PState); 
				goto end; 
			}
		}
		Valid = TRUE;
end:
	 	if (!Valid) {
			printf("ReadConfig(): Line #%d (Ignoring Line): %s\n", 
							Line, msg);
			continue;
		}
		t_TimeStamp=(time_t) atol(TimeStamp);
		if (LineType == NODE)
			AddNode(BaseNet, Name, Type, CanQuery(Type), State, 
					t_TimeStamp, Address);
		else {
			/**** Links migrate to MOVED, SortStatus will delete 
			 **** the problems associated with the DOWN link
			 **** and we will eventually discard the old link ***/
			if ( State == MOVED ) {
				if ( TimeNow-t_TimeStamp > MOVED_LINK_WAIT_TIME ) {
					sprintf(msg,"Deleting MOVED link %s\n",Address);
					fprintf(stderr,msg);
					Log( msg, Error_Log);
					continue;
				} else {
					fprintf(stderr,"MOVED link soon to be deleted: TimeNow=%ld t_TimeSTamp=%ld diff=%d\n",TimeNow,t_TimeStamp,TimeNow-t_TimeStamp);
				}
			}
			AddLink(BaseNet, Name, Address, State, 
					t_TimeStamp, Type);
		}
      	}
	unlock(filename);
	return(BaseNet);
}

/************************************************************************
 *      WriteConfig- Write the config file for this Network into this   * 
 *                  Network Structure                                   *
 *									*
 * TimeStamp NODE Type Address Name  State				* 
 *		or							*
 * TimeStamp LINK Type Address Address State				*
 * where:								*
 *									*
 *	TimeStamp is Standard Unix Time					*
 *	NODE/LINK identifies Nodes and Links				*
 *	Address:	NODE IPAddress of entity			*
 *				or AS# of AS				*
 *			LINK IPAddress of entity			*
 *				or AS# of AS				*
 *	Name:		Domain Name 					*
 *				or AS# (AS201)				*
 *	Type:								*
 *			NODE						*
 *			----						*
 *			PCP						*
 *			SCP						*
 *			NSS						*
 *			ENSS						*
 *			CNSS						*
 *			AS						*
 *			EGPPEER						*
 *									*
 *			LINK						*
 *			----						*
 *			MeritNeighbor					*
 *			is-isNeighbor					*
 *			egpNeighbor					*
 *			ASReachability					*
 *	State:								*
 *			NODE						*
 *			----						*
 *			UP	Responded to queries			*
 *			DOWN						*
 *			BUSY	Someone has an up link to node		*
 *			NR	Noone has a link to him			*
 *									*
 *			LINK						*
 *			----						*
 *			UP	Link is up 				*
 *			DOWN	Link was marked down			*
 *			NR	Link was not seen by anyone		*
 *----------------------------------------------------------------------*
 ************************************************************************/

struct NetworkType *WriteConfig(BaseNetwork, Type, filename)
struct NetworkType *BaseNetwork;
char Type;
char *filename;
{
	extern char Version[];
	extern char *PrintNetworkType(), *PrintState(), *PrintNodeType(),
			*PrintLinkType();
	FILE *stream;
	char buffer[30];
	extern char *addrstrtoname(),*AStoname();
	char *p;
	register struct NodeType *n;
	register struct LinkType *l;
	long timenow=time(&timenow);
        struct hostent *hp;
        struct in_addr in, in2;
        char s[BUFSIZ];

	lock(filename);
	if ((stream = fopen(filename,"w")) == NULL) {
		panic("\n\tERROR - output file Unavailable\n");
		/*NOTREACHED*/
	}

	fprintf(stream,"#\n# %s\n#%s Network Status as of %s#\n", Version,
				PrintNetworkType(Type), ctime(&timenow));

	for (n = BaseNetwork->NodeHead; n != NULL; n = n->Next ) {
		if (ResolveNames) {
			if ((strcmp(n->Name, "NotKnown") == 0) ||
			    (strcmp(n->Name, n->IPAddress) == 0 )) {
				if (n->Type == AS) {
					strcpy(n->Name, n->IPAddress);
					p = AStoname(n->IPAddress);
					if (p != NULL) {
						strcat(n->Name,"(");	
						strcat(n->Name,p);	
						strcat(n->Name,")");	
					}
                                	else {
                                        	strcpy(n->Name,"AS(");
                                        	strcat(n->Name,n->IPAddress);
                                        	strcat(n->Name,")");
                                	}
			    	}
			    	else {
					p = addrstrtoname(n->IPAddress);
					if (p != NULL)
						strcpy(n->Name, p);
					else strcpy( n->Name, n->IPAddress);
				}
			} 
#ifdef FUTURE
			if ( strcmp( n->Name, n->IPAddress ) == 0 ) {
        			hp = gethostbyname(n->Name);
        			if (hp == NULL) {
                			printf("Unknown host %s\n", n->IPAddress);
        			}
				if ( hp != NULL ) {
        				if (hp->h_addrtype != AF_INET) {
                				printf("Address type not INET\n");
					}
					else {
        					memcpy((char *) &in, hp->h_addr_list[0], hp->h_length);
						strcpy(n->IPAddress, inet_ntoa(in));
					}
        			}

			}
#endif
		}
		if (strlen(n->Name) <=0) 
			continue;
		fprintf(stream,"%s NODE %s\t%s\t%s\t%s\n",
			ltoa(n->TimeStamp, buffer), PrintNodeType(n->Type),
			n->Name,n->IPAddress, PrintState(n->State));
	}
	fprintf(stream, "#\n#LINKS\n#\n");
        for (l = BaseNetwork->LinkHead; l != NULL; l = l->Next)
		/*if ( l->State != MOVED )*/
               		fprintf(stream,"%s LINK %s\t%s\t%s\t%s\n",
				ltoa(l->TimeStamp, buffer ),
				PrintLinkType(l->Type),l->Node1, l->Node2, 
				PrintState(l->State));
   fclose(stream);
   unlock(filename);
   return(BaseNetwork);
}
