/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

/************************************************************************
 *	color.c - This module provides the coloring in of the nodes	*
 *		  and links.  						*
 *                                                                      *
 ************************************************************************/
#include <stdio.h>
#include <math.h>
#include "CommonDefs.h"
#include "motif.h"
#include "xmap.h"

extern Widget NodePB[];

#define LINENODEDELTA 2         /* Distance between Node and link       */
#define LINKSENSITIVITY 20

GC upgc, downgc, nrgc;          /* Graphics Context for all lines       */
static int LineWidth = 0;       /* Should be an option somehow          */
static unsigned long DOWNPixel; /* Color to use when entity is DOWN     */
static unsigned long UPPixel;   /* Color to use when entity is UP       */
static unsigned long NRPixel;   /* Color to use when entity is Not Reachable */
static unsigned long BUSYPixel; /* Color to use when entity is BUSY     */
static unsigned long WHITEPixel;/* Foreground color when background is dark */
static unsigned long BLACKPixel;/*  Foreground color when background is light*/

void SetLineWidth(width)
{
	LineWidth = width;
}

/************************************************************************
 *              GetLineSegment - get endpoints of link for drawing      *
 *                                                                      *
 ************************************************************************/
void GetLineSegment(ix1, iy1, n1width, n1height, 
		    ix2, iy2, n2width, n2height,
		    X1, Y1, X2, Y2 )
int ix1, iy1, n1width, n1height, ix2, iy2, n2width, n2height;  	/* INPUT */
int *X1, *Y1, *X2, *Y2;                            		/* RETURN */
{
	int Node1X = ix1 + (n1width / 2);
	int Node2X = ix2 + (n2width / 2);
	int Node1Y = iy1 + (n1height / 2);
	int Node2Y = iy2 + (n2height / 2);

	if ((Node1X == Node2X) || (abs(Node1X - Node2X) < 5)) { /*VERT. LINE*/
		*X1 = *X2 = Node2X = Node1X;
		if (iy1 < iy2) {      /* Node 1 is on top of node 2 */
			*Y1 = iy1 + n1height +LINENODEDELTA;
			*Y2 = iy2 - LINENODEDELTA;
		}
		else {
			*Y2 = iy2 + n2height + LINENODEDELTA;
			*Y1 = iy1 - LINENODEDELTA;
		}
	} else {
		if (Node1Y == Node2Y) {       /* HORIZONTAL LINE */
			*Y1 = *Y2 = Node1Y;
			if (ix1 < ix2) {  /*Node 1 is to the Left of Node 2*/
				*X1 = ix1 + n1width + LINENODEDELTA;
				*X2 = ix2 - LINENODEDELTA;
			}
			else {
				*X2 = ix2 + n2width + LINENODEDELTA;
				*X1 = ix1 - LINENODEDELTA;
			}
		} else {
			*X1 = Node1X; 
			*Y1 = Node1Y;
			*X2 = Node2X; 
			*Y2 = Node2Y;
		}
	}
}

/****************************************************************
 *   Link2Nodes() - convert x,y to closest link in the map      *
 ****************************************************************/
Link2Nodes(x, y, Node1, Node2)
int x, y;
int *Node1, *Node2;
{
	register struct LinkType *L;
	int ix1, iy1, ix2, iy2;
	float x1, y1, x2, y2, x3, y3;
	float ma, mb, ba, bb, x4, y4, dx, dy, dist;
	float bestdist = LINKSENSITIVITY;
	int foundone = 0;
	int n1, n2;             /* indices of the MapNodes we are checking */
	Arg myArgs[MAX_X_ARGS];
	Dimension n1width, n1height, n2width, n2height;
	int X1, Y1, X2, Y2;
	extern struct NetworkType Network;
	extern int ShowASs;

	printf("Searching for closest link\n");
	x3 = x;   /* Account for links going through center of node */
	y3 = y;   /* And make floating point numbers */

	for (L = Network.LinkHead; L != NULL; L = L->Next ) {

		/*
		 *  Only show IS-IS links.  EXCEPTION:  Show AS links
		 *  if the 'ShowASs' flag is set.
		 */
		/*if ( ! ShouldDisplayLink( Net, L ) ) continue;
		/*if (L->Type != ISISLINK && L->Type !=INPLINK ) 
			if (!((L->Type == ASLINK) && ShowASs)) 
				continue; */

		/*
		 *  If either point is zero, skip it.
		 */
		n1 = GetXY(L->Node1, &ix1, &iy1);
		n2 = GetXY(L->Node2, &ix2, &iy2);
		if ((n1 == 0) || (n2 == 0))
			continue;

		XtSetArg(myArgs[0], XmNwidth, &n1width );
		XtSetArg(myArgs[1], XmNheight, &n1height );
		XtGetValues(NodePB[n1], myArgs, 2);
		XtSetArg(myArgs[0], XmNwidth, &n2width);
		XtSetArg(myArgs[1], XmNheight, &n2height);
		XtGetValues(NodePB[n2], myArgs, 2);

		/*
		 *  The following bits of code need some commenting
		 *  in a BIG way.
		 */
		GetLineSegment(ix1, iy1, (int) n1width, (int) n1height,
			       ix2, iy2, (int) n2width, (int) n2height,
			       &X1, &Y1, &X2, &Y2);
		x1 = X1;  
		y1 = Y1;  /* make floating point numbers */
		x2 = X2;  
		y2 = Y2;  /* make floating point numbers */
		if (x2 == x1) {/* Vertical line */
			dist = x3 > x1 ? x3 - x1 : x1 - x3;
			x4 = x1;
			y4 = y3;
		}
		else if (y2 == y1) { /* Horizontal line */
			dist = y3 > y1 ? y3 - y1 : y1 - y3;
			x4 = x3;
			y4 = y1;
		}
		else {
			ma = (y2 - y1)/(x2 - x1);
			mb = -1.0/ma;
			ba = y1 - ma*x1;
			bb = y3 - mb*x3;
			x4 = (bb - ba)/(ma - mb);
			y4 = ma*x4 + ba;

			dx = x3 - x4;
			dy = y3 - y4;
			dist = sqrt((double) (dx*dx + dy*dy));
		}
		if ((((x4 >= x1) && (x4 <= x2)) || ((x4 >= x2) && (x4 <= x1)) ) &&
			   (((y4>=y1)&&(y4<=y2)) || ((y4>=y2)&&(y4<=y1)) ) ) {
			if (dist < bestdist) {
				bestdist = dist;
				*Node1 = n1;
				*Node2 = n2;
				foundone = 1;
			}

		}

	}
	return(foundone);
}

static char ShowNode[ 100 ];	/* The Node user wants to Show */
static int ShowMode=0;
/*
 * SetMapShow - called by main() - User only wants to see map
 *				with this node attached
 */
int SetMapShow( Node ) 
char *Node;
{
	strcpy( ShowNode, Node );
	ShowMode=1;
}

char *GetShowNode()
{
	return(ShowNode);
}

/* 
 *	Does this link match show characteristics
 */
int ShowLink( L )
struct LinkType *L;
{
	if (( strcmp( L->Node1, ShowNode ) == 0 ) ||
	    ( strcmp( L->Node2, ShowNode ) == 0 )) return(1);  /*ddd */
	else return(0);
}

/*
 *	FitsShowCriteria
 * if filtering is turned on and 
 * 	if this node is, or is attached to the node in question	
 * 		We have a match
 */
int NodeFitsShowCriteria( Network, Node )
struct NetworkType *Network;
struct NodeType *Node;
{
register struct LinkType *LPtr;

       	for (LPtr=Network->LinkHead; LPtr != NULL; LPtr = LPtr->Next) {
		if ( ( strcmp(LPtr->Node1, Node->IPAddress ) == 0 ) ||
		     ( strcmp(LPtr->Node2, Node->IPAddress ) == 0 ) )
			if ( ShowLink( LPtr ) == 1 )
				return(1);   		/* Attached to node */
	}
	return(0);	/* Doesn't match Show Criteria */
}

int ShouldDisplayNode( Net, N, ShowASs )
struct NetworkType *Net;
struct NodeType *N;
int ShowASs;
{

	if ( ShowMode )					/* Show Node */
		return ( NodeFitsShowCriteria(Net, N) ); 
	if ( (N->Type==AS) && (ShowASs==0) ) return(0);	/* Show ASs ? */
	if ( N->Type==SCP ) return(0);		/* Don't Show SCPs  */
	return(1);	/* Otherwise - display all nodes */
}

/*
 *  Don't show AS links.  EXCEPTION:  the AS
 *  links if the 'ShowASs' flag is set.
 */
int ShouldDisplayLink( Net, L, ShowASs )
struct NetworkType *Net;
struct LinkType *L;
{
	if ( ShowMode ) 
		return( ShowLink( L ) );
	if ( L->Type == INPLINK ) 	/* Don't Show PCP-SCP LINKS */
		if ((strlen(L->Node1)==4) || (strlen(L->Node2)==4)) return(0);
	if (( ShowASs == 0 ) && ( L->Type == ASLINK )) return(0);
	return(1);
}

/***************************************************************************
 *  ColorNode() - Change the color of the node to the color for its state  *
 ***************************************************************************/
void ColorNode( N, ShowASs)
struct NodeType *N;
int ShowASs;
{
	Arg myArgs[MAX_X_ARGS];
	int i, x, y, donotset=0;
	extern int MapHasChanged;
	extern void AddNodePB();

	if ((i = GetXY(N->IPAddress, &x, &y)) == 0 ) {
		AddNodePB(AddMapNode(N->Name, N->IPAddress, -1,-1));
		MapHasChanged = 1;
		i = GetXY(N->IPAddress, &x, &y);
	}

	XtSetArg( myArgs[1], XmNforeground, BLACKPixel );
	switch( N->State ) {
	case DOWN:
		XtSetArg( myArgs[0], XmNbackground, DOWNPixel );
		break;
	case UP:
		XtSetArg( myArgs[0], XmNbackground, UPPixel );
		break;
	case NR:
		XtSetArg( myArgs[0], XmNbackground, NRPixel );
		XtSetArg( myArgs[1], XmNforeground, WHITEPixel );
		break;
	case BUSY:
		XtSetArg( myArgs[0], XmNbackground, BUSYPixel );
		break;
	default:
		printf("Unknown state for node: %d\n",N->State);
		donotset=1;
		break;
	}
	if (donotset==0) XtSetValues( NodePB[i], myArgs, 2 );
}


/***************************************************************************
 *  ColorNodes() - Draw each of the nodes in the backbone                   *
 ***************************************************************************/
/*ARGSUSED*/
void ColorNodes(drawarea, mynet, ShowASs)
Widget *drawarea;
struct NetworkType *mynet;
int ShowASs;
{
	register struct NodeType *N;

	for (N = mynet->NodeHead; N != NULL; N = N->Next)
		if ( ShouldDisplayNode( mynet, N, ShowASs ) )
			ColorNode( N, ShowASs );
}

/*****************************************************************************
 * ColorLink()-Color the link acording to its state using the appropriate GC *
 *****************************************************************************/
void ColorLink(drawarea, L)
Widget *drawarea;
struct LinkType *L;
{
	int x1, y1, x2, y2, X1, Y1, X2, Y2, n1, n2;
	Arg myArgs[MAX_X_ARGS];
	Dimension n1width, n1height;
	Dimension n2width, n2height;
	extern int GetLinkWidth();
	XGCValues values;

	n1 = GetXY(L->Node1, &x1, &y1);
	n2 = GetXY(L->Node2, &x2, &y2);

	if ((n1 != 0) && (n2 != 0)) {
		/* Find the width and height of these buttons */
		XtSetArg( myArgs[0], XmNwidth, &n1width );
		XtSetArg( myArgs[1], XmNheight, &n1height );
		XtGetValues( NodePB[n1], myArgs, 2 );

		XtSetArg( myArgs[0], XmNwidth, &n2width );
		XtSetArg( myArgs[1], XmNheight, &n2height );
		XtGetValues( NodePB[n2], myArgs, 2 );

		GetLineSegment(x1, y1, (int) n1width, (int) n1height, x2, y2,
			(int) n2width, (int) n2height, &X1, &Y1, &X2, &Y2);
		values.line_width = GetLinkWidth(L->Node1, L->Node2);

		switch (L->State) {
		case UP:
			XChangeGC(XtDisplay(*drawarea), upgc, GCLineWidth, 
								&values);
			XDrawLine(XtDisplay(*drawarea), XtWindow(*drawarea), 
					upgc, X1, Y1, X2, Y2);
			break;
		case DOWN:
			XChangeGC(XtDisplay(*drawarea), downgc, GCLineWidth, 
								&values);
			XDrawLine(XtDisplay(*drawarea), XtWindow(*drawarea), 
					downgc, X1, Y1, X2, Y2);
			break;
		case NR:
			XChangeGC(XtDisplay(*drawarea), nrgc, GCLineWidth, 
								&values);
			XDrawLine(XtDisplay(*drawarea), XtWindow(*drawarea), 
					nrgc, X1, Y1, X2, Y2);
			break;
		default:
			fprintf(stderr,"Unknown state of link:%d\n", L->State);
		}
		return;
	}
	fprintf(stderr, "color.c: ColorLink() couldn't find node %s or %s\n", L->Node1, L->Node2);
}

/************************************************************************
 * ColorLinks() - Color each of the links in the network                *
 ************************************************************************/
void ColorLinks(drawarea, mynet, ShowASs)
Widget *drawarea;
struct NetworkType *mynet;
int ShowASs;
{
register struct LinkType *L;

	for (L = mynet->LinkHead; L != NULL; L = L->Next ) {
		if ( ShouldDisplayLink(mynet, L, ShowASs ) ) 
			ColorLink(drawarea, L);
	}
}

/************************************************************************
 *  AllocateColors() - Allocate the colors we will use - do this once   *
 *                      We set the Global Pixel veluaes, and the        *
 *                      Graphics Contexts for all drawing.              *
 ************************************************************************/
void AllocateColors(drawarea)
Widget drawarea;
{
	Display *dpy = XtDisplay(drawarea);
	int scr = DefaultScreen(dpy);
	Colormap cmap = DefaultColormap(dpy,scr);
	XColor colordef,reg_db_def;
	XGCValues gcv;

	BLACKPixel=BlackPixel(dpy,scr);
	WHITEPixel=WhitePixel(dpy,scr);
	/* Allocate colors to use for all drawing and coloring */
	if ( XAllocNamedColor(dpy,cmap,"green",&colordef,&reg_db_def)==0) {
		printf("Color green not found in DB...using white\n");
		UPPixel=WHITEPixel;
	}
	else UPPixel=colordef.pixel;

	if ( XAllocNamedColor(dpy,cmap,"red",&colordef,&reg_db_def)==0) {
		printf("Color red not found in DB...using black\n");
		DOWNPixel=BLACKPixel;
	}
	else DOWNPixel=colordef.pixel;

	if ( XAllocNamedColor(dpy,cmap,"blue",&colordef,&reg_db_def)==0) {
		printf("Color blue not found in DB...using black\n");
		NRPixel=BLACKPixel;
	}
	else NRPixel=colordef.pixel;
	if ( XAllocNamedColor(dpy,cmap,"Yellow",&colordef,&reg_db_def)==0) {
		printf("Color yellow not found in DB...using white\n");
		BUSYPixel=WHITEPixel;
	}
	else BUSYPixel=colordef.pixel;

	/* Set up graphics contexts to use for all line drawing */
	gcv.foreground = DOWNPixel;
	gcv.background = 2;
	gcv.line_style = LineOnOffDash;
	gcv.line_width = LineWidth;
	downgc = XCreateGC( XtDisplay(drawarea), XtWindow(drawarea),
	    GCForeground | GCBackground | GCLineStyle | GCLineWidth,
	    &gcv );

	gcv.foreground = UPPixel;
	gcv.background = 2;
	gcv.line_style = LineSolid;
	gcv.line_width = LineWidth;
	upgc = XCreateGC( XtDisplay(drawarea), XtWindow(drawarea),
	    GCForeground | GCBackground | GCLineStyle | GCLineWidth,
	    &gcv );

	gcv.foreground = NRPixel;
	gcv.background = 2;
	gcv.line_style = LineDoubleDash;
	gcv.line_width = LineWidth;
	nrgc = XCreateGC( XtDisplay(drawarea), XtWindow(drawarea),
	    GCForeground | GCBackground | GCLineStyle,
	    &gcv );

}

