/************************************************************************
 * FastPINGAPI	- Support routines for asynchronous pinging of nodes	*
 ***********************************************************************/
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/file.h>
#include <netdb.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <memory.h>

extern int verbose;

static int Ident = 0;		/* our pid is the identifier */
static pingsocketoptions = 0;

/************************************************************************
 * MakePINGSocket() - Make a socket for all ping communications		*
 ************************************************************************/
MakePINGSocket()
{
	int on = 1, s;
	struct protoent *proto;
	extern void exit();

	if ((proto = getprotobyname("icmp")) == NULL) {
		fprintf(stderr, "icmp: unknown protocol\n");
		exit(10);
	}
	if ((s = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
		fprintf(stderr,"MakePINGSocket(): ");
		perror("socket");
		fprintf(stderr,"FastPING API Error: Perhaps you didn't make this program as root.\nYou need to do a chown root pingd;chmod 4755 pingd\nto run pingd setuid root.");
		exit(5);
	}
	if (pingsocketoptions & SO_DEBUG)
		setsockopt(s, SOL_SOCKET, SO_DEBUG, &on, sizeof(on));
	if (pingsocketoptions & SO_DONTROUTE)
		setsockopt(s, SOL_SOCKET, SO_DONTROUTE, &on, sizeof(on));
	return( s );
}


/************************************************************************
 * ParsePINGResponse() - Return the sequence number of the responding 	*
 *			node or if unidentified				*
 ************************************************************************/
ParsePINGResponse( packet, packetlen, from )
unsigned char *packet;		/* Packet that was received */
int packetlen;			/* Length of the packet */
struct sockaddr_in *from; 	/* Who it came from */
{
	struct ip *ip;
	struct icmp *icp;
	int hlen;
	char *inet_ntoa();

	from->sin_addr.s_addr = ntohl(from->sin_addr.s_addr);

	ip = (struct ip *) packet;	/* Get a pointer to the ip packet */
	hlen = ip->ip_hl << 2;		/* get header length */
	if (verbose)
		printf("%d byte Packet from %s ->headerlength=%d\n", packetlen, 
			inet_ntoa(ntohl(from->sin_addr.s_addr)), hlen);
	if (packetlen < hlen + ICMP_MINLEN) {
		if (verbose)
			printf("packet too short (%d bytes) from %s\n", packetlen,
			    inet_ntoa(ntohl(from->sin_addr.s_addr)));
		return( 0 );
	}
	packetlen -= hlen;
	icp = (struct icmp *) (packet + hlen);
	if (icp->icmp_type != ICMP_ECHOREPLY)  {
#ifdef DETAILED_PACKET
		if (verbose) {
			printf("%d bytes from %s: ", packetlen,
			    inet_ntoa(ntohl(from->sin_addr.s_addr)));
			printf("icmp_type=%d (%s)\n",
			    icp->icmp_type, pr_type(icp->icmp_type) );
			for( i=0; i<12; i++)
				printf("x%2.2x: x%8.8x\n", 
					i*sizeof(long), *lp++ );
			printf("icmp_type=%d icmp_code=%d\n", 
					icp->icmp_type,icp->icmp_code );
		}
#endif
		return( 0 );
	}
	if(icp->icmp_id != Ident) {
		if (verbose)
			printf("Not my ICMP_ECHO request\n");
		return(0);                 /* 'Twas not our ECHO */
	}
	if (verbose)
		printf("It was my ICMP_ECHO request\n");
	return(icp->icmp_seq);
}


/***********************************************************************
 *    in_cksum - Compute the checksum for this packet                  *
 ***********************************************************************/
static unsigned int in_cksum(addr, len)
u_short *addr;
int len;
{
	register int nleft = len;
	register u_short *w = addr;
	register u_short answer;
	register int sum = 0;
	extern time_t time();

	/*
         *  Our algorithm is simple, using a 32 bit accumulator (sum),
         *  we add sequential 16 bit words to it, and at the end, fold
         *  back all the carry bits from the top 16 bits into the lower
         *  16 bits.
         */
	while( nleft > 1 )  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if( nleft == 1 )
		sum += *(u_char *)w;

	/*
         * add back carry outs from top 16 bits to low 16 bits
         */
	sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
	sum += (sum >> 16);                     /* add carry */
	answer = ~sum;                          /* truncate to 16 bits */
	return (answer);
}


/************************************************************************
 * MakePINGPacket() - make the Ping packet to user's specs		*
 ************************************************************************/
MakePINGPacket(Address, seq, datalen, hp, whereto)
char *Address;			/* IPAddress of packet */
int seq;			/* sequence number for this packet */
int datalen;			/* Length of data portion of ICMP message */
struct icmp *hp;		/* Pointer to a packet header */
struct sockaddr *whereto;	/* Socket info for sending packet */
{
	struct sockaddr_in *to;

	/*
	 *  Perform a few sanity checks:  Be sure our pointers aren't NULL
	 *  and be sure the desired packet length is greater than the size
	 *  of a ICMP packet header.  Be sure that the IP address is OK too.
	 */
	if ((Address == NULL) || (hp == NULL) || (whereto == NULL))
		return(-1);
	if (datalen <= sizeof(struct icmp))
		return(-2);

	/*
	 *  Fill in the socket structure.
	 */
	to = (struct sockaddr_in *) whereto;
	memset((char *) to, 0, sizeof(struct sockaddr));
	if ((to->sin_addr.s_addr = inet_addr(Address)) == -1L)
		return(-3);
	to->sin_family = AF_INET;
	to->sin_port = 0;

	hp->icmp_type = ICMP_ECHO;
	hp->icmp_code = 0;
	hp->icmp_cksum = 0;
	hp->icmp_seq = seq;
	if (Ident == 0)
		Ident = getpid() & 0xffff;
	hp->icmp_id = Ident;
	/*
	 *  It's OK to compute the checksum here:  it is merely the sum of
	 *  each collection of 16-bit shorts in the packet with some further
	 *  tweaking.  Since the data section of the packet will contain
	 *  ONLY zeros, it will not affect the checksum anyhow.  If the
	 *  data IS non-zero, then the checksum must be computed at a later
	 *  point (i.e., after this header has been prepended to the data).
	 */
	hp->icmp_cksum = in_cksum((u_short *) hp, sizeof(struct icmp));
	return(0);
}

#ifdef STANDALONE

main(argc, argv)
int argc;
char *argv[];
{
	int datalen;
	unsigned char *packet;
	struct sockaddr whereto;
	struct icmp icmp;

	datalen = MakePINGPacket("35.1.1.135", 1, 64, &icmp, &whereto );
}

#endif
