/************************************************************************
 * Discovery - build and maintain a network LINKS file			*
 *	This code does a basic network discovery algorithm.		*
 *************************************************************************/

#include <stdio.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <time.h>

#include "CommonDefs.h"
#include "snmptable2.h"
#include "Linkfileio.h"

/*
 *	MIB Variables we request from all nodes
 */
char *MIBRoutingTable[]={
		"ipRouteNextHop",
		"ifType",
		"ipAdEntIfIndex"
		};

char *progname;
char *network, *SubNetMask, *SubNet;
struct in_addr MySubNet, MySubNetMask;

#define MAXCYCLES 10
int CurrentCycleCount;

struct GenericNodeType {
        char Address[40];               	/* Definitive Node Address */
        struct InterfaceType *Interfaces;	/* List of Node Interfaces */
        struct NextHopType *Neighbors;		/* Adjacencies */
	struct GenericNodeType *Next;
} *GenericNodeHead=NULL;

struct InterfaceType {
        int ifnum;                      /* Interface Index */
        int ifType;                     /* Interface Type */
        char Address[40];               /* RemoteAddress */
        struct InterfaceType *Next;
};

struct NextHopType {
        char RemoteAddress[40];         /* Remote Address */
        struct NextHopType *Next;
};

struct AddressChain {
	char *Address;
	struct AddressChain *Next;
} *AddressListHead=NULL;

struct LinkPtrStruct {
	struct GenericNodeType *LocNode;	
	struct NextHopType *HPtr;	
	struct GenericNodeType *RemNode;
	struct NextHopType *RemHPtr;	
	struct InterfaceType *LocIf, *RemIf;	
};

struct NodeInterfaceTuple {
	struct GenericNodeType *Node;
	struct InterfaceType  *If;
};
/*
 *	InSubNet() - Is the given address in the subnet the user requested?
 */
int InSubNet(node)
     char *node;
{
struct in_addr addr;
static int virgin=1;

	if ( virgin ) {
		MySubNet.s_addr = inet_addr( SubNet );
		MySubNetMask.s_addr = inet_addr( SubNetMask );
	}
        addr.s_addr = inet_addr( node );
	if (( addr.s_addr & MySubNetMask.s_addr ) == MySubNet.s_addr )
		return(1);
	else
		return(0);

#ifdef OLD
/*        if (IN_CLASSA(addr.s_addr))
                addr.s_addr &= IN_CLASSA_NET;
        else if (IN_CLASSB(addr.s_addr))
                addr.s_addr &= IN_CLASSB_NET;
        else if (IN_CLASSC(addr.s_addr))
                addr.s_addr &= IN_CLASSC_NET;
        else
                fprintf(stderr, "Unknown class address %s\n", node);
	if ( addr.s_addr == MySubNet.s_addr ) return(1);
	else return(0); */
#endif
}

/*
 *   FindGenericNode - search for the node with this Definitive Address
 */
struct GenericNodeType *FindGenericNode( NodeName )
char *NodeName;
{
register struct GenericNodeType *P;

	for( P=GenericNodeHead; P; P=P->Next )
		if ( strcmp( NodeName, P->Address ) == 0 ) return( P );
	return(NULL);

}

/*
 *	AddGenericNode - Create a Generic Node struct
 */
struct GenericNodeType *AddGenericNode( NodeName )
char *NodeName;
{
struct GenericNodeType *P;

	if ((P=(struct GenericNodeType *)malloc( sizeof(struct GenericNodeType)))==NULL) {
		fprintf(stderr,"malloc failed in AddGenericNode\n");
		exit(1);
	}
	strcpy( P->Address, NodeName );
	P->Interfaces=NULL;
	P->Neighbors=NULL;
	P->Next=NULL;
	if ( GenericNodeHead==NULL ) GenericNodeHead=P;
	else {
		P->Next=GenericNodeHead;
		GenericNodeHead=P;
	}
	return( P );
}

/*
 *	AddInterface - Create an Interface struct
 */
struct InterfaceType *AddInterface(N, interfacenum)
     struct GenericNodeType *N;
     int interfacenum;
{
struct InterfaceType *I,*IPtr,*IPrev;

	if ( interfacenum == 0 ) {
		fprintf( stderr,"AddInterface(): Interface Index should never be 0\n");
		exit(1);
	}
	if ((I=(struct InterfaceType *)malloc(sizeof(struct InterfaceType)))==NULL) {
		fprintf( stderr, "malloc failed in AddInterface()\n");
		exit(1);
	}
	I->ifnum=interfacenum;
	I->ifType=0;
	I->Address[0]='\0';
	I->Next=NULL;
	if ( N->Interfaces == NULL )	/* Chain in the first interface */
		N->Interfaces=I;
	else {
		IPrev=NULL;
		for ( IPtr=N->Interfaces; IPtr!=NULL; IPtr=IPtr->Next) 
			if ( I->ifnum < IPtr->ifnum ) break;
			else IPrev=IPtr;

		if ( IPrev == NULL ) {	/* At head */
			I->Next=IPtr;
			N->Interfaces=I;
		} else {
			I->Next=IPrev->Next; /* Elsewhere */
			IPrev->Next=I;
		}
#ifdef ORIG
		for ( IPtr=N->Interfaces; IPtr->Next!=NULL; IPtr=IPtr->Next );
		IPtr->Next=I;
#endif
	}
	return( I );
}

/*
 *	FindInterface - find interface struct given interface index
 */
struct InterfaceType *FindInterface(N, ifNum)
     struct GenericNodeType *N;
     int ifNum;
{
register struct InterfaceType *I;

	
	for( I=N->Interfaces; I != NULL; I=I->Next )
		if ( I->ifnum == ifNum ) return(I);
	return(NULL);
}

/*
 *	AddIfType - We just discovered an interface on this box
 *		Add/update this interface struct
 */
void AddIfType(N, inst, value)
     struct GenericNodeType *N;
     char *inst;
     char *value;
{
struct InterfaceType *I;
int interfacenum=atoi(inst );

	if ( interfacenum == 0 ) {
		fprintf( stderr,"AddIfType(): Interface Index should never be 0\n");
		exit(1);
	}
	if ( ( I=FindInterface( N, interfacenum ) ) == NULL )
		I=AddInterface( N, interfacenum );
	I->ifType=atoi(value);
	/*printf("AddIfType(): %s Interface: <%s> #%d type=%d\n",N->Address,I->Address,I->ifnum,I->ifType);*/
}

/*
 *	AddIPAddr - we just discovered an interface address on this box
 *		Add/update the interface struct
 */
void AddIPAddr(N, inst, interfacenum )
     struct GenericNodeType *N;
     char *inst;
     int interfacenum;
{
struct InterfaceType *I;

	if ( interfacenum == 0 ) {
		fprintf( stderr,"AddIPAddr(): Interface index should never be 0\n");
		exit(1);
	}
	if ( ( I=FindInterface( N, interfacenum ) ) == NULL )
		I=AddInterface( N, interfacenum );
	strcpy(I->Address,inst);
	/*printf("AddIPAddr(): %s Interface: <%s> #%d type=%d\n",N->Address,I->Address,I->ifnum,I->ifType);*/
}

/*
 *	AddNextHop - We just discovered an adjacency
 *		Add/Update a neighbor struct
 */
void AddNextHop(RespondingNode, NextHop)
     struct GenericNodeType *RespondingNode;
     char *NextHop;
{
register struct NextHopType *N,*NewHop;
register struct InterfaceType *I;

	if ( ! InSubNet( NextHop ) ) return;	/* Assure within SubNet */
						/* Don't count my ifaces */
	for( I=RespondingNode->Interfaces; I!=NULL; I=I->Next )
		if ( strcmp( NextHop, I->Address ) == 0 ) return;

	for( N=RespondingNode->Neighbors; N!=NULL;  N=N->Next)
	    if ( strcmp(N->RemoteAddress, NextHop ) == 0 )
		return;	/* Next Hop Already in list! */

	if ((NewHop=(struct NextHopType *)malloc( sizeof(struct NextHopType))) == NULL ) {
		fprintf( stderr,"malloc() failed, exitting\n");
		exit(1);
	}
	strcpy( NewHop->RemoteAddress, NextHop );
	NewHop->Next=NULL;
	if ( RespondingNode->Neighbors == NULL ) 
		RespondingNode->Neighbors=NewHop;
	else {
		NewHop->Next=RespondingNode->Neighbors; /* Chain it in front */
		RespondingNode->Neighbors=NewHop;	/* New Head */
	}
}

/*
 *	GenericReport - For debugging 
 */
void GenericReport()
{
struct GenericNodeType *N;
struct NextHopType *H;
struct InterfaceType *I;

	for( N=GenericNodeHead; N != NULL; N=N->Next )  {
		for( H=N->Neighbors; H != NULL; H=H->Next ) 
			printf("%s LINK GENLINK %s ?? %s ??\n" ,
				N->Address,N->Address,H->RemoteAddress);
		for( I=N->Interfaces; I != NULL; I=I->Next )
			printf("%s INTERFACE %s %d %d\n" ,
				N->Address,I->Address, I->ifnum,I->ifType);
	}
}

/*
 *	PruneInterfaceTree
 */
void PruneInterfaceTree( )
{
struct GenericNodeType *N;
struct NextHopType *H;
struct InterfaceType *I,*IPrev;

	for( N=GenericNodeHead; N != NULL; N=N->Next )  {
		IPrev=NULL;
		for( I=N->Interfaces; I != NULL; I=I->Next ) {
			if ( I->Address==NULL || strlen(I->Address)==0 ) {
				fprintf(stderr, "Pruning %s interface # %d Type %d - No Address: %s\n",N->Address, I->ifnum, I->ifType, I->Address);

				/* a bit sloppy - should free mem but we
				 * will exit soon anyway */
				if ( IPrev==NULL ) 
					N->Interfaces=I->Next;
				else 
					IPrev->Next=I->Next;
			}
			IPrev=I;
		}
	}
}

/*
 *	FindNodeWithInterface() - match interface to node struct
 *
 * Search for caller's address in the Interface List for each node
 * return pointer to node and ifnum and iftype set.
 * if not found set ifType and ifNum to 0
 */
struct NodeInterfaceTuple *FindNodeWithAddress(Address)
char *Address;
{
register struct GenericNodeType *NPtr;
static struct NodeInterfaceTuple Answer;
struct InterfaceType *IPtr;
	
	for( NPtr=GenericNodeHead; NPtr != NULL; NPtr=NPtr->Next )  {
		for( IPtr=NPtr->Interfaces; IPtr!=NULL; IPtr=IPtr->Next ) {
			if ( strcmp( IPtr->Address, Address ) == 0 ) {
				Answer.Node=NPtr;
				Answer.If=IPtr;
				return( &Answer );
			}
		}
	}
	return( NULL );
}

/*
 *	Set Node struct with this name
 */
void SetNodeName(NPtr)
struct GenericNodeType *NPtr;
{
register struct InterfaceType *I;

	for ( ; NPtr!=NULL; NPtr=NPtr->Next ) {
		if ( NPtr->Interfaces == NULL ) {
			fprintf(stderr,"Weird - node %s has no interfaces\n",
							NPtr->Address);
			continue;
		}
		for( I=NPtr->Interfaces; I!=NULL; I=I->Next ) {
			if (I->ifType!=22 && I->ifType!=18 && I->Address[0]!='\0' ) {
				strcpy(NPtr->Address,I->Address);
				break;
			}
			if ( NPtr->Address!=I->Address )
				strcpy(NPtr->Address,NPtr->Interfaces->Address);
		}
	}
}

/*
 *	MakeAddressChainStruct - Create a generic address chain
 */
struct AddressChain *MakeAddressChainStruct( addr )
char *addr;
{
struct AddressChain *A=(struct AddressChain *)malloc( sizeof(struct AddressChain));
char *Addr=(char *) malloc( strlen(addr) +1 );

	if ( A==NULL || Addr==NULL ) {
		fprintf(stderr,"malloc failed in MakeAddress()\n");
		return(NULL);
	}
	strcpy(Addr,addr);
	A->Address=Addr;
	return(A);
}

/*
 *	Mark this node as non-responsive by creating an entry
 *	This is so we don't keep trying to discover a non-responsive node
 */
void AddNonResponsiveNode( addr )
char *addr;
{
struct AddressChain *A=MakeAddressChainStruct( addr );

	if ( AddressListHead==NULL ) 
		AddressListHead=A;
	else {
		A->Next=AddressListHead;
		AddressListHead=A;
	}
}

/* 
 *  Returns a boolean indicating whether or not this node is responding or not 
 */
int NonResponsive( addr )
char *addr;
{
register struct AddressChain *Ptr;

	for( Ptr=AddressListHead; Ptr!=NULL; Ptr=Ptr->Next )
		if ( strcmp( addr, Ptr->Address ) == 0 ) return(1);
	return(0);
}

/*
 *	MakeNextHostList() - make an address list with all the nodes we want
 *			to query on this next pass
 */
struct AddressListType *MakeNextHostList()
{
register struct GenericNodeType *NPtr;
register struct NextHopType *HPtr;
register struct GenericNodeType *N;
register struct InterfaceType *I;
struct AddressListType *AddressListHead=NULL;
int FoundIt;

	/* 
	 *	For each NextHop that we haven't seen as one of our interfaces
	 *	add it to the list of nodes we query  
	 */
	for( NPtr=GenericNodeHead; NPtr != NULL; NPtr=NPtr->Next )  {
		for( HPtr=NPtr->Neighbors; HPtr != NULL; HPtr=HPtr->Next ) {
			/* 	
			 * 	Search for this Next Hop ?one of our interfaces
			 */
			FoundIt=0;	
			for( N=GenericNodeHead; N != NULL; N=N->Next )  {
				for( I=N->Interfaces; I!=NULL; I=I->Next ) {
					if ( strcmp( HPtr->RemoteAddress, I->Address ) == 0 ) {
						FoundIt=1;
						break;  /* known next hop */
					}
				}
				if ( FoundIt ) break; /* No need to look on */
			}
			if ( !FoundIt ) {
				if ( InSubNet(HPtr->RemoteAddress)) {
				if ( ! NonResponsive( HPtr->RemoteAddress ) )
					AddressListHead=AddAddress( AddressListHead, 
					HPtr->RemoteAddress );
				} else {
					if (verbose) printf("host not in subnet: host=%s subnet=%s\n",HPtr->RemoteAddress,inet_ntoa(MySubNet));
				}
			}
		}	/* Process each Next Hop */
	}
	return(AddressListHead);
}

/************************************************************************
 * AddMIBRoutingTableData() - Add MIB Routing Table into nexthop table	*
 ************************************************************************/
void AddMIBRoutingTableData(RespondingNode, VarName, instance, value)
     char *RespondingNode;
     char *VarName;
     char *instance;
     char *value;
{
struct GenericNodeType *N;

	/* Find this Node and append the next hop if not already there */
	if (( N=FindGenericNode(RespondingNode) ) == NULL ) 
		N=AddGenericNode(RespondingNode);
	if ( strcmp(VarName,"ipRouteNextHop")==0)
		AddNextHop( N, value );
	if ( strcmp(VarName,"ifType")==0)
		AddIfType( N, instance, value );
	if ( strcmp(VarName,"ipAdEntIfIndex")==0)
		AddIPAddr( N, instance, atoi( value ) );
}

/************************************************************************
 *      FormatMIBRoutingTableData() - Primary Call for processing 	*
 *	ALL MIBRoutingTable Data      					*
 ************************************************************************/
void FormatMIBRoutingTableData(AddressListHead)
     struct AddressListType *AddressListHead;
{
struct AddressListType *A;
struct VarList *v;
struct bindings *b;

   for( A=AddressListHead; A!=NULL; A=A->Next ) {
        for( v=A->ResponseListHead; v!=NULL; v=v->Next)
                for( b=v->Head; b!=NULL; b=b->Next )
			 AddMIBRoutingTableData(A->Address,v->VarName,b->instance+strlen(v->Prefix),b->value);
   }
}

/************************************************************************
 *  FindNextHopLink() - Real work horse.  Match up next hops with other	*
 *			nodes' local interfaces				*
 ************************************************************************/
struct LinkPtrStruct *FindNextHopLink( LocNode, HPtr ) 
struct GenericNodeType *LocNode;	/* IN */
struct NextHopType *HPtr;		/* IN */
{
static struct LinkPtrStruct L;
struct NodeInterfaceTuple *NI;
register struct GenericNodeType *RemNode;	
register struct NextHopType *RemHPtr;		
register struct InterfaceType *LocIf, *RemIf;

	if ((NI=FindNodeWithAddress( HPtr->RemoteAddress ))==0) {
		fprintf(stderr,"Can't find node with Interface %s\n",
			HPtr->RemoteAddress);
		return(NULL);
	}
	RemNode=NI->Node;
	RemIf=NI->If;
	/****************************************************************/
	/*  	So far we have the Remote Node, and Remote Interface 	*/
	/* Now, look through this Remote Node's next hops for a local @ */
	/****************************************************************/
	for ( RemHPtr=RemNode->Neighbors; RemHPtr!=NULL; RemHPtr=RemHPtr->Next)
		for( LocIf=LocNode->Interfaces; LocIf!=NULL; LocIf=LocIf->Next )
			if ( strcmp( LocIf->Address, RemHPtr->RemoteAddress ) == 0 ) {
				L.LocNode=LocNode;
				L.HPtr=HPtr;
				L.RemNode=RemNode;
				L.RemHPtr=RemHPtr;
				L.LocIf=LocIf;
				L.RemIf=RemIf; 
				return(&L);	/* Found it */
			}
	return(NULL);	
}

/************************************************************************
 *	LinksReport()      
 *
 ************************************************************************/
void LinksReport(network)
     char *network;
{
register struct GenericNodeType *LocNode;
register struct NextHopType *HPtr;
time_t TimeNow=time(&TimeNow);
struct LinkPtrStruct *LPtr;
struct LinkDetail *L;

	SetNodeName( GenericNodeHead );
	for( LocNode=GenericNodeHead; LocNode != NULL; LocNode=LocNode->Next )  {
		for( HPtr=LocNode->Neighbors; HPtr != NULL; HPtr=HPtr->Next){
			if ( LPtr=FindNextHopLink( LocNode, HPtr ) ) {
				/*** We found the link information we need */
				if ( strcmp(LocNode->Address,LPtr->RemNode->Address) == 0 )
					continue; /* Loopback */
				if ( LPtr->LocIf->ifType == 0 ||
					LPtr->LocIf->ifnum == 0 ||
					LPtr->RemIf->ifnum == 0 ) {
					fprintf(stderr,"0 type or interface\n");
					continue;
				}
				if ( strcmp(LocNode->Address,LPtr->RemNode->Address) < 0 ) {
					printf("%d %ld %s %s %d %d %s %s %ld 0 0\n", 
						LPtr->LocIf->ifType,
						TimeNow, 
						LPtr->LocNode->Address,
						LPtr->LocIf->Address, 
						LPtr->LocIf->ifnum, 
						LPtr->RemIf->ifnum, 
						LPtr->RemIf->Address, 
						LPtr->RemNode->Address, 
						TimeNow );
					if (( L=FindLinkDetail(
						LPtr->LocNode->Address, 
						LPtr->LocIf->ifnum ))==NULL ) {
					fprintf(stderr,"This is a NEW LINK!\n");
					AddLinkDetail( LPtr->LocIf->ifType, 
						TimeNow, LPtr->LocNode->Address,
                                		LPtr->LocIf->Address, 	
						LPtr->LocIf->ifnum,
                                		TimeNow, 			
						LPtr->RemNode->Address, 
						LPtr->RemIf->Address,
                                		LPtr->RemIf->ifnum, 0,0 );
					}
					else {
						UpdateLink( 
						LPtr->LocNode->Address,
						LPtr->LocIf->Address,
                                                LPtr->LocIf->ifnum,
						LPtr->RemIf->ifnum,
						LPtr->RemIf->Address,
						LPtr->RemNode->Address );
					}
				}
				else fprintf(stderr,"Link found: %s to %s but order is reversed.waiting\n",LocNode->Address, LPtr->RemNode->Address);
			}
			else fprintf(stderr,"Couldn't find next hop link from %s to %s\n",LocNode->Address, HPtr->RemoteAddress );
		}
	}
	Write_LinkDetail_File(network);
}


/*
 * Usage()
 */
void Usage()
{
	fprintf(stderr,"Usage: %s [-v] [-d] [-n network] -h <node> ... -h <node> -r <retries> -t <timeout> -c <community> -s <subnet> -m <subnetmask>\n\
where: \n\
-s subnet defines the subnet to discover\n\
-m subnetmask defines the network mask to determine if node is in subnet\n\
-t timeout is in seconds\n\
-c community defines the community name to use for the SNMP queries\n\
-r retries defines how many query attempts to make before giving up\n",progname);
	exit(1);
}

/**************************************************************************
 * main() 
 **************************************************************************/
int main( argc, argv )
int argc;
char *argv[];
{
char *community=NULL;
struct VarList *QueryListHead=NULL;
struct AddressListType *AddressListHead=NULL,*A=NULL,*APtr;
struct GenericNodeType *N;
int i;
extern int MaxSecsB4Retry;
extern int MaxSNMPRetries;

struct LinkDetail *LinkDetailHead,*LPtr;

	progname=argv[0];
        argv++; argc--;
        while(argv[0][0] == '-') {
                switch(argv[0][1]) {
                case 'd':
                case 'v':       
				fprintf(stderr,"Verbose Mode=ON\n");
				verbose=1;
                                argv++; argc--;
                                break;
                case 'h':
                        AddressListHead=AddAddress( AddressListHead, argv[1] );
                        argv++; argc--;
                        argv++; argc--;
                        break;
                case 'c':
                        community=argv[1];
                        argv++; argc--;
                        argv++; argc--;
                        break;
                case 'r':
                        argv++; argc--;
                        if ( ( argc != 0 ) && ( atoi(argv[0]) != 0 )) {
                                MaxSNMPRetries=atoi(argv[0]);
                                argv++; argc--;
                        }
                        else fprintf(stderr,"Invalid retry specification\n");
                        break;
                case 't':
                        argv++; argc--;
                        if ( ( argc != 0 ) && ( atoi(argv[0]) != 0 )) {
                                 MaxSecsB4Retry=atoi(argv[0]);
                                argv++; argc--;
                        }
                        else fprintf(stderr,"Invalid retry specification\n");
                        break;
		case 'm':
                        argv++; argc--;
                        if ( argc != 0 )  {
				SubNetMask=argv[0];
                                argv++; argc--;
                        }
			break;
		case 's':
                        argv++; argc--;
                        if ( argc != 0 )  {
				SubNet=argv[0];
                                argv++; argc--;
                        }
			break;
                case 'n':
                        argv++; argc--;
                        if ( argc != 0 ) {
                                 network=argv[0];
                                argv++; argc--;
                        }
                        else fprintf(stderr,"Invalid network specification\n");
                        break;
                default:
                        fprintf(stderr,"unknown option %s ignored\n(Valid options are: -v )\n",argv[0]);
                        argv++; argc--;
                        break;
                }
        }
	if ( community==NULL) {
		fprintf(stderr,"You MUST specify a community name \n");
		Usage();
	}
	if ( network==NULL) {
		fprintf(stderr,"You MUST specify a network name \n");
		Usage();
	}
	if ( SubNet == NULL || SubNetMask == NULL ) {
		fprintf(stderr,"You MUST specify a SubNet and SubNet Mask\n");
		Usage();
	}

	MaxSecsB4Retry=10;	/* Allow a long time for queries */
	MaxSNMPRetries=10;

	/*
	 *	Read from network LINKS file for a list of nodes to query
	 */
	LinkDetailHead=Read_LinkDetail_File( network );
	for( LPtr=LinkDetailHead; LPtr!=NULL; LPtr=LPtr->Next ) {
		if ( (N=FindGenericNode(LPtr->LocalNode)) == NULL )
			N=AddGenericNode( LPtr->LocalNode );
		if ( FindInterface( N, LPtr->LocalDSUNumber) == NULL )
			AddInterface( N ,LPtr->LocalDSUNumber);
		AddIPAddr( N, LPtr->LocalIPAddr,LPtr->LocalDSUNumber);

		if ( (N=FindGenericNode(LPtr->RemoteNode)) == NULL )
			N=AddGenericNode( LPtr->RemoteNode );
		if ( FindInterface( N,LPtr->RemoteDSUNumber) == NULL )
			AddInterface( N, LPtr->RemoteDSUNumber);
		AddIPAddr( N, LPtr->RemoteIPAddr, LPtr->RemoteDSUNumber);

		/*
		 *	Add local and remote nodes to list to query
		 *	Don't add the same node twice!
		 *
		 *	LOCAL SIDE
		 */
		for (APtr=AddressListHead; APtr!=NULL; APtr=APtr->Next)
			if ( strcmp(APtr->Address, LPtr->RemoteNode) == 0 )
				break;
		if ( APtr==NULL ) 
			AddressListHead=AddAddress( AddressListHead, LPtr->RemoteNode);
		/*
		 *	REMOTE SIDE
		 */
		for (APtr=AddressListHead; APtr!=NULL; APtr=APtr->Next)
			if ( strcmp(APtr->Address, LPtr->LocalNode) == 0 )
				break;
		if ( APtr==NULL ) {
			AddressListHead=AddAddress( AddressListHead, LPtr->LocalNode);
		}
	}
	if ( AddressListHead == NULL ) {
		fprintf(stderr,"No hosts to query\n");
		exit(1);
	}

	for(i=0; i<((sizeof(MIBRoutingTable))/(sizeof(char *))); i++)
       		QueryListHead=AddVar(QueryListHead,MIBRoutingTable[i]);


	for( ; ; ) {
		if (++CurrentCycleCount > MAXCYCLES ) break;
		fprintf(stderr,"--------------- Start of collection cycle #%d-----------\n",CurrentCycleCount);
		for( A=AddressListHead; A!=NULL; A=A->Next ) 
			printf("%s\n",A->Address);
		if ( AddressListHead == NULL ) break;

		AddressListHead=GetTable( AddressListHead, community, QueryListHead );
		for( A=AddressListHead; A!=NULL; A=A->Next ) {
			if (A->QueryListHead!=NULL) {
				fprintf(stderr,"%s did not respond completely to Neighbor Query\n",A->Address);
				AddNonResponsiveNode( A->Address );
			}
			else FormatMIBRoutingTableData( A );
		}
		PruneInterfaceTree();  /* Remove Interfaces with no IP Addr */
		GenericReport();
		FreeAddressList( AddressListHead );
       		AddressListHead=MakeNextHostList();
		if ( AddressListHead == NULL ) 
			break; /* All neighbors tried */
	}
	if (verbose) GenericReport();
	LinksReport(network);
	exit(0);
}

