/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
File:    interface.c
System:  xmib ver. 1.00
Author:  Ranen Goren
Date:    20/07/92

File remarks: contains the Motif interface routines for xmib
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/


#include <stdio.h>
#include <string.h>
#include <varargs.h>
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h> 
#include <Xm/Xm.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/ScrolledW.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/SelectioB.h>
#include <Xm/MessageB.h>
#include <Xm/Text.h>
#include <X11/cursorfont.h>
#include <Tree.h>
#include <snmp.h>         /* for the receive/get snmpCall() commands */
#include "mibinc.h"
#include "xmib.h"
#include "xmib.bmp"       /* the xmib icon & logo bitmap */
#include "online.hlp"     /* on-line help text */

SNMP_PARAMS snmpParams;

static XtResource resources[] = {
  { "root",         "Root",          XtRString, sizeof(String),
	XtOffset(snmpParamsPtr, root),          XtRString, "mib-2"          },
  { "timeout",      "Timeout",       XtRInt,    sizeof(int),
	XtOffset(snmpParamsPtr, timeout),       XtRString, "5"              },
  { "retries",      "Retries",       XtRInt,    sizeof(int),
	XtOffset(snmpParamsPtr, retries),       XtRString, "3"              },
  { "port",         "Port",          XtRInt,    sizeof(int),
	XtOffset(snmpParamsPtr, port),          XtRString, "161"            },
  { "community",    "Community",     XtRString, sizeof(String),
	XtOffset(snmpParamsPtr, community),     XtRString, "public"         },
  { "host",         "Host",          XtRString, sizeof(String),
	XtOffset(snmpParamsPtr, host),          XtRString, "No host given"  },
  { "minDepth",     "MinDepth",      XtRInt,    sizeof(int),
	XtOffset(snmpParamsPtr, min_depth),     XtRString, "1"              },
  { "maxDepth",     "MaxDepth",      XtRInt,    sizeof(int),
	XtOffset(snmpParamsPtr, max_depth),     XtRString, "7"              },
  { "defaultDepth", "DefaultDepth",  XtRInt,    sizeof(int),
	XtOffset(snmpParamsPtr, default_depth), XtRString, "2"              },
  { "travInterval", "TravInterval",  XtRInt,    sizeof(int),
	XtOffset(snmpParamsPtr, travInterval),  XtRString, "0"              }
};


XtAppContext app;
Widget toplevel, mainBox, treeW, sw, scale, varSpec, varSetButton;
Widget setDialog, jumpDialog, paramDialog, logo, varBox, controlBox, rightBox;
Widget hostW, communityW, timeoutW, retriesW, portW;
char   *varValue = NULL;
Widget lastTouched = NULL;
char   *which;
XtIntervalId activeTimer = 0;
MIBNODE *mibRoot;


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* initializes the tree widget, calls fillTree to fill it, and manages it */

void MibShowXTree(root)
  MIBNODE *root;
{
    int        n;
    Arg        wargs[10];
    int        maxDepth;

    mibRoot = root;

            /* create the tree widget */
    n=0;    
    XtSetArg(wargs[n], XtNverticalSpace,   20);            n++;
    XtSetArg(wargs[n], XtNhorizontalSpace, 100);           n++;
    treeW = XtCreateWidget("tree", XstreeWidgetClass, sw, wargs, n);

       /* read the current maximum tree depth */
    XtVaGetValues(scale, XmNvalue, &maxDepth, NULL);
    
    fillTree(treeW, root, maxDepth);
    XtManageChild(treeW);
    XtRealizeWidget(toplevel);
    centerNode(NULL, mibRoot, NULL);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* recursive function that destroys a tree node's widget and then calls
   itself on all children of the node. calling MibDestroyXTree() on a tree
   root results in the destruction of all widgets in the subtree.       */

void MibDestroyXTree(root)
  MIBNODE *root;
{
    MIBNODE *cur;
    
    if ( (root == NULL) || (root->widget == NULL) )
	return;
    XtDestroyWidget(root->widget);
    for (cur = root->child1;  cur && cur->widget;  cur = cur->brother)
	MibDestroyXTree(cur);
    if (root == mibRoot)
	XtDestroyWidget(treeW);
}






/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* attaches a widget to each node of the tree, calling by recursion on
   all children of root, until the initial passed level is reached. The
   created widgets are inserted to the tree widget, and their pointers
   are also stored in the MIB tree structure. Also, the pointer to the
   tree entry for each var is stored as a userData in this widget.       */

void fillTree(treeW, root, level)
  Widget  treeW;
  MIBNODE *root;
{
    Widget        w;
    Arg           wargs[10];
    int           n = 0;
    Widget        superNode;
    XtActionsRec  actions;
    char          translations[1000];
    char          *widgetName;
    
    if (root == NULL)     /* tree end */
	return;

       /* register the translations */
    actions.string =  "newTree";
    actions.proc   =  newTree;
    XtAppAddActions(app, &actions, 1);
    actions.string =  "updateVar";
    actions.proc   =  updateVar;
    XtAppAddActions(app, &actions, 1);
    actions.string =  "centerNode";
    actions.proc   =  centerNodeTran;
    XtAppAddActions(app, &actions, 1);

    if (root == mibRoot)
	superNode = NULL;
    else
	superNode = MibFindNode(root->father->name)->widget;
    
    sprintf(translations,
	    "<Btn3Down>(2+): ArmAndActivate() updateVar(toggle) Disarm() \n\
	     <Btn3Down>:       ArmAndActivate() updateVar(noToggle)  Disarm() \n\
             <Btn1Down>:       ArmAndActivate() newTree(%s)          Disarm() \n\
	     <Btn2Down>:       ArmAndActivate() centerNode()         Disarm()",
	    (root != mibRoot ? root->name : 
	     (root->father == NULL ? root->name : root->father->name)));
		
    widgetName = (root->child1 == NULL ? "leaf" : "node");
    
    /*
     * Create a widget for the node, specifying the
     * given super_node constraint.
     */
    n = 0;
    XtSetArg(wargs[n], XmNtranslations,	XtParseTranslationTable(translations)); n++;
    XtSetArg(wargs[n], XtNsuperNode,    superNode);                             n++;
    XtSetArg(wargs[n], XmNuserData,     root);   /* save the node as data */    n++;
    w  =  XtCreateManagedWidget(widgetName, xmPushButtonWidgetClass, 
			 treeW, wargs, n);
    xs_wprintf(w, "%s", root->name);
    root->widget = w;
    
    /*
     * Recursively create the subnodes.
     */
    if (level > 1)
	for (root = root->child1;  root != NULL;  root = root->brother)
	    fillTree(treeW, root, level-1);
    else
	if (root->child1 != NULL)
	    root->child1->widget = NULL;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) performs some cleanup procedures, and then exits normally */

void quit(w, root, cbs)
  Widget w;
  Widget root;
  XtPointer cbs;
{
    MibDestroyXTree(mibRoot);            /* remove widgets */
    snmpClose();                         /* free the tree related resources */
    XtCloseDisplay(XtDisplay(toplevel)); 
    XFlush(XtDisplay(toplevel));
    exit(0);       /* goodbye */
}






/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* creates almost all widgets of xmib */

void initWidgets(argc, argv)
  int *argc;
  char **argv;
{
    Widget     mainQuit, mainHelp, center, snmpParamW, jumpButton;
    Pixmap     icon;
    Pixmap     logoPixmap;
    Pixel      logoFg, logoBg;
    XmString   scaleTitle;
    XmString   varSpecStr;
    Arg        wargs[10];
    int        n;
    
    toplevel = XtAppInitialize(&app, "Xmib", NULL, 0, 
			       argc, argv, NULL, NULL, 0);

    XtGetApplicationResources(toplevel, &snmpParams, resources,
			      XtNumber(resources), NULL, 0);

    checkCommandLine(*argc, argv);    /* look for overrides */
    
        /* set the xmib icon */
    icon = XCreateBitmapFromData(XtDisplay(toplevel),
				 RootWindowOfScreen(XtScreen(toplevel)),
				 xmib_bits, xmib_width, xmib_height);
    XtVaSetValues(toplevel, XmNiconPixmap, icon, NULL);

    n=0;      /* mainBox */
    mainBox = XtCreateManagedWidget("mainBox", xmFormWidgetClass,
				    toplevel, wargs, n);

    n=0;      /* rightBox - contains all widgets to the right of the tree */
    XtSetArg(wargs[n], XmNtopAttachment,    XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM);        n++;
    rightBox = XtCreateManagedWidget("rightBox", xmFormWidgetClass,
				     mainBox, wargs, n);

    n=0;      /* controlBox - contains the Quit, Help, Center... widgets */
    XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNentryAlignment,   XmALIGNMENT_CENTER);   n++;
    controlBox = XtCreateManagedWidget("controlBox", xmRowColumnWidgetClass,
				       rightBox, wargs, n);
   
    n=0;      /* varBox - contains the variable info, xmib params, set button */
    XtSetArg(wargs[n], XmNtopAttachment,    XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM);        n++;
    varBox = XtCreateManagedWidget("varBox", xmFormWidgetClass,
				   rightBox, wargs, n);

    n=0;      /* logo - the pixmap that appears between varBox and controlBox */
    XtSetArg(wargs[n], XmNtopAttachment,    XmATTACH_WIDGET);      n++; 
    XtSetArg(wargs[n], XmNtopWidget,        varBox);               n++;
    XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_WIDGET);      n++; 
    XtSetArg(wargs[n], XmNbottomWidget,     controlBox);           n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNlabelType,        XmPIXMAP);             n++;
    logo = XtCreateManagedWidget("logo", xmLabelWidgetClass,
				 rightBox, wargs, n);
    XtVaGetValues(logo, XmNforeground, &logoFg, XmNbackground, &logoBg, NULL);
    logoPixmap = XCreatePixmapFromBitmapData(XtDisplay(toplevel), 
					 RootWindowOfScreen(XtScreen(toplevel)),
					 xmib_bits, xmib_width, xmib_height, 
					 logoFg, logoBg, 
					 DefaultDepthOfScreen(XtScreen(toplevel)));
    XtVaSetValues(logo, XmNlabelType,   XmPIXMAP,
		        XmNlabelPixmap, logoPixmap,
		        NULL);
   
    n=0;    /* sw - a scrolledWindow widget to contain the tree widget */
    XtSetArg(wargs[n], XmNscrollingPolicy,  XmAUTOMATIC);          n++;
    XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNtopAttachment,    XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_WIDGET);      n++; 
    XtSetArg(wargs[n], XmNrightWidget,      rightBox);             n++;
    sw = XtCreateManagedWidget("scroll",
			       xmScrolledWindowWidgetClass,
			       mainBox, wargs, n);

       /* the varBox widgets */
    snmpParamW = dispSnmpParams(varBox);
    varSpecStr = XmStringCreateLtoR("Name: \n\nValue:\n\nType:\nAccess:\nStatus:", 
				    XmSTRING_DEFAULT_CHARSET);

    n=0;     /* varSpec - a multi-lined label, contains a variables info */
    XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM);         n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM);         n++; 
    XtSetArg(wargs[n], XmNlabelString,      varSpecStr);            n++;
    XtSetArg(wargs[n], XmNtopAttachment,    XmATTACH_WIDGET);       n++; 
    XtSetArg(wargs[n], XmNtopWidget,        snmpParamW);            n++;
    XtSetArg(wargs[n], XmNalignment,        XmALIGNMENT_BEGINNING); n++;
    varSpec = XtCreateManagedWidget("varSpec", xmLabelWidgetClass,
				    varBox, wargs, n);
    safefree(varSpecStr);
    
    n=0;     /* varSetButton - is pressed to change a var's value */
    XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM);        n++;
    XtSetArg(wargs[n], XmNtopAttachment,    XmATTACH_WIDGET);      n++; 
    XtSetArg(wargs[n], XmNtopWidget,        varSpec);              n++;
    XtSetArg(wargs[n], XmNsensitive,        False);                n++;
    varSetButton = XtCreateManagedWidget("varSetButton", xmPushButtonWidgetClass,
					 varBox, wargs, n);
    XtAddCallback(varSetButton, XmNactivateCallback, setVarValue, NULL);
   
    n=0;     /* setDialog - the dialog that appears when setting a variable */
    XtSetArg(wargs[n], XmNselectionLabelString,
	     XmStringCreateSimple("New value: "));  n++;
    XtSetArg(wargs[n], XmNdialogStyle,   XmDIALOG_FULL_APPLICATION_MODAL);      n++;
    setDialog = XmCreatePromptDialog(varBox, "setDialog", wargs, n);
    XtUnmanageChild(XmSelectionBoxGetChild(setDialog, XmDIALOG_HELP_BUTTON));
    XtAddCallback(setDialog, XmNokCallback, okSetVarValue, NULL);
   
    
       /* the controlBox widgets */
    scaleTitle = XmStringCreateSimple("tree depth");
    n=0;     /* scale - an XmScale widget that allows changing the tree depth */
    XtSetArg(wargs[n], XmNmaximum,        snmpParams.max_depth);          n++;
    XtSetArg(wargs[n], XmNminimum,        snmpParams.min_depth);          n++;
    XtSetArg(wargs[n], XmNvalue,          snmpParams.default_depth);      n++;
    XtSetArg(wargs[n], XmNorientation,    XmHORIZONTAL);                  n++;
    XtSetArg(wargs[n], XmNshowValue,      True);                          n++;
    XtSetArg(wargs[n], XmNtitleString,    scaleTitle);                    n++;
    scale = XtCreateManagedWidget("scale", xmScaleWidgetClass,
				     controlBox, wargs, n);     
    XtAddCallback(scale, XmNvalueChangedCallback, redrawTree, NULL);

    n=0;     /* jumpButton - press it to jump to a specific variable */
    jumpButton = XtCreateManagedWidget("jumpButton", xmPushButtonWidgetClass,
				       controlBox, wargs, n);
    XtAddCallback(jumpButton, XmNactivateCallback, jumpRequest, NULL);

    n=0;     /* jumpDialog - asks where to jump */
    XtSetArg(wargs[n], XmNselectionLabelString,
	     XmStringCreateSimple("Jump to node: "));  n++;
    XtSetArg(wargs[n], XmNdialogStyle,   XmDIALOG_FULL_APPLICATION_MODAL);      n++;
    jumpDialog = XmCreatePromptDialog(controlBox, "jumpDialog", wargs, n);
    XtUnmanageChild(XmSelectionBoxGetChild(jumpDialog, XmDIALOG_HELP_BUTTON));
    XtAddCallback(jumpDialog, XmNokCallback, jumpRequestOk, NULL);
   
    n=0;     /* center - centers the current tree root in the tree window */
    center = XtCreateManagedWidget("centerButton", xmPushButtonWidgetClass,
				   controlBox, wargs, n);
    XtAddCallback(center, XmNactivateCallback, centerNode, mibRoot);

    n=0;     /* mainHelp - opens a help window */
    mainHelp = XtCreateManagedWidget("mainHelp", xmPushButtonWidgetClass,
				     controlBox, wargs, n);
    XtAddCallback(mainHelp, XmNactivateCallback, openHelp, NULL);

    n=0;     /* mainQuit - quits xmib (by calling quit()) */
    mainQuit = XtCreateManagedWidget("mainQuit", xmPushButtonWidgetClass,
				     controlBox, wargs, n);
    XtAddCallback(mainQuit, XmNactivateCallback, quit, NULL);
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* realizes toplevel and starts the event main-loop */

void startMainLoop()
{
    XtRealizeWidget(toplevel);
    XtAppMainLoop(app);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* changes the cursor shape related to <widget>, to <shape>. If shape == None,
   changes the cursor to the default cursor                                 */

void setCursor(widget, shape)
  Widget        widget;
  unsigned int  shape;
{  
    XSetWindowAttributes attrs;

    if (shape != None)
	attrs.cursor = XCreateFontCursor(XtDisplay(widget), shape);
    else
	attrs.cursor = None;
    XChangeWindowAttributes(XtDisplay(widget), XtWindow(widget),
			    CWCursor, &attrs);
    XFlush(XtDisplay(widget));
}    







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (translation proc) Destroys the current tree, and creates a new tree with
   (*args) as new root. <args> is a pointer to a string, which is the new
   root variable name. The widget, event, and num_args params are ignored.    */

void newTree(widget, event, args, num_args)
   Widget widget;
   XButtonEvent *event;
   String *args;
   int *num_args;
{
    while (activeTimer != 0)    /* wait for last timer to stop */
	XtAppProcessEvent(app, XtIMAll);
    setCursor(toplevel, XC_watch);
    if (lastTouched != NULL)
	invertWidgetColors(lastTouched);
    lastTouched = NULL;
    XtUnmanageChild(treeW);
    MibDestroyXTree(mibRoot);
    MibShowXTree(MibFindNode(*args));
    XFlush(XtDisplay(toplevel));
    setCursor(toplevel, None);
}






/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) redraws the tree, keeping the current root as the new tree's root */

void redrawTree(w, clientData, cbs)
  Widget    w;
  XtPointer clientData, cbs;
{
    char *rootName = mibRoot->name;
    
    newTree(NULL, NULL, &rootName, 1);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (translation proc) updates info for the variable whose related node widget 
   is <widget>. The information is displayed in the info window.
   (*argv) can point is either "toggle" or "noToggle", depending on the
   action which called updateVar (see fillTree() for more info).
   If "toggle" is used, the <widget> label toggles between expanded and
   shrunk mode (shrunk=display var name only, expanded=display all var info).
   The information is obtained by calling snmpCall() with a PKT_GETRQS command. */

void updateVar(widget, event, args, num_args)
  Widget widget;
  XButtonEvent *event;
  String *args;
  int *num_args;
{
    MIBNODE  *node;
    XmString labelString;
    static   char type[20], mode[20], status[20];
    char     channel[256];    
    char     *value;
    Boolean  longForm;
    
    setCursor(toplevel, XC_watch);
    if (varValue != NULL)
	safefree(varValue);
    if (lastTouched != NULL)
	invertWidgetColors(lastTouched);
    invertWidgetColors(lastTouched = widget);
    XtVaGetValues(widget, XmNuserData, &node, NULL);
    node2text(node, type, mode, status);
    strcpy(channel, node->name);
    if ( snmpCall(PKT_GETRQS, channel) )
	warn(varBox, channel);
    if ( (value = strstr(channel, "]=")) == NULL )
	value = "ERROR";
    else
	value += 2;   /* skip the ]= */
    if (! strcmp(value, "NULL0"))
    {
	value = "-";
	varValue = strdup("");
    }
    else
	varValue = strdup(value);
       /* update the var box */
    xs_wprintf(varSpec, "\nName: %s\n\nValue:  %s\n\nType:   %s\nAccess: %s\n\
Status: %s\n", node->name, value, type, mode, status);
    if ( (node->mode == 2) || (node->mode == 3) )
	XtVaSetValues(varSetButton, XmNsensitive, True, NULL);
    else
	XtVaSetValues(varSetButton, XmNsensitive, False, NULL);
       /* now update the tree node */
    XtVaGetValues(widget, XmNlabelString, &labelString, NULL);
    longForm = ((XmStringLineCount(labelString) >= 1) ? 1 : 0);
    if (! strcmp(*args, "toggle"))   /* toggle long <-> short form */
	longForm = !longForm;
    if (longForm)
    {
	XtVaSetValues(widget, XmNalignment, XmALIGNMENT_BEGINNING, NULL);
	xs_wprintf(widget, "\nName: %s\n\nValue:  %s\n\nType:   %s\nAccess: %s\n\
Status: %s\n", node->name, value, type, mode, status);
    }
    else
    {
	XtVaSetValues(widget, XmNalignment, XmALIGNMENT_CENTER, NULL);
	xs_wprintf(widget, "%s", node->name);
    }
    safefree(labelString);
    setCursor(toplevel, None);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* translates the type, mode and status codes stored in the MIBNODE structure
   <node>, into the strings <type>, <mode> and <status>, which must be big
   enough to hold the results. (the translation tables appear in xmib.h).    */

void node2text(node, type, mode, status)
  MIBNODE *node;
  char *type, *mode, *status;
{
    int i;
    
    if (type != NULL)
    {
	for (i = 0;  mibType[i].value != 0;  i++)
	    if (mibType[i].value == node->type)
		break;
	if (mibType[i].value == 0)
	    strcpy(type, "-");
	else
	    strcpy(type, mibType[i].name);
    }
    
    if (mode != NULL)
    {
	for (i = 0;  mibMode[i].value != 0;  i++)
	    if (mibMode[i].value == node->mode)
		break;
	if (mibMode[i].value == 0)
	    strcpy(mode, "-");
	else
	    strcpy(mode, mibMode[i].name);
    }
    
    if (status != NULL)
    {
	for (i = 0;  mibStatus[i].value != 0;  i++)
	    if (mibStatus[i].value == node->status)
		break;
	if (mibStatus[i].value == 0)
	    strcpy(status, "-");
	else
	    strcpy(status, mibStatus[i].name);
    }
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) displayes a dialog asking for a new value for the variable.
   The initial text in the text-field is the variable's current value.         */

void setVarValue(w, clientData, cbs)
  Widget w;
  XtPointer clientData, cbs;
{
    XmString oldValueAsX;
    
    oldValueAsX = XmStringCreateSimple(varValue);
    XtVaSetValues(setDialog, XmNtextString, oldValueAsX,
		  XmNdefaultButton, 
		  XmSelectionBoxGetChild(setDialog, XmDIALOG_OK_BUTTON), NULL);
    XtManageChild(setDialog);
    XFlush(XtDisplay(setDialog));
    XtAppAddTimeOut(app, snmpParams.travInterval, gotoTextField, setDialog);
    gotoTextField(setDialog);
    safefree(oldValueAsX);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) the setDialog "ok" button callback. calls snmpCall() with
   PKT_SETRQS as the command (first argument), and an appropriate string
   as the 2nd param. (the channel string, in the format "varName=value").      */

void okSetVarValue(w, clientData, cbs)
  Widget                        w;
  XtPointer                     clientData;
  XmSelectionBoxCallbackStruct  *cbs;
{
    MIBNODE   *node;
    XmString  varNameX;
    char      *varName;
    char      *newVal;
    char      channel[256];
    char      *updateCommand = "box";
    
    setCursor(toplevel, XC_watch);
    XtVaGetValues(varSpec, XmNlabelString, &varNameX, NULL);
    XmStringGetLtoR(varNameX, XmSTRING_DEFAULT_CHARSET, &varName);
    XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &newVal);
    if ((node = MibFindNode(varName+6)) == NULL)
	exit(1);
    strcpy(channel, varName+6);
    strcat(channel, "=");
    strcat(channel, newVal);
#ifdef DEBUG
    printf("Set request: %s\n", channel);
#endif
    if ( snmpCall(PKT_SETRQS, channel) )
	warn(varBox, channel);
    safefree(varNameX);
    safefree(varName);
    safefree(newVal);
    updateVar(node->widget, NULL, &updateCommand, 1);
    setCursor(toplevel, None);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* creates widgets (and registers callbacks) for the Host, Community, Timeout,...
   buttons at the top-right of the xmib window. The buttons also show the
   current value for each parameter.                                       */

Widget dispSnmpParams(varBox)
  Widget varBox;
{
    Arg    wargs[10];
    int    n;
    Widget snmpParamW;
    
    n=0;
    XtSetArg(wargs[n], XmNtopAttachment,    XmATTACH_FORM);         n++;
    XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM);         n++;
    XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM);         n++;
    XtSetArg(wargs[n], XmNalignment,        XmALIGNMENT_BEGINNING); n++;
    snmpParamW = XtCreateManagedWidget("paramW",
				       xmRowColumnWidgetClass,
				       varBox, wargs, n);
    n=0;
    hostW = XtCreateManagedWidget("hostW", xmPushButtonWidgetClass, 
				  snmpParamW, wargs, n);
    XtAddCallback(hostW, XmNactivateCallback, setParamValue, "host");

    n=0;
    communityW = XtCreateManagedWidget("communityW", xmPushButtonWidgetClass,
				       snmpParamW, wargs, n);
    XtAddCallback(communityW, XmNactivateCallback, setParamValue, "community");

    n=0;
    timeoutW = XtCreateManagedWidget("timeoutW", xmPushButtonWidgetClass,	     
				     snmpParamW, wargs, n);
    XtAddCallback(timeoutW, XmNactivateCallback, setParamValue, "timeout");

    n=0;
    retriesW = XtCreateManagedWidget("retriesW", xmPushButtonWidgetClass,
				     snmpParamW, wargs, n);
    XtAddCallback(retriesW, XmNactivateCallback, setParamValue, "retries");

    n=0;
    portW = XtCreateManagedWidget("portW", xmPushButtonWidgetClass,
				  snmpParamW, wargs, n);
    XtAddCallback(portW, XmNactivateCallback, setParamValue, "port");

    n=0;
    XtSetArg(wargs[n], XmNselectionLabelString,
	     XmStringCreateSimple("New value: "));  n++;
    XtSetArg(wargs[n], XmNdialogStyle,   XmDIALOG_FULL_APPLICATION_MODAL);      n++;
    paramDialog = XmCreatePromptDialog(varBox, "paramDialog", wargs, n);
    XtUnmanageChild(XmSelectionBoxGetChild(paramDialog, XmDIALOG_HELP_BUTTON));
    XtAddCallback(paramDialog, XmNokCallback, setParamValueOk, NULL);
    updateParams();

    return snmpParamW;
}    







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* updates the labels of the Host, Community, Timeout,... buttons according
   to these parameters actual value */

void updateParams()
{
    SNMP_PARAMS *s = &snmpParams;

    xs_wprintf(hostW, "Host:      %s", s->host);
    xs_wprintf(communityW, "Community: %s", s->community);
    xs_wprintf(timeoutW, "Timeout:   %d sec.", s->timeout);
    xs_wprintf(retriesW, "Retries:   %d", s->retries);
    xs_wprintf(portW, "Port:      %d", s->port);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
void checkCommandLine(argc, argv)
  int argc;
  char *argv[];
{
    int c;
    extern char *optarg;
    extern int optind;
    int errflg = 0;

    while ((c = getopt(argc, argv, "h:c:t:r:p:")) != -1)
	switch (c) {
	  case 'h':
	    snmpParams.host =      strdup(optarg);
	    break;
	  case 'c':
	    snmpParams.community = strdup(optarg);
	    break;
	  case 't':
	    snmpParams.timeout =       atoi(optarg);
	    break;
	  case 'r':
	    snmpParams.retries =       atoi(optarg);
	    break;
	  case 'p':
	    snmpParams.port =          atoi(optarg);
	    break;
	  case 'o':
	    snmpParams.root =          strdup(optarg);
	    break;
	  case 'd':
	    snmpParams.default_depth = atoi(optarg);
	    break;
	  case '?':
	    errflg++;
	}
    if (errflg || (optind < argc)) {
	fprintf(stderr, "usage:   xmib  [-h host] [-c community] [-t timeout] [-r retries] [-p port] [-o root] [-d depth]\n");
	exit (2);
    }
}








/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* creates and pops-up a warning dialog (with a single button - "Ok").
   The message is specified by <message>, and the dialog is created as a child
   of the widget <parent>.                                                     */

void warn(parent, message)
  Widget parent;
  char *message;
{
    Arg wargs[10];
    int n;
    Widget   warnDialog;
    XmString messageAsX;
    
    messageAsX = XmStringCreateSimple(message);
    n=0;
    XtSetArg(wargs[n], XmNdialogStyle,   XmDIALOG_FULL_APPLICATION_MODAL);   n++;
    XtSetArg(wargs[n], XmNmessageString, messageAsX);                        n++;
    warnDialog = XmCreateErrorDialog(parent, "warnDialog", wargs, n);
    XtUnmanageChild(XmMessageBoxGetChild(warnDialog, XmDIALOG_HELP_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(warnDialog, XmDIALOG_CANCEL_BUTTON));
    XtAddCallback(warnDialog, XmNokCallback, warnOk, NULL);
    XtManageChild(warnDialog);
    safefree(messageAsX);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) the "ok" button callback of the warnDialog widget. Should only
   destroy warnDialog (passed as <w>), as each call to warn() creates a new
   dialog.                                                                     */

void warnOk(w, clientData, cbs)
  Widget w;
  XtPointer clientData, cbs;
{
    XtDestroyWidget(w);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) Centers the widget related to <node> tree node in the tree window.
   the <w> and <cbs> params. are ignored. Flashes the centered widget.         */

void centerNode(w, node, cbs)
  Widget w;
  MIBNODE   *node;
  XtPointer cbs;
{
    Widget     verScroll, horScroll, workWindow;
    Position   nodeX, nodeY, nodeHeight, nodeWidth;
    int        minScrollX, maxScrollX, sliderSizeX;
    int        minScrollY, maxScrollY, sliderSizeY;
    Position   maxDrawX, maxDrawY;
    
    int        newValX, newValY;
    XmScrollBarCallbackStruct scrollCbs;

    if (node == NULL)
	node = mibRoot;
    XtVaGetValues(node->widget, XmNx,      &nodeX, 
		                XmNy,      &nodeY,
		                XmNheight, &nodeHeight,
		                XmNwidth,  &nodeWidth,
		                NULL);
    nodeX += nodeWidth/2;
    nodeY += nodeHeight/2;
    XtVaGetValues(sw, XmNverticalScrollBar,   &verScroll, 
		      XmNhorizontalScrollBar, &horScroll,
		      XmNworkWindow,          &workWindow,
		      XmNwidth,               &maxDrawX,
		      XmNheight,              &maxDrawY,
		      NULL);
    XtVaGetValues(horScroll, XmNmaximum,               &maxScrollX,
		             XmNminimum,               &minScrollX,
		             XmNsliderSize,            &sliderSizeX,
		             NULL);   
    XtVaGetValues(verScroll, XmNmaximum,               &maxScrollY,
		             XmNminimum,               &minScrollY,
		             XmNsliderSize,            &sliderSizeY,
		             NULL);   

         /* set the horizontal scrollbar */
    newValX = nodeX - maxDrawX/2;
    newValX = max(0, newValX);
    newValX = min(maxScrollX-sliderSizeX, newValX);
    XtVaSetValues(horScroll, XmNvalue, newValX, NULL);
    scrollCbs.reason = XmCR_VALUE_CHANGED;
    scrollCbs.value  = newValX;
    scrollCbs.event  = NULL;
    XtCallCallbacks(horScroll, XmNvalueChangedCallback, &scrollCbs);

        /* set the vertical scrollbar */
    newValY = nodeY - maxDrawY/2;
    newValY = max(0, newValY);
    newValY = min(maxScrollY-sliderSizeY, newValY);
    XtVaSetValues(verScroll, XmNvalue, newValY, NULL);
    scrollCbs.reason = XmCR_VALUE_CHANGED;
    scrollCbs.value  = newValY;
    scrollCbs.event  = NULL;
    XtCallCallbacks(verScroll, XmNvalueChangedCallback, &scrollCbs);
    flashWidget(node->widget, 5, 50);
    if (lastTouched != NULL)
	invertWidgetColors(lastTouched);
    invertWidgetColors(lastTouched = node->widget);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (translation proc) centers the node related to <widget>. other params. are
   ignored.                                                                    */

void centerNodeTran(widget, event, args, num_args)
  Widget widget;
  XButtonEvent *event;
  String *args;
  int *num_args;
{
    MIBNODE *node;
         /* the widget ptr is stored as userData in each widget */
    XtVaGetValues(widget, XmNuserData, &node, NULL);
    centerNode(NULL, node, NULL);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* flashes the widget <w> <times> times, with <interval> msec. between each
   color inverse. "flash" = inverting colors twice, ie. the background color
   becomes the foreground color and vice-versa, twice. (see invertWidgetColors()) */

void flashWidget(w, times, interval)
  Widget w;
  int    times, interval;
{
    XtIntervalId id;
    int isFirstTimer;   /* the 1st timer set is the last to finish counting */
    
    for (isFirstTimer = 1, times *= 2 ;  times > 0;  times--, isFirstTimer = 0)
    {
	id = XtAppAddTimeOut(app,
			     interval * times,
			     timedInvertWidgetColors,
			     w);
	if (isFirstTimer)    /* the first set timer is the one to exit last */
	    activeTimer = id;
    }
}	







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (timer) calls invertWidgetColors(), and if the calling timer is the last
   timer activated, then reset activeTimer (no active timers left).            */

void timedInvertWidgetColors(w, id)
  Widget         w;
  XtIntervalId   *id;
{
    invertWidgetColors(w);
    if (*id == activeTimer)
	activeTimer = 0;
}




    


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* inverts the <w> widget's colors, ie. background color becomes foreground color,
   and vice-versa.                                                             */

void invertWidgetColors(w)
  Widget         w;
{
    Pixel foreground, background;
    
    if (w == NULL)
	XtError("NULL widget passed to invertWidgetColors");
    if (! XtIsManaged(w))
	return;
    XtVaGetValues(w, XmNforeground, &foreground,
		     XmNbackground, &background,
		     NULL);
    XtVaSetValues(w, XmNforeground, background,
		     XmNbackground, foreground,
		     NULL);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) pops-up the jump dialog, that asks for a MIB variable to jump to */

void jumpRequest(w, clientData, cbs)
  Widget w;
  XtPointer clientData, cbs;
{
    XtVaSetValues(jumpDialog, XmNdefaultButton, 
		  XmSelectionBoxGetChild(jumpDialog, XmDIALOG_OK_BUTTON), NULL);
    XtManageChild(jumpDialog);
    XFlush(XtDisplay(jumpDialog));
    XtAppAddTimeOut(app, snmpParams.travInterval, gotoTextField, jumpDialog);
    gotoTextField(jumpDialog);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) the "ok" button callback of the jumpDialog widget. If the entered
   MIB variable is valid, calls newTree() to set a new tree with that variable
   as the root. Otherwise, display an error message.                           */

void jumpRequestOk(w, clientData, cbs)
  Widget w;
  XtPointer clientData;
  XmSelectionBoxCallbackStruct *cbs;
{
    char    *nodeName;
    MIBNODE *node;

    XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &nodeName);
    if ((node = MibFindNode(nodeName)) == NULL)
	warn(controlBox, "Jump failed: MIB variable not defined");
    else
	newTree(NULL, NULL, &nodeName, 1);
    safefree(nodeName);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) pops-up a dialog that asks for a new value for an SNMP parameter
   (host, community, timeout, retries, port), according to the button pressed.
   The clientData received is a string that is "host", "community", etc.,
   according to the button pressed. setParamValue() stores this clientData
   in the global variable 'which', for easy access by setParamValueOk().       */
   
void setParamValue(w, clientData, cbs)
  Widget w;
  char *clientData;
  XmSelectionBoxCallbackStruct *cbs;
{
    char buf[100];
    XmString titleX;
    XmString emptyXmString;

    emptyXmString = XmStringCreateSimple("");
    which = clientData;
    sprintf(buf, "update %s", which);
    titleX = XmStringCreateSimple(buf);
    XtVaSetValues(paramDialog, XmNdialogTitle, titleX, NULL);
    XtVaSetValues(paramDialog, XmNdefaultButton, 
		  XmSelectionBoxGetChild(paramDialog, XmDIALOG_OK_BUTTON), 
		  XmNtextString, emptyXmString,
		  NULL);
    safefree(titleX);
    safefree(emptyXmString);
    XtManageChild(paramDialog);
    XFlush(XtDisplay(paramDialog));
    XtAppAddTimeOut(app, snmpParams.travInterval, gotoTextField, paramDialog);
    gotoTextField(paramDialog);
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) the "ok" button callback of the paramDialog widget. All 3 params.
   of this callback are ignored. setParamValueOk() stores the value found in
   the text-field of the paramDialog widget in the appropriate param-variable
   according to the string stored in the global variable 'which'.
   ( Yes, I know it's ugly, but all in all, so is all this program :-( )       */

void setParamValueOk(w, clientData, cbs)
  Widget w;
  XtPointer clientData;
  XmSelectionBoxCallbackStruct *cbs;
{
    char     *newVal;

    XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &newVal);
    if (!strcmp(which, "host"))
	strcpy(snmpParams.host, newVal);
    if (!strcmp(which, "community"))
	strcpy(snmpParams.community, newVal);
    if (!strcmp(which, "timeout"))
	snmpParams.timeout = atoi(newVal);
    if (!strcmp(which, "retries"))
	snmpParams.retries = atoi(newVal);
    if (!strcmp(which, "port"))
	snmpParams.port = atoi(newVal);
    safefree(newVal);
    updateParams();
       /* changing the Host or Port params. requires reopening snmp */
    if ( (!strcmp(which, "host")) || (!strcmp(which, "port")) )
    {
	setCursor(toplevel, XC_watch);   /* no, I *can't* call newTree() instead, */
	if (lastTouched != NULL)         /* the proof is left as an ex. to you    */
	    invertWidgetColors(lastTouched);
	lastTouched = NULL;
	XtUnmanageChild(treeW);
	MibDestroyXTree(mibRoot);
	strcpy(snmpParams.root, mibRoot->name);  /* reopen host with same root */
	snmpClose();    /* reopen port */
	mibRoot = snmpOpen();
	MibShowXTree(mibRoot);
	XFlush(XtDisplay(toplevel));
	setCursor(toplevel, None);
    }
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* traverses to the text-field of the dialog widget <w>.
   gotoTextField() assumes that <w> *is* a dialog widget, so don't cheat it... */

void gotoTextField(w)
  Widget w;
{
    Widget textChild;

    textChild = XmSelectionBoxGetChild(w, XmDIALOG_TEXT);
    if (! XmProcessTraversal(textChild, XmTRAVERSE_CURRENT))
	fprintf(stderr, "Traversal failed!\n");
}







/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* opens a help window that shows the contents of the onlineHelpText global
   string variable. The help window consists of a scrolledText widget that
   contains the text, and a 'quit' button that closes the help window.
   In the first time openHelp() is called, it creates all needed widgets.
   Further calls merely manage and pop-up the associated help shell.           */

void openHelp()
{
    Arg       wargs[10];
    Cardinal  n;
    Widget    helpSh = NULL;
    Widget    helpBox, helpScroll, helpQuit;
    Position  xPos, yPos;
    Dimension width, height;
    
    if (helpSh != NULL)    /* openHelp() has already been called */
    {
	XtManageChild(helpScroll);
	XtPopup(helpSh, XtGrabNone);               /* popup shell */
	return;        /* no need to create widgets */
    }    
    
    /* bad luck... have to create all widgets */
    XtVaGetValues(toplevel,         /* sets pop-up shell relative to toplevel */
		  XtNwidth, &width,
		  XtNheight, &height,
		  NULL);
    
    XtTranslateCoords(toplevel,
		      (Position) width/2,
		      (Position) height/2,
		      &xPos, 
		      &yPos);

    n=0;    
    XtSetArg(wargs[n], XtNx,         xPos);                           n++;
    XtSetArg(wargs[n], XtNy,         yPos);                           n++;
    helpSh = XtCreatePopupShell("helpSh",
                                transientShellWidgetClass, 
				toplevel,
				wargs,
				n);

              /* Create a container box for the help widgets defined below */
    n=0;
    helpBox = XtCreateManagedWidget("helpBox",
				    xmFormWidgetClass,
				    helpSh,
				    wargs,
				    n);
    
                                 /* the helpQuit button (closes it) */
    n = 0;
    XtSetArg(wargs[n], XmNleftAttachment,   (XtArgVal) XmATTACH_FORM);    n++;
    XtSetArg(wargs[n], XmNrightAttachment,  (XtArgVal) XmATTACH_FORM);    n++;   
    XtSetArg(wargs[n], XmNbottomAttachment, (XtArgVal) XmATTACH_FORM);    n++;   
    helpQuit = XtCreateManagedWidget("helpQuit",
				     xmPushButtonWidgetClass,
				     helpBox,
				     wargs,
				     n);
        /* register the button callback */
    XtAddCallback(helpQuit, XmNactivateCallback, closeHelp, helpSh);

    n=0;
    XtSetArg(wargs[n], XmNrows,             12);                            n++;
    XtSetArg(wargs[n], XmNcolumns,          70);                            n++;
    XtSetArg(wargs[n], XmNeditable,         False);                         n++;  
    XtSetArg(wargs[n], XmNeditMode,         (XtArgVal) XmMULTI_LINE_EDIT);  n++;   
    XtSetArg(wargs[n], XmNtopAttachment,    (XtArgVal) XmATTACH_FORM);      n++;   
    XtSetArg(wargs[n], XmNleftAttachment,   (XtArgVal) XmATTACH_FORM);      n++;
    XtSetArg(wargs[n], XmNrightAttachment,  (XtArgVal) XmATTACH_FORM);      n++;   
    XtSetArg(wargs[n], XmNbottomAttachment, (XtArgVal) XmATTACH_WIDGET);    n++;   
    XtSetArg(wargs[n], XmNbottomWidget,     (XtArgVal) helpQuit);           n++;
    helpScroll = XmCreateScrolledText(helpBox,
				      "helpText",
				      wargs,
				      n);
    XtVaSetValues(helpScroll,
		  XmNscrollingPolicy,        XmAUTOMATIC,
		  XmNscrollBarDisplayPolicy, XmAS_NEEDED,
		  NULL);
    XmTextSetString(helpScroll, onlineHelpText);

    XtManageChild(helpScroll);
    XtPopup(helpSh, XtGrabNonexclusive);                     /* popup shell */
}    






/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback)  the "quit" button callback of the help window. Just pops-
   down the help-shell, so further calls to openHelp only pops it up.   */

void closeHelp(w, client_data, call_data)
  Widget     w;
  XtPointer  client_data;
  XtPointer  call_data;
{
    XtPopdown(XtParent(XtParent(w)));   /* the button is a grandchild of helpSh */
}
