/* 1435, Tue 11 May 93

   TREEMON.C:  AU Monitor, using a height-balanced tree

   Copyright (C) 1992,1993 by Nevil Brownlee,
   Computer Centre,  University of Auckland */

#include <stdio.h>
#include <alloc.h>

#define NULL  0

#include "trsnap.h"
#include "trtree.h"

struct node *new_tree()  /* Initialise a new tree */
{
   struct node *q;
   if ((q = malloc(sizeof(struct node_header)+sizeof(struct root_info)))
	 == NULL) {
      printf("No memory for new tree!\n");
      return (struct node *)NULL;
      }
   q->h.llink = q->h.rlink = NULL;
   q->h.balance = 0;  q->h.type = N_ROOT;
   q->i.r.height = q->i.r.nbr_nodes =
      q->i.r.compares = q->i.r.searches = 0;
   return (struct node *)q;
   }


struct node *new_node(t)  /* Space for new node, type t */
int t;
{
   struct node_header *q;
   int size = sizeof(struct node_header);
   if (t == N_FLOW) size += sizeof(struct flow);
   else if (t == N_BACK_FLOW) size += sizeof(struct back_flow);
   if ((q = malloc(size)) == NULL) {
      printf("No memory for new node!\n");
      return (struct node *)NULL;
      }
   q->llink = q->rlink = NULL;
   q->balance = 0;  q->type = t;
   return (struct node *)q;
   }

/* Height-balanced tree algorithm from Knuth (1973) Vol 3 */

#define link(a,p)  (a == -1 ? p->h.llink : p->h.rlink)
#define setlink(a,p, v)  if (a == -1) p->h.llink = v; else p->h.rlink = v

struct node *hbtree(n,compfn, type,insfn, tree)  /* Find node n in tree */
struct node *n;
int (*compfn)();  /* Compare two nodes */
int type;  /* Type of node to add: N_VOID -> don't add new node */
void (*insfn)();  /* Initialise node */
struct node *tree;
{
   struct node *t,  /* Father of s */
      *s,  /* Place where rebalancing may be neccessary */
      *p,
      *q,*r;
   char a;
   t = tree;  /* A1: Initialise */
   s = p = t->h.rlink;
   tree->i.r.searches += 1;
   if (p == NULL) {  /* Empty tree */
      if (type == N_VOID) return NULL;
      if ((q = new_node(type)) == NULL) return q;
      tree->h.rlink = q;
      insfn(q,n);
      tree->i.r.nbr_nodes = tree->i.r.height = 1;
      return q;
      }
   for (;;) {  /* A2: Compare */
      tree->i.r.compares += 1;
      if ((a = (*compfn)(n,p)) == 0) return p;  /* Success */
      if (a < 0) {
	 if ((q = p->h.llink) == NULL) {  /* A3: Move left */
	    if (type == N_VOID) return NULL;
	    if ((q = new_node(type)) == NULL) return q;
	    p->h.llink = q;
	    break;
	    }
	 }
      else {  /* A4: Move right */
	 if ((q = p->h.rlink) == NULL) {
	    if (type== N_VOID) return NULL;
	    if ((q = new_node(type)) == NULL) return q;
	    p->h.rlink = q;
	    break;
	    }
	 }
      if (q->h.balance != 0) {
	 t = p;  s = q;
	 }
      p = q;
      }
   insfn(q,n);  /* A5: Insert */
   tree->i.r.nbr_nodes += 1;
   tree->i.r.compares += 1;
   if ((*compfn)(n,s) < 0)  /* A6: Adjust balance factors */
      r = p = s->h.llink;
   else
      r = p = s->h.rlink;
   while (p != q) {
      tree->i.r.compares += 1;
      if ((*compfn)(n,p) < 0) {
	 p->h.balance = -1;  p = p->h.llink;
	 }
      else {
	 p->h.balance = 1;  p = p->h.rlink;
	 }
      }
   tree->i.r.compares += 1;
   a = (*compfn)(n,s) < 0 ? -1 : 1;  /* A7: Balancing act */
   if (s->h.balance == 0) {  /* Tree has grown higher */
      s->h.balance = a;
      tree->i.r.height +=1;
      return q;
      }
   if (s->h.balance == -a) {  /* Tree balance has improved */
      s->h.balance = 0;
      return q;
      }
   /* Tree has become unbalanced */
   if (r->h.balance == a) {  /* A8: Single rotation */
      p = r;
      setlink(a,s, link(-a,r));
      setlink(-a,r, s);
      s->h.balance = r->h.balance = 0;
      }
   else {  /* A9: Double rotation */
      p = link(-a,r);
      setlink(-a,r, link(a,p));
      setlink(a,p, r);
      setlink(a,s, link(-a,p));
      setlink(-a,p, s);
      if (p->h.balance == a) {
	 s->h.balance = -a;  r->h.balance = 0;
	 }
      else if (p->h.balance == 0) {
	 s->h.balance = r->h.balance = 0;
	 }
      else {
	 s->h.balance = 0;  r->h.balance = a;
	 }
      p->h.balance = 0;
      }
   if (s == t->h.rlink) t->h.rlink = p;  /* A10: Finishing touch */
   else t->h.llink = p;
   return q;
   }

void tree_walk(t, f)
struct node *t;
void (*f)();
{
   if (t->h.llink) tree_walk(t->h.llink, f);
   f(t);
   if (t->h.rlink) tree_walk(t->h.rlink, f);
   }

struct node *ftree; /* Root of tree */

void tree_stats(unsigned long *searches, unsigned long *compares)
{
   *searches = ftree->i.r.searches;
   *compares = ftree->i.r.compares;
   }

int pt_compare(n,p)
struct node *n,*p;
{
   if (n->i.f.kind < p->i.f.kind) return -1;
   if (n->i.f.kind > p->i.f.kind) return 1;
   if (n->i.f.type < p->i.f.type) return -1;
   if (n->i.f.type > p->i.f.type) return 1;
   return 0;
   }

void insert_key(q,n)  /* Insert key data into q from n */
struct node *q, *n;
{
   q->i.f.flowindex = n->i.f.flowindex;
   q->i.f.kind = n->i.f.kind;
   q->i.f.type = n->i.f.type;
   q->i.f.packets = q->i.f.bytes = q->i.f.s_packets = q->i.f.s_bytes = 0;
   }

#define MAXPKTLEN  1526

#define P_BLUE_BOOK  0  /* Flow kind values */
#define P_SNAP       1
#define P_802_2      2

struct s_flow_info {
   char id[12];
   unsigned char kind;
   unsigned int type;
   };

struct s_flow_info flow_info[] = {
   {"Novell IPX",  P_802_2,     0xFFFF},
   {"IP",          P_BLUE_BOOK, 0x0800},
   {"VAX LAVC",    P_BLUE_BOOK, 0x6007},
   {"Ethertalk 2", P_SNAP,      0x809B},

   {"DEC LAT",     P_BLUE_BOOK, 0x6004},
   {"DECnet",      P_BLUE_BOOK, 0x6003},

   {"Time 0101",   P_802_2,     0x0101},
   {"AARP 2",      P_SNAP,      0x80F3},
   {"ARP",         P_BLUE_BOOK, 0x0806},
   {"DEC LANBrdg", P_BLUE_BOOK, 0x8038},
   {"DEC MOP Rmt", P_BLUE_BOOK, 0x6002},

   {"Loopback",    P_BLUE_BOOK, 0x9000},
   {"DEC MSDOS",   P_BLUE_BOOK, 0x8041},
   {"DEC MOP Ld",  P_BLUE_BOOK, 0x6001},
   {"DEC NetBIOS", P_BLUE_BOOK, 0x8040},
   {"RARP",        P_BLUE_BOOK, 0x8035},
   {"Banyan",      P_BLUE_BOOK, 0x0BAD},
   {"DEC Time",    P_BLUE_BOOK, 0x803E},
   {"Ethertalk 1", P_BLUE_BOOK, 0x809B},
   {"AARP 1",      P_BLUE_BOOK, 0x80F3},
   {"TRW Ld req",  P_BLUE_BOOK, 0x0D00},
   {"TRW Ld data", P_BLUE_BOOK, 0xAD23},
   {"Other 802.2", P_802_2,     0     },

   {0,0,0}  /* End marker */
   };

void setup_flows()
{
   struct node data;
   struct s_flow_info *fip;
   int flow_nbr = 0;

   for (fip = flow_info;  fip->id[0];  ++fip) {
      data.i.f.kind = fip->kind;
      data.i.f.type = fip->type;
      data.i.f.flowindex = ++flow_nbr;
      hbtree(&data,pt_compare, N_FLOW,insert_key, ftree);
      }
   }

void traffic_monitor(unsigned char far *pp, int len)
{
   unsigned char pkt_kind;
   unsigned int pkt_type;
   struct node data, *n;

   pkt_type = pp[12-SNAPFROM]<<8 | pp[13-SNAPFROM];
   if (pkt_type > MAXPKTLEN) pkt_kind = P_BLUE_BOOK;
   else {  /* 802.2 */
      if (pp[14-SNAPFROM] == 0xAA && pp[15-SNAPFROM] == 0xAA) {  /* SNAP packet */
	 pkt_kind = P_SNAP;
	 pkt_type = pp[20-SNAPFROM]<<8 | pp[21-SNAPFROM];
	 }
      else {
	 pkt_kind = P_802_2;
	 if (pp[14-SNAPFROM] == 0xFF && pp[15-SNAPFROM] == 0xFF)  /* Novell IPX */
	    pkt_type = 0xFFFF;
	 else if (pp[14-SNAPFROM] == 0x01 && pp[15-SNAPFROM] == 0x01)  /* Time */
	    pkt_type = 0x0101;
	 else
	    pkt_type = 0;
	 }
      }

   data.i.f.flowindex = 0;
   data.i.f.kind = pkt_kind;
   data.i.f.type = pkt_type;
   n = hbtree(&data,pt_compare, N_FLOW,insert_key, ftree);
   if (n != NULL) {
      n->i.f.packets += 1;  n->i.f.bytes += len;
      }
   }

void t_save_counts(t)
struct node *t;
{
   t->i.f.s_packets = t->i.f.packets;
   t->i.f.s_bytes = t->i.f.bytes;
   }

void save_time()
{
   s_tod_h = tod_h;  s_tod_m = tod_m;  s_tod_s = tod_s;
   elapsed_sec = 0;
   }

void save_counts()
{
   tree_walk(ftree->h.rlink, t_save_counts);
   save_time();  /* For screen display */
   }

void show_stats(t)
struct node *t;
{
   scpos(0,24);
   printf("Stats: %u nodes, %u lvls, %lu cmps/srch",
      t->i.r.nbr_nodes,t->i.r.height,t->i.r.compares/t->i.r.searches);
   }

void show_time()
{
   scpos(0,24);
   printf("%lu seconds since %02d%02d:%02d",
      elapsed_sec, s_tod_h,s_tod_m,s_tod_s);
   }

int sy;

void t_show_pcount(t)
struct node *t;
{
   long p,b;
   p = t->i.f.packets - t->i.f.s_packets;
   if (p == 0) return;  /* No counts yet */
   if (sy == 24) w_roll(0,7, 40,24, 1);
   else scpos(0,++sy);
   b = t->i.f.bytes - t->i.f.s_bytes;
   if (t->i.f.flowindex) printf("%-11s %8lu %10lu",
      flow_info[t->i.f.flowindex-1].id, p,b);
   else printf("  %1d  %04x   %8lu %10lu",
      t->i.f.kind, t->i.f.type, p,b);
   }

void show_pcount()
{
   w_clear(0,sy = 7, 40,24);  /* Clear window */
   tree_walk(ftree->h.rlink, t_show_pcount);
   }

void t_show_ptype(t)
struct node *t;
{
   long p,b;
   unsigned pps,bps, av_pkt_sz, util_100;

   p = t->i.f.packets - t->i.f.s_packets;
   if (p == 0) return;  /* No counts yet */
   b = t->i.f.bytes - t->i.f.s_bytes;
   pps = p/elapsed_sec;  bps = b/elapsed_sec;
   av_pkt_sz = b/p;
   util_100 =  /* % util * 100 */
   ((24l*pps + bps)/25*2l + 5l)/10l;
   if (util_100 < 5) return;  /* >= 0.05% */
   if (t->i.f.flowindex) printf("%-11s %6u %5u %5u.%02u\n",
      flow_info[t->i.f.flowindex-1].id, pps, av_pkt_sz,
      util_100/100,util_100%100);
   else printf("%4d  %04x  %6u %5u %5u.%02u\n",
      t->i.f.kind, t->i.f.type, pps, av_pkt_sz,
      util_100/100,util_100%100);
   }

void show_ptype()
{
   w_clear(0,7, 40,24);  /* Clear window */
   scpos(14,7);
   printf("rate  size   %% util");
   scpos(0,9);
   tree_walk(ftree->h.rlink, t_show_ptype);
   show_time();
   }

void init_monitor()
{
   ftree = new_tree();
   setup_flows();
   set_tod();
   save_time();  /* For screen display */
   }

void show_help()
{
   w_clear(0,7, 40,24);  /* Clear window */
   scpos(0,7);
   printf("Copyright (C) 1992,1993 by Nevil Brownlee\n");
   printf("Computer Centre, University of Auckland\n\n");
   printf("Keyboard commands ..\n");
   printf("  p: show packet counts\n");
   printf("  z: restart counters\n\n");
   printf("' ': show utilastion by packet type\n");
   printf("Tab: show utilisation, reset counters\n\n");;
   printf("Esc: stop monitoring, exit to DOS\n\n");
   printf("  b: show Bad packet counts\n");
   printf("  m: show Memory usage\n");
   printf("  v: show monitor Version\n");
   }

void handle_kb(ch)
int ch;
{
   switch (tolower(ch)) {
   case '\t':
      show_ptype();
      save_counts();
      break;
   case ' ':
      show_ptype();
      break;
   case 'p':
      show_pcount();
      break;
   case 's':
      show_stats(ftree);
      break;
   case 't':
      show_time();
      break;
   case 'z':
      save_counts();
      break;
   case '?':
      show_help();
      break;
   default:
      break;
      }
   }
