/* 1730, Fri 3 Sep 93

   NM.C:  First attempt at a manager for the AU Accounting Meter

   Copyright (C) 1992,1993 by Nevil Brownlee,
   Computer Centre,  University of Auckland */

/***********************************************************
	Copyright 1988 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/
#include <sys/param.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>

#include <stdio.h>
#include <ctype.h>
#include <sys/time.h>
#include <errno.h>

#include <string.h>
#include <malloc.h>

#include "snmp.h"
#include "snmpimpl.h"
#include "asn1.h"
#include "snmpclnt.h"
#include "snmpapi.h"
#include "mib.h"

#ifndef BSD4_3
#define BSD4_2
#endif

#define RULEFILE "rules.txt"

extern int  errno;
int	snmp_dump_packet = 0;

#define  INT_ACCT  1, 3, 6, 1, 3, 99
#define  AUKUNI    1, 3, 6, 1, 4, 1, 411

oid o_sysDescr[]	  = {MIB, 1, 1, 0};
oid o_sysUpTime[]	  = {MIB, 1, 3, 0};
oid o_ifNumber[]	  = {MIB, 2, 1, 0};

oid o_InactivityTimeout[] = {INT_ACCT, 1,  3, 0};
oid o_LastCollectTime[]   = {INT_ACCT, 1,  6, 0};
oid o_riRuleSize[]        = {INT_ACCT, 1,  7, 1, 2, 0xFF};
oid o_riActionSize[]      = {INT_ACCT, 1,  7, 1, 3, 0xFF};
oid o_CurrentRuleSet[]    = {INT_ACCT, 1,  8, 0};
oid o_EmergencyRuleSet[]  = {INT_ACCT, 1,  9, 0};

oid o_ftFlowIndex[]            = {INT_ACCT, 2, 1, 1,  1, 0xFFFF};
oid o_ftFlowStatus[]           = {INT_ACCT, 2, 1, 1,  2, 0xFFFF};
oid o_ftLowInterface[]         = {INT_ACCT, 2, 1, 1,  3, 0xFFFF};
oid o_ftLowAdjacentType[]      = {INT_ACCT, 2, 1, 1,  4, 0xFFFF};
oid o_ftLowAdjacentAddress[]   = {INT_ACCT, 2, 1, 1,  5, 0xFFFF};
oid o_ftLowAdjacentMask[]      = {INT_ACCT, 2, 1, 1,  6, 0xFFFF};
oid o_ftLowPeerType[]          = {INT_ACCT, 2, 1, 1,  7, 0xFFFF};
oid o_ftLowPeerAddress[]       = {INT_ACCT, 2, 1, 1,  8, 0xFFFF};
oid o_ftLowPeerMask[]          = {INT_ACCT, 2, 1, 1,  9, 0xFFFF};
oid o_ftLowDetailType[]        = {INT_ACCT, 2, 1, 1, 10, 0xFFFF};
oid o_ftLowDetailAddress[]     = {INT_ACCT, 2, 1, 1, 11, 0xFFFF};
oid o_ftLowDetailMask[]        = {INT_ACCT, 2, 1, 1, 12, 0xFFFF};
oid o_ftHighInterface[]        = {INT_ACCT, 2, 1, 1, 15, 0xFFFF};
oid o_ftHighAdjacentType[]     = {INT_ACCT, 2, 1, 1, 16, 0xFFFF};
oid o_ftHighAdjacentAddress[]  = {INT_ACCT, 2, 1, 1, 17, 0xFFFF};
oid o_ftHighAdjacentMask[]     = {INT_ACCT, 2, 1, 1, 18, 0xFFFF};
oid o_ftHighPeerType[]         = {INT_ACCT, 2, 1, 1, 19, 0xFFFF};
oid o_ftHighPeerAddress[]      = {INT_ACCT, 2, 1, 1, 20, 0xFFFF};
oid o_ftHighPeerMask[]         = {INT_ACCT, 2, 1, 1, 21, 0xFFFF};
oid o_ftHighDetailType[]       = {INT_ACCT, 2, 1, 1, 22, 0xFFFF};
oid o_ftHighDetailAddress[]    = {INT_ACCT, 2, 1, 1, 23, 0xFFFF};
oid o_ftHighDetailMask[]       = {INT_ACCT, 2, 1, 1, 24, 0xFFFF};
oid o_ftPDUScale[]             = {INT_ACCT, 2, 1, 1, 29, 0xFFFF};
oid o_ftOctetScale[]           = {INT_ACCT, 2, 1, 1, 30, 0xFFFF};
oid o_ftRuleSet[]              = {INT_ACCT, 2, 1, 1, 31, 0xFFFF};
oid o_ftFlowType[]             = {INT_ACCT, 2, 1, 1, 32, 0xFFFF};
oid o_ftUpOctets[]             = {INT_ACCT, 2, 1, 1, 33, 0xFFFF};
oid o_ftUpPDUs[]               = {INT_ACCT, 2, 1, 1, 34, 0xFFFF};
oid o_ftDownOctets[]           = {INT_ACCT, 2, 1, 1, 35, 0xFFFF};
oid o_ftDownPDUs[]             = {INT_ACCT, 2, 1, 1, 36, 0xFFFF};
oid o_ftFirstTime[]            = {INT_ACCT, 2, 1, 1, 37, 0xFFFF};
oid o_ftLastTime[]             = {INT_ACCT, 2, 1, 1, 38, 0xFFFF};

oid o_ftCreateTime[]	  = {INT_ACCT, 2, 2, 1, 1, 0xFFFFFFFF, 0xFFFF};
oid o_ftCreateIndex[]	  = {INT_ACCT, 2, 2, 1, 2, 0xFFFFFFFF, 0xFFFF};

oid o_ftActiveTime[]	  = {INT_ACCT, 2, 3, 1, 1, 0xFFFFFFFF, 0xFFFF};
oid o_ftActiveIndex[]	  = {INT_ACCT, 2, 3, 1, 2, 0xFFFFFFFF, 0xFFFF};
oid o_ftActiveFlowBlob[]  = {INT_ACCT, 2, 3, 1, 3, 0xFFFFFFFF, 0xFFFF};

oid o_rtSelector[]        = {INT_ACCT, 3, 1, 1, 3, 0xFF, 0xFFFF};
oid o_rtRuleMask[]        = {INT_ACCT, 3, 1, 1, 4, 0xFF, 0xFFFF};
oid o_rtMatchedValue[]    = {INT_ACCT, 3, 1, 1, 5, 0xFF, 0xFFFF};
oid o_rtRuleAction[]      = {INT_ACCT, 3, 1, 1, 6, 0xFF, 0xFFFF};
oid o_rtJumpIndex[]       = {INT_ACCT, 3, 1, 1, 7, 0xFF, 0xFFFF};

oid o_atLowInterface[]         = {INT_ACCT, 4, 1, 1,  3, 0xFF, 0xFFFF};
oid o_atLowAdjacentType[]      = {INT_ACCT, 4, 1, 1,  4, 0xFF, 0xFFFF};
oid o_atLowAdjacentAddress[]   = {INT_ACCT, 4, 1, 1,  5, 0xFF, 0xFFFF};
oid o_atLowAdjacentMask[]      = {INT_ACCT, 4, 1, 1,  6, 0xFF, 0xFFFF};
oid o_atLowPeerType[]          = {INT_ACCT, 4, 1, 1,  7, 0xFF, 0xFFFF};
oid o_atLowPeerAddress[]       = {INT_ACCT, 4, 1, 1,  8, 0xFF, 0xFFFF};
oid o_atLowPeerMask[]          = {INT_ACCT, 4, 1, 1,  9, 0xFF, 0xFFFF};
oid o_atLowDetailType[]        = {INT_ACCT, 4, 1, 1, 10, 0xFF, 0xFFFF};
oid o_atLowDetailAddress[]     = {INT_ACCT, 4, 1, 1, 11, 0xFF, 0xFFFF};
oid o_atLowDetailMask[]        = {INT_ACCT, 4, 1, 1, 12, 0xFF, 0xFFFF};
oid o_atHighInterface[]        = {INT_ACCT, 4, 1, 1, 15, 0xFF, 0xFFFF};
oid o_atHighAdjacentType[]     = {INT_ACCT, 4, 1, 1, 16, 0xFF, 0xFFFF};
oid o_atHighAdjacentAddress[]  = {INT_ACCT, 4, 1, 1, 17, 0xFF, 0xFFFF};
oid o_atHighAdjacentMask[]     = {INT_ACCT, 4, 1, 1, 18, 0xFF, 0xFFFF};
oid o_atHighPeerType[]         = {INT_ACCT, 4, 1, 1, 19, 0xFF, 0xFFFF};
oid o_atHighPeerAddress[]      = {INT_ACCT, 4, 1, 1, 20, 0xFF, 0xFFFF};
oid o_atHighPeerMask[]         = {INT_ACCT, 4, 1, 1, 21, 0xFF, 0xFFFF};
oid o_atHighDetailType[]       = {INT_ACCT, 4, 1, 1, 22, 0xFF, 0xFFFF};
oid o_atHighDetailAddress[]    = {INT_ACCT, 4, 1, 1, 23, 0xFF, 0xFFFF};
oid o_atHighDetailMask[]       = {INT_ACCT, 4, 1, 1, 24, 0xFF, 0xFFFF};
oid o_atPDUScale[]             = {INT_ACCT, 4, 1, 1, 29, 0xFF, 0xFFFF};
oid o_atOctetScale[]           = {INT_ACCT, 4, 1, 1, 30, 0xFF, 0xFFFF};
oid o_atRuleSet[]              = {INT_ACCT, 4, 1, 1, 31, 0xFF, 0xFFFF};

oid o_pcNearMem[]         = {AUKUNI, 2, 1, 0};
oid o_pcFarMem[]          = {AUKUNI, 2, 2, 0};
oid o_pcBadPackets[]      = {AUKUNI, 2, 3, 0};
oid o_pcNoBufPackets[]    = {AUKUNI, 2, 4, 0};
oid o_pcLostPackets[]     = {AUKUNI, 2, 5, 0};
oid o_pcPacketBacklog[]   = {AUKUNI, 2, 6, 0};
oid o_pcChkSearches[]     = {AUKUNI, 2, 7, 0};
oid o_pcChkCompares[]     = {AUKUNI, 2, 8, 0};

#define MAC_ADDR_LEN     6
#define PEER_ADDR_LEN    4
#define DETAIL_ADDR_LEN  2

#define RULE_ADDR_LEN    6

struct flow_info {
   unsigned char AdjAddrType,
      LowAdjAddress[MAC_ADDR_LEN], LowAdjMask[MAC_ADDR_LEN],
      HighAdjAddress[MAC_ADDR_LEN], HighAdjMask[MAC_ADDR_LEN];
   unsigned char PeerAddrType,
      LowPeerAddress[PEER_ADDR_LEN], LowPeerMask[PEER_ADDR_LEN],
      HighPeerAddress[PEER_ADDR_LEN], HighPeerMask[PEER_ADDR_LEN];
   unsigned char DetailAddrType,
      LowDetailAddress[DETAIL_ADDR_LEN], LowDetailMask[DETAIL_ADDR_LEN],
      HighDetailAddress[DETAIL_ADDR_LEN], HighDetailMask[DETAIL_ADDR_LEN];
   unsigned long
      FwdPackets,FwdBytes,  BackPackets,BackBytes,
      FirstTime,LastTime;
   };

struct rule_info {
   int RuleSet, RuleNbr;
   unsigned char RuleSelector,
      RuleMask[RULE_ADDR_LEN], RuleMatchedValue[RULE_ADDR_LEN], 
      RuleAction, RuleJumpIndex;
   };

struct action_info {
   int ActionSet, ActionNbr;
   unsigned char AdjAddrType,
      LowAdjAddress[MAC_ADDR_LEN], LowAdjMask[MAC_ADDR_LEN],
      HighAdjAddress[MAC_ADDR_LEN], HighAdjMask[MAC_ADDR_LEN];
   unsigned char PeerAddrType,
      LowPeerAddress[PEER_ADDR_LEN], LowPeerMask[PEER_ADDR_LEN],
      HighPeerAddress[PEER_ADDR_LEN], HighPeerMask[PEER_ADDR_LEN];
   unsigned char DetailAddrType,
      LowDetailAddress[DETAIL_ADDR_LEN], LowDetailMask[DETAIL_ADDR_LEN],
      HighDetailAddress[DETAIL_ADDR_LEN], HighDetailMask[DETAIL_ADDR_LEN];
   };

#define AT_IGNORE     0  /* Addr_type values */
#define AT_IP         2
#define AT_NOVELL     6
#define AT_DECNET     5
#define AT_ETHERTALK  7

#define RS_NULL           0
#define RS_LOWPEERTYPE    7
#define RS_LOWPEERADDR    8
#define RS_HIPEERADDR    20

#define RA_COUNT      1
#define RA_TALLY      2
#define RA_AGGREGATE  3
#define RA_SUCCEED    4
#define RA_FAIL       5
#define RA_PUSHTO     6
#define RA_POPTO      7
#define RA_GOTO       8

#define RF_SET       -1
#define RF_RULES     -2
#define RF_ACTIONS   -3

#define FTFLOWINDEX		1  /* Flow table attribute values */
#define FTFLOWSTATUS		2

#define FTLOWINTERFACE          3
#define FTLOWADJACENTTYPE       4
#define FTLOWADJACENTADDRESS    5
#define FTLOWADJACENTMASK       6
#define FTLOWPEERTYPE		7
#define FTLOWPEERADDRESS	8
#define FTLOWPEERMASK		9
#define FTLOWDETAILTYPE        10
#define FTLOWDETAILADDRESS     11
#define FTLOWDETAILMASK        12
#define FTLOWSUBSCRIBERID      13
#define FTLOWSUBSCRIBERMASK    14
#define FTHIINTERFACE          15
#define FTHIADJACENTTYPE       16
#define FTHIADJACENTADDRESS    17
#define FTHIADJACENTMASK       18
#define FTHIPEERTYPE	       19
#define FTHIPEERADDRESS	       20
#define FTHIPEERMASK	       21
#define FTHIDETAILTYPE         22
#define FTHIDETAILADDRESS      23
#define FTHIDETAILMASK         24
#define FTHISUBSCRIBERID       25
#define FTHISUBSCRIBERMASK     26

#define FTSUBSCRIBERID         27
#define FTSUBSCRIBERMASK       28

#define FTPDUSCALE             29
#define FTOCTETSCALE           30
#define FTRULESET              31
#define FTFLOWTYPE             32

#define FTUPOCTETS	       33
#define FTUPPDUS	       34
#define FTDOWNOCTETS	       35
#define FTDOWNPDUS	       36

#define FTFIRSTTIME	       37
#define FTLASTTIME	       38

#define ATN  1  /* Number.  Attribute type values */
#define ATA  2  /* Address */
#define ANI  0  /* Not implemented */

struct attrib_info {
   unsigned char *name;
   unsigned char index;
   unsigned char type;
   unsigned char len;
   };

struct attrib_info attribs[1+FTLASTTIME-FTFLOWINDEX] = {
   "flowindex",              FTFLOWINDEX,           ATN, 2,
   "flowstatus",             FTFLOWSTATUS,          ATN, 1,
   "sourceinterface",        FTLOWINTERFACE,        ATN, 1,
   "sourceadjacenttype",     FTLOWADJACENTTYPE,     ATN, 1,
   "sourceadjacentaddress",  FTLOWADJACENTADDRESS,  ATA, MAC_ADDR_LEN,
   "sourceadjacentmask",     FTLOWADJACENTMASK,     ATA, MAC_ADDR_LEN,
   "sourcepeertype",         FTLOWPEERTYPE,         ATN, 1,
   "sourcepeeraddress",      FTLOWPEERADDRESS,      ATA, PEER_ADDR_LEN,
   "sourcapeermask",         FTLOWPEERMASK,         ATA, PEER_ADDR_LEN,
   "sourcedetailtype",       FTLOWDETAILTYPE,       ATN, 1,
   "sourcedetailaddress",    FTLOWDETAILADDRESS,    ATA, DETAIL_ADDR_LEN,
   "sourcedetailmask",       FTLOWDETAILMASK,       ATA, DETAIL_ADDR_LEN,
   "sourcesubscriberid",     FTLOWSUBSCRIBERID,     ANI, 0,
   "sourcesubscribermask",   FTLOWSUBSCRIBERMASK,   ANI, 0,
   "destinterface",          FTHIINTERFACE,         ATN, 1,
   "destadjacenttype",       FTHIADJACENTTYPE,      ATN, 1,
   "destadjacentaddress",    FTHIADJACENTADDRESS,   ATA, MAC_ADDR_LEN,
   "destadjacentmask",       FTHIADJACENTMASK,      ATA, MAC_ADDR_LEN,
   "destpeertype",           FTHIPEERTYPE,          ATN, 1,
   "destpeeraddress",        FTHIPEERADDRESS,       ATA, PEER_ADDR_LEN,
   "sourcapeermask",         FTHIPEERMASK,          ATA, PEER_ADDR_LEN,
   "destdetailtype",         FTHIDETAILTYPE,        ATN, 1,
   "destdetailaddress",      FTHIDETAILADDRESS,     ATA, DETAIL_ADDR_LEN,
   "destdetailmask",         FTHIDETAILMASK,        ATA, DETAIL_ADDR_LEN,
   "destsubscriberid",       FTHISUBSCRIBERID,      ANI, 0,
   "destsubscribermask",     FTHISUBSCRIBERMASK,    ANI, 0,
   "subscriberid",           FTSUBSCRIBERID,        ANI, 0,
   "subscribermask",         FTSUBSCRIBERMASK,      ANI, 0,
   "pduscale",               FTPDUSCALE,            ANI, 0,
   "octetscale",             FTOCTETSCALE,          ANI, 0,
   "flowruleset",            FTRULESET,             ATN, 1,
   "flowtype",               FTFLOWTYPE,            ATN, 1,
   "tooctets",               FTUPOCTETS,            ATN, 4,
   "topdus",                 FTUPPDUS,              ATN, 4,
   "fromoctets",             FTDOWNOCTETS,          ATN, 4,
   "frompdus",               FTDOWNPDUS,            ATN, 4,
   "firsttime",              FTFIRSTTIME,           ATN, 4,
   "lasttime",               FTLASTTIME,            ATN, 4
   };

#define FLOWBLOBSZ  25

unsigned char column_blob[
   (FLOWBLOBSZ+2)*(2+RULE_ADDR_LEN)];

#ifdef AU_MSDOS  /* PC ntoh routines swap the byte order */
#define netshort(x)  x
#define netlong(x)   x
#else            /* SunOS ntoh routines don't */
#define netshort(x)  htons(x)
#define netlong(x)   htonl(x)
#endif

#define NAME_LN  64

struct meter_status {
   struct meter_status *next;

   char name[NAME_LN];
   char community[NAME_LN];
   char rulefile[NAME_LN];

   struct snmp_session *ss;

   short status;

   char descr[NAME_LN];  /* From meter_info() */
   unsigned long uptime;
   unsigned long LastCollectTime;  /* By any collector, not neccessarily us */
   unsigned char LastCollectPeer[PEER_ADDR_LEN];

   int ruleset, nrules, nactions;

   unsigned long OurLastCollectTime;

   FILE *keys;
   FILE *flows;
   FILE *log;
   };

struct meter_status *first_meter;

/* Values for status */

#define	MT_MANAGE    0x0001  /* Manage this meter */
#define	MT_INFO	     0x0002  /* Have basic info */
#define	MT_UP	     0x0004  /* Meter is running */
#define	MT_REPORTED  0x0008  /* Current status reported */

#ifdef MTR_LIST
/* #define  CTLFILE  "/usr/local/src/snmp/manager.cfg" */
#define  CTLFILE  "manager.cfg"
#endif

unsigned short getshort(unsigned char *ucp);
unsigned long getlong(unsigned char *ucp);

void init_meter();
void monitor_meter();
void monitor(struct meter_status *ms);
void print_meters();
void meter_print(struct meter_status *ms);

void printaddress(FILE *f,unsigned char *a,unsigned char len);
void getaddress(unsigned char *a,unsigned char len);

int scan_rulefile(struct meter_status *ms, int doset, int list);
void parse_rulefile(struct meter_status *ms, int list);
int add_rule(struct meter_status *ms, struct rule_info *ri);
int add_action(struct meter_status *ms, struct action_info *ai);

int same_acct_oid(oid *a, oid *b);
int blob_info(struct meter_status *ms, unsigned char *fb, 
   unsigned long ft, int *fn);
int flow_info(struct meter_status *ms, struct flow_info *fi,
   unsigned long ft, int *fn, int getkey);
   
char *uptime_string(unsigned long timeticks, char *buf);

int verbose = 0, testing = 0;

void main(argc,argv)
int argc;
char *argv[];
{
   struct meter_status *ms;
   struct snmp_session session, *ssp;

   int arg;
   char meter[NAME_LN], community[NAME_LN], rfname[NAME_LN],
      filename[NAME_LN*2];
   int interval = 0;
   int syntax = 0;

   meter[0] = community[0] = rfname[0] = NULL;

#ifdef MTR_LIST
   parse_ctlfile();
#endif

   init_mib();

   /*  usage: nm meter-name community */

   for (arg = 1; arg < argc; arg++) {
      if (argv[arg][0] == '-') {
	 switch (argv[arg][1]) {
	 case 'd':
	    snmp_dump_packet++;
	    break;
	 case 'c':
	    if ((interval = atoi(argv[arg]+2)) == 0) interval = 120;
	    break;
	 case 'r':
	    strcpy(rfname, argv[++arg]);
	    break;
	 case 's':
	    syntax++;
	    break;
	 case 't':
	    testing++;
	    break;
	 case 'v':
	    verbose++;
	    break;
	 default:
	    printf("Invalid option: -%c\n", argv[arg][1]);
	    break;
	    }
	 continue;
	 }
      if (meter[0] == NULL) strcpy(meter,argv[arg]);
      else if (community[0] == NULL) strcpy(community,argv[arg]);
      }

   if (meter[0] == NULL) {
      printf("nm meter-name community\n\n");
      exit(0);
      }

   ms = (struct meter_status *)calloc(sizeof(struct meter_status), 1);
   strcpy(ms->name,meter);
   strcpy(ms->community,community);
   if (*rfname) strcpy(ms->rulefile, rfname);
   else strcpy(ms->rulefile, RULEFILE);

   if (syntax) {  /* For testing rule files */
      scan_rulefile(ms,0,1);  exit(0);
      }

   ms->OurLastCollectTime = 2L;

   strcpy(filename,meter);
   strcat(filename,".log");
   if ((ms->log = fopen(filename,"wa")) == NULL) {
      printf("Failed to open %s\n",filename);
      exit(0);
      }
   strcpy(filename,meter);
   strcat(filename,".keys");
   if ((ms->keys = fopen(filename,"wa")) == NULL) {
      printf("Failed to open %s\n",filename);
      exit(0);
      }
   strcpy(filename,meter);
   strcat(filename,".flows");
   if ((ms->flows = fopen(filename,"wa")) == NULL) {
      printf("Failed to open %s\n",filename);
      exit(0);
      }

   bzero((char *)&session, sizeof(struct snmp_session));
   session.peername = ms->name;
   session.community = ms->community;
   session.community_len = strlen(ms->community);
   session.retries = SNMP_DEFAULT_RETRIES;
   session.timeout = SNMP_DEFAULT_TIMEOUT;
   session.authenticator = NULL;
   snmp_synch_setup(&session);
   ssp = snmp_open(&session);
   if (ssp == NULL) {
      printf("Couldn't open snmp to %s\n", session.peername);
      exit(-1);
      }
   ms->ss = ssp;

   ms->status = MT_MANAGE;
   first_meter = ms;  /* $$$$$$$ */

#ifdef MTR_LIST
   if (!explicit) add_all_meters();
#endif

   init_meter();

   if (!interval) {
      print_meters();
      exit (0);
      }

   parse_rulefile(ms,TRUE);  /* Download the rules we want to use */

   for (;;) {
      sleep(interval);

      monitor_meter();
      if (verbose) print_meters();
      }
   }

#ifdef MTR_LIST
void parse_ctlfile()
{
   struct meter_status *ms, *mp;
   FILE *fp;
   char linebuffer[256];
   char label[128];
   char community[128];
   int n;

   if ((fp = fopen(CTLFILE, "r")) == NULL) {
      perror(CTLFILE);
      exit(1);
      }

   while (fgets(linebuffer, sizeof(linebuffer), fp) != NULL) {
      if (linebuffer[0] < '0' || linebuffer[0] > 'z') continue;
      n = sscanf(linebuffer, "%s %s", label, community);
      if (n < 1) continue;
      ms = (struct meter_status *)calloc(sizeof(struct meter_status), 1);
      ms->name = savestr(label);
      ms->community = n > 1 ? savestr(community) : savestr("public");
      ms->next = NULL;

      if ((mp = first_meter) == NULL) first_meter = ms;
      else {
	 while (mp->next) mp = mp->next;
	 mp->next = ms;
	 }
      }
   }

void add_all_meters()
{
   struct meter_status *ms;
   for (ms = first_meter; ms != NULL; ms = ms->next)
      ms->status = MT_MANAGE;
   }

void add_meter(name)
char *name;
{
   struct meter_status *ms;

   for (ms = first_meter; ms != NULL; ms = ms->next) {
      if (strcasecmp(name,ms->name) == 0) break;
      }

   if (ms == NULL) {
      struct hostent *hp;
      u_long addr;

      if ((addr = (u_long)inet_addr(name)) == (u_long)-1) {
	 hp = gethostbyname(name);
	 if (hp != NULL) {
	    for (ms = first_meter; ms != NULL; ms = ms->next) {
#ifdef h_addr
	       char **ap;
#endif
	       if (strcasecmp(hp->h_name, ms->name) == 0) break;
#ifdef h_addr
	       if (hp->h_addr_list == NULL) continue;

	       for (ap = hp->h_addr_list; *ap; ap++) {
		  if (addr == *(u_long *)(*ap)) break;
		  }
	       if (*ap) break;
#else
	       if (addr == *(u_long *)(hp->h_addr)) break;
#endif
					
	       }
	    if (ms == NULL) {
	       printf("No config info for %s\n", name);
	       return;
	       }
	    }
         else {
	    printf("No config info for %s\n", name);
	    return;
	    }
	 }
      }

   ms->status = MT_MANAGE;
   }
#endif

void init_meter()
{
   struct meter_status *ms;
   int haveone = 0;

   for (ms = first_meter; ms; ms = ms->next) {
      if (!(ms->status & MT_MANAGE)) continue;
      haveone++;
      if (meter_info(ms)) {
         ms->status |= (MT_UP | MT_INFO);
         meter_print(ms);
         }
      }

   if (!haveone) {
      printf("No meters to monitor\n");
      exit(5);
      }
   }


void monitor_meter()
{
   struct meter_status *ms;
   for (ms = first_meter; ms; ms = ms->next)
      if (ms->status & MT_MANAGE) monitor(ms);
   }

/* CAUTION:  Following struct doesn't work because of alignment on 32-bit words
 *
 * struct active_flow_data {  
 *    unsigned short flow_nbr;
 *    unsigned long fwd_bytes, back_bytes;
 *    }; 
 */

#define flow_nbr_x    0
#define fwd_bytes_x   (flow_nbr_x + sizeof(unsigned short))
#define back_bytes_x  (fwd_bytes_x + sizeof(unsigned long))
#define flow_data_sz  (back_bytes_x + sizeof(unsigned long))

#define FLOWBLOBSZ  25

unsigned short getshort(ucp)
unsigned char *ucp;
{
   return ucp[0]<<8 | ucp[1];
   }

unsigned long getlong(ucp)
unsigned char *ucp;
{
   return ucp[0]<<24 | ucp[1]<<16 | ucp[2]<<8 | ucp[3];
   }

void monitor(ms)  /* Called every interval for each meter */
struct meter_status *ms;
{
   time_t t;
   char *ts;
   unsigned short newflows,activeflows, n;
   int fn;
   struct flow_info fi;
   unsigned char flowblob[FLOWBLOBSZ*flow_data_sz], *afdp;
   unsigned long fwd_bytes, back_bytes;

   time(&t);  ts = ctime(&t);
   if (!(ms->status & MT_INFO)) {  /* No info */
      if (meter_info(ms))  /* Got some */
	 ms->status |= (MT_UP | MT_INFO);
      return;
      }
   if (meter_info(ms) == 0) {  /* Lost contact */
      if (ms->status & MT_UP) {  /* Was up */
	 fprintf(ms->log,"%19.19s -- %s: No response\n", ts,ms->name);
	 if (verbose) printf("%19.19s -- %s: No response\n", ts,ms->name);
	 }
      ms->status &= ~MT_UP;  /* Mark as 'down' */
      return;
      }
   if (!(ms->status & MT_UP)) {  /* Have contact now, was down */
      fprintf(ms->log,"%19.19s -- %s: Regained contact\n", ts,ms->name);
      if (verbose) printf("%19.19s -- %s: Regained contact\n", ts,ms->name);
      }
   ms->status |= MT_UP;

   /* Meter processing .. */

   set_collect_time(ms,1);  /* Tell meter we're starting a collection */

   fprintf(ms->keys, 
      "## %19.19s %s: New flows from %lu to %lu##\n",
      ts,ms->name, ms->LastCollectTime,ms->uptime);
   fn = 1;  newflows = 0;
   while (flow_info(ms, &fi, ms->OurLastCollectTime,&fn, 1)) {
      ++newflows;
      fprintf(ms->keys, "%u %lu %u", fn, fi.FirstTime,fi.PeerAddrType);
      printaddress(ms->keys, fi.LowPeerAddress,PEER_ADDR_LEN);
      printaddress(ms->keys, fi.HighPeerAddress,PEER_ADDR_LEN);
      fprintf(ms->keys,"\n");
      }
   fflush(ms->keys);

   fprintf(ms->flows, 
      "## %19.19s %s: Active flows from %lu to %lu##\n",
      ts,ms->name, ms->LastCollectTime,ms->uptime);
   fn = 1;  activeflows = 0;
   while (blob_info(ms,flowblob, ms->OurLastCollectTime,&fn)) {
      for (n = 0, afdp = flowblob;  n != FLOWBLOBSZ; 
            ++n, afdp += flow_data_sz) {
         fn = ntohs(getshort(afdp + flow_nbr_x));
         if (fn == 0) break;  /* No more active flows in blob */
         ++activeflows;
         fprintf(ms->flows,"%u %lu %lu\n", fn,
            ntohl(getlong(afdp + fwd_bytes_x)),
            ntohl(getlong(afdp + back_bytes_x)) );
         }
      if (fn == 0) break;  /* Don't bother to look for more */
      }	
   fflush(ms->flows);

   ms->OurLastCollectTime = ms->uptime - 1;
      /* -1 to make sure we don't miss any flows.  We may
         collect some of them twice, but we don't mind that */

   if (verbose) printf(
      "%19.19s %s: %d new and %d active flows from %lu to %lu\n",
      ts,ms->name, newflows,activeflows, ms->LastCollectTime,ms->uptime);
   fprintf(ms->log, 
      "## %19.19s %s: %d new and %d active flows from %ld to %ld##\n",
      ts,ms->name, newflows,activeflows, ms->LastCollectTime,ms->uptime);
   fflush(ms->log);  
   }

#define ADD_VAR(v)            snmp_add_null_var(pdu, v, sizeof(v)/sizeof(oid))
#define ADD_X_VAR(v,n1)       { v[sizeof(v)/sizeof(oid) - 1] = n1; \
            ADD_VAR(v); }
#define ADD_X2_VAR(v,n1,n2)   { v[sizeof(v)/sizeof(oid) - 2] = n1; \
            v[sizeof(v)/sizeof(oid) - 1] = n2; \
            ADD_VAR(v); }

#define SET_INT(v)            { vars->type = INTEGER; \
  	    vars->val.integer = (long *)malloc(vars->val_len = sizeof(long)); \
	    *(vars->val.integer) = (long)v; }
#define SET_TIMETICKS(v)      { vars->type = TIMETICKS; \
  	    vars->val.integer = (long *)malloc(vars->val_len = sizeof(long)); \
	    *(vars->val.integer) = (long)v; }
#define SET_STRING(v,len)     { vars->type = STRING; \
	    vars->val.string = (u_char *)malloc(RULE_ADDR_LEN) ;\
	    bcopy(v, (char *)vars->val.string, vars->val_len = len); }

#define STRING_VAL(v)         bcopy(vars->val.string, v, vars->val_len)
#define INT_VAL(v)            v = *(vars->val.integer)

int set_collect_time(ms,v)  /* Set LastCollectTime for meter - */
struct meter_status *ms;    /*    this tells meter a collection is starting */
int v;
{
   int i, count, status;
   struct snmp_pdu *pdu, *response;
   struct variable_list *vars;
   pdu = snmp_pdu_create(SET_REQ_MSG);
   ADD_VAR(o_LastCollectTime);
   vars = pdu->variables;
   SET_TIMETICKS(v);

   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
         printf("Set %d: starting collection\n", ms->ruleset);
	 }
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 if (response->errstat == SNMP_ERR_NOSUCHNAME) {
	    printf("This name does not exist: ");
	    for (count=1, vars = response->variables;
	       vars && count != response->errindex;
		  vars = vars->next_variable, count++) ;
	    if (vars) print_objid(vars->name, vars->name_length);
	    printf("\n");
	    }
	 }
      }
   else return 0;
   snmp_free_pdu(response);
   return 1;
   }

int set_rule_info(ms,setset)  /* 1 to set rule+action set */
struct meter_status *ms;      /* 0 to set rule and actions tables sizes */
int setset;
{
   int i, count, status;
   struct snmp_pdu *pdu, *response;
   struct variable_list *vars;
   pdu = snmp_pdu_create(SET_REQ_MSG);
   if (setset) {
      ADD_VAR(o_CurrentRuleSet);
         vars = pdu->variables;
         SET_INT(ms->ruleset);
      }
   else {
      ADD_X_VAR(o_riRuleSize, ms->ruleset);
         vars = pdu->variables;
         SET_INT(ms->nrules);
      ADD_X_VAR(o_riActionSize, ms->ruleset);
         vars = vars->next_variable;
         SET_INT(ms->nactions);
      }

   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
         if (setset) ("Meter is now using rule+action set %d\n", ms->ruleset);
         else printf("Set %d: sizes set to %d rules + %d actions\n", 
            ms->ruleset,ms->nrules,ms->nactions);
	 }
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 if (response->errstat == SNMP_ERR_NOSUCHNAME) {
	    printf("This name does not exist: ");
	    for (count=1, vars = response->variables;
	       vars && count != response->errindex;
		  vars = vars->next_variable, count++) ;
	    if (vars) print_objid(vars->name, vars->name_length);
	    printf("\n");
	    }
	 }
      }
   else return 0;
   snmp_free_pdu(response);
   return 1;
   }

int add_rule(ms,ri)
struct meter_status *ms;
struct rule_info *ri;
{
   int i, count, status;
   struct snmp_pdu *pdu, *response;
   struct variable_list *vars;

   pdu = snmp_pdu_create(SET_REQ_MSG);
   ADD_X2_VAR(o_rtSelector, ri->RuleSet,ri->RuleNbr);
      vars = pdu->variables;
      SET_INT(ri->RuleSelector);
   ADD_X2_VAR(o_rtRuleMask, ri->RuleSet,ri->RuleNbr);
      vars = vars->next_variable;
      SET_STRING(ri->RuleMask,RULE_ADDR_LEN);
   ADD_X2_VAR(o_rtMatchedValue, ri->RuleSet,ri->RuleNbr);
      vars = vars->next_variable;
      SET_STRING(ri->RuleMatchedValue,RULE_ADDR_LEN);
   ADD_X2_VAR(o_rtRuleAction, ri->RuleSet,ri->RuleNbr);
      vars = vars->next_variable;
      SET_INT(ri->RuleAction);
   ADD_X2_VAR(o_rtJumpIndex, ri->RuleSet,ri->RuleNbr);
      vars = vars->next_variable;
      SET_INT(ri->RuleJumpIndex);

   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
         printf("Rule %d added to table %d\n", ri->RuleNbr,ri->RuleSet);
	 }
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 if (response->errstat == SNMP_ERR_NOSUCHNAME) {
	    printf("This name does not exist: ");
	    for (count=1, vars = response->variables;
	       vars && count != response->errindex;
		  vars = vars->next_variable, count++) ;
	    if (vars) print_objid(vars->name, vars->name_length);
	    printf("\n");
	    }
	 }
      }
   else return 0;
   snmp_free_pdu(response);
   return 1;
   }

int add_action(ms,ai)
struct meter_status *ms;
struct action_info *ai;
{
   int i, count, status;
   struct snmp_pdu *pdu, *response;
   struct variable_list *vars;

   pdu = snmp_pdu_create(SET_REQ_MSG);
   ADD_X2_VAR(o_atLowPeerType, ai->ActionSet,ai->ActionNbr);
      vars = pdu->variables;
      SET_INT(ai->PeerAddrType);
   ADD_X2_VAR(o_atLowPeerAddress, ai->ActionSet,ai->ActionNbr);
      vars = vars->next_variable;
      SET_STRING(ai->LowPeerAddress,PEER_ADDR_LEN);
   ADD_X2_VAR(o_atLowPeerMask, ai->ActionSet,ai->ActionNbr);
      vars = vars->next_variable;
      SET_STRING(ai->LowPeerMask,PEER_ADDR_LEN);
   ADD_X2_VAR(o_atHighPeerAddress, ai->ActionSet,ai->ActionNbr);
      vars = vars->next_variable;
      SET_STRING(ai->HighPeerAddress,PEER_ADDR_LEN);
   ADD_X2_VAR(o_atHighPeerMask, ai->ActionSet,ai->ActionNbr);
      vars = vars->next_variable;
      SET_STRING(ai->HighPeerMask,PEER_ADDR_LEN);

   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
         printf("Action %d added to table %d\n", ai->ActionNbr,ai->ActionSet);
	 }
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 if (response->errstat == SNMP_ERR_NOSUCHNAME) {
	    printf("This name does not exist: ");
	    for (count=1, vars = response->variables;
	       vars && count != response->errindex;
		  vars = vars->next_variable, count++) ;
	    if (vars) print_objid(vars->name, vars->name_length);
	    printf("\n");
	    }
	 }
      }
   else return 0;
   snmp_free_pdu(response);
   return 1;
   }

int same_acct_oid(oid *a, oid *b)  
   /* Compare oids for equality within internet-accounting MIB */
{
   int j;
   for (j = 6; j != 10; ++j) {
      if (a[j] != b[j]) return 0;
      }
   return 1;
   }

int blob_info(struct meter_status *ms, unsigned char *fb,
   unsigned long ft, int *fn)
{
   int i, count, status;
   struct snmp_pdu *pdu, *response;
   struct variable_list *vars;

   if (testing) printf("blob_info(ms,fb,ft=%lu,fn=%d)\n",ft,*fn);
   pdu = snmp_pdu_create(GETNEXT_REQ_MSG);
   ADD_X2_VAR(o_ftActiveFlowBlob,ft,*fn)
   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
	 vars = response->variables;
         if (!same_acct_oid(vars->name,o_ftActiveFlowBlob)) 
            return 0;  /* No more flows with last-active time > ft */
	 STRING_VAL(fb);
	 }
      else if (response->errstat == SNMP_ERR_NOSUCHNAME) return 0;
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 printf("This name does not exist: ");
	 for (count=1, vars = response->variables;
	    vars && count != response->errindex;
	 vars = vars->next_variable, count++) ;
	 if (vars) print_objid(vars->name, vars->name_length);
	 printf("\n");
	 }
      }
   else return 0;
   snmp_free_pdu(response);
   return 1;
   }

int flow_info(struct meter_status *ms, struct flow_info *fi,
   unsigned long ft, int *fn, int getkey)
  /* getkey = 1 to get keys created after ft, 
              0 to get flows active after ft */
{
   int j, count, status;
   struct snmp_pdu *pdu, *response;
   struct variable_list *vars;

   if (testing) printf("flow_info(ms,fi,ft=%lu,fn=%d,getkey=%d)\n",
      ft,*fn,getkey);

   pdu = snmp_pdu_create(GETNEXT_REQ_MSG);
   if (getkey) ADD_X2_VAR(o_ftCreateIndex,ft,*fn)
   else ADD_X2_VAR(o_ftActiveIndex,ft,*fn)
   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
	 vars = response->variables;
         if (getkey) {
            if (!same_acct_oid(vars->name,o_ftCreateIndex)) 
               return 0;  /* No more flows with create time > ft */
            }
         else {
            if (!same_acct_oid(vars->name,o_ftActiveIndex)) 
               return 0;  /* No more flows with last-active time > ft */
	    }
	 INT_VAL(*fn);  /* Index of next flow created or active after ft */
	 }
      else if (response->errstat == SNMP_ERR_NOSUCHNAME) return 0;
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 printf("This name does not exist: ");
	 for (count=1, vars = response->variables;
	    vars && count != response->errindex;
	 vars = vars->next_variable, count++) ;
	 if (vars) print_objid(vars->name, vars->name_length);
	  printf("\n");
	 }
      }
   else return 0;
   snmp_free_pdu(response);

   pdu = snmp_pdu_create(GET_REQ_MSG);
   if (getkey) {
      ADD_X_VAR(o_ftLowPeerType,*fn);
      ADD_X_VAR(o_ftLowPeerAddress,*fn);
      ADD_X_VAR(o_ftHighPeerAddress,*fn);
      ADD_X_VAR(o_ftFirstTime,*fn);
      }
   else {
      ADD_X_VAR(o_ftUpOctets,*fn);
      ADD_X_VAR(o_ftDownOctets,*fn);
      ADD_X_VAR(o_ftLastTime,*fn);
      }
   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
	 vars = response->variables;
	 if (getkey) {
	    INT_VAL(fi->PeerAddrType);
	    vars = vars->next_variable;
	    STRING_VAL(fi->LowPeerAddress);
	    vars = vars->next_variable;
	    STRING_VAL(fi->HighPeerAddress);
	    vars = vars->next_variable;
	    INT_VAL(fi->FirstTime);
	    vars = vars->next_variable;
	    }
	 else {
	    INT_VAL(fi->FwdBytes);
	    vars = vars->next_variable;
	    INT_VAL(fi->BackBytes);
	    vars = vars->next_variable;
	    INT_VAL(fi->LastTime);
	    }
	 }
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 if (response->errstat == SNMP_ERR_NOSUCHNAME) {
	    printf("This name does not exist: ");
	    for (count=1, vars = response->variables;
	       vars && count != response->errindex;
		  vars = vars->next_variable, count++) ;
	    if (vars) print_objid(vars->name, vars->name_length);
	    printf("\n");
	    }
	 }
      }
   else return 0;
   snmp_free_pdu(response);
   return 1;
   }

int meter_info(ms)
struct meter_status *ms;
{
   int i, count, status;
   struct snmp_pdu *pdu, *response;
   struct variable_list *vars;

   if (testing) printf("meter_info(ms)\n");
   pdu = snmp_pdu_create(GET_REQ_MSG);
   ADD_VAR(o_sysDescr);
   ADD_VAR(o_sysUpTime);
   ADD_VAR(o_LastCollectTime);
   status = snmp_synch_response(ms->ss, pdu, &response);
   if (status == STAT_SUCCESS) {
      if (response->errstat == SNMP_ERR_NOERROR) {
	 vars = response->variables;
	 STRING_VAL(ms->descr);
         vars = vars->next_variable;
	 INT_VAL(ms->uptime);
         vars = vars->next_variable;
	 INT_VAL(ms->LastCollectTime);
	 }
      else {
	 printf("Error in packet, reason = %s\n",
	    snmp_errstring(response->errstat));
	 if (response->errstat == SNMP_ERR_NOSUCHNAME) {
	    printf("This name does not exist: ");
	    for (count=1, vars = response->variables;
	       vars && count != response->errindex;
		  vars = vars->next_variable, count++) ;
	    if (vars) print_objid(vars->name, vars->name_length);
	    printf("\n");
	    }
	 }
      }
   else return 0;
   snmp_free_pdu(response);
   return 1;
   }

void print_meters()
{
   struct meter_status *ms;
   for (ms = first_meter; ms; ms = ms->next)
      if (ms->status & MT_MANAGE) meter_print(ms);
   }

void meter_print(ms)
struct meter_status *ms;
{
   struct timeval tv;
   char buf[32];
   time_t t;   char tsbuf[32];

   time(&t);  strcpy(tsbuf,ctime(&t));

   gettimeofday(&tv, (struct timezone *)0);
   tv.tv_sec -= ms->uptime / 100;
   if ((ms->uptime % 100)*10000 > tv.tv_usec) {
      tv.tv_sec--;
      tv.tv_usec += 1000000;
      }
   tv.tv_usec -= (ms->uptime % 100)*10000;

   fprintf(ms->log,
      "%19.19s -- %s: %s\n\tUp %s (since %.24s)\n",
	 tsbuf,ms->name, ms->descr, uptime_string(ms->uptime, buf),
	 ctime(&tv.tv_sec));
   if (verbose) printf(
      "%19.19s -- %s: %s\n\tUp %s (since %.24s)\n",
	 tsbuf,ms->name, ms->descr, uptime_string(ms->uptime, buf),
	 ctime(&tv.tv_sec));
   }


char *uptime_string(timeticks, buf)
unsigned long timeticks;
char *buf;
{
   int	seconds, minutes, hours, days;

   timeticks /= 100;
   days = timeticks / (60 * 60 * 24);
   timeticks %= (60 * 60 * 24);

   hours = timeticks / (60 * 60);
   timeticks %= (60 * 60);

   minutes = timeticks / 60;
   seconds = timeticks % 60;

   if (days == 0){
      sprintf(buf, "%d:%02d:%02d", hours, minutes, seconds);
      }
   else if (days == 1) {
      sprintf(buf, "%d day, %d:%02d:%02d",
	 days, hours, minutes, seconds);
      }
   else {
      sprintf(buf, "%d days, %d:%02d:%02d",
	 days, hours, minutes, seconds);
      }
   return buf;
   }

void printaddress(f,a,len)
FILE *f;
unsigned char *a, len;
{
   int j;
   fprintf(f," %u", a[0]);
   for (j = 1; j != len; ++j) fprintf(f,".%u", a[j]);
   }


FILE *rfp;  /* For rules file */
char inbuf[256], *ibp;
int lic, ic,  /* Last input char, current input char */
   iblisted, rule_line, rferrors;

int nextchar()
{
   lic = ic;
   for (;;) {
      if (lic == '\n') {
	 if (fgets(inbuf, sizeof(inbuf), rfp) == NULL) return ic = EOF;
	 iblisted = 0;  ++rule_line;
	 ibp = inbuf;
	 }
      ic = *ibp++;
      if (ic == '#') lic = '\n';  /* Ignore comments */
      else return ic;
      }
   }

int wordis(p,w)
char *p, *w;
{
   return strncmp(p,w,strlen(w)) == 0;
   }

int getword()
{
   char wbuf[30], *wp = wbuf;
   for (;;) {
      *wp++ = tolower(ic);
      nextchar(rfp);
      if (ic == EOF) return EOF;
      if (!isalpha(ic)) break;
      }

   if (wordis(wbuf,"null")) return RS_NULL;
   if (wordis(wbuf,"lowpeertype")) return RS_LOWPEERTYPE;
   if (wordis(wbuf,"lowpeeraddress")) return RS_LOWPEERADDR;
   if (wordis(wbuf,"highpeeraddress")) return RS_HIPEERADDR;

   if (wordis(wbuf,"ip")) return AT_IP;
   if (wordis(wbuf,"novell")) return AT_NOVELL;
   if (wordis(wbuf,"decnet")) return AT_DECNET;
   if (wordis(wbuf,"ethertalk")) return AT_ETHERTALK;

   if (wordis(wbuf,"count")) return RA_COUNT;
   if (wordis(wbuf,"tally")) return RA_TALLY;
   if (wordis(wbuf,"aggregate")) return RA_AGGREGATE;
   if (wordis(wbuf,"succeed")) return RA_SUCCEED;
   if (wordis(wbuf,"fail")) return RA_FAIL;
   if (wordis(wbuf,"pushto")) return RA_PUSHTO;
   if (wordis(wbuf,"popto")) return RA_POPTO;
   if (wordis(wbuf,"goto")) return RA_GOTO;

   if (wordis(wbuf,"set")) return RF_SET;
   if (wordis(wbuf,"rules")) return RF_RULES;
   if (wordis(wbuf,"actions")) return RF_ACTIONS;

   if (!iblisted) {
      printf("RULEFILE line %d: %s\n", rule_line,inbuf);
      iblisted = 1;
      }
   printf("Unknown word %s !!!\n", wbuf);
   ++rferrors;
   return 0;
   }

int getnbr()
{
   int v = 0;
   for (;;) {
      if (ic == EOF) return EOF;
      if (isdigit(ic)) break;
      else if (isalpha(ic)) return getword();
      else nextchar(rfp);
      }
   for (;;) {
      v = v*10 + ic-'0';
      if (nextchar(ic) == EOF) return EOF;
      if (!isdigit(ic)) break;
      }
   if (v > 255) {
      if (!iblisted) {
	 printf("RULEFILE line %d: %s\n", rule_line,inbuf);
	 iblisted = 1;
	 }
      printf("Number > 255 !!!\n");
      ++rferrors;
      }
   return v;
   }

void getaddress(a,len)
unsigned char *a, len;
{
   int j;
   for (j = 0; j != len; ++j) {
      a[j] = getnbr();
      if (ic != '.') {  /* End of address; pad with zeroes */
         for (++j; j != len; ++j) a[j] = 0;
         return;
         }
      }
   }

int scan_rulefile(ms,doset,list)
struct meter_status *ms;
int doset,list;
{
   struct rule_info ri;
   struct action_info ai;
   int rule_set, nrules, nactions, n, kind;

   if ((rfp = fopen(ms->rulefile, "r")) == NULL) {
      perror(ms->rulefile);
      printf("   -> Using meter's default rules <-\n\n");
      return 0;  /* Fail */
      }

   rferrors = 0;
   ic = '\n';  rule_set = 2;  /* Default rule+action set */
   nrules = nactions = 0;  kind = RF_RULES;
   for (;;) {
      do {  /* First char of a line */
	 nextchar(rfp);
	 if (ic == EOF) break;
	 } while (lic != '\n');
      if (ic == EOF) break;
      n = getnbr();  /* What kind of line is it? */
      if (n == RF_SET) {
         ri.RuleSet = ai.ActionSet = rule_set = getnbr();
         kind = RF_RULES;  
         continue;
         }
      else if (n == RF_RULES) {
         kind = RF_RULES;  continue;
         }
      if (n == RF_ACTIONS) {
         kind = RF_ACTIONS;   continue;
         }

      if (kind == RF_RULES) {
         ri.RuleSelector = n;
         getaddress(ri.RuleMask,RULE_ADDR_LEN);
         if (ic == EOF) break;
         getaddress(ri.RuleMatchedValue,RULE_ADDR_LEN);
         if (ic == EOF) break;
         ri.RuleAction = getnbr();
         if (ic == EOF) break;
         ri.RuleJumpIndex = getnbr();
         if (ic == EOF) break;
         ri.RuleNbr = ++nrules;
         if (list) {
	    printf("Rule %d,%d:  %d  ", rule_set,nrules,ri.RuleSelector);
	    printaddress(stdout, ri.RuleMask, RULE_ADDR_LEN);
	    printaddress(stdout, ri.RuleMatchedValue,RULE_ADDR_LEN);
	    printf(" %d %d\n", ri.RuleAction,ri.RuleJumpIndex);
	    }
         if (doset) add_rule(ms,&ri);  /* Add rule to meter's rule table */
	 }

      if (kind == RF_ACTIONS) {
         ai.PeerAddrType = n;
         if (ic == EOF) break;
         getaddress(ai.LowPeerAddress,PEER_ADDR_LEN);
         if (ic == EOF) break;
         getaddress(ai.LowPeerMask,PEER_ADDR_LEN);
         if (ic == EOF) break;
         getaddress(ai.HighPeerAddress,PEER_ADDR_LEN);
         if (ic == EOF) break;
         getaddress(ai.HighPeerMask,PEER_ADDR_LEN);
         if (ic == EOF) break;
         ai.ActionNbr = ++nactions;
         if (list) {
	    printf("Action %d,%d:  %d  ", rule_set,nactions,ai.PeerAddrType);
	    printaddress(stdout, ai.LowPeerAddress,PEER_ADDR_LEN);
	    printaddress(stdout, ai.LowPeerMask,PEER_ADDR_LEN);
            printf("  ");
	    printaddress(stdout, ai.HighPeerAddress,PEER_ADDR_LEN);
	    printaddress(stdout, ai.HighPeerMask,PEER_ADDR_LEN);
	    printf("\n");
	    }
      if (doset) add_action(ms,&ai);  /* Add rule to meter's action table */
	 }
     }

   fclose(rfp);
   if (rferrors == 0) {
      ms->ruleset = rule_set;  ms->nrules = nrules;  ms->nactions = nactions;
      return 1;  /* Succeed */
      }
   return 0;  /* Fail */
   }

void parse_rulefile(ms,list)
struct meter_status *ms;
int list;
{
   if (scan_rulefile(ms,0,list)) {  /* Rulefile is OK */
      if (set_rule_info(ms,0)) {  /* Set rule+action table sizes */
         scan_rulefile(ms,1,0);  /* Download the rules and actions */
         set_rule_info(ms,1);  /* Tell meter to start using new rules */
         }
      }
   }
