/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * States for the state machines
 */

/*
 * Exports:
 *	int	checktransition(oldstate, newstate: state_t)
 *	int	parentstate2timeout(state: state_t)
 *	int	childstate2timeout(state: state_t)
 *
 * For the state of the clients and servers
 *	int 	alloc_childstate(nproc: int)
 *	int 	childstate_free()
 *	int 	set_state(addr: bcpaddr_t; state: state_t)
 *	int 	check_state(addr: bcpaddr_t; state: state_t)
 *	int 	set_fg_client_state(state: state_t)
 *	int 	set_bg_client_state(state: state_t)
 *	int 	check_fg_client_state(state: state_t)
 *	int 	check_bg_client_state(state: state_t)
 *	int 	set_result(addr: bcpaddr_t; stat: statistics_t *)
 *	int 	set_addr(addr: bcpaddr_t; address: address_t *)
 *	int 	get_results(a_stats: statistics_t *)
 *	void 	get_respresults(a_stats: statistics_t *)
 *	struct address_t *get_addr(index:int)
 *
 *	state_t	get_state(addr: bcpaddr_t)
 *	int	set_statebackground(addr: bcpaddr_t)
 *					takes e.g. BA_CLIENT | BA_SERVER
 *	int	get_statebackground(addr: bcpaddr_t)
 *
 *	char 	*state2str(state: state_t)
 */

#include <general.h>
#include <state.h>
#include <ipc.h>

/*  */

/*
 * The state of the client and servers are recorded in
 */
typedef struct {
    state_t		cs_state;	
    int			cs_background;
    statistics_t	*cs_result;
    struct address_t	*cs_addr;
} childstate_t;

#define NULLCS (childstate_t *)NULL

static childstate_t *server_states = NULLCS, *client_states = NULLCS;
static int server_nstate = 0, client_nstate = 0;

/*  */

int checktransition(old, new)
    state_t old, new;
{

    tprintf("checktransitions(%s, %s)\n", state2str(old), state2str(new));

#ifdef NO_CHK
    return OK;
    /* NOTREACHED */
#endif    

    if (new == old)
	return OK;

    if (new == DEAD)
	return OK;
    
    switch (old) {
    case UNKNOWN:
	if (new == IDLE || new == INIT || new == SETUP)
	    return OK;
	else return NOTOK;
	
    case IDLE:
	if (new == ABORTING || new == WAIT_INIT || new == INIT)
	    return OK;
	else return NOTOK;

    case CREATING:
	if (new == ABORTING || new == SETUP)
	    return OK;
	else return NOTOK;
	
    case SETUP:
	if (new == ABORTING || new == WAIT_INIT || new == INIT || new == IDLE)
	    return OK;
	else return NOTOK;
	
    case WAIT_INIT:
	if (new == ABORTING || new == INIT || new == IDLE)
	    return OK;
	else return NOTOK;
	

    case INIT:
	if (new == ABORTING || new == WAIT_STARTED || new == TAREDOWN
	    || new == RESETTING || new == EXECUTING || new == IDLE
	    || new == UNKNOWN || new == DEAD || new == SETUP)
	    return OK;
	else return NOTOK;
	
    case WAIT_STARTED:
	if (new == ABORTING || new == EXECUTING)
	    return OK;
	else return NOTOK;
	
    case EXECUTING:
	if (new == ABORTING || new == WAIT_STOPPED || new == WAIT_TERM
	    || new == FAILING || new == INIT || new == IDLE)
	    return OK;
	else return NOTOK;
	
    case WAIT_STOPPED:
	if (new == ABORTING || new == INIT)
	    return OK;
	else return NOTOK;
	
    
    case WAIT_TERM:
	if (new == ABORTING || new == INIT)
	    return OK;
	else return NOTOK;
	

    case TAREDOWN:
	if (new == ABORTING || new == IDLE || new == UNKNOWN || new == DEAD)
	    return OK;
	else return NOTOK;
	
    case RESETTING:
	if (new == ABORTING || new == IDLE)
	    return OK;
	else return NOTOK;
	
    case FAILING:
	if (new == ABORTING || new == INIT)
	    return OK;
	else return NOTOK;
	
    case ABORTING:
	if (new == IDLE)
	    return OK;
	else return NOTOK;

    case DEAD:
	return OK;

    default:
	dprintf(EF_IN3, INTERNAL, "unknown state", "checktransition");
	return NOTOK;
    }
    /* NOTREACHED */
} /* checktransition */

/*  */

int parentstate2timeout(state)
    state_t state;
{

    switch (state) {
    case EXECUTING:
	return NOTOK;	/* infinite timeout */

    default:
	return 30;
    }
    /* NOTREACHED */
} /* parentstate2timeout */

/*  */

int childstate2timeout(state)
    state_t state;
{

    switch (state) {
    case EXECUTING:
#ifdef notdef
	return 0;	/* just do a poll - we have been
			   interupted by a message */
#endif notdef
    case IDLE:
	return NOTOK;	/* infinity */
	
    default:
	return 10;
    }
    /* NOTREACHED */
} /* childstate2timeout */

/*  */

char *state2str(state)
    state_t state;
{
    char buf[100];
    
    switch (state) {
    case UNKNOWN:
	return "UNKNOWN";
    case IDLE:
	return "IDLE";
    case CREATING:
	return "CREATING";
    case SETUP:
	return "SETUP";
    case WAIT_INIT:
	return "WAIT_INIT";

    case INIT:
	return "INIT";
    case WAIT_STARTED:
	return "WAIT_STARTED";
    case EXECUTING:
	return "EXECUTING";
    case WAIT_STOPPED:
	return "WAIT_STOPPED";
    
    case WAIT_TERM:
	return "WAIT_TERM";

    case TAREDOWN:
	return "TAREDOWN";
    case RESETTING:
	return "RESETTING";
    case FAILING:
	return "FAILING";
    case ABORTING:
	return "ABORTING";

    case DEAD:
	return "DEAD";
	
    default:
	sprintf(buf, "unknown (%d)", state);
	return buf;
    }
} /* state2str */

/*  */

int alloc_childstate(nproc)
    int nproc;
{
    childstate_t *tmp;
    int i;
    
    tprintf("alloc_childstate(%d)\n", nproc);
    dprintf("alloc_childstate: clients %d 0x%x, servers %d 0x%x\n",
	    client_nstate, client_states, server_nstate, server_states);
    
    if (nproc > client_nstate) {
	tmp = (childstate_t *)malloc(nproc*sizeof(childstate_t));
	if (tmp == NULLCS) {
	    eprintf(EF_IN3, RESOURCE, "Out of memory", "alloc_childstate");
	    return NOTOK;
	}
	for (i = 0; i < client_nstate; i++) {
	    tmp[i] = client_states[i];
	}
	for (i = client_nstate; i < nproc; i++) {
	    tmp[i].cs_state = DEAD;
	    tmp[i].cs_background = 0;
	    tmp[i].cs_result = NULL;
	    tmp[i].cs_addr = NULL;
	}
	if (client_states != NULLCS)
	    free((char *)client_states);
	client_nstate = nproc;
	client_states = tmp;
    }
    if (nproc > server_nstate) {
	tmp = (childstate_t *)malloc(nproc*sizeof(childstate_t));
	if (tmp == NULLCS) {
	    eprintf(EF_IN3, RESOURCE, "Out of memory", "alloc_childstate");
	    return NOTOK;
	}
	for (i = 0; i < server_nstate; i++) {
	    tmp[i] = server_states[i];
	}
	for (i = server_nstate; i < nproc; i++) {
	    tmp[i].cs_state = DEAD;
	    tmp[i].cs_background = 0;
	    tmp[i].cs_result = NULL;
	    tmp[i].cs_addr = NULL;
	}
	if (server_states != NULLCS)
	    free((char *)server_states);
	server_nstate = nproc;
	server_states = tmp;
    }

    /* reset state if necessery */
    for (i = 0; i < client_nstate; i++) {
	if (client_states[i].cs_state != DEAD) {
	    stprintf("Resetting state for BA_CLIENT:%d\n", i);
	    client_states[i].cs_state = DEAD;
	}
    }
    for (i = 0; i < server_nstate; i++) {
	if (server_states[i].cs_state != DEAD) {
	    stprintf("Resetting state for BA_SERVER:%d\n", i);
	    server_states[i].cs_state = DEAD;
	}
    }
    tprintf("alloc_childstate - done\n");
    return OK;
} /* alloc_childstate */

/*  */

childstate_free()
{
    int i;
    
    for (i = 0; i < server_nstate; i++) {
	if (server_states[i].cs_addr != NULL) 
	    address_free(server_states[i].cs_addr);
	if (server_states[i].cs_result != NULL) 
	    statistics_free(server_states[i].cs_result);
    }
    if (server_nstate != 0)
	free((char *)server_states);
    server_nstate = 0;

    for (i = 0; i < client_nstate; i++) {
	if (client_states[i].cs_addr != NULL) 
	    address_free(client_states[i].cs_addr);
	if (client_states[i].cs_result != NULL) 
	    statistics_free(client_states[i].cs_result);
    }
    if (client_nstate != 0)
	free((char *)client_states);
    client_nstate = 0;
} /* childstate_free */

/*  */

int set_state(addr, state)
    bcpaddr_t addr;
    state_t state;
{
    tprintf("set_state(%s, %s)\n", bcpaddr2str(addr),
	    state2str(state));

    stprintf("Setting state for %s to %s\n", bcpaddr2str(addr),
	    state2str(state));

    if ((addr.ba_addr & BA_CLIENT)
	&& addr.ba_index >= client_nstate) {
	eprintf(EF_IN3, INTERNAL, "ba_index too large", "set_state");
	dprintf("\tba_index %d, client_nstate %d\n",
		addr.ba_index, client_nstate);
	return NOTOK;
    }
    if ((addr.ba_addr & BA_SERVER)
	&& addr.ba_index >= server_nstate) {
	eprintf(EF_IN3, INTERNAL, "ba_index too large", "set_state");
	dprintf("\tba_index %d, server_nstate %d\n",
		addr.ba_index, server_nstate);
	return NOTOK;
    }

    if (addr.ba_addr & (BA_CLIENT | BA_CLIENTS)) {
	if (set_client_state(addr, state) == NOTOK)
	    return NOTOK;
    }
    if (addr.ba_addr & (BA_SERVER | BA_SERVERS)) {
	if (set_server_state(addr, state) == NOTOK)
	    return NOTOK;
    }
    return OK;
} /* set_state */

/*  */

static int set_client_state(addr, state)
    bcpaddr_t addr;
    state_t state;
{
    int start, stop, i;
    
    if (addr.ba_addr & BA_CLIENTS) {
	start = 0;
	stop = client_nstate;
    } else {
	start = addr.ba_index;
	stop = start+1;
    }
    for (i = start; i < stop; i++) {
	if (checktransition(client_states[i].cs_state, state) == NOTOK) {
	    eprintf(EF_IN3, PROTOCOL, "Bad state transition attempted",
		    "set_client_state");
	    eprintf("\t%s => %s\n",
		    state2str(client_states[i].cs_state), state2str(state));
	    return NOTOK;
	}
	client_states[i].cs_state = state;
    }
    return OK;
} /* set_client_state */

/*  */

int set_fg_client_state(state)
    state_t state;
{
    bcpaddr_t addr;
    int i;

    tprintf("set_fg_client_state(%s)\n", state2str(state));

    stprintf("Setting the state for all foreground clients to %s\n",
	     state2str(state));

    addr.ba_addr = BA_CLIENT;
    for (i = 0; i < client_nstate; i++) {
	if (!client_states[i].cs_background) {
	    addr.ba_index = i;
	    if (set_client_state(addr, state) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_fg_client_state",
			"set_fg_client_state");
		eprintf("\t%s -> %s failed!\n", bcpaddr2str(addr),
			state2str(state));
		return NOTOK;
	    }
	}
    }
    return OK;
} /* set_fg_client_state */

 /*  */

int set_bg_client_state(state)
    state_t state;
{
    bcpaddr_t addr;
    int i;

    tprintf("set_bg_client_state(%s)\n", state2str(state));

    stprintf("Setting the state for all background clients to %s\n",
	     state2str(state));

    addr.ba_addr = BA_CLIENT;
    for (i = 0; i < client_nstate; i++) {
	if (client_states[i].cs_background) {
	    addr.ba_index = i;
	    if (set_client_state(addr, state) == NOTOK) {
		eprintf(EF_IN3, INTERNAL, "set_bg_client_state",
			"set_bg_client_state");
		eprintf("\t%s -> %s failed!\n", bcpaddr2str(addr),
			state2str(state));
		return NOTOK;
	    }
	}
    }
    return OK;
} /* set_bg_client_state */

 /*  */

static int set_server_state(addr, state)
    bcpaddr_t addr;
    state_t state;
{
    int start, stop, i;
    
    if (addr.ba_addr & BA_SERVERS) {
	start = 0;
	stop = server_nstate;
    } else {
	start = addr.ba_index;
	stop = start+1;
    }
    for (i = start; i < stop; i++) {
	if (checktransition(server_states[i].cs_state, state) == NOTOK) {
	    eprintf(EF_IN3, PROTOCOL, "Bad state transition attempted",
		    "set_server_state");
	    eprintf("\t%s => %s\n",
		    state2str(server_states[i].cs_state),
		    state2str(state));
	    return NOTOK;
	}
	server_states[i].cs_state = state;
    }
    return OK;
} /* set_server_state */

/*  */

int check_state(addr, state)
    bcpaddr_t addr;
    state_t state;
{
    tprintf("check_state(%s, %s)\n", bcpaddr2str(addr),
	    state2str(state));

    stprintf("Checking if the state for %s is %s\n", bcpaddr2str(addr),
	    state2str(state));

    if ((addr.ba_addr & BA_CLIENT)
	&& addr.ba_index >= client_nstate) {
	eprintf(EF_IN3, INTERNAL, "ba_index too large", "check_state");
	dprintf("\tba_index %d, client_nstate %d\n",
		addr.ba_index, client_nstate);
	return NOTOK;
    }
    if ((addr.ba_addr & BA_SERVER)
	&& addr.ba_index >= server_nstate) {
	eprintf(EF_IN3, INTERNAL, "ba_index too large", "check_state");
	dprintf("\tba_index %d, server_nstate %d\n",
		addr.ba_index, server_nstate);
	return NOTOK;
    }

    if (addr.ba_addr & (BA_CLIENT | BA_CLIENTS)) {
	if (check_client_state(addr, state) == NOTOK)
	    return NOTOK;
    }
    if (addr.ba_addr & (BA_SERVER | BA_SERVERS)) {
	if (check_server_state(addr, state) == NOTOK)
	    return NOTOK;
    }
    return OK;
} /* check_state */

/*  */

static int check_client_state(addr, state)
    bcpaddr_t addr;
    state_t state;
{
    register int i;

    addr.ba_addr &= BA_CLIENTS | BA_CLIENT;
    
    tprintf("check_client_state(%s, %s)\n",
	    bcpaddr2str(addr), state2str(state));

    if (addr.ba_addr & BA_CLIENTS) {
	for (i = 0; i < client_nstate; i++)
	    if (client_states[i].cs_state != state) {
		dprintf("check_client_state: %d in %s\n",
			i, state2str(client_states[i].cs_state));
		
		return NOTOK;
	    }
    } else {
	i = addr.ba_index;
	if (client_states[i].cs_state != state) {
	    dprintf("check_client_state: %d in %s\n",
		    i, state2str(client_states[i].cs_state));

	    return NOTOK;
	}
    }
    stprintf("%s is/are in the %s state\n",
	     bcpaddr2str(addr), state2str(state));

    return OK;
} /* check_client_state */

/*  */

static int check_server_state(addr, state)
    bcpaddr_t addr;
    state_t state;
{
    register int i;

    addr.ba_addr &= BA_SERVERS | BA_SERVER;
    
    tprintf("check_server_state(%s, %s)\n",
	    bcpaddr2str(addr), state2str(state));

    if (addr.ba_addr & BA_SERVERS) {
	for (i = 0; i < server_nstate; i++)
	    if (server_states[i].cs_state != state) {
		dprintf("check_server_state: %d in %s\n",
			i, state2str(server_states[i].cs_state));
		
		return NOTOK;
	    }
    } else {
	i = addr.ba_index;
	if (server_states[i].cs_state != state) {
	    dprintf("check_server_state: %d in %s\n",
		    i, state2str(server_states[i].cs_state));

	    return NOTOK;
	}
    }
    stprintf("%s is/are in the %s state\n",
	     bcpaddr2str(addr), state2str(state));

    return OK;
} /* check_server_state */

/*  */

int check_fg_client_state(state)
    state_t state;
{
    int i;

    tprintf("check_fg_client_state(%s)\n", state2str(state));

    for (i = 0; i < client_nstate; i++) {
	if (!client_states[i].cs_background) {
	    if (client_states[i].cs_state != state) {
		dprintf("check_fg_client_state: CLIENT:%d in %s state\n", i,
			state2str(client_states[i].cs_state));
		return NOTOK;
	    }
	}
    }
    stprintf("All foreground clients are in the %s state\n",
	     state2str(state));
    return OK;
} /* check_fg_client_state */

/*  */

int check_bg_client_state(state)
    state_t state;
{
    int i;

    tprintf("check_bg_client_state(%s)\n", state2str(state));

    for (i = 0; i < client_nstate; i++) {
	if (client_states[i].cs_background) {
	    if (client_states[i].cs_state != state) {
		dprintf("check_bg_client_state: CLIENT:%d in %s state\n", i,
			state2str(client_states[i].cs_state));
		return NOTOK;
	    }
	}
    }
    stprintf("All background clients are in the %s state\n",
	     state2str(state));
    return OK;
} /* check_bg_client_state */

/*  */

struct address_t *get_addr(index)
{
    if (index >= server_nstate)
	return NULL;
    return server_states[index].cs_addr;
}

/*  */

int get_results(a_stats)
    statistics_t **a_stats;
{
    statistics_t *stats;
    int nproc = client_nstate;
    int i, j;
    
    tprintf("get_result(0x%x)\n", a_stats);

    stats = (statistics_t *)malloc(sizeof(statistics_t) * nproc);
    if (stats == NULL) {
	eprintf(EF_IN4, INTERNAL, RESOURCE, "malloc() failed", "get_results");
	return NULL;
    }
    for (i = 0, j = 0; j < nproc; j++) {
	if (client_states[j].cs_result != NULL) {
	    stats[i] = *client_states[j].cs_result;
	    i++;
	}
    }
    dprintf("get_result: got %d results of %d\n", i, nproc);

    *a_stats = stats;
    return i;
} /* get_result */

/*  */

get_respresults(a_stats)
    statistics_t **a_stats;
{
    statistics_t *stats;
    int nproc = client_nstate;
    int i, j;
    
    tprintf("get_respresult(0x%x)\n", a_stats);

    stats = (statistics_t *)malloc(sizeof(statistics_t) * nproc);
    if (stats == NULL) {
	eprintf(EF_IN4, INTERNAL, RESOURCE, "malloc() failed", 
		"get_respresults");
	return NOTOK;
    }
    for (i = 0, j = 0; j < nproc; j++) {
	if (server_states[j].cs_result != NULL) {
	    stats[i] = *server_states[j].cs_result;
	    i++;
	}
    }
    dprintf("get_respresult: got %d results of %d\n", i, nproc);

    *a_stats = stats;
    return OK;
} /* get_respresult */

/*  */

int set_result(addr, stat)
    bcpaddr_t addr;
    statistics_t *stat;
{
    int index = addr.ba_index;

    tprintf("set_result(0x%x:%d, 0x%x)\n",
	    addr.ba_addr, addr.ba_index, stat);

    if (index >= client_nstate) {
	eprintf(EF_IN4, INTERNAL, PARAMETER, "address",
		"set_result - index to large");
	return NOTOK;
    }
    
    if (addr.ba_addr == BA_CLIENT) {
        if (client_states[index].cs_result != NULL) {
	    statistics_free(client_states[index].cs_result);
	}
	client_states[index].cs_result = stat;
    } else {
        if (server_states[index].cs_result != NULL) {
	    statistics_free(server_states[index].cs_result);
	}
	server_states[index].cs_result = stat;
    }
    return OK;
} /* set_result */

/*  */

int set_addr(addr, address)
    bcpaddr_t addr;
    struct address_t *address;
{
    int index = addr.ba_index;

    tprintf("set_addr(0x%x:%d, 0x%x)\n",
	    addr.ba_addr, addr.ba_index, address);

    if (!(addr.ba_addr & BA_SERVER)) {
	eprintf(EF_IN4, INTERNAL, PARAMETER, "address",
		"set_addr - not BA_SERVER");
	return NOTOK;
    }	
    if (index >= server_nstate) {
	eprintf(EF_IN4, INTERNAL, PARAMETER, "address",
		"set_addr - index to large");
	return NOTOK;
    }
    
    if (server_states[index].cs_addr != NULL) {
	address_free(server_states[index].cs_addr);
    }

    server_states[index].cs_addr = address;

    return OK;
} /* set_addr */

/*  */

state_t get_state(addr)
    bcpaddr_t addr;
{
    switch (addr.ba_addr) {
    case BA_CLIENT:
	if (addr.ba_index >= client_nstate) {
	    eprintf(EF_IN3, INTERNAL, "client index out of bounds",
		    "get_state");
	    return UNKNOWN;
	}
	return client_states[addr.ba_index].cs_state;

    case BA_SERVER:
	if (addr.ba_index >= server_nstate) {
	    eprintf(EF_IN3, INTERNAL, "server index out of bounds",
		    "get_state");
	    return UNKNOWN;
	}
	return server_states[addr.ba_index].cs_state;
    }
} /* get_state */

/*  */

int get_statebackground(addr)
    bcpaddr_t addr;
{
    switch (addr.ba_addr) {
    case BA_CLIENT:
	if (addr.ba_index >= client_nstate) {
	    eprintf(EF_IN3, INTERNAL, "client index out of bounds",
		    "get_statebackground");
	    return 0;
	}
	return client_states[addr.ba_index].cs_background;

    case BA_SERVER:
	if (addr.ba_index >= server_nstate) {
	    eprintf(EF_IN3, INTERNAL, "server index out of bounds",
		    "get_statebackground");
	    return 0;
	}
	return server_states[addr.ba_index].cs_background;
    }
} /* get_statebackground */

/*  */

int set_statebackground(addr)
    bcpaddr_t addr;
{
    tprintf("set_statebackground(%s)\n", bcpaddr2str(addr));

    if (addr.ba_addr & BA_CLIENT) {
	if (addr.ba_index < 0 || addr.ba_index >= client_nstate) {
	    eprintf(EF_IN3, INTERNAL, PARAMETER, "set_statebackground");
	    return NOTOK;
	}
	client_states[addr.ba_index].cs_background = 1;
    }
    if (addr.ba_addr & BA_SERVER) {
	if (addr.ba_index < 0 || addr.ba_index >= server_nstate) {
	    eprintf(EF_IN3, INTERNAL, PARAMETER, "set_statebackground");
	    return NOTOK;
	}
	server_states[addr.ba_index].cs_background = 1;
    }
	
    return OK;
} /* set_statebackground */
    

	
