/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The Benchmark Control Protocol - handle routes to the client and server
 * processes
 */

#include <general.h>
#include <ipc.h>


/*
 * Exports:
 *	route_enter(addr, dest: bcpaddr_t) -> OK/NOTOK
 *	route_remove(addr:bcpaddr_t) -> OK/NOTOK
 *	route_remove_dest(addr:bcpaddr_t) -> OK/NOTOK
 *	route_remove_all()
 *	route_lookup(addr: bcpaddr_t; a_dest: bcpaddr_t *) -> OK/NOTOK
 *	route_multicast_lookup(addr: bcpaddr_t; a_ndest: int *;
 *				a_dests: bcpaddr_t **)
 *		Returns malloc'ed data in a_dests
 */

/*  Local data */

#define MAXROUTES	256

struct route {
    bcpaddr_t	rt_addr, rt_dest;
    int 	rt_inuse;
};

static int nroute = 0;
static struct route routes[MAXROUTES];


void routes_print();

/*  */

static int lookup(addr)
    bcpaddr_t addr;
{
    int i;

    for (i = 0; i < nroute; i++)
	if (routes[i].rt_inuse && addrcmp(addr, routes[i].rt_addr) ==0)
	    return i;
    return NOTOK;
} /* lookup */

/*  */

static int lookup_dest(addr)
    bcpaddr_t addr;
{
    int i;

    for (i = 0; i < nroute; i++)
	if (routes[i].rt_inuse && addrcmp(addr, routes[i].rt_dest) ==0)
	    return i;
    return NOTOK;
} /* lookup_dest */

/*  */

int route_enter(addr, dest)
    bcpaddr_t addr, dest;
{
    int i;

    tprintf("route_enter(0x%x:%d, 0x%x:0x%x)\n", addr.ba_addr, addr.ba_index,
	    dest.ba_addr, dest.ba_index);

    /* initialize if necessery */
    if (nroute < MAXROUTES) {
	for (i = nroute; i < MAXROUTES; i++)
	    routes[i].rt_inuse = 0;
	nroute = MAXROUTES;
    }
    
    i = lookup(addr);

    if (i == NOTOK) {
	/* find free slot */
	for (i = 0; i < nroute; i++)
	    if (!routes[i].rt_inuse) {
		routes[i].rt_inuse++;
		routes[i].rt_addr = addr;
		routes[i].rt_dest = dest;
		dprintf("Entered route for %s: %s\n", bcpaddr2str(addr),
			bcpaddr2str(dest));
		if (Debug)
		    routes_print();
		return OK;
	    }
	eprintf(EF_IN3, INTERNAL, RESOURCE, "routing table");
	eprintf("\tAll %d slots being used\n", MAXROUTES);
	return NOTOK;
    }
    routes[i].rt_dest = dest;
    dprintf("Entered (replaced) route for %s: %s\n", bcpaddr2str(addr),
	    bcpaddr2str(dest));
    if (Debug)
	routes_print();
    
    return OK;
} /* route_enter */

/*  */

int route_remove(addr)
    bcpaddr_t addr;
{
    int i;

    tprintf("route_remove(0x%x:%d)\n", addr.ba_addr, addr.ba_index);

    i = lookup(addr);

    if (i == NOTOK) {
	dprintf("route_remove: not here\n");
	return NOTOK;
    }
    routes[i].rt_inuse = 0;
    return OK;
} /* route_remove */

/*  */

int route_remove_dest(addr)
    bcpaddr_t addr;
{
    int i;

    tprintf("route_remove_dest(0x%x:0x%x)\n", addr.ba_addr, addr.ba_index);

    if (addr.ba_addr != BA_SYSTEM) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "route_remove_dest",
		bcpaddr2str(addr));
	return NOTOK;
    }
    i = lookup_dest(addr);

    if (i == NOTOK) {
	dprintf("route_remove_dest: not here\n");
	return NOTOK;
    }
    routes[i].rt_inuse = 0;
    while ((i = lookup_dest(addr)) != NOTOK)
        routes[i].rt_inuse = 0;
	
    return OK;
} /* route_remove_dest */

/*  */

int route_remove_all()
{
    int i;

    tprintf("route_remove_all()\n");

    for (i = 0; i < nroute; i++)
	routes[i].rt_inuse = 0;
    return OK;
} /* route_remove_all */

/*  */

int route_lookup(addr, a_dest)
    bcpaddr_t addr, *a_dest;
{
    int i;

    tprintf("route_lookup(0x%x:%d, 0x%x)\n", addr.ba_addr, addr.ba_index,
	    a_dest);

    i = lookup(addr);

    if (i == NOTOK) {
	dprintf("route_lookup: not found\n");
	return NOTOK;
    }
    *a_dest = routes[i].rt_dest;
    return OK;
} /* route_lookup */

/*  */

int route_multicast_lookup(select, a_ndest, a_dests)
    bcpaddr_t select, **a_dests;
    int *a_ndest;
{
    int i, j, ndest = 0;
    static bcpaddr_t buf[NOFILE];	/* NOFILE == MAXHOSTS */
    bcpaddr_t *dests;
    
    tprintf("route_multicast_lookup(0x%x:%d, 0x%x, 0x%x)\n",
	    select.ba_addr, select.ba_index,
	    a_ndest, a_dests);

    if (Debug)
	routes_print();
    
    for (i = 0; i < nroute; i++) {
	if (routes[i].rt_inuse &&
	    routes[i].rt_addr.ba_addr & select.ba_addr) {
	    for (j = 0; j < ndest; j++) {
		if (addrcmp(routes[i].rt_dest, buf[j]) == 0) {
		    dprintf("\tSkipping duplicate route %s\n",
			    bcpaddr2str(routes[i].rt_dest));
		    break;
		}
	    }
	    if (j == ndest)
		buf[ndest++] = routes[i].rt_dest;
	}
    }
    dests = (bcpaddr_t *)malloc(sizeof(bcpaddr_t)*ndest);
    if (dests == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "route_multicast_lookup");
	*a_ndest = 0;
	return NOTOK;
    }
    bcopy(buf, dests, sizeof(bcpaddr_t)*ndest);

    dprintf("route_multicast_lookup: found %d destinations\n", ndest);
    *a_ndest = ndest;
    *a_dests = dests;
    return OK;
} /* route_multicast_lookup */

/*  */

void routes_print()
{
    int i;

    printf("Routing table:\n");
    for (i = 0; i < nroute; i++) {
	if (routes[i].rt_inuse) {
	    printf("\t%d: dest %s addr %s\n",
		   i, bcpaddr2str(routes[i].rt_dest),
		   bcpaddr2str(routes[i].rt_addr));
	}
    }
} /* routes_print */


