/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The Benchmark Control Protocol - pseudo transport using a multitude of
 * underlying communication mechanisms.
 */

#include <general.h>
#include <ipc.h>

/*
 * Exports:
 *	bcp_init(myaddr: bcpaddr_t) -> OK/NOTOK
 * 	send_msg(msg: *msg_t) -> OK / NOTOK 		(Sets msg_from field)
 * 	bcp_send_msg(msg: *msg_t) -> OK / NOTOK
 *	await_recv_msg(msg: *msg_t; secs: int) -> OK/NOTOK 	(From BA_ANY)
 *	bcp_await_recv_msg(from: bcpaddr_t; msg: *msg_t; secs: int)
 *		-> OK/NOTOK
 *
 *	printmyname()		Prints the address on stdout
 *	printmynameerr()	Prints the address on stderr
 *	reset_myaddr()
 *	bcpaddr_t get_myaddr()
 */

/*  Implementation dependent #include's */

#include <ipc_sig.h>

/*  Imports */

#ifdef BSD
extern int h_errno;
#endif BSD

/* Forwards */

static void free_msg();
static void check_closed();

/*  Module local data */

static bcpaddr_t myaddr = { NOTOK, 0 };

/* Imported from ipc_system.c */

extern int 		system_server_nproc;
extern commstate_t 	*system_server_procs;
extern int 		system_client_nproc;
extern commstate_t 	*system_client_procs;


/*  */

int bcp_init(addr)
    bcpaddr_t addr;
{
    bcpaddr_t dummy;
    
    tprintf("bcp_init(0x%x, %d)\n", addr.ba_addr, addr.ba_index);

    switch (addr.ba_addr) {
    case BA_BENCH:
	host_close_all();
	/* set signal handling */
	setup_bench_sighdl();
	break;

    case BA_SYSTEM:
	if (transport_server_init() == NOTOK) {
	    dprintf(EF_IN3, COMMUNICATION,
		    "Can't initialize the transport server", "bcp_init");
	    return NOTOK;
	}
	/* set signal handling */
	setup_demon_sighdl();
	break;

    case BA_CLIENT:
    case BA_SERVER:
	/*
	 * Avoid inheriting messages when being forked
	 */
	(void)local_clear();
	
	(void)host_close_all();
	(void)transport_server_close();

	if (addr.ba_index >= ((addr.ba_addr == BA_CLIENT) ?
	    			system_client_nproc : system_server_nproc)) {
	    dprintf(EF_IN4, INTERNAL, PARAMETER, "ba_index is too large",
		    "bcp_init");
	    return NOTOK;
	}
	/* setup signal handlers */
	setup_child_sighdl();

	break;

    default:
	dprintf(EF_IN4, INTERNAL, PARAMETER, "addr", "bcp_init");
	return NOTOK;
    }
    myaddr = addr;
    return OK;
} /* bcp_init */

/*  */

int close_readends()
{
    int i;
    int ret = OK;

    tprintf("close_readends()\n");
    
    for (i = 0; i < system_client_nproc; i++) {
	if (system_client_procs[i].c_read != NOTOK) {
	    if (close(system_client_procs[i].c_read) == NOTOK)
		ret = NOTOK;
	    system_client_procs[i].c_read = NOTOK;
	}
    }
    for (i = 0; i < system_server_nproc; i++) {
	if (system_server_procs[i].c_read != NOTOK) {
	    if (close(system_server_procs[i].c_read) == NOTOK)
		ret = NOTOK;
	    system_server_procs[i].c_read = NOTOK;
	}
    }
    return ret;
} /* close_readends */

    
/*  */

int send_msg(msg)
    msg_t *msg;
{
    msg->msg_from = myaddr;
    return send_msg_flg(msg, 0);
} /* send_msg */

int bcp_send_msg(msg)
    msg_t *msg;
{
    return send_msg_flg(msg, 0);
} /* bcp_send_msg */

/*  */
 
#define SF_FORWARDED	0x1

static int send_msg_flg(msg, flags)
    msg_t *msg;
    int flags;
{
    bcpaddr_t dest, to;
    
    tprintf("bcp_send_msg(0x%x, %d)\n", msg, flags);

    iprintf("Sending message code %s from %s to %s\n\tlength %d, <%s>\n",
	    code2str(msg->msg_code),
	    bcpaddr2str(msg->msg_from),
	    bcpaddr2str(msg->msg_to),
	    msg->msg_datalen,
	   ( msg->msg_datalen == 0) ? "" : msg->msg_data);


    dest = msg->msg_to;
    switch (myaddr.ba_addr) {
    case BA_BENCH:
	if (dest.ba_addr & BA_BENCH) {
	    if (enqueue_local(msg) == NOTOK) {
		eprintf(EF_IN4, INTERNAL, PROTOCOL,
			"Failed delivering a message to local",
			"bcp_send_msg");
	    }
	    dest.ba_addr &= ~myaddr.ba_addr;
	}

	if (dest.ba_addr & BA_CLIENT) {
	    to.ba_addr = BA_CLIENT;
	    to.ba_index = dest.ba_index;
	    if (send_with_routing(to, msg, flags) == NOTOK) {
		eprintf(EF_IN3, PROTOCOL,
			"send_with_routing", "bcp_send_msg");
	    }
	}
	if (dest.ba_addr & BA_SERVER) {
	    to.ba_addr = BA_SERVER;
	    to.ba_index = dest.ba_index;
	    if (send_with_routing(to, msg, flags) == NOTOK) {
		eprintf(EF_IN3, PROTOCOL,
			"send_with_routing", "bcp_send_msg");
	    }
	}
	if (dest.ba_addr & BA_SYSTEM) {
	    to.ba_addr = BA_SYSTEM;
	    to.ba_index = dest.ba_index;
	    if (send_with_routing(to, msg, flags) == NOTOK) {
		eprintf(EF_IN3, PROTOCOL,
			"send_with_routing", "bcp_send_msg");
	    }
	}
	if (dest.ba_addr & (BA_CLIENTS | BA_SERVERS | BA_SYSTEMS)) {
	    to.ba_addr = dest.ba_addr & (BA_CLIENTS | BA_SERVERS | BA_SYSTEMS);
	    to.ba_index = 0;
	    if (send_with_multi_routing(to, msg, flags) == NOTOK) {
		eprintf(EF_IN3, PROTOCOL,
			"send_with_multi_routing", "bcp_send_msg");
	    }
	}
	break;
	
    case BA_SYSTEM:
	if (dest.ba_addr & BA_SYSTEM 
#ifndef MULTIPLE_IPADDR
	    && dest.ba_index == myaddr.ba_index
#endif
	    && SYSTEM_CODE(msg->msg_code)) {
	    (void)handle_system_msg(msg);
	    dest.ba_addr &= ~BA_SYSTEM;
	}
	
	if (local_addr(dest)) {
	    if (MSG_ASYNC(msg->msg_code)) {
		if (sig_send_msg(msg) == NOTOK) 
		    eprintf(EF_IN4, INTERNAL, PROTOCOL,
			    "Failed delivering a message using sig",
			    "bcp_send_msg");
	    } else {
		if (dest.ba_addr & BA_CLIENTS) {
		    int i;
		    for (i = system_client_nproc-1; i >= 0; i--)
			if (fd_send_msg(&system_client_procs[i].c_write, msg) ==
			    NOTOK) {
			    eprintf(EF_IN3, PROTOCOL,
				    "Failed delivering msg using c_write",
				    "bcp_send_msg");
			}
		} else if (dest.ba_addr & BA_CLIENT) {
		    if (dest.ba_index >= system_client_nproc
			&& system_client_nproc > 0) {
			eprintf(EF_IN4, INTERNAL, PROTOCOL,
				"Too large ba_index",
				"bcp_send_msg");
			return NOTOK;
		    }
		    if (fd_send_msg(&system_client_procs[dest.ba_index].c_write,
				    msg) == NOTOK) {
			eprintf(EF_IN3, PROTOCOL,
				"Failed delivering msg using c_write",
				"bcp_send_msg");
		    }
		}
		
		if (dest.ba_addr & BA_SERVERS) {
		    int i;
		    for (i = system_server_nproc-1; i >= 0; i--)
			if (fd_send_msg(&system_server_procs[i].c_write, msg) ==
			    NOTOK) {
			    eprintf(EF_IN3, PROTOCOL,
				    "Failed delivering msg using c_write",
				    "bcp_send_msg");
			}
		} else if (dest.ba_addr & BA_SERVER) {
		    if (dest.ba_index >= system_server_nproc
			&& system_server_nproc > 0) {
			eprintf(EF_IN4, INTERNAL, PROTOCOL,
				"Too large ba_index",
				"bcp_send_msg");
			return NOTOK;
		    }
		    if (fd_send_msg(&system_server_procs[dest.ba_index].c_write,
				    msg) == NOTOK) {
			eprintf(EF_IN3, PROTOCOL,
				"Failed delivering msg using c_write",
				"bcp_send_msg");
		    }
		}
		
	    }
	}
	if (remote_addr(dest) &&
	    !(flags & SF_FORWARDED && remote_addr(msg->msg_from))) {
	    /* forward via BENCH */
	    if (transport_send_msg(msg) == NOTOK) {
		eprintf(EF_IN3, PROTOCOL,
			"Failed delivering msg using transport",
			"bcp_send_msg");
	    }
	}

	break;
	
    case BA_CLIENT:
    case BA_SERVER:
	if ((dest.ba_addr & myaddr.ba_addr)
	    && dest.ba_index == myaddr.ba_index) {
	    if (enqueue_local(msg) == NOTOK) {
		eprintf(EF_IN4, INTERNAL, PROTOCOL,
			"Failed delivering a message to local",
			"bcp_send_msg");
	    }
	    dest.ba_addr &= ~myaddr.ba_addr;
	    break;
	} 
	
	/* forward via SYSTEM */
	if (dest.ba_addr) {
	    if (myaddr.ba_addr & BA_CLIENT) {
		if (fd_send_msg(&system_client_procs[myaddr.ba_index].c_write,
				msg) == NOTOK) {
		    eprintf(EF_IN3, PROTOCOL,
			    "Failed delivering a message using a c_write",
			    "bcp_send_msg");
		}
	    } else {
		if (fd_send_msg(&system_server_procs[myaddr.ba_index].c_write,
				msg) == NOTOK) {
		    eprintf(EF_IN3, PROTOCOL,
			    "Failed delivering a message using a c_write",
			    "bcp_send_msg");
		}
	    }
	}
	break;
	
    default:
	eprintf(EF_IN4, INTERNAL, PROTOCOL, "Bad myaddr", "bcp_send_msg");
	return NOTOK;
    }


    return OK;
} /* bcp_send_msg */

/*  */

static int send_with_multi_routing(to, msg, flags)
    bcpaddr_t to;
    msg_t *msg;
    int flags;
{
    bcpaddr_t dest, *dests, from;
    int ndest, i;
    int *a_sock;
    
    tprintf("send_with_multi_routing{0x%x:%d, 0x%x, %d)\n", to.ba_addr, to.ba_index,
	    msg, flags);

    dest.ba_addr = 0; dest.ba_index = 0;
    if (to.ba_addr & BA_CLIENTS)
	dest.ba_addr |= BA_CLIENT;
    if (to.ba_addr & BA_SERVERS)
	dest.ba_addr |= BA_SERVER;
    if (to.ba_addr & BA_SYSTEMS)
	dest.ba_addr |= BA_CLIENT | BA_SERVER;

    route_multicast_lookup(dest, &ndest, &dests);

    if (flags & SF_FORWARDED) {
	/* Make sure not to send it back to the sender! */
	if (route_lookup(msg->msg_from, &from) == NOTOK) {
	    eprintf(EF_IN4, INTERNAL, IPC, "route_lookup",
		    "send_with_multi_routing - No route to sender!");
	    return NOTOK;
	}
    }
    for (i = 0; i < ndest; i++) {
	if (flags & SF_FORWARDED && addrcmp(from, dests[i]) == 0) {
	    dprintf("Not sending out on incomming connection ...\n");
	    continue;
	}
	
	a_sock = host_lookup(dests[i]);
	if (a_sock == NULL) {
	    eprintf(EF_IN4, INTERNAL, IPC, "host_lookup",
		    "send_with_multi_routing");
	    dprintf("\Failed for 0x%x:0x%x\n", dests[i].ba_addr,
		    dests[i].ba_index);
	    free((char *)dests);
	    return NOTOK;
	}
	if (fd_send_msg(a_sock, msg) == NOTOK) {
	    eprintf(EF_IN4, INTERNAL, COMMUNICATION, "fd_send_msg",
		    "send_with_multi_routing");
	    free((char *)dests);
	    return NOTOK;
	}
    }
    free((char *)dests);

    return OK;
} /* send_with_multi_routing */

/*  */
    
static int send_with_routing(to, msg, flags)
    bcpaddr_t to;
    msg_t *msg;
    int flags;
{
    bcpaddr_t dest;
    int *a_sock;

    switch(to.ba_addr) {
	
    case BA_SYSTEM:
	dest = to;
	break;

    case BA_CLIENT:
    case BA_SERVER:
	if (route_lookup(to, &dest) == NOTOK) {
	    eprintf(EF_IN4, INTERNAL, IPC, "route_lookup",
		    "send_with_routing - No route to singleton!");
	    return NOTOK;
	}
	break;

    default:
	eprintf(EF_IN4, INTERNAL, PARAMETER, "to",
		"send_with_routing");
	return NOTOK;
    }
    a_sock = host_lookup(dest);
    if (a_sock == NULL) {
	eprintf(EF_IN4, INTERNAL, IPC, "host_lookup",
		"send_with_routing");
	dprintf("\Failed for 0x%x:0x%x\n", dest.ba_addr,
		dest.ba_index);
	return NOTOK;
    }
    if (fd_send_msg(a_sock, msg) == NOTOK) {
	eprintf(EF_IN4, INTERNAL, COMMUNICATION, "fd_send_msg",
		"send_with_routing");
	return NOTOK;
    }

    return OK;
} /* send_with_routing */

/*  */
	 
int await_recv_msg(msg, secs)
    msg_t *msg;
    int secs;
{
    bcpaddr_t addr;

    addr.ba_addr = BA_STAR;
    addr.ba_index = 0;
    return bcp_await_recv_msg(addr, msg, secs);
    
} /* await_recv_msg */

/*  */

int bcp_await_recv_msg(from, msg, secs)
    bcpaddr_t from;	/* from parameter is not used */
    msg_t *msg;
    int secs;
{
    bcpaddr_t dest;
    struct timeval tvs, *tv;
    int readfs, hostfs, transportfs;	/* file descriptor maks */
    int nfds, nread, i, nfd;
    
    tprintf("bcp_await_recv_msg(0x%x:%d, 0x%x, %d)\n",
	    from.ba_addr, from.ba_index, msg, secs);

    if (secs == NOTOK)
	tv = NULL;
    else {
	tv = &tvs;
	tv->tv_sec = secs;
	tv->tv_usec = 0;
    }

 retry:
    
    /* create a select mask to use */
    readfs = 0; hostfs = 0; transportfs = 0;
    nfds = 0;

    if (myaddr.ba_addr == BA_SYSTEM) {
	/*
	 * Don't use select for accepting the connection.
	 * Signals will cause accept() to be interupted - pick up the
	 * corresponding message from BA_SYSTEM:0.
	 */
	if (transport_server_listen() == NOTOK) {
	    eprintf(EF_IN3, COMMUNICATION, "transport_server_listen",
		    "bcp_await_recv_msg");
	    return NOTOK;
	}
    }
    
    switch (myaddr.ba_addr) {
    case BA_BENCH:
	host_mask(&hostfs, &nfd);
	readfs |= hostfs;
	if (nfd > nfds)
	    nfds = nfd;

	/* fall thru ... */

    case BA_SYSTEM:
	transport_mask(&transportfs, &nfd);
	readfs |= transportfs;
	if (nfd >= nfds)
	    nfds = nfd;
	
	for (i = 0; i < system_client_nproc; i++) {
	    if (system_client_procs[i].c_read != NOTOK) {
		readfs |= 1 << system_client_procs[i].c_read;
		if (system_client_procs[i].c_read >= nfds)
		    nfds = system_client_procs[i].c_read + 1;
	    } else {
		dprintf("bcp_recv_msg: NOTOK fid for index %d\n", i);
	    }
	}
	for (i = 0; i < system_server_nproc; i++) {
	    if (system_server_procs[i].c_read != NOTOK) {
		readfs |= 1 << system_server_procs[i].c_read;
		if (system_server_procs[i].c_read >= nfds)
		    nfds = system_server_procs[i].c_read + 1;
	    } else {
		dprintf("bcp_recv_msg: NOTOK fid for index %d\n", i);
	    }
	}
	
	break;
	
    case BA_CLIENT:
	if (system_client_procs[myaddr.ba_index].c_read != NOTOK) {
	    readfs |= 1 << system_client_procs[myaddr.ba_index].c_read;
	    if (system_client_procs[myaddr.ba_index].c_read >= nfds)
		nfds = 1 + system_client_procs[myaddr.ba_index].c_read;
	} else {
	    dprintf("bcp_recv_msg (SP): NOTOK fid for index %d\n",
		    myaddr.ba_index);
	}
	break;
	
    case BA_SERVER:
	if (system_server_procs[myaddr.ba_index].c_read != NOTOK) {
	    readfs |= 1 << system_server_procs[myaddr.ba_index].c_read;
	    if (system_server_procs[myaddr.ba_index].c_read >= nfds)
		nfds = 1 + system_server_procs[myaddr.ba_index].c_read;
	} else {
	    dprintf("bcp_recv_msg (SP): NOTOK fid for index %d\n",
		    myaddr.ba_index);
	}
	break;
    }
    
    if (local_nummsg() > 0)
	goto received;

    dprintf("select call mask 0x%x, nfds %d\n", readfs, nfds);
    
    nread = select(nfds, &readfs, NULL, NULL, tv);
    if (nread == NOTOK && errno != EINTR) {
	fprintf(stderr, EF_SYSCALL, INTERNAL, "select", "bcp_await_recv_msg",
		getsyserr());
	return NOTOK;
    }
    
    dprintf("select num %d, returned mask 0x%x\n", nread, readfs);
    
    if (nread == 0) { 	/* timed out */
	msg->msg_code = MSG_TIMEOUT;
	msg->msg_datalen = 0;
	return OK;
    }

 received:
    if (local_nummsg() > 0) {
	if (dequeue_local(msg) == NOTOK) {
	    eprintf(EF_IN3, INTERNAL, PROTOCOL, "dequeue_local",
		    "bcp_await_recv_msg");
	} else 
	    goto got_one;
    }

    if (readfs & hostfs) {
	int *a_sock;
	bcpaddr_t from;
	
	from = host_mask2addr(readfs & hostfs);
	a_sock = host_lookup(from);
	if (a_sock == NULL) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "host_mask2sock",
		    "bcp_await_recv");
	} else if (fd_recv_msg(a_sock, msg) == NOTOK) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "fd_recv_msg on socket",
		    "bcp_await_recv");
	} else {
	    check_closed(msg, from.ba_addr, from.ba_index);
	    goto got_one;
	}
    }
    if (readfs & transportfs) {
	if (transport_recv_msg(transportfs, msg) == NOTOK) {
	    eprintf(EF_IN4, INTERNAL, PROTOCOL, "transport_recv_msg",
		    "bcp_await_recv");
	} else {
	    check_closed(msg, BA_BENCH, 0L);
	    goto got_one;
	}
    }


    for (i = 0; i < system_client_nproc; i++) {
	if (system_client_procs[i].c_read != NOTOK) {
	    if (readfs & (1 << system_client_procs[i].c_read))
		if (fd_recv_msg(&system_client_procs[i].c_read, msg) == NOTOK) {
		    eprintf(EF_IN4, INTERNAL, PROTOCOL,
			    "fd_recv_msg on c_read",
			    "bcp_await_recv");
		} else {
		    if (myaddr.ba_addr & BA_SYSTEM)
			check_closed(msg, BA_CLIENT, (long)i);
		    else
			check_closed(msg, BA_SYSTEM, 0L);
		    goto got_one;
		}
	}
    }
    for (i = 0; i < system_server_nproc; i++) {
	if (system_server_procs[i].c_read != NOTOK) {
	    if (readfs & (1 << system_server_procs[i].c_read))
		if (fd_recv_msg(&system_server_procs[i].c_read, msg) == NOTOK) {
		    eprintf(EF_IN4, INTERNAL, PROTOCOL,
			    "fd_recv_msg on c_read",
			    "bcp_await_recv");
		} else {
		    if (myaddr.ba_addr & BA_SYSTEM)
			check_closed(msg, BA_SERVER, (long)i);
		    else
			check_closed(msg, BA_SYSTEM, 0L);
		    goto got_one;
		}
	}
    }

    goto retry;
    
 got_one:
    iprintf("Received message code %s from %s to %s\n\tlength %d, <%s>\n",
	    code2str(msg->msg_code),
	    bcpaddr2str(msg->msg_from),
	    bcpaddr2str(msg->msg_to),
    	    msg->msg_datalen,
	   ( msg->msg_datalen == 0) ? "" : msg->msg_data);


    /* forward if necessery */
    if (myaddr.ba_addr & BA_FORWARDER) {
	msg_t fmsgs, *fmsg = &fmsgs;

	*fmsg = *msg;
	fmsg->msg_to.ba_addr &= ~(myaddr.ba_addr | fmsg->msg_from.ba_addr);

	if (fmsg->msg_to.ba_addr) {
	    iprintf("Forwarding a message to %s\n",
		    bcpaddr2str(fmsg->msg_to));
	    
	    if (send_msg_flg(fmsg, SF_FORWARDED) == NOTOK) {
		eprintf(EF_IN3, PROTOCOL,
			"Failed sending when forwarding a message",
			"bcp_await_recv");
		fmsg->msg_to = msg->msg_from;
		fmsg->msg_from = myaddr;
		fmsg->msg_code = MSG_FAILURE;
		fmsg->msg_datalen = 0;
		(void)bcp_send_msg(fmsg);
	    }
	}
	
	if (myaddr.ba_addr == BA_SYSTEM && SYSTEM_CODE(msg->msg_code)) 
	    if ((msg->msg_to.ba_addr & BA_SYSTEM
#ifndef MULTIPLE_IPADDR
		 && msg->msg_to.ba_index == myaddr.ba_index
#endif
		 )
		|| msg->msg_to.ba_addr & BA_SYSTEMS) {
		(void)handle_system_msg(msg);
		msg->msg_to.ba_addr &= ~(BA_SYSTEM | BA_SYSTEMS);
	    }

	if (!(msg->msg_to.ba_addr & myaddr.ba_addr & BA_BENCH ||
	      (msg->msg_to.ba_addr & myaddr.ba_addr & BA_SYSTEM
#ifndef MULTIPLE_IPADDR
	       && msg->msg_to.ba_index == myaddr.ba_index
#endif
	       ) ||
	      (msg->msg_to.ba_addr & BA_SYSTEMS && myaddr.ba_addr == BA_SYSTEM))) {
	    dprintf("Throwing away the message: only forwarded\n");
	    if (msg->msg_datalen)
		free(msg->msg_data);
	    msg->msg_datalen = 0;
	    goto retry;
	}
    }
    
    return OK;
} /* bcp_await_recv_msg */

/*  */

	
static void free_msg(msg)
    msg_t *msg;
{
    if (msg->msg_datalen && msg->msg_data != NULL)
	free(msg->msg_data);

    msg->msg_datalen = 0;
    msg->msg_data = NULL;
    free((char *)msg);
} /* free_msg */

/*  */

static void check_closed(msg, baddr, bindex)
    msg_t *msg;
    int baddr;
    long bindex;
{
    tprintf("check_closed(0x%x, 0x%x:%d)\n", msg, baddr, bindex);

    if (msg->msg_code == MSG_INTERNAL_CLOSED) {
	msg->msg_code = MSG_CLOSED;
	msg->msg_from.ba_addr = baddr;
	msg->msg_from.ba_index = bindex;
	msg->msg_to = myaddr;
    }
} /* check_closed */

/*  */

char *code2str(code)
    int code;
{
    char buf[100];

    switch (code) {
    case MSG_NOTOK:
	return "MSG_NOTOK";
    case MSG_OK:
	return "MSG_OK";

    case MSG_CREATE:
	return "MSG_CREATE";

    case MSG_SPEC:
	return "MSG_SPEC";
    case MSG_ADDR:
	return "MSG_ADDR";

    case MSG_FAILURE:
	return "MSG_FAILURE";

    case MSG_TIMEOUT:
	return "MSG_TIMEOUT";
    case MSG_ALARM:
	return "MSG_ALARM";

    case MSG_PROERROR:
	return "MSG_PROERROR";
    case MSG_FATAL:
	return "MSG_FATAL";

    case MSG_START:
	return "MSG_START";
    case MSG_STARTED:
	return "MSG_STARTED";
    case MSG_STOP:
	return "MSG_STOP";
    case MSG_STOPPED:
	return "MSG_STOPPED";

    case MSG_ABORT:
	return "MSG_ABORT";
    case MSG_ABORTED:
	return "MSG_ABORTED";
    case MSG_TERMINATE:
	return "MSG_TERMINATE";
    case MSG_TERMINATED:
	return "MSG_TERMINATED";

    case MSG_DESTROY:
	return "MSG_DESTROY";
    case MSG_RESET:
	return "MSG_RESET";

    case MSG_CLOSE:
	return "MSG_CLOSE";
    case MSG_CLOSED:
	return "MSG_CLOSED";
    case MSG_INTERNAL_CLOSED:
	return "MSG_INTERNAL_CLOSED";

    case MSG_TIMEQ:
	return "MSG_TIMEQ";
    case MSG_TIME:
	return "MSG_TIME";
	
    default:
	sprintf(buf, "unknown (%d)", code);
	return buf;
    }
} /* code2str */

/*  */

static char *names[] = {
    "BENCH",
    "CLIENT",
    "CLIENTS",
    "SERVER",
    "SERVERS",
    "SYSTEM",
    "SYSTEMS",
};

static int values[] = {
    BA_BENCH,
    BA_CLIENT,
    BA_CLIENTS,
    BA_SERVER,
    BA_SERVERS,
    BA_SYSTEM,
    BA_SYSTEMS,
};
    
#define BUFSIZE 128
static int currbuf = 0;

char *bcpaddr2str(addr)
    bcpaddr_t addr;
{
    static char buf[4][BUFSIZE];
    int num = 0, i;
    char *str = buf[currbuf];

    currbuf = (currbuf+1) % 4;

    str[0] = '\0';

    if (addr.ba_addr == NOTOK) { /* Not initialized */
	return "(unknown)";
    }
    
    for (i = 0; i < sizeof(values)/sizeof(int); i++) {
	if (addr.ba_addr & values[i]) {
	    if (num++)
		strcat(str, " | ");
	    strcat(str, names[i]);
	    if (values[i] & (BA_CLIENT | BA_SERVER))
		sprintf(str + strlen(str), ":%d", addr.ba_index);
	    else if (values[i] & BA_SYSTEM)
		sprintf(str + strlen(str), ":0x%x", addr.ba_index);	    
	}
    }
    return str;
} /* bcpaddr2str */


/*  */

printmyname()
{
    if (myaddr.ba_addr == BA_BENCH)
	printf("%-8s ", bcpaddr2str(myaddr));
    else
	printf("%-17s ", bcpaddr2str(myaddr));
} /* printmyname */

printmynameerr()
{
    if (myaddr.ba_addr == BA_BENCH)
	fprintf(stderr, "ERROR %-8s ", bcpaddr2str(myaddr));
    else
	fprintf(stderr, "ERROR %-17s ", bcpaddr2str(myaddr));
} /* printmynameerr */

reset_myaddr()
{
    myaddr.ba_addr = NOTOK;
    myaddr.ba_index = 0;
} /* reset_myaddr */

bcpaddr_t get_myaddr()
{
    return myaddr;
}
