/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * bench.c
 *
 * The benchmark client program.
 * Reads lines from standard input specifying the benchmarks (see doc/informat)
 * executes the benchmark, computes some statistics and outputs the result on
 * standard output (format specified in doc/outformat).
 *
 * Switches:
 *	-e Don't echo specification
 *	-c Clock difference compensation
 *	-r Print statistics from Responder
 *
 *	-d Debug on (all)			(dprintf)
 *	-t Trace on - tracing procedures	(tprintf)
 *	-p Protocol Debug on 			(pprintf)
 *	-s State Trace on			(stprintf)
 *	-i IPC Trace on 			(iprintf)
 *	-m Measurement procedure debug		(mprintf)
 *
 * Parameters:
 *	default client hostname
 *	default server hostname
 * If only one hostname is specified it is taken to be the default server
 * hostname. The default values for both one is the name of the local machine.
 */
/********************
 *
 *	Revision: Responder switch 'r' added 0989/ PerG
 *
 */

#define MAX_RETRIES 3

#include <general.h>

#include <y.tab.h>

extern yyparse();
struct spec *spec;		/* top level specification from parser */

int Trace = 0;
int PrintResponder = 0;
int Debug = 0;
int ProtoDebug = 0, StateTrace = 0, IPCTrace = 0, ProcDebug = 0;

int ClockDiff = 0;
int NoEcho = 0;

char *myname;

#define HOSTNAMLEN 40

char DefServerHost[HOSTNAMLEN];
char DefClientHost[HOSTNAMLEN];
char LocalHost[HOSTNAMLEN];

main(argc,argv)
    int argc;
    char **argv;
{
    int nretry, ret;
    
    myname  = argv[0];

    argv++, argc--;
    while (argc > 0 && *argv[0] == '-') {
	getflags(argv[0]);
	argv++, argc--;
    }
    
    if (argc > 2) {
	fprintf(stderr,
		"Usage: %s [-certdpism] [ [<default client host>] <default server host>]\n",
		myname);
	exit(1);
    }
    
    if (gethostname(LocalHost, HOSTNAMLEN) == NOTOK) {
	fprintf(stderr, "Internal error: gethostname\n");
	exit(1);
    }

    if (argc == 0) {
	strncpy(DefClientHost, LocalHost, HOSTNAMLEN);
	strncpy(DefServerHost, LocalHost, HOSTNAMLEN);
    } else if (argc == 1) {
	strncpy(DefClientHost, LocalHost, HOSTNAMLEN);
	strncpy(DefServerHost, argv[0], HOSTNAMLEN);
    } else {
	strncpy(DefClientHost, argv[0], HOSTNAMLEN);
	strncpy(DefServerHost, argv[1], HOSTNAMLEN);
    }	

    init();

    if (yyparse() != OK) {
	fprintf(stderr, "Syntax error!\n");
	exit(1);
    }

    nretry = 0;
    
    while (spec != NULLSP) {
	extern char protostr[];
	
	switch (ret = bench_server(spec)) {
	case OK:
	    spec = spec_free(spec);
	    nretry = 0;
	    break;
	case DONE:
	    spec = NULLSP;
	    printf("Benchmark using the %s protocol - Failed\n",
		   protostr);
	    break;
	case NOTOK:
	    if (nretry++ >= MAX_RETRIES) {
		printf("Benchmark using the %s protocol - Failed\n",
		       protostr);
		spec = spec_free(spec);
		nretry = 0;
	    } else {
		
		printf("Benchmark using the %s protocol - Failed - retrying\n",
		       protostr);
	    }
	    break;
	}
    }
    symtab_free();
    childstate_free();
    procstate_free();
    result_init();	/* doesn't work if composers have been used? */
    host_close_all();	/* free space in host table */
    
    if (ret == OK)
	exit(0);
    else
	exit(1);
} /* main */

/*  */

getflags(str)
    char *str;
{
    str++;	/* skip '-' */
    while (*str != '\0') {
	switch (*str) {
	case 'c':
	    ClockDiff = 1;
	    break;
	case 'e':
	    NoEcho = 1;
	    break;
	  case 'r':
	    PrintResponder = 1;
	    break;
	case 'd':
	    Debug = 1;
	    Trace = 1;
	    ProtoDebug = 1;
	    IPCTrace = 1;
	    StateTrace = 1;
	    ProcDebug = 1;
	    break;
	case 't':
	    Trace = 1;
	    break;
	case 'p':
	    ProtoDebug = 1;
	    break;
	case 'i':
	    IPCTrace = 1;
	    break;
	case 's':
	    StateTrace = 1;
	    break;
	case 'm':
	    ProcDebug = 1;
	    break;
	    
	default:
	    fprintf(stderr, "%s: unknown flag -%c\n", myname, *str);
	    exit(1);
	}
	str++;
    }
} /* getflags */

/*  */

init()
{
    init_stdio();
    
    /* init the symbol table */
    symtab_init();
    /* Enter tokens into the symbol table */
    tokeninit();
    /* Enter benchmark and distribution names into the symbol table */
    autoinit();

    /* init random number generator */
    initrand();
}

/*  */

tokeninit()
{
    symbol_enter("bytes", SET_TOKEN, BYTES, NULLP, NULLP);
    symbol_enter("ascii", SET_TOKEN, ASCII, NULLP, NULLP);
    symbol_enter("int", SET_TOKEN, INT, NULLP, NULLP);
    symbol_enter("filename", SET_TOKEN, FILENAME, NULLP, NULLP);
    symbol_enter("file", SET_TOKEN, FILENAME, NULLP, NULLP);
    symbol_enter("memory", SET_TOKEN, MEMORY, NULLP, NULLP);
    symbol_enter("virtual", SET_TOKEN, VIRTUAL, NULLP, NULLP);
    symbol_enter("float", SET_TOKEN, FLOAT, NULLP, NULLP);
    symbol_enter("string", SET_TOKEN, STRING, NULLP, NULLP);
    symbol_enter("timelimit", SET_TOKEN, TIMELIMIT, NULLP, NULLP);
    symbol_enter("to", SET_TOKEN, TO, NULLP, NULLP);
    symbol_enter("from", SET_TOKEN, FROM, NULLP, NULLP);
}

/*  */

/* Dummy routines */

client_procedure(addr)
{
    eprintf(EF_IN3, INTERNAL, "client_procedure called", "bench program!");
}

server_procedure(addr)
{
    eprintf(EF_IN3, INTERNAL, "server_procedure called", "bench program!");
}
