/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Extracting metrics
 */
/*
 * TODO
 */

#include <general.h>


/*
 * Unsupported metrics:
 * 	Metrics2Opackets()
 *	Metrics2Ipackets()
 */

/*
 * Exports:
 *	StartMeasurements(mt)
 *	StopMeasurements(mt)
 *	Metrics2*(mt)
 *	MetricsSetDelay(mt, delay)
 *	StartProfiling() -> OK/NOTOK
 *	StopProfiling() -> OK/NOTOK
 */
#ifdef SYSV
long times();
#endif SYSV


StartMeasurements(mt)
    metrics_t *mt;
{
    struct timezone tz;
#ifdef SYSV
    struct tms buffers, *buffer = &buffers;
    long rtime;
#endif SYSV

    tprintf("StartMeasurements(0x%x)\n", mt);
    
    mt->mt_exit = MT_EXIT_OK;
    mt->mt_datasent = mt->mt_datarcvd = 0;
    mt->mt_ipackets = mt->mt_opackets = 0;
    mt->mt_delay = 0;
#ifdef BSD
    if (getrusage(RUSAGE_SELF, &mt->mt_usage) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "StartMeasurements", "getrusage", 
	       getsyserr());
	return;
    }
#endif BSD
#ifdef SYSV
    if ((rtime = times(buffer)) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "StartMeasurements", "times", 
	       getsyserr());
	return;
    }
    mt->mt_stime.tv_sec = buffer->tms_stime/HZ;
    mt->mt_stime.tv_usec = 1000000*(buffer->tms_stime%HZ)/HZ;
    mt->mt_utime.tv_sec = buffer->tms_utime/HZ;
    mt->mt_utime.tv_usec = 1000000*(buffer->tms_utime%HZ)/HZ;
    mt->mt_starttime.tv_sec = rtime/HZ;
    mt->mt_starttime.tv_usec = 1000000*(rtime%HZ)/HZ;
#endif SYSV
#ifdef BSD
    if (gettimeofday(&mt->mt_starttime, &tz) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "StartMeasurements", "gettimeofday",
	       getsyserr());
	return;
    }
#endif BSD
}
	

    
StopMeasurements(mt)
    metrics_t *mt;
{
    struct timezone tz; /* Not used */
#ifdef BSD
    struct rusage ru;
#endif BSD
#ifdef SYSV
    struct tms buffers, *buffer = &buffers;
    struct timeval stime, utime;
    long rtime;
#endif SYSV

    tprintf("StopMeasurements(0x%x)\n", mt);

#ifdef BSD
    if (gettimeofday(&mt->mt_stoptime, &tz) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "StopMeasurements", "gettimeofday",
	       getsyserr());
	return;
    }
    if (getrusage(RUSAGE_SELF, &ru) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "StopMeasurements", "getrusage", 
	       getsyserr());
	return;
    }
#endif BSD
#ifdef SYSV
    if ((rtime = times(buffer)) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "StartMeasurements", "times", 
	       getsyserr());
	return;
    }
    stime.tv_sec = buffer->tms_stime/HZ;
    stime.tv_usec = 1000000*(buffer->tms_stime%HZ)/HZ;
    utime.tv_sec = buffer->tms_utime/HZ;
    utime.tv_usec = 1000000*(buffer->tms_utime%HZ)/HZ;
    mt->mt_stoptime.tv_sec = rtime/HZ;
    mt->mt_stoptime.tv_usec = 1000000*(rtime%HZ)/HZ;
#endif SYSV

    /* Compute the difference between start and stop */
#ifdef SYSV
    timediff(&mt->mt_utime, &utime);
    timediff(&mt->mt_stime, &stime);
#endif SYSV
#ifdef BSD
    timediff(&mt->mt_utime, &ru.ru_utime);
    timediff(&mt->mt_stime, &ru.ru_stime);

    mt->mt_minflt = ru.ru_minflt - mt->mt_minflt;
    mt->mt_majflt = ru.ru_majflt - mt->mt_majflt;
    mt->mt_nswap = ru.ru_nswap - mt->mt_nswap;
    mt->mt_inblock = ru.ru_inblock - mt->mt_inblock;
    mt->mt_oublock = ru.ru_oublock - mt->mt_oublock;
    mt->mt_msgsent = ru.ru_msgsnd - mt->mt_msgsent;
    mt->mt_msgrcvd = ru.ru_msgrcv - mt->mt_msgrcvd;
    mt->mt_nsignals = ru.ru_nsignals - mt->mt_nsignals;
    mt->mt_nvcsw = ru.ru_nvcsw - mt->mt_nvcsw;
    mt->mt_nivcsw = ru.ru_nivcsw - mt->mt_nivcsw;

    mt->mt_maxrss = MAX(mt->mt_maxrss, ru.ru_maxrss);
#endif BSD    
}



timediff(start, stop)
    struct timeval *start, *stop;	/* difference in start */
{
    start->tv_sec = stop->tv_sec - start->tv_sec;
    start->tv_usec = stop->tv_usec - start->tv_usec;
}

timeMAX(max, next)
    struct timeval *max, *next;
{
    if (max->tv_sec > next->tv_sec)
	return;
    if (max->tv_sec == next->tv_sec && max->tv_usec > next->tv_usec)
	return;

    max->tv_sec = next->tv_sec;
    max->tv_usec = next->tv_usec;
}

timeMIN(min, next)
    struct timeval *min, *next;
{
    if (min->tv_sec < next->tv_sec)
	return;
    if (min->tv_sec == next->tv_sec && min->tv_usec < next->tv_usec)
	return;

    min->tv_sec = next->tv_sec;
    min->tv_usec = next->tv_usec;
}

long tv2millisecs(tv)
    struct timeval *tv;
{
    return (tv->tv_sec*1000 + (tv->tv_usec + 500)/1000);
}

int Metrics2Exit(mt)
    metrics_t *mt;
{
    return mt->mt_exit;
}
double Metrics2Starttime(mt)
    metrics_t *mt;
{
    return (mt->mt_starttime.tv_sec +
	(mt->mt_starttime.tv_usec + .500)/1000000);
}
double Metrics2Stoptime(mt)
    metrics_t *mt;
{
    return (mt->mt_stoptime.tv_sec +
	(mt->mt_stoptime.tv_usec + .500)/1000000);
}
long Metrics2Elapsed(mt)
    metrics_t *mt;
{
    return tv2millisecs(&mt->mt_stoptime) - tv2millisecs(&mt->mt_starttime);
}
long Metrics2Cputime(mt)
    metrics_t *mt;
{
    return Metrics2Usertime(mt) + Metrics2Systime(mt);
}
long Metrics2Usertime(mt)
    metrics_t *mt;
{
    return tv2millisecs(&mt->mt_utime);
}
long Metrics2Systime(mt)
    metrics_t *mt;
{
    return tv2millisecs(&mt->mt_stime);
}
long Metrics2Sentdata(mt)
    metrics_t *mt;
{
    return mt->mt_datasent;
}
long Metrics2Recvdata(mt)
    metrics_t *mt;
{
    return mt->mt_datarcvd;
}
#ifdef BSD
long Metrics2Sentmsg(mt)
    metrics_t *mt;
{
    return mt->mt_msgsent;
}
long Metrics2Recvmsg(mt)
    metrics_t *mt;
{
    return mt->mt_msgrcvd;
}
long Metrics2Recvsignals(mt)
    metrics_t *mt;
{
    return mt->mt_nsignals;
}
long Metrics2Opackets(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
#ifdef notdef
    return mt->mt_opackets;
#endif
}
long Metrics2Ipackets(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
#ifdef notdef
    return mt->mt_ipackets;
#endif
}

long Metrics2Oblock(mt)
    metrics_t *mt;
{
    return mt->mt_oublock;
}
long Metrics2Iblock(mt)
    metrics_t *mt;
{
    return mt->mt_inblock;
}
long Metrics2Pfio(mt)
    metrics_t *mt;
{
    return mt->mt_majflt;
}
long Metrics2Pfrecl(mt)
    metrics_t *mt;
{
    return mt->mt_minflt;
}
long Metrics2Swaps(mt)
    metrics_t *mt;
{
    return mt->mt_nswap;
}
long Metrics2Ivcsw(mt)
    metrics_t *mt;
{
    return mt->mt_nivcsw;
}
long Metrics2Vcsw(mt)
    metrics_t *mt;
{
    return mt->mt_nvcsw;
}
long Metrics2Memusage(mt)
    metrics_t *mt;
{
    return mt->mt_maxrss;
}
#endif BSD
#ifdef SYSV
long Metrics2Sentmsg(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Recvmsg(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Recvsignals(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Opackets(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Ipackets(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}

long Metrics2Oblock(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Iblock(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Pfio(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Pfrecl(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Swaps(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Ivcsw(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Vcsw(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
long Metrics2Memusage(mt)
    metrics_t *mt;
{
    return MT_UNSUPP;
}
#endif SYSV

double Metrics2Delay(mt)
    metrics_t *mt;
{
    return mt->mt_delay;
}

/*  */

int StartProfiling()
{
  int ret;
  
  dprintf("StartProfiling()\n");

#ifdef sun
  ret = system("/usr/etc/kgmon -r\n");
#else
  ret = system("/etc/kgmon -r\n");
#endif
  if (ret != 0) {
    eprintf("StartProfiling: failed resetting kernel profiling (0x%x)\n", ret);
    return NOTOK;
  }

#ifdef sun
  ret = system("/usr/etc/kgmon -b\n");
#else
  ret = system("/etc/kgmon -b\n");
#endif
  if (ret != 0) {
    eprintf("StartProfiling: failed starting kernel profiling (0x%x)\n", ret);
    return NOTOK;
  }

  printf("kernel profiling started\n");

  return OK;
} /* StartProfiling */


static char gprofstr[128];
static char filenametemplate[] = "/usr/tmp/spims%x.XXXXXX";

char *mktemp();

int StopProfiling()
{
  int ret;
  int hextime;
  char *filename;

  dprintf("StopProfiling()\n");

#ifdef sun
  ret = system("/usr/etc/kgmon -h\n");
#else
  ret = system("/etc/kgmon -h\n");
#endif
  if (ret != 0) {
    eprintf("StopProfiling: failed stopping kernel profiling (0x%x)\n", ret);
    return NOTOK;
  }
  
  printf("Kernel profiling stopped\n");

#ifdef sun
  ret = system("(cd /usr/tmp; /usr/etc/kgmon -p)\n");
#else
  ret = system("(cd /usr/tmp; /etc/kgmon -p)\n");
#endif
  if (ret != 0) {
    eprintf("StopProfiling: failed printing profiling info (0x%x)\n", ret);
    return NOTOK;
  }

  filename = (char *)malloc(strlen(filenametemplate) + 1 +10);
  if (filename == NULL) {
    eprintf(EF_IN3, INTERNAL, RESOURCE, "StopProfiling");
    return NOTOK;
  }

  hextime = time(0);

  sprintf(filename, filenametemplate, hextime);

  filename = mktemp(filename);

  printf("StopProfiling: result written on %s\n", filename);

  sprintf(gprofstr, "(cd /usr/tmp; gprof /vmunix >%s)\n", filename);

  dprintf("StopProfiling: gprofstr=%s\n", gprofstr);

  ret = system(gprofstr);
  if (ret != 0) {
    eprintf("StopProfiling: failed running gprof (0x%x)\n", ret);
    return NOTOK;
  }
  return OK;
} /* StopProfiling */




