unit main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, Menus, ImgList, ActnList, ExtCtrls, StdCtrls, Grids, contnrs,
  Buttons, ToolWin;

type
  TMainForm = class(TForm)
    ImageList: TImageList;
    ActionList: TActionList;
      FileNewAction: TAction;
      FileOpenAction: TAction;
      FileSaveAction: TAction;
      FileSaveAsAction: TAction;
      FileExitAction: TAction;
      EditCopyAction: TAction;
      EditFindAction: TAction;
      EditAgainAction: TAction;
      NextLatinAction: TAction;
      NextMissingAction: TAction;
      ViewToolBarAction: TAction;
      SetupFontAction: TAction;
      ShowGridAction: TAction;
      ShowOriginalAction: TAction;
      HelpCreditsAction: TAction;
      HelpAboutAction: TAction;
    MainMenu: TMainMenu;
      FileMenu: TMenuItem;
        FileNewItem: TMenuItem;
        FileOpenItem: TMenuItem;
        FileSaveItem: TMenuItem;
        FileSaveAsItem: TMenuItem;
        N1: TMenuItem;
        FileExitItem: TMenuItem;
      EditMenu: TMenuItem;
        EditCopyItem: TMenuItem;
        N3: TMenuItem;
        EditFindItem: TMenuItem;
        FindAgainItem: TMenuItem;
        NextLatinItem: TMenuItem;
        NextMissingItem: TMenuItem;
      OptionsMenu: TMenuItem;
        ShowToolbarItem: TMenuItem;
        N2: TMenuItem;
        OptionsFontItem: TMenuItem;
        OptionsGridItem: TMenuItem;
        OptionsOriginalItem: TMenuItem;
      HelpMenu: TMenuItem;
        HelpCreditsItem: TMenuItem;
        HelpAboutItem: TMenuItem;
    ToolBar: TToolBar;
      FileNewButton: TToolButton;
      FileOpenButton: TToolButton;
      FileSaveButton: TToolButton;
      FileSaveAsButton: TToolButton;
      FileExitToolButton: TToolButton;
        ToolButton6: TToolButton;
      FindFirstButton: TToolButton;
      FindNextButton: TToolButton;
      FindLationButton: TToolButton;
      FindMissingButton: TToolButton;
      EditCopyButton: TToolButton;
        ToolButton10: TToolButton;
      ToolBarButton: TToolButton;
      SetupFontButton: TToolButton;
      SetupGridButton: TToolButton;
      SetupOrgButton: TToolButton;
        ToolButton14: TToolButton;
      HelpCreditsButton: TToolButton;
      HelpAboutButton: TToolButton;
    DrawGrid: TDrawGrid;
    StatusBar: TStatusBar;
    TemplateDialog: TOpenDialog;
    TranslateDialog: TOpenDialog;
    SaveAsDialog: TSaveDialog;
    FontDialog: TFontDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);

    procedure FileNewActionExecute(Sender: TObject);
    procedure FileOpenActionExecute(Sender: TObject);
    procedure FileSaveActionUpdate(Sender: TObject);
    procedure FileSaveActionExecute(Sender: TObject);
    procedure FileSaveAsActionUpdate(Sender: TObject);
    procedure FileSaveAsActionExecute(Sender: TObject);
    procedure FileExitActionExecute(Sender: TObject);
    procedure EditCopyActionUpdate(Sender: TObject);
    procedure EditCopyActionExecute(Sender: TObject);
    procedure EditFindActionUpdate(Sender: TObject);
    procedure EditFindActionExecute(Sender: TObject);
    procedure EditAgainActionUpdate(Sender: TObject);
    procedure EditAgainActionExecute(Sender: TObject);
    procedure NextLatinActionUpdate(Sender: TObject);
    procedure NextLatinActionExecute(Sender: TObject);
    procedure NextMissingActionUpdate(Sender: TObject);
    procedure NextMissingActionExecute(Sender: TObject);
    procedure ViewToolBarActionUpdate(Sender: TObject);
    procedure ViewToolBarActionExecute(Sender: TObject);
    procedure SetupFontActionExecute(Sender: TObject);
    procedure ShowGridActionUpdate(Sender: TObject);
    procedure ShowGridActionExecute(Sender: TObject);
    procedure ShowOriginalActionUpdate(Sender: TObject);
    procedure ShowOriginalActionExecute(Sender: TObject);
    procedure HelpCreditsActionExecute(Sender: TObject);
    procedure HelpAboutActionExecute(Sender: TObject);
    procedure ToolBarButtonClick(Sender: TObject);

    procedure DrawGridDrawCell(Sender: TObject; ACol, ARow: Integer;
      Rect: TRect; State: TGridDrawState);
    procedure DrawGridGetEditText(Sender: TObject; ACol, ARow: Integer;
      var Value: string);
    procedure DrawGridSetEditText(Sender: TObject; ACol, ARow: Integer;
      const Value: string);
    procedure DrawGridKeyPress(Sender: TObject; var Key: Char);
  private
    FEditText: string;
    FCancelEdit: Boolean;
    FFileName: string;
    FModified: Boolean;
    FFlags: TList;
    FTemplate: TStringList;
    FKeywords: TStringList;
    FOriginal: TStringList;
    FTranslat: TStringList;
    FShowText: Boolean;
    function GetGridStr(ACol, ARow: Integer): string;
    procedure SetGridStr(ACol, ARow: Integer; const S: string);
    function KillTranslation: Boolean;
    function LoadTemplate: Boolean;
    function LoadTranslation: Boolean;
    function SaveTranslation(Force: Boolean): Boolean;
    function GetFTransLoaded: Boolean;
    procedure LocateMode(Mode: Integer);
    procedure ActualFontSize;
    procedure FindText(Again: Boolean);
  public
    property TransModified: Boolean read FModified;
    property TransLoaded: Boolean read GetFTransLoaded;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  end;

var
  MainForm: TMainForm;

resourcestring
  CTransClearConfirm = 'Current translation not saved. Save this?';
  CLineNumberGridTitle = 'Line #';
  COriginalGridTitle = 'Original';
  CTranslatGridTitle = 'Translated';
  CFindQueryTitle = 'Find Text';
  CFindQueryPrompt = '&Text to find:';
  CStatusBarText = 'Phrases: %d, Untranslated: %d, Missing: %d';
  CFormatLanguageStr = '    case %s: $tmp = %s; break;';
  CFormatDefineStr = 'define(%s,%s);';
  CFormatWebsiteStr = '[[%s]] = [[%s]]';
  CFormatPSlashStr = 'case strtolower(%s): $tmp = %s; break;';
  CTextNotFound = 'Text not found';

implementation
{ ========================================================================= }
uses about, masks, find, credits, clipbrd, registry;
const
  CKeyTreeRoot = 'Software\irnis.net\nukelang';
{$R *.dfm}
procedure TMainForm.FormCreate(Sender: TObject);
var
  F: TFont;
  R: TRegistry;
begin
  R := TRegistry.Create;
  try
    if R.OpenKeyReadOnly(CKeyTreeRoot) then begin
      Top := R.ReadInteger('Top');
      Left := R.ReadInteger('Left');
      Width := R.ReadInteger('Width');
      Height := R.ReadInteger('Height');
      ToolBar.Visible := R.ReadBool('ToolBar');
      DrawGrid.GridLineWidth := R.ReadInteger('GridLine');
      FShowText := not R.ReadBool('ShowKeys');

      F := DrawGrid.Font;
      F.Name := R.ReadString('FontName');
      F.Charset := R.ReadInteger('FontChar');
      F.Color := R.ReadInteger('FontColor');
      F.Pitch := TFontPitch(R.ReadInteger('FontPitch'));
      F.Size := R.ReadInteger('FontSize');
      if R.ReadBool('FontStyleBold')
        then F.Style := F.Style + [fsBold]
        else F.Style := F.Style - [fsBold];
      if R.ReadBool('FontStyleItalic')
        then F.Style := F.Style + [fsItalic]
        else F.Style := F.Style - [fsItalic];
      if R.ReadBool('FontStyleUnder')
        then F.Style := F.Style + [fsUnderline]
        else F.Style := F.Style - [fsUnderline];
      if R.ReadBool('FontStyleStrike')
        then F.Style := F.Style + [fsStrikeOut]
        else F.Style := F.Style - [fsStrikeOut];
    end;
  finally
    R.Free;
  end;
end;

procedure TMainForm.FormResize(Sender: TObject);
begin
  if not ((csCreating in ControlState) or
          (csDestroyingHandle in ControlState)) then ActualFontSize
end;

procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  CanClose := KillTranslation
end;

procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  F: TFont;
  R: TRegistry;
begin
  R := TRegistry.Create;
  try
    if R.OpenKey(CKeyTreeRoot, TRUE) then begin
      R.WriteInteger('Top', Top);
      R.WriteInteger('Left', Left);
      R.WriteInteger('Width', Width);
      R.WriteInteger('Height', Height);
      R.WriteBool('ToolBar', ToolBar.Visible);

      R.WriteInteger('GridLine', DrawGrid.GridLineWidth);
      R.WriteBool('ShowKeys', FShowText);

      F := DrawGrid.Font;
      R.WriteString('FontName', F.Name);
      R.WriteInteger('FontChar', F.Charset);
      R.WriteInteger('FontColor', F.Color);
      R.WriteInteger('FontPitch', Integer(F.Pitch));
      R.WriteInteger('FontSize', F.Size);
      R.WriteBool('FontStyleBold', fsBold in F.Style);
      R.WriteBool('FontStyleItalic', fsItalic in F.Style);
      R.WriteBool('FontStyleUnder', fsUnderline in F.Style);
      R.WriteBool('FontStyleStrike', fsStrikeout in F.Style);
    end;
  finally
    R.Free;
  end;
end;

procedure TMainForm.FileNewActionExecute(Sender: TObject);
begin
  if KillTranslation then LoadTemplate;
end;

procedure TMainForm.FileOpenActionExecute(Sender: TObject);
begin
  if KillTranslation and LoadTemplate then LoadTranslation
end;

procedure TMainForm.FileSaveActionUpdate(Sender: TObject);
begin
  FileSaveAction.Enabled := TransModified;
end;

procedure TMainForm.FileSaveActionExecute(Sender: TObject);
begin
  SaveTranslation(FALSE)
end;

procedure TMainForm.FileSaveAsActionUpdate(Sender: TObject);
begin
  FileSaveAsAction.Enabled := TransLoaded;
end;

procedure TMainForm.FileSaveAsActionExecute(Sender: TObject);
begin
  SaveTranslation(TRUE)
end;

procedure TMainForm.FileExitActionExecute(Sender: TObject);
begin
  Close
end;

procedure TMainForm.EditCopyActionUpdate(Sender: TObject);
begin
  EditCopyAction.Enabled := TransLoaded;
end;

procedure TMainForm.EditCopyActionExecute(Sender: TObject);
var
  S: string;
  I, J: Integer;
begin
  S := '';
  for I := DrawGrid.Selection.Top to DrawGrid.Selection.Bottom do begin
    for J := DrawGrid.Selection.Left to DrawGrid.Selection.Right do begin
      S := S + GetGridStr(J, I);
      if J <> DrawGrid.Selection.Right then S := S + #9;
    end;
    if I <> DrawGrid.Selection.Bottom then S := S + #13#10
  end;
  Clipboard.AsText := S;
end;

procedure TMainForm.EditFindActionUpdate(Sender: TObject);
begin
  EditFindAction.Enabled := TransLoaded;
end;

procedure TMainForm.EditFindActionExecute(Sender: TObject);
begin
  if FindForm.ShowModal = mrCancel then Exit;
  FindText(FALSE)
end;

procedure TMainForm.EditAgainActionUpdate(Sender: TObject);
begin
  EditAgainAction.Enabled := TransLoaded and
                             (Length(Trim(FindForm.FindEdit.Text)) > 0);
end;

procedure TMainForm.EditAgainActionExecute(Sender: TObject);
begin
  FindText(TRUE);
end;

procedure TMainForm.NextLatinActionUpdate(Sender: TObject);
begin
  NextLatinAction.Enabled := TransLoaded;
end;

procedure TMainForm.NextLatinActionExecute(Sender: TObject);
begin
  LocateMode(1)
end;

procedure TMainForm.NextMissingActionUpdate(Sender: TObject);
begin
  NextMissingAction.Enabled := TransLoaded;
end;

procedure TMainForm.NextMissingActionExecute(Sender: TObject);
begin
  LocateMode(0)
end;

procedure TMainForm.SetupFontActionExecute(Sender: TObject);
begin
  FontDialog.Font.Assign(DrawGrid.Font);
  if FontDialog.Execute then begin
    DrawGrid.Font.Assign(FontDialog.Font);
    DrawGrid.Update;
    ActualFontSize;
  end
end;

procedure TMainForm.ShowGridActionUpdate(Sender: TObject);
begin
  ShowGridAction.Checked := DrawGrid.GridLineWidth  <> 0;
  SetupGridButton.Down := ShowGridAction.Checked;
end;

procedure TMainForm.ShowGridActionExecute(Sender: TObject);
begin
  ShowGridAction.Checked := not ShowGridAction.Checked;
  if ShowGridAction.Checked then DrawGrid.GridLineWidth := 1
                            else DrawGrid.GridLineWidth := 0
end;

procedure TMainForm.ShowOriginalActionUpdate(Sender: TObject);
begin
  ShowOriginalAction.Checked := FShowText;
  SetupOrgButton.Down := FShowText;
end;

procedure TMainForm.ShowOriginalActionExecute(Sender: TObject);
begin
  FShowText := not FShowText;
  DrawGrid.Invalidate
end;

procedure TMainForm.ViewToolBarActionUpdate(Sender: TObject);
begin
  ViewToolbarAction.Checked := Toolbar.Visible;
end;

procedure TMainForm.ViewToolBarActionExecute(Sender: TObject);
begin
  Toolbar.Visible := not Toolbar.Visible
end;

procedure TMainForm.HelpCreditsActionExecute(Sender: TObject);
begin
  CreditsForm.ShowModal
end;

procedure TMainForm.HelpAboutActionExecute(Sender: TObject);
begin
  AboutForm.ShowModal
end;

procedure TMainForm.ToolBarButtonClick(Sender: TObject);
begin
  ToolBar.Hide
end;
{ ------------------------------------------------------------------------- }
procedure TMainForm.DrawGridDrawCell(Sender: TObject; ACol, ARow: Integer;
  Rect: TRect; State: TGridDrawState);
var
  S: string;
  H: Integer;
  C: TColor;
begin
  S := GetGridStr(ACol, ARow);
  H := ((Rect.Bottom - Rect.Top) - DrawGrid.Canvas.TextHeight(S)) div 2;
  C := DrawGrid.Canvas.Font.Color;
  try
    if (ACol > 1) and (ARow > 0) and not (gdFocused in State) then
      case Integer(FTranslat.Objects[ARow-1]) of
        0: DrawGrid.Canvas.Font.Color := clRed;
        1: DrawGrid.Canvas.Font.Color := clNavy;
        2: DrawGrid.Canvas.Font.Color := clGreen;
      end;
    DrawGrid.Canvas.TextRect(Rect, Rect.Left + 2, Rect.Top + H, S)
  finally
    DrawGrid.Canvas.Font.Color := C
  end;
end;

procedure TMainForm.DrawGridGetEditText(Sender: TObject; ACol,
  ARow: Integer; var Value: string);
begin
  Value := GetGridStr(ACol, ARow);
  FEditText := Value;
end;

procedure TMainForm.DrawGridSetEditText(Sender: TObject; ACol,
  ARow: Integer; const Value: string);
begin
  if FCancelEdit then FCancelEdit := FALSE else
  if ACol = 2 then SetGridStr(ACol, ARow, Value) else begin
    FCancelEdit := TRUE;
    DrawGrid.EditorMode := FALSE;    
  end
end;

procedure TMainForm.DrawGridKeyPress(Sender: TObject; var Key: Char);
begin
  if DrawGrid.EditorMode and (Key = #27) then begin
    FCancelEdit := TRUE;
    FTranslat[DrawGrid.Row-1] := FEditText;
    DrawGrid.EditorMode := FALSE;
  end
end;

{ ------------------------------------------------------------------------- }
function TMainForm.GetFTransLoaded: Boolean;
begin
  Result := FTemplate.Count > 0
end;

function TMainForm.GetGridStr(ACol, ARow: Integer): string;
begin
  case ACol of
    0: if ARow = 0 then Result := CLineNumberGridTitle
                   else Result := IntToStr(Integer(FKeywords.Objects[ARow-1]));
    1: if ARow = 0 then Result := COriginalGridTitle else
       if FShowText then Result := FOriginal[ARow-1] else Result := FKeyWords[ARow-1];
    2: if ARow = 0 then Result := CTranslatGridTitle
                   else Result := FTranslat[ARow-1];
  end;
end;

procedure TMainForm.SetGridStr(ACol, ARow: Integer; const S: string);
begin
  if (ACol > 1) and (ARow > 0) and (S <> FTranslat[ARow-1]) then begin
    FTranslat.Objects[ARow-1] := Pointer(2);
    FTranslat[ARow-1] := S;
    FModified := TRUE
  end;
end;

function TMainForm.KillTranslation: Boolean;
begin
  Result := FALSE;
  if TransModified then case MessageDlg(CTransClearConfirm, mtConfirmation,
    [mbYes, mbNo, mbCancel], 0) of
    mrYes: if not SaveTranslation(TRUE) then Exit;
    mrNo:;
    mrCancel: Exit
  end;
  FModified := FALSE;

  DrawGrid.RowCount := 1;
  FFlags.Clear;
  FTemplate.Clear;
  FKeywords.Clear;
  FOriginal.Clear;
  FTranslat.Clear;
  FFileName := '';
  StatusBar.SimpleText := '';
  Result := TRUE;
end;

function ParseStr(const S: string; var C, D: string): Integer;
var
  I, L: Integer;
begin
  Result := 0; I := 1; L := Length(S);
  // Trim spaces
  while (I <= L) and (S[I] <= ' ') do Inc(I);
  while (L >= I) and (S[L] <= ' ') do Dec(L);
  if I > L then Exit; // if empty string, skip
  // Find "case" or "define" keyword, or "[[" chars (for PHPWebSite)
  if Copy(S, I, 15) = 'case strtolower' then Result := 4 else
  if Copy(S, I, 4) = 'case' then Result := 1 else
  if Copy(S, I, 6) = 'define' then Result := 2 else
  if Copy(S, I, 2) = '[[' then Result := 3;
  if Result = 0 then Exit; // if not, skip

  C := ''; D := '';
  // Find translation string start
  if Result = 3 then begin
    Inc(I, 2);
    while I < L do begin
      if (S[I] = ']') and (S[I+1] = ']') then Break;
      C := C + S[I]; Inc(I)
    end;
  end else begin
    while (I <= L) and (S[I] <> '"') do Inc(I); if I > L then Exit;
    // Replace '\"' by "
    Inc(I);
    while I <= L do begin
      if (S[I] = '\') and (I < L) and (S[I+1] = '"') then begin
        Inc(I); C := C + '"'
      end else if S[I] = '"' then Break else C := C + S[I];
      Inc(I)
    end;
    Inc(I);
  end;
  if Length(C) = 0 then Exit; // empty string

  // find translated text start
  if Result = 3 then begin
    while (I < L) and ((S[I] <> '[') or (S[I+1] <> '[')) do Inc(I); Inc(I, 2);
    while I < L do begin
      if (S[I] = ']') and (S[I+1] = ']') then Break;
      D := D + S[I]; Inc(I)
    end;
  end else begin
    while (I <= L) and (S[I] <> '"') do Inc(I); if I > L then Exit;
    // Also replace '\"' by '"'
    Inc(I);
    while I <= L do begin
      if (S[I] = '\') and (I < L) and (S[I+1] = '"') then begin
        Inc(I); D := D + '"'
      end else if S[I] = '"' then Break else D := D + S[I];
      Inc(I)
    end;
  end;
end;

function MakeQuotes(const S: string): string;
var
  I: Integer;
begin
  Result := '"';
  for I := 1 to Length(S) do begin
    if S[I] = '"' then Result := Result + '\';
    Result := Result + S[I]
  end;
  Result := Result + '"';
end;

function FindStr(const List: TStrings; const S: string; var Index: Integer): Boolean;
var
  I: Integer;
begin
  Result := FALSE;
  for I := 0 to List.Count - 1 do
    if AnsiCompareStr(List[I], S) = 0 then begin
      Result := TRUE;
      Index := I;
      Break
    end;
end;

function TMainForm.LoadTemplate: Boolean;
var
  C, D: string;
  Flag: Integer;
  Stream: TFileStream;
  Index, Idx, I: Integer;
  Rating: array[0..4] of Integer;
begin
  Result := FALSE;
  // Select template file
  if not TemplateDialog.Execute then Exit;
  // Loading template file
  Stream := TFileStream.Create(TemplateDialog.FileName, fmOpenRead);
  try
    FTemplate.LoadFromStream(Stream);
  finally
    Stream.Free
  end;
  // Reset rating table
  for I := 0 to 3 do Rating[I] := 0;
  // Creating translation table
  for Index := 0 to FTemplate.Count - 1 do begin
    FTemplate.Objects[Index] := Ptr(-1); // Non-translation string by default
    Flag := ParseStr(FTemplate[Index], C, D);
    Inc(Rating[Flag]);
    if Flag = 0 then Continue;
    if FindStr(FKeywords, C, Idx) then Continue;
    // Add string to translation table
    try
      Idx := FKeywords.Add(C); // add string and get his index
      FKeywords.Objects[Idx] := Ptr(Index); // store cross-index
      FOriginal.Insert(Idx, D); // insert original string
      FTranslat.Insert(Idx, D); // insert translated string
      FFlags.Insert(Idx, Pointer(Flag));
    except
      // duplicates not allowed
      on EStringListError do Continue;
    end;
  end;
  //
  Index := 1;
  for I := 2 to 3 do if Rating[I] > Rating[Index] then Index := I;
  FShowText := Index = 2;
  // Synhronize template indexes with translation table
  for I := 0 to FKeywords.Count - 1 do begin
    Index := Integer(FKeywords.Objects[I]);
    FTemplate.Objects[Index] := Ptr(I); // store index in template
  end;
  // Synhronize DrawGrid with translation table
  DrawGrid.RowCount := FKeywords.Count + 1;
  if DrawGrid.RowCount > 1 then DrawGrid.FixedRows := 1;
  DrawGrid.Col := 2;
  ActualFontSize;
  // Return OK
  Result := TRUE
end;

function TMainForm.LoadTranslation: Boolean;
var
  C, D: string;
  List: TStringList;
  Stream: TFileStream;
  Index, Idx, I, K, Flag: Integer;
  Latin, Missing: Integer;
begin
  Result := FALSE;
  // Select translation file
  TranslateDialog.FileName := FFileName;
  if not TranslateDialog.Execute then Exit;
  FFileName := TranslateDialog.FileName;
  // create temporary string list
  List := TStringList.Create;
  try
    // load translation from file
    Stream := TFileStream.Create(FFileName, fmOpenRead);
    try
      List.LoadFromStream(Stream);
    finally
      Stream.Free
    end;
    // Update translation table
    for Index := 0 to List.Count - 1 do begin
      // get next string
      Flag := ParseStr(List[Index], C, D);
      if Flag = 0 then Continue;
      // find original text in translation table
      if not FindStr(FKeywords, C, Idx) then Continue; // not found, skip
      // update translation table
      if D <> FTranslat[Idx] then K := 2 else K := 1; // assume text is translated
      FTranslat[Idx] := D; FTranslat.Objects[Idx] := Ptr(K);
    end;
  finally
    List.Free
  end;
  // repaint drawgrid
  DrawGrid.Invalidate;
  // getting some statistics :)
  Latin := 0; Missing := 0;
  for I := 0 to FTranslat.Count - 1 do case Integer(FTranslat.Objects[I]) of
    0: Inc(Missing);
    1: Inc(Latin);
  end;
  StatusBar.SimpleText := Format(CStatusBarText,
    [FTranslat.Count, Latin, Missing]);
  // return OK
  Result := TRUE
end;

function TMainForm.SaveTranslation(Force: Boolean): Boolean;
var
  S: string;
  List: TStringList;
  Stream: TFileStream;
  I, J: Integer;
begin
  Result := FALSE;
  // Select file
  if Force or (Length(FFileName) = 0) then begin
    SaveAsDialog.FileName := FFileName;
    if not SaveAsDialog.Execute then Exit;
    FFileName := SaveAsDialog.FileName;
  end;
  // create temporary stringlist
  List := TStringList.Create; List.Sorted := FALSE;
  try
    // fill temporary string list with data
    for I := 0 to FTemplate.Count - 1 do begin
      // get translation table index for current template line
      J := Integer(FTemplate.Objects[I]);

      if J = -1 then List.Add(FTemplate[I]) // simple copy this line
      // this line has entry in translation table
      else begin
        case Integer(FFlags[J]) of
          1: S := CFormatLanguageStr;
          2: S := CFormatDefineStr;
          3: S := CFormatWebSiteStr;
          4: S := CFormatPSlashStr
          else S := '';
        end;
        if Integer(FFlags[J]) = 3
          then List.Add(Format(S, [FKeywords[J], FTranslat[J]]))
          else List.Add(Format(S, [MakeQuotes(FKeywords[J]),
                                   MakeQuotes(FTranslat[J])]))
      end
    end;
    // store string list to stream
    Stream := TFileStream.Create(FFileName, fmCreate);
    try
      List.SaveToStream(Stream)
    finally
      Stream.Free
    end;
  finally
    List.Free
  end;
  FModified := FALSE;
  Result := TRUE;
end;

procedure TMainForm.LocateMode(Mode: Integer);
var
  I: Integer;
begin
  I := DrawGrid.Row;
  while (I < FTranslat.Count) and
        (Integer(FTranslat.Objects[I]) <> Mode) do Inc(I);
  if I < FTranslat.Count then DrawGrid.Row := I + 1;
end;

procedure TMainForm.ActualFontSize;
var
  J, L: Integer;
begin
  DrawGrid.DefaultRowHeight := DrawGrid.Canvas.TextHeight('9') * 4 div 3;
  L := DrawGrid.Canvas.TextWidth('999999');
  J := DrawGrid.Canvas.TextWidth(CLineNumberGridTitle);
  if J > L then L := J; DrawGrid.ColWidths[0] := L + 20;
  DrawGrid.ColWidths[1] :=
    (DrawGrid.ClientWidth - DrawGrid.ColWidths[0] -
     DrawGrid.GridLineWidth * 3) div 2;
  DrawGrid.ColWidths[2] := DrawGrid.ColWidths[1];
  DrawGrid.Invalidate
end;

procedure TMainForm.FindText(Again: Boolean);
var
  S, C: string;
  MS, RS, OT, TT: Boolean;
  Index, I, L: Integer;
begin
  Index := DrawGrid.Row - 1; if Again then Inc(Index);
  RS := not FindForm.RegistryBox.Checked;
  OT := FindForm.OriginalBox.Checked;
  TT := FindForm.TranslatedBox.Checked;
  C := FindForm.FindEdit.Text; if RS then C := AnsiUpperCase(C);

  MS := FALSE; I := 1; L := Length(C);
  if not FindForm.MaskOrdBox.Checked then while I <= L do begin
    if (C[I] = '*') or (C[I] = '?') then begin MS := TRUE; Break end;
    Inc(I)
  end;

  while Index < FKeywords.Count do begin
    if OT then begin
      if FShowText then S := FOriginal[Index] else S := FKeyWords[Index];
      S := FKeywords[Index]; if RS then S := AnsiUpperCase(S);
      if (MS and MatchesMask(S, C)) or
         ((not MS) and (Pos(C, S) > 0)) then Break
    end;
    if TT then begin
      S := FTranslat[Index]; if RS then S := AnsiUpperCase(S);
      if (MS and MatchesMask(S, C)) or
         ((not MS) and (Pos(C, S) > 0)) then Break
    end;
    Inc(Index)
  end;
  if Index >= FKeywords.Count then ShowMessage(CTextNotFound)
                              else DrawGrid.Row := Index + 1;
end;

constructor TMainForm.Create(AOwner: TComponent);
begin
  inherited;
  FFlags := TList.Create;
  FTemplate := TStringList.Create; FTemplate.Sorted := FALSE;
  FKeywords := TStringList.Create; FKeywords.Sorted := FALSE;
  FOriginal := TStringList.Create; FOriginal.Sorted := FALSE;
  FTranslat := TStringList.Create; FTranslat.Sorted := FALSE;
end;

destructor TMainForm.Destroy;
begin
  FTranslat.Free;
  FOriginal.Free;
  FKeywords.Free;
  FTemplate.Free;
  inherited;
end;
{ ========================================================================= }
end.


