/*
** Arr_home.t defines the player's house, where the game begins.
**
** Copyright (c) 1998, Stephen Granade. All rights reserved.
*/

// I started out writing C code, so I prefer TADS to use C-style
// operators instead of its own style (i.e. '=' instead of ':=' etc.)
// This pragma statement does just that.
#pragma C+

// Some #define's for the wobbling stack of dishes
#define CAN_WOBBLE 4

/*
** This room demonstrates an adaptive description. Whenever the flag 'landed'
** is nil, an extra sentence is printed in the room's ldesc.
*/
startroom: houseRm
    landed = nil        // When the aliens land, set this to true
    hasNightSky = true  // You can see the sky from here--see below
    gotoDir = &east     // How the player gets to the ship. For more
                        //  information, take a look at doGoto in the shipRoom
                        //  class in arr_h.t
    gotoString = 'east'
    sdesc = "Kitchen"
    ldesc = {
        "You&rsquo;ve spent many an hour at the kitchen sink, elbow-deep in
            suds, staring out the window";
        if (!self.landed)
            ". Add this hour to the total, thanks to mom and her speech about
                familial responsibility. At least she isn&rsquo;t making you
                wipe down the cabinets again. ";
        else ", ignoring the ugly appliances and dirty cabinets. ";
        "A dismayingly large stack of dirty plates towers on one side of the
            sink";
        if (left_stack.location == self)
            ", balanced by a much smaller stack of clean plates on the
                other side";
        ". Next to the <<link('east doorway', 'e')>> is the avocado-green
            refrigerator your family has owned forever. ";
    }
    // Don't let the player go east until the alien ship lands
    canGoEast = true
    east = {
        if (!self.landed) {
            "Like you&rsquo;re gonna abandon the dishes and get sent to your
                room for the next year or more. ";
            return nil;
        }
        return hallway;
    }
    out = (self.east)
    // I'm going to modify enterRoom to handle the case that mom's in here
    //  cleaning up. If she is in the kitchen, the player gets in trouble
    enterRoom(actor) = {
        inherited.enterRoom(actor);
        if (actor == Me && mom.location == self) {
            "\bMom looks up from all the broken crockery littering the floor
                and glares at you. She looks madder than you&rsquo;ve seen
                her in a while, at least since the Hamster Incident. ";
            if (Me.leftSceneOfCrime)
                "<q>Stay put. I told you I don&rsquo;t want you to
                    leave.</q>\b";
            else "<q>When
                I get done cleaning this up, we&rsquo;re going to have a
                little talk. Don&rsquo;t you go anywhere.</q>\b";
        }
    }
    leaveRoom(actor) = {
        if (actor == Me && mom.location == self) {
            "Mom shouts after you:\ <q>Kid! Get back in here! Don&rsquo;t
                make me chase you down!</q>\b";
            Me.leftSceneOfCrime = true;
        }
        pass leaveRoom;
    }
;


// The window is here so that the player can look out it.
kitchen_window: fixeditem
    noun = 'window'
    adjective = 'kitchen'
    location = startroom
    sdesc = "kitchen window"
    ldesc = {
        "The window may have been clear once, but your mom&rsquo;s gleeful
            use of oil while cooking has added a dirty yellow glaze, making it
            hard to <<link('see through', 'look through window')>>. ";
    }
    whatdesc = "It&rsquo;s just like a wall, only you can see through it.
        Imagine! "
    verDoClean(actor) = {
        "Eww, gross. That oil&rsquo;d coat your hands more or less
            permanently. ";
    }
    verDoOpen(actor) = {
        "That&rsquo;s not gonna happen. All the windows have been painted
            shut for years. ";
    }
    verDoLookthru(actor) = {}
    doLookthru(actor) = {
        if (startroom.landed) {  // Different desc. if the aliens have landed
            "You can see the alien ship on your back lawn. ";
            if (global.graphicsOn) {
                "\n";
                if (real_ship.rampDown)
                    gprintf('Resources/Ship Window Ramp.jpg', 'WIDTH="400"
                        HEIGHT="278"');
                else gprintf('Resources/Ship Window.jpg', 'WIDTH="400"
                    HEIGHT="278"');
            }
        }
        else {
            "The stars of the night sky glitter dully through the window. ";
            if (global.graphicsOn) {
                "\n";
                gprintf('Resources/Shipless Window.jpg', 'WIDTH="400"
                    HEIGHT="278"');
            }
        }
    }
;

// The night sky will appear in any room which has the flag 'hasNightSky'
// set to true.
night_sky: distantItem, floatingItem
    noun = 'sky' 'star' 'stars'
    adjective = 'night'
    location = {
        if (Me.location.hasNightSky)
            return Me.location;
        return nil;
    }
    sdesc = "night sky"
    ldesc = "The night sky is filled with stars. "
;

kitchen_sink: fixeditem
    noun = 'sink' 'suds' 'water'
    adjective = 'kitchen' 'porcelain' 'discolored'
    location = startroom
    sdesc = "kitchen sink"
    ldesc = "The discolored porcelain sink is filled with suds. To its
        right is a stack of dirty dishes; to its left, a stack of clean
        dishes. "
    whatdesc = "Mom makes you use it for washing dishes. "
    verDoLookin(actor) = {
        "Suds and water. ";
    }
;

sink_faucet: fixeditem
    noun = 'faucet' 'spigot'
    adjective = 'kitchen'
    location = startroom
    sdesc = "kitchen faucet"
    ldesc = "It&rsquo;s just there for
        versimila&mdash;versumil&mdash;virsi&mdash;realism. Don&rsquo;t
        bother with it. "
    whatdesc = "When you turn the handle, water comes out. "
    dobjGen(a, v, i, p) = {
        if (v != inspectVerb && !v.issysverb) {
    	    "Don&rsquo;t bother. You&rsquo;ll never need to do anything with
    	        this faucet. ";
    	    exit;
    	}
    }
    iobjGen(a, v, d, p) = {
        "Don&rsquo;t bother. You&rsquo;ll never need to do anything with
            this faucet. ";
        exit;
    }
;

// trackCleaning allows me to change the dirty dishes' description each time
//  the player looks at the stack, then cleans one. It is set to be true right
//  after the player examines the stack; if the player then washes a dish,
//  numCleaned is increased and trackCleaning is set back to nil, so we don't
//  go through more than one description between glances at the stack.
right_stack: fixeditem
    numCleaned = 0
    trackCleaning = nil
    noun = 'plate' 'dish' 'plates' 'dishes' 'pfaltzgraff'
    plural = 'stacks'
    adjective = 'stack' 'dirty' 'right'
    location = startroom
    sdesc = "stack of dirty dishes"
    ldesc = {
        switch (self.numCleaned) {
            case 0:
            "The stack of dishes to the right of the sink is taller than it
                has any right to be. How did you and your parents mess up so
                many dishes in one meal? ";
            break;

            case 1:
            "You&rsquo;re not sure, but that stack of dishes may be
                <b>taller</b> than it was before you started washing them. ";
            break;

            case 2:
            "Taller. Definitely taller. ";
            break;

            default:
            "Even if the dirty dishes are growing in number, you&rsquo;d best
                keep cleaning them anyway. ";
            break;
        }
        self.trackCleaning = true;
    }
    takedesc = "The stack of dishes is much too heavy for you to lift. "
    whatdesc = {
        // Display a different message if the player asks about pfaltzgraff
        if (find(objwords(1), 'pfaltzgraff') != nil)
            "Pfaltzgraff is this hoity-toity brand of plates that
                mom likes. It&rsquo;s pretty stout stuff. ";
        else "Occasionally you and your family eat food off of them, which
            is why you have to wash them. If you get busy, you might be done
            sometime before the sun goes out. ";
    }
    verDoCount(actor) = {}
    doCount(actor) = {
        "There may not be 69,105 of them, but there sure are a lot. ";
    }
    verDoDry(actor) = {
        "You haven&rsquo;t washed them yet! ";
    }
    verDoTake(actor) = {
        if (left_stack.location == nil)
            "You&rsquo;ve already broken a bunch of dishes, but you want to
                keep cleaning them? Forget that&mdash;nothing&rsquo;s gonna
                get you out of this. ";
    }
    doTake(actor) = {
        "You pull, freeing the top plate from the gummy, foody stack with
            a wet slurping sound.
            One quick dip in the suds, one quick rinse from the faucet, and
            you toss the plate on the <q>clean</q> stack. ";
        if (self.trackCleaning) {
            self.numCleaned++;
            self.trackCleaning = nil;
        }
        left_stack.wobbleNum++;
        if (left_stack.wobbleNum == 1) {
            "The clean stack, in return, shimmies a bit. You keep piling
                slick clean dishes on that stack and, chances are,
                you&rsquo;ll soon have dish pieces on your hands. ";
        }
        else if (left_stack.isWobbling) {
            "Unfortunately, doing so only makes the stack wobble more. ";
            left_stack.fallCount++;
        }
        else if (left_stack.wobbleNum >= CAN_WOBBLE)
            "As you do, the clean stack of dishes wobbles for a moment
                before settling down. You could probably get that stack
                dancing if you wanted. ";
    }
    doSynonym('Take') = 'Clean'
    verDoBreak(actor) = {
        if (left_stack.location == nil)
            "Don&rsquo;t you think you&rsquo;ve broken enough dishes for
                one night? ";
        else "Nah. The food that&rsquo;s stuck on the dirty dishes would
            probably hold them together. ";
    }
    verDoEat(actor) = {
        "If only you could get rid of them that way. ";
    }
    verDoPoke(actor) = {
        "Those dirty dishes are way too stuck together for you to be able
            to move them around like that. ";
    }
    doSynonym('Poke') = 'Touch' 'Push' 'Pull' 'Move'
;

// wobbleNum keeps track of how many you've cleaned. After a certain point,
// the player can start the stack wobbling. fallCount keeps track of when
// the wobbling stack reaches the point that it falls.
left_stack: fixeditem
    smashedOne = nil
    isWobbling = nil
    wobbleNum = 0
    fallCount = 0
    noun = 'plate' 'dish' 'plates' 'dishes' 'pfaltzgraff'
    plural = 'stacks'
    adjective = 'stack' 'clean' 'left' 'wobbling' 'shaking' 'swaying'
    location = startroom
    sdesc = "stack of clean dishes"
    ldesc = {
        "You&rsquo;ve spent so much time sighing heavily, trying to get
            your mom to let you put this chore off for an hour at least, that
            you haven&rsquo;t cleaned many of them. ";
        if (self.isWobbling)
            "Despite this, the stack is swaying back and forth alarmingly. ";
    }
    takedesc = "The stack of dishes is much too heavy for you to lift. "
    // This next line allows the parser to assume the player is talking about
    //  the dirty stack of dishes if they type 'what is pfaltzgraff'.
    verDoWhatis(actor) = { self.whatdesc; }
    whatdesc = "They once held food, but now they&rsquo;re all clean. Until
        you mess them up with food again. "
    verDoCount(actor) = {}
    doCount(actor) = {
        "A depressingly small number. ";
    }
    verDoDry(actor) = {}
    doDry(actor) = {
        "You and mom have never been the dish-drying types, so there
            aren&rsquo;t any towels around. You&rsquo;ll just have to wait
            for them to air-dry. ";
    }
    verDoTake(actor) = {}
    doTake(actor) = {
        if (!self.isWobbling)
            "Ew. No thanks. You&rsquo;ve cleaned them, so why have anything
                else to do with them? ";
        else "You&rsquo;re afraid to try to grab one of the swaying dishes. ";
    }
    verDoClean(actor) = {
        "Too late. They&rsquo;re already clean. ";
    }
    verDoBreak(actor) = {}
    doBreak(actor) = {
        if (!self.isWobbling) {
            if (!self.smashedOne) {
                self.smashedOne = true;
                "You smash a plate on the floor. Yeah! Smash the plate! Smash
                    the plate!
                    \bUnfortunately, the sound was much too quiet to be heard
                    outside the kitchen. You sweep up the shards and get rid
                    of them, hoping mom won&rsquo;t notice. ";
            }
            else "One is enough, unless you can find a way of making a much
                bigger noise with the dishes. ";
        }
        else "They&rsquo;re on their way to doing that for you. ";
    }
    verDoPoke(actor) = {}
    doPoke(actor) = {
        if (self.isWobbling) {
            "You wait paitently, then jab at the swaying stack of dishes at
                just the right time to bring it to a halt. ";
            self.isWobbling = nil;
            self.fallCount = 0;
            unnotify(self, &wobbleDaemon);
        }
        else if (self.wobbleNum < CAN_WOBBLE)
            "Your poke at the dishes sends them swaying, but the motion
                damps out pretty quickly. ";
        else {
            "You poke at the stack of clean dishes, which begins to sway back
                and forth,
                back and forth. You&rsquo;d think they&rsquo;d stop, but
                your house is always shaking just the tiniest bit, and that
                must help keep them moving. They&rsquo;ll probably hit the
                ground before too much longer. ";
            self.isWobbling = true;
            notify(self, &wobbleDaemon, 0);
        }
    }
    doSynonym('Poke') = 'Touch' 'Push' 'Pull' 'Move'
    // A function to handle the wild, wobbling stack o' dishes
    wobbleDaemon = {
        self.fallCount++;
        switch (self.fallCount) {
            case 2:
                if (Me.location == startroom)
                    "\bThe stack of clean dishes is wobbling fit to beat the
                        band. ";
                break;
            case 4:
                if (Me.location == startroom)
                    "\bThose wobbling dishes are swaying dramatically. It
                        won&rsquo;t be long before they fall. ";
                break;
            case 6:
                self.makeAMess;
                break;
        }
    }
    makeAMess = {
        if (Me.location == startroom) {
            "\bThe clean stack of dishes leans way over the edge of the
                counter and pauses, as if thinking about whether to fall
                or not. The answer must be yes, because they all slide to
                a fractured end on the floor below. The sound is
                horrendous. ";
            "<SOUND SRC=\"mp3/Dishes Break.mp3\" LAYER=\"FOREGROUND\">";
            "\bOh boy. You&rsquo;d better not let mom see you now. ";
        }
        else if (isclass(Me.location, houseRm) || Me.location == backyard) {
            "\bFrom the kitchen, you hear a tremendous clattering banging
                crash. Something very bad has happened to your clean
                dishes, and something even worse will probably happen to
                you if mom catches you. ";
            "<SOUND SRC=\"mp3/Quieter Dishes Break.mp3\"
                LAYER=\"FOREGROUND\">";
        }
        unnotify(self, &wobbleDaemon);
        self.moveInto(nil);
        broken_dishes.moveInto(startroom);
        notify(mom, &moveToKitchen, 0);
    }
;

broken_dishes: fixeditem
    noun = 'plate' 'plates' 'dish' 'dishes' 'piece' 'pieces'
    adjective = 'broken'
    sdesc = "broken dishes"
    ldesc = "All those dishes you cleaned are lying broken on the floor.
        That can&rsquo;t be good. "
    heredesc = "Broken dishes are scattered across the floor. "
    takedesc = "Mom told you never to pick up broken dishes. You&rsquo;re
        liable to cut yourself. "
    whatdesc = "Uh-oh. They&rsquo;re not much use now, that&rsquo;s what
        they are. "
    verDoWhatis(actor) = {
        if (self.location == nil)
            "They aren&rsquo;t. At least not yet. ";
    }
;

cabinets: fixeditem, openable
    canOpen = nil        // Set when the aliens ask for rock salt
    isThem = true
    isopen = nil
    noun = 'cabinet' 'cabinets'
    adjective = 'pressboard'
    location = startroom
    sdesc = "cabinets"
    ldesc = {
        if (self.isopen) "Open";
        else "Closed";
        " pressboard cabinets. From time to time, mom makes you sponge them
            down to remove the oily residue that builds up on them. ";
        if (length(self.contents) > 0 && self.isopen)
            "Sitting in the cabinets you see <<listcont(self)>>. ";
    }
    whatdesc = "The cabinets are for holding all that food mom buys. "
    verDoClean(actor) = {
        "No thanks. With luck, mom won&rsquo;t ask you to clean them for at
            least another month. ";
    }
    verDoOpen(actor) = {
        if (!self.canOpen)
            "Your mom would skin you. Besides, you don&rsquo;t want to give
                her any reason to put those locks back on there for your
                <q>safety.</q> So leave &rsquo;em for now. ";
        else pass verDoOpen;
    }
    verDoLookin(actor) = {
        if (!self.isopen)
            "They&rsquo;re closed. ";
        else pass verDoLookin;
    }
;

rock_salt: item
    noun = 'can'
    adjective = 'rock' 'salt'
    location = cabinets
    sdesc = "can of rock salt"
    ldesc = "The cardboard can is blue and has a picture of this girl
        who&rsquo;s
        pouring rock salt all over the ground. She obviously hasn&rsquo;t
        heard your mom&rsquo;s speech about how much stuff like that costs.
        Or maybe she&rsquo;s
        ruining the crops of her enemies, like Mrs.\ Flavin talked about in
        history class. "
    whatdesc = "Salt in rock form. "
    verDoEat(actor) = { "Bleah! No thanks. "; }
    verDoOpen(actor) = { "No need. "; }
;

generic_cupcakes: item
    noun = 'box'
    adjective = 'generic' 'cupcakes' 'cakes' 'cup'
    location = cabinets
    sdesc = "box of generic cupcakes"
    ldesc = "That Hostess stuff is too expensive&mdash;all mom will buy you
        are these cheap knock-offs. "
    whatdesc = "Bad generic rip-off deserts mom buys when she is feeling
        cheap. "
    verDoOpen(actor) = {
        "Don&rsquo;t worry about the separate cupcakes. Just leave them in the
            box. ";
    }
    verDoEat(actor) = { "You&rsquo;ve already had three tonight! "; }
;

other_stuff: fixeditem
    isListed = true
    noun = 'stuff' 'noodles' 'noodle' 'stuffing'
    adjective = 'other' 'raman' 'stove' 'top'
    location = cabinets
    sdesc = "other stuff"
    adesc = "some other stuff"
    thedesc = "some other stuff"
    ldesc = "Raman noodles, Stove Top Stuffing&mdash;you know, all that stuff
        mom uses to cook dinner. "
    takedesc = "Nah. You don&rsquo;t like to eat most of that stuff, let alone
        put it away when mom brings it home from the store. "
    whatdesc = "That&rsquo;s all that stuff mom buys to make dinner from. "
    verDoEat(actor) = {
        "Nuh-uh. It tastes bad enough when it&rsquo;s cooked. ";
    }
;

fridge: fixeditem
    noun = 'refrigerator' 'fridge'
    adjective = 'avocado' 'green' 'avocado-green'
    location = startroom
    sdesc = "avocado-green refrigerator"
    ldesc = {
        "Everything in the kitchen used to be the same pukey-green color
            as the fridge. Thankfully, all that other stuff was
            replaced. Too bad the fridge still works, or you wouldn&rsquo;t
            ever have to see that color again. Mom has covered up as much of
            the fridge as she could with some of your artwork from class. ";
        if (global.graphicsOn) {
            "\n";
            gprintf('Resources/Fridge.jpg', 'WIDTH="240" HEIGHT="330"');
        }
    }
    whatdesc = "If you&rsquo;re a Brit or somesuch, then think of it as the
        icebox. "
    verDoOpen(actor) = {
        "Nuh-uh. You mess around in there and you won&rsquo;t get to see
            Captain Dangerous&rsquo; adventures for the rest of your life,
            maybe longer. ";
    }
    verDoClose(actor) = { "It&rsquo;s already closed. "; }
    verDoBreak(actor) = { "You&rsquo;ve tried. You&rsquo;ve tried. "; }
    verDoLookunder(actor) = {
        "There&rsquo;s nothing under the fridge. Maybe in the bad old days of
            adventures there&rsquo;d be ye olde limb-whacking sworde of
            antiquity tucked away under there, but
            in these enlightened times we don&rsquo;t hide important things
            under other things.
        \bWell, most of the time, anyway. ";
    }
    verDoLookin(actor) = {
        "You can&rsquo;t without opening it, and you won&rsquo;t open it
            &rsquo;cause mom would skin you alive. ";
    }
;

artwork: decoration
    noun = 'artwork' 'painting' 'paintings' 'drawing' 'drawings'
    location = startroom
    sdesc = "artwork"
    ldesc = "Just some stuff you drew for class. No big deal. Though you did
        get an A+ on the big blue swirly one. "
    whatdesc = "Some stuff you drew for a class assignment. "
;

kitchen_doorway: fixeditem
    noun = 'doorway'
    location = startroom
    sdesc = "doorway"
    ldesc = "It goes east. "
    whatdesc = "Don&rsquo;t be silly. "
    verDoEnter(actor) = {
        startroom.east;
    }
    doEnter(actor) = {
        actor.travelTo(hallway);
    }
;

hallway: houseRm
    gotoDir = &north
    gotoString = 'north'
    sdesc = "Hallway"
    ldesc = "A short hall lined with dark wood panelling. It connects the
        kitchen, to the <<link('west','w')>>, with the living room, to
        the <<link('east', 'e')>>. The backyard is <<link('north', 'n')>> of
        here. "
    canGoNorth = true
    canGoEast = true
    canGoWest = true
    north = backyard
    east = living_room
    west = startroom
    // Again, enterRoom modified to handle Mom grounding the player
    enterRoom(actor) = {
        inherited.enterRoom(actor);
        if (actor == Me && mom.location == self && mom.isReturning) {
            "\bMom looks at you angrily. You duck your head and scuffle
                one foot against the floor. ";
            mom.groundKid;
        }
    }
;

dark_wood_panelling: decoration
    noun = 'panelling'
    adjective = 'dark' 'wood'
    location = hallway
    sdesc = "dark wood panelling"
    ldesc = "Gloomy. At least your mom doesn&rsquo;t make you clean it. "
    whatdesc = "What is it? It&rsquo;s an ugly alternative to paint or
        wallpaper, that&rsquo;s what it is. "
    verDoClean(actor) = {
        "That&rsquo;s right, start cleaning something voluntarily. You need
            another thing to clean like you need your spleen.\b
        Wait. You need your spleen. Okay, you need another thing to clean
            like you need your appendix. ";
    }
;

living_room: houseRm
    hasShip = true
    gotoDir = &west
    gotoString = 'west'
    sdesc = "Lived-in Room"
    ldesc = "The living room is the rumpled center of your home. Comfy floral
        furniture, a few shelves filled with knick-knacks, a window, two
        parents, and a TV. The hallway to the kitchen is <<link('west', 'w')>>,
        and the unimportant rest
        of your home (mentioned here merely for the sake of completeness) is
        south. "
    canGoSouth = true
    canGoWest = true
    south = {
        "That part of your house is so unimportant right now that it
            hasn&rsquo;t been implemented. Don&rsquo;t you read the
            long descriptions of rooms? ";
        return nil;
    }
    west = hallway
    // Set some stuff up
    firstseen = {
        if (mom.location == living_room && left_stack.location != nil)
            "\bMom says, <q>You better be done with those dishes,
                kid,</q> and smiles. ";
        notify(dad, &doCloseEncountersRef, 5 + _rand(3));
    }
    // Modify enterRoom to handle Mom grounding the player
    enterRoom(actor) = {
        inherited.enterRoom(actor);
        if (actor == Me && mom.location == self && mom.isReturning) {
            "\bMom looks at you angrily. You duck your head and scuffle
                one foot against the floor. <q>You know better than to run
                away and hide
                when you&rsquo;ve done something wrong,</q> she says. You
                turn beet red. ";
            mom.groundKid;
        }
    }
;

furniture: fixeditem
    noun = 'furniture'
    adjective = 'comfy' 'floral'
    location = living_room
    sdesc = "comfy floral furniture"
    ldesc = "It sprawls across the carpet, filling the room. "
    whatdesc = "It&rsquo;s for sitting on. "
    verDoSiton(actor) = { "No time! There&rsquo;s a ship in your backyard! "; }
    doSynonym('Siton') = 'Lieon' 'Board'
    verDoClean(actor) = {
        "That&rsquo;s okay. The amount of change you&rsquo;d probably find
            would hardly be worth having something else to add to your list of
            chores. ";
    }
;

carpet: decoration
    noun = 'carpet' 'rug'
    location = living_room
    sdesc = "carpet"
    ldesc = "Don&rsquo;t look too closely! Mom might see you eyeing the
        carpet, and then wham!\ bam!\ you&rsquo;ll be vacuuming it tomorrow
        night. No thank you, ma&rsquo;am. "
    whatdesc = "Floor covering. "
    verDoClean(actor) = {
        "You&rsquo;re really developing an unhealthy preoccupation with
            cleaning things. Have you stopped taking your Ritalin? ";
    }
    verDoLookunder(actor) = {
        "You find a trap door!
        \bOkay, not really. ";
    }
;

shelves: fixeditem
    noun = 'shelf' 'shelves' 'board' 'boards'
    adjective = 'pine'
    location = living_room
    sdesc = "shelves"
    ldesc = "A couple of pine boards nailed into the wall. Knick-knacks fill
        them, well-nigh spilling over the edges and onto the carpet. "
    whatdesc = "See, you put stuff on it and the stuff stays up. "
    verIoPutOn(actor) = {
        "If only there were room. The knick-knacks leave no space for anything
            (or anyone) else. ";
    }
    verDoClean(actor) = { "Nuh-uh. "; }
;

// takenOne is true if the player has taken a knick-knack
knick_knacks: fixeditem
    takenOne = nil
    noun = 'knick-knacks' 'collectibles' 'figurines' 'plate' 'plates' 'smurf' 'smurfs' 'doll' 'dolls' 'memento' 'mementos'
    adjective = 'hummel' 'franklin' 'mint' 'plastic' 'precious' 'moments'
    location = living_room
    sdesc = "knick-knacks"
    ldesc = "Collectibles out the wazoo. Well, out the shelves. Hummel
        figurines, Franklin Mint plates, even a bunch of plastic smurfs. One
        entire shelf is filled with Precious Moments dolls, all of them
        staring at you with wide eyes, exuding eerie cuteness. "
    whatdesc = "If only I knew. "
    verDoTake(actor) = {
        if (self.takenOne)
            "You&rsquo;ve already pressed your luck by taking one. Take two
                and mom&rsquo;s sure to notice. ";
        else if (mom.location == living_room)
            "Mom glances at you. <q>Leave that stuff alone, kid.</q> ";
        else if (dad.location == living_room)
            "Dad says, <q>Don&rsquo;t mess with your mom&rsquo;s stuff.</q> ";
    }
    doTake(actor) = {
        "You look wildly around, making sure mom and dad aren&rsquo;t around,
            then snag one of the Precious Moments dolls from off the shelf. ";
        if (global.graphicsOn) {
            "\n";
            gprint('Resources/Holding Doll.jpg');
        }
        memento.moveInto(actor);
        self.takenOne = true;
        incscore(4);
    }
    verDoClean(actor) = {
        "You don&rsquo;t touch mom&rsquo;s collection, especially not to
            clean it. ";
    }
;

memento: item
    noun = 'memento' 'doll'
    adjective = 'precious' 'moments' 'single'
    sdesc = "single Precious Moments doll"
    ldesc = "One singular Precious Moments doll, thieved from your mom&rsquo;s
        collection. She won&rsquo;t miss it. Probably. "
    whatdesc = "See, folks collect these things. Dunno why. "
;

teevee: fixeditem
    noun = 'tv' 'teevee' 'telly' 'television' 'tube'
    adjective = 'boob'
    location = living_room
    sdesc = "TV"
    ldesc = "The TV sits there, completely silent. It&rsquo;s not natural, I
        tell you. Behind that blank screen the adventures of Captain
        Dangerous and Billy Bob are going on, and you&rsquo;re missing them! "
    verDoTurnon(actor) = {
        if (mom.location == living_room || dad.location == living_room)
            "Not with your parents in here! ";
        else "You start to turn on the TV, but stop. The guilt is too much.
            Besides, the noise would probably bring them both running like
            a shot. ";
    }
    verDoTurnoff(actor) = {
        "Oh, there&rsquo;s a clever idea. What&rsquo;s next, opening open
            doors? ";
    }
    doSynonym('Turnon') = 'Switch'
    verDoClean(actor) = { "Nuh-uh. "; }
    doBreak(actor) = {
        "Yeah! Smash the set! Smash the set! ";
    }
;

living_room_window: fixeditem
    noun = 'window'
    adjective = 'living' 'room'
    location = living_room
    sdesc = "living room window"
    ldesc = "The window looks out on the backyard and on the ship. "
    whatdesc = "It&rsquo;s just like a wall, only you can see through it.
        Imagine! "
    verDoOpen(actor) = {
        "That&rsquo;s not gonna happen. All the windows have been painted
            shut for years. ";
    }
    verDoLookthru(actor) = {}
    doLookthru(actor) = {
        "The alien ship on your back lawn, big as life. ";
        if (global.graphicsOn) {
            "\n";
            if (real_ship.rampDown)
                gprintf('Resources/Ship Window Ramp.jpg', 'WIDTH="400"
                    HEIGHT="278"');
            else gprintf('Resources/Ship Window.jpg', 'WIDTH="400"
                HEIGHT="278"');
        }
    }
    verDoClean(actor) = { "No thanks. "; }
;

dad_paper: decoration
    noun = 'paper' 'newspaper'
    location = living_room
    sdesc = "newspaper"
    ldesc = "Your dad is reading it with deliberate care. "
    whatdesc = "Part of the 4th estate. "
    verDoTake(actor) = { "Wait your turn! "; }
    verIoAskFor(actor) = {}
    verDoRead(actor) = { "Wait your turn! "; }
;

backyard: room
    cutOff = nil        // This is true if the aliens' energy curtain is
                        //  blocking the player's way back into the house
    gotoDir = &north
    gotoString = 'north'
    sdesc = "Backyard"
    ldesc = {
        "Normally the backyard is boring, but not tonight! Tonight the
            lawn has been crisped by the arrival of the silvery ship. It sits
            there, ";
        if (real_ship.rampDown)
            "<<link('ramp down', 'climb ramp')>>, ";
        "gleaming under the ";
        if (current_control.clockwise && !bright_lights.oneBroken)
            "harsh";
        else "bright";
        " backyard lights your dad installed a while back. Its shadow
            stretches across the brick patio and Weber grill. Your house
            is to the <<link('south', 's')>>";
        if (self.cutOff)
            ", blocked by a curtain of energy. ";
        else ". ";
    }
    leaveRoom(actor) = {    // Can't leave if the hose is attached to the faucet
        if (!inside(garden_hose, actor))
            pass leaveRoom;
        if (garden_hose.attachedToFaucet) {
            "Not until you either drop the hose or you detach it from the
                faucet. ";
            exit;
        }
        if (garden_hose.attachedToShip) {
            "The end of the garden hose slithers out of the ship&rsquo;s fuel
                tank and follows you.\b";
            garden_hose.detachFromShip;
        }
    }
    canGoNorth = { return (real_ship.rampDown && zigurt.location != self); }
    canGoSouth = (!self.cutOff)
    north = {
        if (!real_ship.rampDown) {
            "The ship&rsquo;s ramp isn&rsquo;t down. ";
            return nil;
        }
        if (zigurt.location == self) {
            "The aliens move, blocking your path. ";
            return nil;
        }
        return ship_hallway;
    }
    south = {
        if (self.cutOff) {
            "The energy curtain prevents you. ";
            return nil;
        }
        return hallway;
    }
    firstseen = {
        notify(alien_daemon, &lowerTheRamp, 4);
    }
;

energy_curtain: fixeditem
    noun = 'curtain'
    adjective = 'energy'
    sdesc = "curtain of energy"
    ldesc = "A crackling sheet of pure energy, harnessed and bent to the will
        of Zigurt and Floban. Right now, their will is that you not go back
        into the house. "
    whatdesc = "Some kinda energy that won&rsquo;t let you pass. "
    verDoTouch(actor) = { "You receive a mild shock. "; }
    verDoClimb(actor) = {
        "It shocks you, throwing you to the ground. ";
    }
;

lawn: fixeditem
    noun = 'lawn' 'semicircle' 'grass' 'crabgrass'
    adjective = 'burnt' 'crisped' 'big' 'black' 'green' 'zoysia'
    location = backyard
    sdesc = "lawn"
    ldesc = "The smooth green that dad spends so much time on has a big
        black semicircle burnt into it. Even mom&rsquo;s prized azaleas are no
        more&mdash;the ship crushed them when it landed. "
    whatdesc = "Crabgrass, mostly. "
    verIoPutOn(actor) = {}
    ioPutOn(actor, dobj) = {
        "You drop <<dobj.thedesc>>. ";
        dobj.moveInto(backyard);
    }
;

bright_lights: distantItem
    isThem = true
    oneBroken = nil             // True if one of them is broken
    noun = 'light' 'lights'
    adjective = 'bright' 'kleig' 'backyard' 'busted' 'broken'
    location = backyard
    sdesc = "bright lights"
    ldesc = {
        "Your dad&rsquo;s always adding this and that to the house,
            trying to improve it. A while back, he added these lights to the
            backyard, presumably because he wanted to show the mosquitos
            where to gather. ";
        if (self.oneBroken)
            "Someone has busted one of the lights. ";
        else if (current_control.clockwise)
            "Right now the lights are bright enough that you can&rsquo;t look
                anywhere near them without having to blink away
                afterimages. ";
        "The current control for the lights is mounted below them. ";
    }
    whatdesc = "Bright kleig lights your dad added to the backyard in one of
        his do-it-yourself fits. "
    verDoClimb(actor) = {
        "You can&rsquo;t get up to them. ";
    }
    verDoTurnon(actor) = {
        "The lights are already on. If you want to fiddle with their
            brightness, you need to play with the current control knob. ";
    }
    verDoTurnoff(actor) = {
        "Only dad can actually turn the lights off, but if you want to fiddle
            with their brightness, you need to play with the current control
            knob. ";
    }
;

current_control: fixeditem
    clockwise = nil
    noun = 'control' 'knob'
    adjective = 'current'
    location = backyard
    sdesc = "current control"
    ldesc = {
        "It&rsquo;s just a knob to control how bright the lights get. Right
            now it&rsquo;s turned <<self.clockwise ? "" :
            "counter">>clockwise. ";
    }
    whatdesc = "It&rsquo;s a knob which controls how bright the lights are. "
    verDoTurn(actor) = {
        "You can turn the knob clockwise or counterclockwise. ";
    }
    verDoTurnCW(actor) = {
        if (zigurt.location == backyard)
            "An unseen force pushes you back around to the front of the
                ship. <<zig('Do not go skulking about just yet,
                please,')>> Zigurt tells you. ";
        else if (!real_ship.rampDown)
            "The current control is behind the spaceship, and
                you&rsquo;re not sure you want to go poking around back
                there just yet. ";
        else if (self.clockwise)
            "The control is already turned as far clockwise as it&rsquo;ll
                go. ";
    }
    verDoTurnCCW(actor) = {
        if (zigurt.location == backyard)
            "An unseen force pushes you back around to the front of the
                ship. <<zig('Do not go skulking about just yet,
                please,')>> Zigurt tells you. ";
        else if (!real_ship.rampDown)
            "The current control is behind the spaceship, and
                you&rsquo;re not sure you want to go poking around back
                there just yet. ";
        else if (!self.clockwise)
            "The control is already turned as far counterclockwise as
                it&rsquo;ll go. ";
    }
    doTurnCW(actor) = {
        self.clockwise = true;
        "You twist the current control all the way clockwise. ";
        if (bright_lights.oneBroken) {
            "The lights brighten a bit, but not as much as they used to. ";
            return;
        }
        "The lights brighten to the point that the backyard is more lit
            up than it is during the day. ";
        notify(self, &breakALight, 5);
    }
    doTurnCCW(actor) = {
        self.clockwise = nil;
        "When you turn the current control counterclockwise, the lights
            dim";
        if (bright_lights.oneBroken)
            " a bit";
        ". ";
    }
    // The aliens come out to smash the lights
    breakALight = {
        if (!self.clockwise || bright_lights.oneBroken)
            return;
        if (zigurt.location != rec_room || Me.location == rec_room) {
            notify(self, &breakALight, 1);
            return;
        }
        bright_lights.oneBroken = true;
        incscore(4);
        if (Me.location == ship_hallway) {
            "\bYou step back as Zigurt and Floban come bustling out of the
                rec room. ";
            if (VALLEYSPEAK)
                zig('Like, totally excuuse us, okay?');
            else zig('Pardon, Ambassador,');
            "\ says Zigurt as they walk down the ramp and into the backyard.
                From outside you hear something go FZOOM!, then FZOOM!\ again.
                Shortly thereafter Zigurt and Floban return. Zigurt looks
                a little put-out, though Floban is grinning happily. They
                re-enter the rec room. ";
            return;
        }
        if (Me.location != backyard) {
            "\bFrom somewhere outside you hear something go FZOOM!, then
                FZOOM!\ again. ";
            return;
        }
        "\bYou are startled to see Zigurt come down the ramp, squinting and
            wiping away tears. Floban follows him, waving his tentacles
            about madly. ";
        if (VALLEYSPEAK)
            flo('Like, oh my gawd! This light is, like, way bright, and it is
                sooo keeping me from, y&rsquo;know, thinking.');
        else flo('What did I tell you? The light out here has
            become unbearable. How do you expect me to concentrate on our
            plans with the sound of so many photons bombarding our
            ship&rsquo;s repulso-field?');
        "\b";
        if (global.graphicsOn) {
            "<img src=\"Resources/Shoot Light.jpg\" align=\"right\">";
        }
        if (VALLEYSPEAK)
            zig('Totally,');
        else zig('Your point is taken,');
        " Zigurt replies thoughtfully. He
            brings out a large ray gun, complete with flanges and
            doohickeys. He braces its barrel against one tentacle and
            fires at the lights. With a FZOOM!\ a shower of sparks falls from
            a nearby tree.
        \bIn response, Zigurt clears his throat and wipes his tearing eyes
            once more. ";
        if (VALLEYSPEAK)
            zig('Whoa. That totally sucks. Lemme try again...');
        else zig('Somewhat...tricky...');
        " he mutters, taking
            aim once more. FZOOM!\ and more sparks fly from the tree,
            followed by a very put-out flock of nesting birds.
        \bZigurt is drawing a bead on the lights once more when something
            hits one of them and breaks it with a quiet tinkle. In the
            reduced glare, both you and Zigurt turn to look at Floban, who
            has dropped a tentaclefull of medium-sized rocks. ";
        // Take a break to play the sounds
        "<SOUND SRC=\"mp3/Laser Shot.mp3\" LAYER=\"FOREGROUND\">
        <SOUND SRC=\"Snds/1 Sec Silence.wav\" LAYER=\"FOREGROUND\">
        <SOUND SRC=\"Snds/1 Sec Silence.wav\" LAYER=\"FOREGROUND\">
        <SOUND SRC=\"mp3/Laser Shot.mp3\" LAYER=\"FOREGROUND\">
        <SOUND SRC=\"Snds/1 and half Sec Silence.wav\" LAYER=\"FOREGROUND\">
        <SOUND SRC=\"mp3/Light Breaks.mp3\" LAYER=\"FOREGROUND\">";
        if (VALLEYSPEAK)
            flo('Awesome! Am I like good or what? Like, being on the
                All-Solar team at the Academy was, y&rsquo;know, way good for
                me,');
        else flo('Not for nothing was I on the All-Solar team at the
            Academy,');
        " he says with a touch of pride.
        \b";
        if (VALLEYSPEAK)
            zig('Rad,');
        else zig('Indeed,');
        " says Zigurt as both ";
        if (backyard_faucet.ison && !(garden_hose.attachedToFaucet &&
            garden_hose.attachedToShip)) {
            "cautiously detour around the water-spewing ";
            if (garden_hose.attachedToFaucet)
                "hose";
            else "faucet";
            " and ";
        }
        "re-enter the ship. ";
    }
;

azaleas: fixeditem
    noun = 'azalea' 'azaleas' 'flower' 'flowers'
    adjective = 'prized' 'prize' 'crushed' 'smushed'
    location = backyard
    sdesc = "azaleas"
    ldesc = "It&rsquo;s pretty hard to see them, since they&rsquo;re smushed
        under the ship. "
    whatdesc = "n.\ a member of <i>Azalea</i>, fam.\ <i>Ericaceae</i>,
        often classified as a subgenus of <i>Rhododendron</i>, a flowering
        deciduous shrub, cultivated for the beauty and variety of its
        flowers [Mod.\ L.\ fr.\ Gk <i>azalea</i>, dry] "
    dobjGen(a, v, i, p) = {
        if (v != inspectVerb && !v.issysverb && v != whatisVerb) {
    	    "Well, duh, it&rsquo;s kinda hard to do anything about the
    	        azaleas, since there&rsquo;s a ship on them. ";
    	    exit;
    	}
    }
    iobjGen(a, v, d, p) = {
        if (v != inspectVerb && !v.issysverb &&
            !((v == askVerb || v == tellVerb) && p == aboutPrep)) {
    	    "Well, duh, it&rsquo;s kinda hard to do anything about the
    	        azaleas, since there&rsquo;s a ship on them. ";
    	    exit;
    	}
    }
    verDoSprayWith(actor, io) = {}
    verIoSprayAt(actor) = {}
;

// The fence and tree aren't described in the room's ldesc; they're only in
// the pictures. But I'll put them in here so people can examine them
fence: fixeditem
    noun = 'fence'
    adjective = 'wooden' 'wood' 'short'
    location = backyard
    sdesc = "short fence"
    ldesc = "The short fence marks the boundaries of your backyard. Dad put
        it up just before he bought you that puppy. Too bad Sparky loved
        chasing cars like he did.\bBut boy did he look happy the day he
        caught that VW microbus. "
    whatdesc = "The fence is just some wood that marks where your yard ends
        and the neighbors&rsquo; begins. "
    verDoClimb(actor) = {
        "Yikes! And get a handfull of splinters? Dad never did get around to
            painting it or anything. ";
    }
;

tree: fixeditem
    noun = 'tree'
    location = backyard
    sdesc = "tree"
    ldesc = "You used to climb that tree a lot, before some of the bottom
        branches died and your dad had to hack them off. "
    whatdesc = "Think of it as a really tall, woody flower. "
    verDoClimb(actor) = {
        "There aren't any branches low enough for you to grab ahold of. ";
    }
;

patio: fixeditem
    noun = 'patio' 'bricks'
    adjective = 'brick'
    location = backyard
    sdesc = "brick patio"
    ldesc = "Dad spent one summer putting it in. He got the bricks from an old
        house, so they were covered in mortar. He offered you ten cents a
        brick to scrape the mortar off&mdash;yeah, like you wanted to spend
        your vacation working for the Man. The Weber is on one side of the
        patio, a free-standing faucet on the other. "
    whatdesc = "An area of brick behind the house. "
    verIoPutOn(actor) = {}
    ioPutOn(actor, dobj) = {
        dobj.moveInto(backyard);
        "You drop <<dobj.thedesc>>. ";
    }
;

weber: fixeditem
    noun = 'weber' 'grill'
    adjective = 'weber' 'red' 'barbecue' 'barbeque' 'bbq'
    location = backyard
    sdesc = "Weber grill"
    ldesc = "Roundish, red, with three legs and wheels. Occasionally dad uses
        it to flash-burn meat into flaky, charcoal lumps. "
    whatdesc = "A grill, for burning meat until it&rsquo;s inedible. "
    verDoOpen(actor) = {
        "No thanks. Last time dad cooked something in it, it almost exploded
            from all the lighter fluid. If you open it and get soot all over
            you, you&rsquo;ll probably have to clean it. ";
    }
    verDoLookin(actor) = {
        "Like you&rsquo;re Superman or something. ";
    }
    verDoTurnon(actor) = {
        "Seeing as how that involves matches and burning and, most
            importantly, opening the grill, you&rsquo;re better off giving
            this plan a miss. ";
    }
    verDoPush(actor) = {
        "You&rsquo;re still a little small to be pushing the Weber around. ";
    }
;

// The first time the player empties the fuel tank, the fuel puddle appears.
//  I mention the ship's hatch the first time the fuel is described.
fuel_puddle: fixeditem
    firstHeredesc = true
    noun = 'puddle'
    adjective = 'small' 'stinky' 'gas' 'gasoline' 'fuel'
    sdesc = "small puddle"
    heredesc = {
        "A small puddle of liquid has soaked the ground ";
        if (self.firstHeredesc) {
            "directly under a newly-opened hatch in the ship. ";
            self.firstHeredesc = nil;
        }
        else "under the ship. ";
    }
    ldesc = "The puddle is small and smells like gas. "
    takedesc = "Don&rsquo;t touch that stuff! "
    whatdesc = "It looks like gas or something. "
    verDoDrink(actor) = { "Yerg. Nuh-uh. "; }
;

// The second time etc., the water puddle appears
water_puddle: fixeditem
    noun = 'puddle'
    adjective = 'large' 'stinky' 'water'
    sdesc = "large puddle"
    heredesc = "A large puddle of liquid lies beneath the ship. "
    ldesc = "The water in the puddle looks pretty scummy, like it&rsquo;s
        been mixed with something nasty. "
    whatdesc = "It looks like water and fuel. "
    verDoDrink(actor) = {
        "You take a quick sip. Yuck. It tastes as nasty as it looks. And
            to top it off, now you&rsquo;ve got this strange metallic
            aftertaste in your mouth, like you&rsquo;ve been chewing
            tinfoil. ";
    }
;

// Be warned: the faucet is pretty complex. It has to handle having hoses
// attached to it, taken off of it, etc. etc.
backyard_faucet: fixeditem
    ison = nil
    hoseIsAttached = nil
    noun = 'faucet' 'spigot' 'tap'
    adjective = 'free-standing' 'rusty' 'rusted'
    location = backyard
    sdesc = "faucet"
    ldesc = {
        "A faucet on the end of a thin steel pipe jutting out of the ground.
            It&rsquo;s kinda bent, thanks to that time you tried to climb it
            when you were four. ";
        if (self.ison && !(garden_hose.attachedToFaucet &&
            garden_hose.attachedToShip)) {
            "Water is pouring from ";
            if (self.hoseIsAttached)
                "the far end of the attached hose. ";
            else "it. ";
        }
        else if (self.hoseIsAttached)
            "A green garden hose dangles from it. ";
    }
    whatdesc = "When you turn the handle, water comes out. "
    verDoClimb(actor) = {
        "Dad made it very clear what would happen to you, and
            more specifically to your butt, if you ever, and he meant ever,
            climbed it again. ";
    }
    verDoTurn(actor) = {
        if (self.ison)
            self.verDoTurnoff(actor);
        else self.verDoTurnon(actor);
    }
    doTurn(actor) = {
        if (self.ison)
            self.doTurnoff(actor);
        else self.doTurnon(actor);
    }
    verDoTurnon(actor) = {
        if (self.ison)
            "The faucet is already on. ";
        else if (dad.location == self.location)
            "That&rsquo;s not a good idea while dad&rsquo;s here. He&rsquo;s
                warned you about playing with the faucet. ";
    }
    doTurnon(actor) = {
        "You turn on the faucet with a twist. ";
        if (self.hoseIsAttached && garden_hose.attachedToShip) {
            if (real_ship.fillShip)
                "You hear water gurgle through the hose and into the ship. ";
        }
        else {
            "Water begins gushing from ";
            if (self.hoseIsAttached)
                "the far end of the hose. ";
            else "it. ";
        }
        self.ison = true;
        if (zigurt.location == backyard && zigurt.makingComments &&
            !zigurt.mentionedFaucet) {
            zigurt.mentionedFaucet = true;
            "\b<<zig('Oh, look, Floban&mdash;the child offers us
                refreshment!')>> says Zigurt.
            \b<<flo('Hush. You merely display your vast unknowledge,')
                >> Floban snaps back. <<flo('The human is obviously performing
                the ritual of the lawn.')>>
            \b<<zig('Mmm. Quite possibly. As long as I am not wetted in the
                process, I have little concern one way or the other.')>> ";
        }
    }
    verDoTurnoff(actor) = {
        if (!self.ison)
            "The faucet isn&rsquo;t on. ";
    }
    doTurnoff(actor) = {
        "The faucet is a little rusty, and harder to turn off than to turn
            on, but you manage it. ";
        self.ison = nil;
    }
    verDoAttachTo(actor, io) = {
        if (io != garden_hose) {
            "There&rsquo;s no need to attach ";
            if (io == nil)
                "that";
            else io.thedesc;
            " to the faucet. ";
        }
    }
    verIoAttachTo(actor) = {}
    ioAttachTo(actor, dobj) = {
        if (dobj != garden_hose)
            "There is no need to attach <<dobj.thedesc>> to the faucet. ";
        else dobj.ioAttachTo(actor, self);
    }
    doSynonym('AttachTo') = 'FastenTo' 'PlugIn' 'ScrewInto'
    ioSynonym('AttachTo') = 'FastenTo' 'PlugIn' 'ScrewInto'
    verIoPutOn(actor) = {}
    ioPutOn(actor, dobj) = {
        if (dobj != garden_hose)
            "There&rsquo;s no good surface on the faucet. ";
        else dobj.ioAttachTo(actor, self);
    }
    verDoDetach(actor) = {
        if (!self.hoseIsAttached)
            "The faucet isn&rsquo;t attached to anything. ";
    }
    doDetach(actor) = { garden_hose.ioDetachFrom(actor, self); }
    verDoDetachFrom(actor, iobj) = {
        if (iobj != garden_hose || !self.hoseIsAttached)
            "\^<<iobj.thedesc>> isn&rsquo;t attached to the faucet. ";
    }
    verIoDetachFrom(actor) = {}
    ioDetachFrom(actor, dobj) = {
        if (dobj != garden_hose)
            "\^<<dobj.thedesc>> isn&rsquo;t attached to the faucet. ";
        else dobj.ioDetachFrom(actor, self);
    }
    doSynonym('Detach') = 'Unfasten' 'Unscrew'
    doSynonym('DetachFrom') = 'UnplugFrom' 'UnscrewFrom'
    ioSynonym('DetachFrom') = 'UnplugFrom' 'UnscrewFrom'
    verDoTakeOut(actor, iobj) = {
        if (iobj != garden_hose)
            pass verDoTakeOut;
        if (!self.hoseIsAttached)
            "The hose isn&rsquo;t attached to the faucet. ";
    }
    ioTakeOut(actor, dobj) = {
        if (dobj != garden_hose)
            pass ioTakeOut;
        dobj.ioDetachFrom(actor, self);
    }
;

// running_water is a floatingItem. It's present in the backyard only if
// the faucet is on and the hose isn't attached both to the faucet and to
// the ship
running_water: fixeditem, floatingItem
    noun = 'water'
    adjective = 'running' 'gushing'
    location = {
        if (backyard_faucet.ison && !(garden_hose.attachedToFaucet &&
            garden_hose.attachedToShip))
            return backyard;
        return nil;
    }
    sdesc = "water"
    ldesc = {
        "Water is running from the ";
        if (backyard_faucet.hoseIsAttached)
            "end of the hose. ";
        else "faucet. ";
    }
    whatdesc = "You telling me you don&rsquo;t know what water is? Get outta
        here. "
    verDoTake(actor) = { "Sure. Like you can do that. "; }
    verDoDrink(actor) = {}
    doDrink(actor) = {
        "You take a sip of the water. It tastes like they dredged it from the
            river and forgot to strain out the crustaceans. ";
    }
    doTurnon -> backyard_faucet
    doTurnoff -> backyard_faucet
    verDoSprayAt(actor, io) = {
        if (!garden_hose.attachedToFaucet)
            "You&rsquo;re going to have a hard time doing that without some
                better way to direct the water. ";
    }
    doSprayAt(actor, io) = {
        garden_hose.doSprayAt(actor, io);
    }
    verIoSprayWith(actor) = {
        if (!garden_hose.attachedToFaucet)
            "You&rsquo;re going to have a hard time doing that without some
                better way to direct the water. ";
    }
    ioSprayWith(actor, dobj) = {
        garden_hose.ioSprayWith(actor, dobj);
    }
;

