/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993, 1994, Elan Feingold (feingold@zko.dec.com)           *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "callbacks.h"
#include "network.h"
#include "utils.h"
#include "riskgame.h"
#include "gui.h"
#include "server.h"
#include "client.h"
#include "debug.h"

/* Globals */
Int iWriteSock, iReadSock, iMode;

/* Private functions */
Int   _RISK_GetMsgType(Int iField); 
void *_RISK_GetMsg(Int iField, Int iIndex1, Int iIndex2);

/****************************************/
typedef struct _ContinentDatabase
{
  String  strName;
  Int     iValue;
  Int     iNumCountries;
} ContinentDatabase;

/****************************************/ 
typedef struct _CountryObject
{
  String   strName;            /* Name of the country */
  Int      iContinent;         /* The continent it forms part of */
  Int      iNumArmies;         /* Number of armies on the country */
  Int      iTextX, iTextY;     /* Where to write the text */
  Int      piOwner;            /* The player it belongs to */
  Int      piAdjCountries[6];  /* All attackable countries */
} CountryObject;

/****************************************/
typedef struct _PlayerObject
{	
  Int      iAttackMode, iDiceMode, iMsgDstMode;
  Boolean  fAlive;
  Int      iClient;
  String   strName, strColor;
  Int      iCountriesOwned, iNumArmies, iNumCards, iColor;
  Int      piCards[MAX_CARDS];
} PlayerObject;

/****************************************/
typedef struct _Game
{
  void               (*BroadcastCallback)(Int, void *);
  void               (*FinishBroadcastCallback)(Int, Int, void *);
  void               (*SendMsgCallback)(Int, Int, void *);
  void               (*MessageCallback)(Int, void *);
  void               (*SendSyncMsgCallback)(Int, Int, void *, Int, Int, 
					    void (*)(Int, void *));
  Int                iNumPlayers;
  Int                iNumLivePlayers;
  ContinentDatabase  pContinents[NUM_CONTINENTS];
  PlayerObject       pPlayers[MAX_PLAYERS];
  CountryObject      pCountries[NUM_COUNTRIES];
} Game;

/* The object */
Game RiskGame = 
{
  NULL, NULL, NULL, NULL, NULL,
  0, 0, 
  {
    "North America",  5, 9,
    "South America",  2, 4,
    "Africa",         3, 6,
    "Australia",      2, 4,
    "Asia",           7, 12,
    "Europe",         5, 7,
  },
  {
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 1 */
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 2 */
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 3 */
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 4 */
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 5 */
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 6 */
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 7 */
    { 1, 3, 0, TRUE, 0, NULL, NULL, 0, 0, 0, 0, {0,0,0,0,0,0} }, /* Player 8 */
  },
  { 
    { "Greenland",   CNT_NORTHAMERICA,  0, 264,  32, -1, 
	{1, 5, 10, 11, -1, -1} },
    { "Iceland",     CNT_EUROPE,        0, 342,  51, -1, 
	{0, 7, 14, -1, -1, -1} },
    { "Siberia",     CNT_ASIA,          0, 622,  91, -1, 
	{3, 6, 13, 21, 22, -1} },
    { "Ural",        CNT_ASIA,          0, 557, 106, -1, 
	{2, 8, 15, 22, -1, -1} },
    { "Alaska",      CNT_NORTHAMERICA,  0,  13,  61, -1, 
	{5, 9, 12, -1, -1, -1} },
    { "Northwest Territories", CNT_NORTHAMERICA, 0, 61, 62, -1, 
	{0, 4, 11, 12, -1, -1} },
    { "Irkutsk",      CNT_ASIA,          0, 700,  71, -1, 
	{2, 9, 13, -1, -1, -1} },
    { "Scandinavia", CNT_EUROPE,        0, 387,  85, -1, 
	{1, 8, 14, 16, -1, -1} },
    { "Ukraine",      CNT_EUROPE,        0, 450, 103, -1, 
	{3, 7, 15, 16, 20, 24}},
    { "Kamchatka",   CNT_ASIA,          0, 731, 121, -1, 
	{4, 6, 13, 21, 23, -1} },
    { "Quebec",    CNT_NORTHAMERICA,  0, 181,  95, -1, 
	{0, 11, 18, -1, -1, -1} },
    { "Ontario",     CNT_NORTHAMERICA,  0, 116,  97, -1, 
	{0, 5, 10, 12, 17, 18} },
    { "Alberta",     CNT_NORTHAMERICA,  0,  55,  90, -1, 
	{4, 5, 11, 17, -1, -1} },
    { "Yakutsk",     CNT_ASIA,          0, 690, 111, -1, 
	{2, 6, 9, 21, -1, -1} },
    { "Great Britain", CNT_EUROPE,      0, 338, 110, -1, 
	{1, 7, 16, 19, -1, -1} },
    { "Afghanistan",  CNT_ASIA,          0, 534, 141, -1,
	{3, 8, 22, 24, 28, -1} },
    { "Northern Europe", CNT_EUROPE,    0, 386, 113, -1,
	{7, 8, 14, 19, 20, -1} },
    { "Western United States", CNT_NORTHAMERICA, 0, 62, 126, -1,
	{11, 12, 18, 25, -1, -1} },
    { "Eastern United States", CNT_NORTHAMERICA, 0, 126, 145, -1,
	{10, 11, 17, 25, -1, -1} },
    { "Western Europe",  CNT_EUROPE,    0, 338, 148, -1,
	{14, 16, 20, 26, -1, -1} },
    { "Southern Europe", CNT_EUROPE,    0, 416, 134, -1,
	{8, 16, 19, 24, 26, 27} },
    { "Mongolia",    CNT_ASIA,          0, 685, 139, -1,
	{2, 9, 13, 22, 23, -1} }, 
    { "China",       CNT_ASIA,          0, 632, 163, -1,
	{2, 3, 15, 21, 28, 29} },
    { "Japan",       CNT_ASIA,          0, 735, 166, -1, 
	{9, 21, -1, -1, -1, -1} },
    { "Middle East", CNT_ASIA,          0, 477, 193, -1, 
	{8, 15, 20, 27, 28, 30} },
    { "Central America", CNT_NORTHAMERICA, 0, 87, 165, -1, 
	{17, 18, 32, -1, -1, -1}}, 
    { "Northern Africa", CNT_AFRICA,    0, 339, 214, -1, 
	{19, 20, 27, 30, 33, 35} },
    { "Egypt",       CNT_AFRICA,        0, 407, 194, -1, 
	{20, 24, 26, 30, -1, -1} },
    { "India",       CNT_ASIA,          0, 572, 194, -1,
	{15, 22, 24, 29, -1, -1} },
    { "Siam",        CNT_ASIA,          0, 637, 207, -1, 
	{22, 28, 31, -1, -1, -1} },
    { "Eastern Africa", CNT_AFRICA,     0, 437, 231, -1, 
	{24, 26, 27, 35, 37, 40} },
    { "Indonesia",   CNT_AUSTRALIA,     0, 672, 262, -1, 
	{29, 36, 38, -1, -1, -1} },
    { "Venezuela",   CNT_SOUTHAMERICA,  0, 188, 230, -1, 
	{25, 33, 34, -1, -1, -1} },
    { "Brazil",      CNT_SOUTHAMERICA,   0, 233, 277, -1, 
	{26, 32, 34, 41, -1, -1} },
    { "Peru",        CNT_SOUTHAMERICA,   0, 194, 294, -1, 
	{32, 33, 41, -1, -1, -1}},
    { "Congo",       CNT_AFRICA,         0, 410, 271, -1, 
	{26, 30, 37, -1, -1, -1} },
    { "New Guinea",  CNT_AUSTRALIA,      0, 751, 276, -1, 
	{31, 38, 39, -1, -1, -1} },
    { "South Africa", CNT_AFRICA,        0, 416, 326, -1, 
	{30, 35, 40, -1, -1, -1} },
    { "Western Australia", CNT_AUSTRALIA, 0, 700, 335, -1, 
	{31, 36, 39, -1, -1, -1} },
    { "Eastern Australia", CNT_AUSTRALIA, 0, 756, 343, -1, 
	{36, 38, -1, -1, -1, -1} },
    { "Madagascar",  CNT_AFRICA,         0, 479, 321, -1, 
	{30, 37, -1, -1, -1, -1} },
    { "Argentina",   CNT_SOUTHAMERICA,   0, 189, 352, -1, 
	{33, 34, -1, -1, -1, -1} },
  }
};


/************************************************************************ 
 *  FUNCTION: _RISK_Replicate
 *  HISTORY: 
 *     05.03.94  ESF  Created.
 *     06.24.94  ESF  Fixed memory leak.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void _RISK_Replicate(Int iField, Int iIndex1, Int iIndex2)
{
  void *pvMess;
  Int   iMessType;

  /* Get the message and the message type */
  pvMess = _RISK_GetMsg(iField, iIndex1, iIndex2);
  iMessType = _RISK_GetMsgType(iField);
  
  /* If I am a client, call the message handler */
  if (iMode & MODE_CLIENT && RiskGame.MessageCallback)
    RiskGame.MessageCallback(iMessType, pvMess);
  
  /* Broadcast if a server, send to server if client */
  if (iMode & MODE_CLIENT && RiskGame.SendMsgCallback)
    RiskGame.SendMsgCallback(iWriteSock, iMessType, pvMess);
  else if (iMode & MODE_SERVER && RiskGame.BroadcastCallback)
    RiskGame.BroadcastCallback(iMessType, pvMess);

  /* Delete the memory the message was taking */
  MEM_Free(pvMess);
}


/************************************************************************ 
 *  FUNCTION: RISK_
 *  HISTORY: 
 *     05.02.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void RISK_Initialize(Int iObjMode, Int iWSock, Int iRSock)
{
  /* Init the sockets and the mode */
  iWriteSock = iWSock;
  iReadSock  = iRSock;
  iMode      = iObjMode;
  
  RISK_ResetObj();
}


/************************************************************************ 
 *  FUNCTION: RISK_
 *  HISTORY: 
 *     05.06.94  ESF  Created.
 *     05.10.94  ESF  Added needed check for existence of callback.
 *  PURPOSE: 
 *     Sends data to a client, for purposes of data synchronicity.
 *  NOTES: 
 ************************************************************************/
void RISK_SelectiveReplicate(Int iSocket, Int iField, 
			     Int iMin1, Int iMax1, 
			     Int iMin2, Int iMax2)
{
  Int i, j;

  /* Loop through getting values to replicate */
  for (i=iMin1; i<=iMin1; i++)
    for (j=iMin2; j<=iMin2; j++)
      if (RiskGame.SendMsgCallback)
	RiskGame.SendMsgCallback(iSocket,
				 _RISK_GetMsgType(iField), 
				 _RISK_GetMsg(iField, i, j)); 
      else
	printf("Error: Undefined callback...\n");
}


/************************************************************************ 
 *  FUNCTION: RISK_
 *  HISTORY: 
 *     05.02.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void RISK_ResetObj(void)
{
  Int i;

  RISK_SetNumPlayers(0);
  RISK_SetNumLivePlayers(0);

  /* Init the needed fields of the players */
  for (i=0; i!=MAX_PLAYERS; i++)
    {
      RISK_SetAttackModeOfPlayer(i, 1);
      RISK_SetStateOfPlayer(i, TRUE);
      RISK_SetClientOfPlayer(i, -1);
      RISK_SetNumCountriesOfPlayer(i, 0);
      RISK_SetNumArmiesOfPlayer(i, 0);
      RISK_SetNumCardsOfPlayer(i, 0);
    }

  /* Init the needed fields of the countries */
  for (i=0; i!=NUM_COUNTRIES; i++)
    {
      RISK_SetNumArmiesOfCountry(i, 0);
      RISK_SetOwnerOfCountry(i, -1);
    }
}


/************************************************************************ 
 *  FUNCTION: RISK_
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *  PURPOSE:
 *     New game, keep players.
 *  NOTES: 
 ************************************************************************/
void RISK_ResetGame(void)
{
  Int i;

  /* Bring the players to life again */
  RISK_SetNumLivePlayers(RISK_GetNumPlayers());

  /* Init the needed fields of the players */
  for (i=0; i!=MAX_PLAYERS; i++)
    {
      RISK_SetStateOfPlayer(i, TRUE);
      RISK_SetNumCountriesOfPlayer(i, 0);
      RISK_SetNumCardsOfPlayer(i, 0);
    }

  /* Init the needed fields of the countries */
  for (i=0; i!=NUM_COUNTRIES; i++)
    {
      RISK_SetNumArmiesOfCountry(i, 0);
      RISK_SetOwnerOfCountry(i, -1);
    }
}


/************************************************************************ 
 *  FUNCTION: RISK_AllocPlayer
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *     05.15.94  ESF  Fixed race condition, go to server for player.
 *  PURPOSE:
 *  NOTES: 
 ************************************************************************/
Int RISK_AllocPlayer(void (*pfMsgHandler)(Int, void *))
{
  /* The callback is needed! */
  if (!RiskGame.SendSyncMsgCallback)
    {
      printf("Fatal Error!  RISK: Need SendSyncMsgCallback to be set.\n");
      UTIL_ExitProgram(-1);
    }

  RiskGame.SendSyncMsgCallback(iWriteSock, MSG_ALLOCPLAYER, NULL,
			       iReadSock, MSG_REPLYPACKET, pfMsgHandler);
  
  return(iReply);
}


/************************************************************************ 
 *  FUNCTION: RISK_FreePlayer
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *     05.15.94  ESF  Fixed race condition, go to server.
 *  PURPOSE:
 *  NOTES: 
 ************************************************************************/
void RISK_FreePlayer(Int i)
{
  MsgFreePlayer mess;

  /* Tell the server that this player ID is free */
  mess.iPlayer = i;
  RiskGame.SendMsgCallback(iWriteSock, MSG_FREEPLAYER, &mess);
}


/************************************************************************ 
 *  FUNCTION: RISK_ProcessMessage
 *  HISTORY: 
 *     05.02.94  ESF  Created.
 *     05.10.94  ESF  Added needed check for existence of callback.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void RISK_ProcessMessage(Int iMessType, void *pvMess, Int iClientSrc)
{
  Int               iIndex1, iIndex2, iValue, iField;
  String            strValue;
  MsgObjIntUpdate  *pIntMess = (MsgObjIntUpdate *)pvMess;
  MsgObjStrUpdate  *pStrMess = (MsgObjStrUpdate *)pvMess;

  /* Broadcast the message to the other clients if I'm a server */
  if (iMode & MODE_SERVER && RiskGame.FinishBroadcastCallback)
    RiskGame.FinishBroadcastCallback(iClientSrc, iMessType, pvMess);

  if (iMessType == MSG_OBJINTUPDATE)
    {
      iField  = pIntMess->iField;
      iIndex1 = pIntMess->iIndex1;
      iIndex2 = pIntMess->iIndex2;
      iValue  = pIntMess->iNewValue;
    }
  else
    {
      iField  = pStrMess->iField;
      iIndex1 = pStrMess->iIndex1;
      iIndex2 = pStrMess->iIndex2;
      strValue  = pStrMess->strNewValue;
    }

  if (RiskGame.MessageCallback)
    RiskGame.MessageCallback(iMessType, pvMess);
  
  switch (iField)
    {
    case PLR_ATTACKMODE:
      RiskGame.pPlayers[iIndex1].iAttackMode = iValue;
      break;
    case PLR_DICEMODE:
      RiskGame.pPlayers[iIndex1].iDiceMode = iValue;
      break;
    case PLR_MSGDSTMODE:
      RiskGame.pPlayers[iIndex1].iMsgDstMode = iValue;
      break;
    case PLR_STATE:
      RiskGame.pPlayers[iIndex1].fAlive = iValue;
      break;
    case PLR_CLIENT:
      RiskGame.pPlayers[iIndex1].iClient = iValue;
      break;
    case PLR_NUMCOUNTRIES:
      RiskGame.pPlayers[iIndex1].iCountriesOwned = iValue;
      break;
    case PLR_NUMARMIES:
      RiskGame.pPlayers[iIndex1].iNumArmies = iValue;
      break;
    case PLR_NUMCARDS:
      RiskGame.pPlayers[iIndex1].iNumCards = iValue;
      break;
    case PLR_COLORINDEX:
      RiskGame.pPlayers[iIndex1].iColor = iValue;
      break;
    case PLR_CARD:
      RiskGame.pPlayers[iIndex1].piCards[iIndex2] = iValue;
      break;
    case CNT_NUMARMIES:
      RiskGame.pCountries[iIndex1].iNumArmies = iValue;
      break;
    case CNT_OWNER:
      RiskGame.pCountries[iIndex1].piOwner = iValue;
      break;
    case GEN_NUMPLAYERS:    
      RiskGame.iNumPlayers = iValue;
      break;
    case GEN_NUMLIVEPLAYERS:
      RiskGame.iNumLivePlayers = iValue;
      break;
    case PLR_NAME:
      if (RiskGame.pPlayers[iIndex1].strName != NULL)
	MEM_Free(RiskGame.pPlayers[iIndex1].strName);
      RiskGame.pPlayers[iIndex1].strName = 
	(String)MEM_Alloc(strlen(strValue)+1);
      strcpy(RiskGame.pPlayers[iIndex1].strName, strValue);
      break;
    case PLR_COLORSTRING:
      if (RiskGame.pPlayers[iIndex1].strColor != NULL)
	MEM_Free(RiskGame.pPlayers[iIndex1].strColor);
      RiskGame.pPlayers[iIndex1].strColor = 
	(String)MEM_Alloc(strlen(strValue)+1);
      strcpy(RiskGame.pPlayers[iIndex1].strColor, strValue);
      break;
    default:
      printf("Trying to modify a Read Only field [%d]...\n", iMessType);
    }
}

/***************************************/
void RISK_SetBroadcastCallback(void (*func)(Int, void *))
{
  RiskGame.BroadcastCallback = func;
}

/***************************************/
void RISK_SetSendMsgCallback(void (*func)(Int, Int, void *))
{
  RiskGame.SendMsgCallback = func;
}

/***************************************/
void RISK_SetSendSyncMsgCallback(void (*func)(Int, Int, void *, Int, Int, 
				 void (*)(Int, void *)))
{
  RiskGame.SendSyncMsgCallback = func;
}

/***************************************/
void RISK_SetMessageCallback(void (*func)(Int, void *))
{
  RiskGame.MessageCallback = func;
}

/***************************************/
void RISK_SetFinishBroadcastCallback(void (*func)(Int, Int, void *))
{
  RiskGame.FinishBroadcastCallback = func;
}

/***************************************/
void RISK_SetNumPlayers(Int iNumPlayers) 
{
  RiskGame.iNumPlayers = iNumPlayers;
  _RISK_Replicate(GEN_NUMPLAYERS, 0, 0);
}

/***************************************/
void RISK_SetAttackModeOfPlayer(Int iPlayer, Int iMode)
{
  RiskGame.pPlayers[iPlayer].iAttackMode = iMode;
  _RISK_Replicate(PLR_ATTACKMODE, iPlayer, 0);
}

/***************************************/
void RISK_SetDiceModeOfPlayer(Int iPlayer, Int iMode)
{
  RiskGame.pPlayers[iPlayer].iDiceMode = iMode;
  _RISK_Replicate(PLR_DICEMODE, iPlayer, 0);
}

/***************************************/
void RISK_SetMsgDstModeOfPlayer(Int iPlayer, Int iMode)
{
  RiskGame.pPlayers[iPlayer].iMsgDstMode = iMode;
  _RISK_Replicate(PLR_MSGDSTMODE, iPlayer, 0);
}

/***************************************/
void RISK_SetStateOfPlayer(Int iPlayer, Boolean fState)
{
  RiskGame.pPlayers[iPlayer].fAlive = fState;
  _RISK_Replicate(PLR_STATE, iPlayer, 0);
}

/***************************************/
void RISK_SetClientOfPlayer(Int iPlayer, Int iClient)
{
  RiskGame.pPlayers[iPlayer].iClient = iClient;
  _RISK_Replicate(PLR_CLIENT, iPlayer, 0);
}

/***************************************/
void RISK_SetNumCountriesOfPlayer(Int iPlayer, Int iNumCountries)
{
  RiskGame.pPlayers[iPlayer].iCountriesOwned = iNumCountries;
  _RISK_Replicate(PLR_NUMCOUNTRIES, iPlayer, 0);
}

/***************************************/
void RISK_SetNumArmiesOfPlayer(Int iPlayer, Int iNumArmies)
{
  RiskGame.pPlayers[iPlayer].iNumArmies = iNumArmies;
  _RISK_Replicate(PLR_NUMARMIES, iPlayer, 0);
}

/***************************************/
void RISK_SetNumCardsOfPlayer(Int iPlayer, Int iNumCards)
{
  RiskGame.pPlayers[iPlayer].iNumCards = iNumCards;
  _RISK_Replicate(PLR_NUMCARDS, iPlayer, 0);
}

/***************************************/
void RISK_SetNameOfPlayer(Int iPlayer, String strName)
{
  if (RiskGame.pPlayers[iPlayer].strName != NULL)
    MEM_Free(RiskGame.pPlayers[iPlayer].strName);
  RiskGame.pPlayers[iPlayer].strName = (String)MEM_Alloc(strlen(strName)+1);
  strcpy(RiskGame.pPlayers[iPlayer].strName, strName);

  _RISK_Replicate(PLR_NAME, iPlayer, 0);
}

/***************************************/
void RISK_SetColorStringOfPlayer(Int iPlayer, String strColor)
{
  if (RiskGame.pPlayers[iPlayer].strColor != NULL)
    MEM_Free(RiskGame.pPlayers[iPlayer].strColor);
  RiskGame.pPlayers[iPlayer].strColor = (String)MEM_Alloc(strlen(strColor)+1);
  strcpy(RiskGame.pPlayers[iPlayer].strColor, strColor);

  _RISK_Replicate(PLR_COLORSTRING, iPlayer, 0);
}

/***************************************/
void RISK_SetColorIndexOfPlayer(Int iPlayer, Int iColor)
{
  RiskGame.pPlayers[iPlayer].iColor = iColor;
  _RISK_Replicate(PLR_COLORINDEX, iPlayer, 0);
}

/***************************************/
void RISK_SetCardOfPlayer(Int iPlayer, Int iCard, Int iValue)
{
  RiskGame.pPlayers[iPlayer].piCards[iCard] = iValue;
  _RISK_Replicate(PLR_CARD, iPlayer, iCard);
}

/***************************************/
void RISK_SetNumArmiesOfCountry(Int iCountry, Int iNumArmies)
{
  RiskGame.pCountries[iCountry].iNumArmies = iNumArmies;
  _RISK_Replicate(CNT_NUMARMIES, iCountry, 0);
}

/***************************************/
void RISK_SetOwnerOfCountry(Int iCountry, Int iOwner)
{
  RiskGame.pCountries[iCountry].piOwner = iOwner;
  _RISK_Replicate(CNT_OWNER, iCountry, 0);
}

/***************************************/
void RISK_SetNumLivePlayers(Int iNumLivePlayers)
{
  RiskGame.iNumLivePlayers = iNumLivePlayers;
  _RISK_Replicate(GEN_NUMLIVEPLAYERS, 0, 0);
}


/***************************************/
Int RISK_GetNumPlayers(void) 
{
  return RiskGame.iNumPlayers;
}

/***************************************/
Int RISK_GetAttackModeOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iAttackMode;
}

/***************************************/
Int RISK_GetMsgDstModeOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iMsgDstMode;
}

/***************************************/
Int RISK_GetDiceModeOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iDiceMode;
}

/***************************************/
Boolean RISK_GetStateOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].fAlive;
}

/***************************************/
Int RISK_GetClientOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iClient;
}

/***************************************/
Int RISK_GetNumCountriesOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iCountriesOwned;
}

/***************************************/
Int RISK_GetNumArmiesOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iNumArmies;
}

/***************************************/
Int RISK_GetNumCardsOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iNumCards;
}

/***************************************/
Int RISK_GetValueOfContinent(Int iContinent)
{
  return RiskGame.pContinents[iContinent].iValue;
}

/***************************************/
String RISK_GetNameOfContinent(Int iContinent)
{
  return RiskGame.pContinents[iContinent].strName;
}

/***************************************/
Int RISK_GetNumCountriesOfContinent(Int iContinent)
{
  return RiskGame.pContinents[iContinent].iNumCountries;
}

/***************************************/
String RISK_GetNameOfPlayer(Int iPlayer) 
{ 
  return RiskGame.pPlayers[iPlayer].strName;
}

/***************************************/
String RISK_GetColorStringOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].strColor;
}

/***************************************/
Int RISK_GetColorIndexOfPlayer(Int iPlayer)
{
  return RiskGame.pPlayers[iPlayer].iColor;
}

/***************************************/
Int RISK_GetCardOfPlayer(Int iPlayer, Int iCard)
{
  return RiskGame.pPlayers[iPlayer].piCards[iCard];
}

/***************************************/
String RISK_GetNameOfCountry(Int iCountry)
{
  return RiskGame.pCountries[iCountry].strName;
}

/***************************************/
Int RISK_GetContinentOfCountry(Int iCountry)
{
  return RiskGame.pCountries[iCountry].iContinent;
}

/***************************************/
Int RISK_GetNumArmiesOfCountry(Int iCountry)
{
  return RiskGame.pCountries[iCountry].iNumArmies;
}

/***************************************/
Int RISK_GetOwnerOfCountry(Int iCountry)
{
  return RiskGame.pCountries[iCountry].piOwner;
}

/***************************************/
Int RISK_GetAdjCountryOfCountry(Int iCountry, Int iIndex)
{
  return RiskGame.pCountries[iCountry].piAdjCountries[iIndex];
}

/***************************************/
Int RISK_GetNumLivePlayers(void)
{
  return RiskGame.iNumLivePlayers;
}

/***************************************/
Int RISK_GetTextXOfCountry(Int iCountry)
{
  return RiskGame.pCountries[iCountry].iTextX;
}

/***************************************/
Int RISK_GetTextYOfCountry(Int iCountry)
{
  return RiskGame.pCountries[iCountry].iTextY;
}

/***************************************/
Int _RISK_GetMsgType(Int iField)
{
  switch (iField)
    {
    case PLR_ATTACKMODE:
    case PLR_DICEMODE:
    case PLR_MSGDSTMODE:
    case PLR_STATE:     
    case PLR_CLIENT:    
    case PLR_NUMCOUNTRIES:
    case PLR_NUMARMIES:   
    case PLR_NUMCARDS:    
    case PLR_COLORINDEX:  
    case PLR_CARD:        
    case CNT_CONTINENT:   
    case CNT_NUMARMIES:   
    case CNT_OWNER:     
    case CNT_ADJCOUNTRY:  
    case CNT_TEXTX:       
    case CNT_TEXTY:       
    case CON_VALUE:         
    case CON_NUMCOUNTRIES:  
    case GEN_NUMPLAYERS:    
    case GEN_NUMLIVEPLAYERS:
      return MSG_OBJINTUPDATE;
      break;
      
    case PLR_NAME:        
    case PLR_COLORSTRING: 
    case CNT_NAME:        
    case CON_NAME:          
      return MSG_OBJSTRUPDATE;
      break;    
    }
  
  printf("Error: Undefined field.\n");
  return(-1);
}

/***************************************/
void *_RISK_GetMsg(Int iField, Int iIndex1, Int iIndex2)
{
  MsgObjIntUpdate *pIntMess = 
    (MsgObjIntUpdate *)MEM_Alloc(sizeof(MsgObjIntUpdate));
  MsgObjStrUpdate *pStrMess = 
    (MsgObjStrUpdate *)MEM_Alloc(sizeof(MsgObjStrUpdate));
  
  pIntMess->iField  = pStrMess->iField  = iField;
  pIntMess->iIndex1 = pStrMess->iIndex1 = iIndex1;
  pIntMess->iIndex2 = pStrMess->iIndex2 = iIndex2;

  switch (iField)
    {
    case PLR_ATTACKMODE:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iAttackMode;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_DICEMODE:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iDiceMode;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_MSGDSTMODE:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iMsgDstMode;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_STATE:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].fAlive;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_CLIENT:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iClient;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_NUMCOUNTRIES:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iCountriesOwned;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_NUMARMIES:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iNumArmies;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_NUMCARDS:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iNumCards;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_COLORINDEX:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].iColor;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case PLR_CARD:
      pIntMess->iNewValue = RiskGame.pPlayers[iIndex1].piCards[iIndex2];
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case CNT_NUMARMIES:
      pIntMess->iNewValue = RiskGame.pCountries[iIndex1].iNumArmies;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case CNT_OWNER:
      pIntMess->iNewValue = RiskGame.pCountries[iIndex1].piOwner;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case GEN_NUMPLAYERS:
      pIntMess->iNewValue = RiskGame.iNumPlayers;
      MEM_Free(pStrMess);
      return (void *)pIntMess;
      break;
    case GEN_NUMLIVEPLAYERS:
      pIntMess->iNewValue = RiskGame.iNumLivePlayers;
      MEM_Free(pStrMess);
      return (void *)pIntMess;      
      break;

    case PLR_NAME:
      pStrMess->strNewValue = RiskGame.pPlayers[iIndex1].strName;
      MEM_Free(pIntMess);
      return (void *)pStrMess;
      break;
    case PLR_COLORSTRING:
      pStrMess->strNewValue = RiskGame.pPlayers[iIndex1].strColor;
      MEM_Free(pIntMess);
      return (void *)pStrMess;
      break;
    }

  printf("Error: Undefined field!\n");
  return NULL;
}
