/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>

#include "libsysactivity.h"

int error = 0;

void print_swap_info(struct sa_swap* swap) {
#ifdef SA_SWAP_NAME
	printf("swap name: %s", swap->name);
#endif
#ifdef SA_SWAP_TYPE
	printf(swap->type == 1 ? ", type: partition" : ", type: file");
#endif
#ifdef SA_SWAP_TOTAL
	printf(", total: %"PRIu64, swap->total);
#endif
#ifdef SA_SWAP_FREE
	printf(", free: %"PRIu64, swap->free);
#endif
	printf("\n");
}

void test_swap_info(struct sa_swap* swap) {
	if (swap->free > swap->total) {
		printf("%s:%d ERROR: The free swap space is larger than the total space\n", __FILE__, __LINE__);
		error = 1;
	}
#ifdef SA_SWAP_NAME
#ifdef __NetBSD__
	size_t len = strlen(swap->name);
#else
	size_t len = strnlen(swap->name, SA_SWAP_NAME);
#endif
	if (len == 0) {
		printf("%s:%d ERROR: The length of the swap name is zero\n", __FILE__, __LINE__);
		error = 1;
	}
	if (len == SA_SWAP_NAME) {
		printf("%s:%d ERROR: The length of the swap name is equals to the maximum name length\n", __FILE__, __LINE__);
		error = 1;
	}
#endif
#ifdef SA_SWAP_TYPE
	if (swap->type != 1 && swap->type != 2) {
		printf("%s:%d ERROR: The value of the swap type is: %d\n", __FILE__, __LINE__, swap->type);
		error = 1;
	}
#endif
}

void* get_swap_info(void* arg) {
	int i;
	int ret;

#ifdef SA_OPEN_SWAP
	ret = sa_open_swap();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_open_swap(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif

	ret = sa_reset_swaps();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_reset_swaps(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}

	struct sa_swap swap;
	ret = sa_get_swap(0, &swap);
	if (ret == ENODEV) {
		printf("%s:%d WARNING: sa_get_swap() couldn't find any swap.\n", __FILE__, __LINE__);
		exit(EXIT_SUCCESS);
	}
	if (ret != 0) {
		printf("%s:%d ERROR: sa_get_swap(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
	print_swap_info(&swap);
	printf("\n");
	test_swap_info(&swap);

	struct sa_swap* swaps = NULL;
	for (i = 0; i < 5; i++) {
		uint16_t number_swaps = 0;
		ret = sa_count_swaps(&number_swaps);
		if (ret != 0) {
			printf("%s:%d ERROR: sa_count_swaps(): %s\n", __FILE__, __LINE__, strerror(ret));
			exit(EXIT_FAILURE);
		}
		printf("number of swaps: %d\n", number_swaps);

		ret = sa_reset_swaps();
		if (ret != 0) {
			printf("%s:%d ERROR: sa_reset_swaps(): %s\n", __FILE__, __LINE__, strerror(ret));
			exit(EXIT_FAILURE);
		}

		swaps = (struct sa_swap*) realloc(swaps, number_swaps * sizeof (struct sa_swap));
		uint16_t written = 0;
		ret = sa_get_swaps(swaps, number_swaps, &written);
		if (ret != 0) {
			printf("%s:%d ERROR: sa_get_swaps(): %s\n", __FILE__, __LINE__, strerror(ret));
			exit(EXIT_FAILURE);
		}
		int j;
		for (j = 0; j < written; j++) {
			printf("swap number %d\n", j);
			print_swap_info(&swaps[j]);
			printf("\n");
			test_swap_info(&swaps[j]);
		}
		sleep(1);
	}
	if (swaps != NULL)
		free(swaps);

#ifdef SA_CLOSE_SWAP
	ret = sa_close_swap();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_close_swap(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_OPEN_SWAP
	ret = sa_open_swap();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_open_swap(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_CLOSE_SWAP
	ret = sa_close_swap();
	if (ret != 0) {
		printf("%s:%d ERROR: sa_close_swap(): %s\n", __FILE__, __LINE__, strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
	return NULL;
}

int main() {
	pthread_t thread1;
	struct timespec delay;
	delay.tv_sec = 0;
	delay.tv_nsec = 500000000;

	pthread_create(&thread1, NULL, get_swap_info, NULL);
	nanosleep(&delay, NULL);
	get_swap_info(NULL);

	if (error)
		return EXIT_FAILURE;

	return EXIT_SUCCESS;
}
