// SPDX-License-Identifier: BSD-3-Clause
/* Copyright (c) 2023, Linaro Ltd. */

#include <sys/mman.h>
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "debugcc.h"

static struct gcc_mux gcc = {
	.mux = {
		.phys =	0x100000,
		.size = 0x1f4200,

		.measure = measure_gcc,

		.enable_reg = 0x62004,
		.enable_mask = BIT(0),

		.mux_reg = 0x62024,
		.mux_mask = 0x3ff,

		.div_reg = 0x62000,
		.div_mask = 0xf,
		.div_val = 2,
	},

	.xo_div4_reg = 0x62008,
	.debug_ctl_reg = 0x62048,
	.debug_status_reg = 0x6204c,
};

static struct debug_mux cam_cc = {
	.phys = 0xade0000,
	.size = 0x20000,
	.block_name = "cam",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x6a,

	.enable_reg = 0x14008,
	.enable_mask = BIT(0),

	.mux_reg = 0x16000,
	.mux_mask = 0xff,

	.div_reg = 0x14004,
	.div_mask = 0xf,
	.div_val = 4,
};

static struct debug_mux disp_cc = {
	.phys = 0xaf00000,
	.size = 0x20000,
	.block_name = "disp",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x6f,

	.enable_reg = 0xd004,
	.enable_mask = BIT(0),

	.mux_reg = 0x11000,
	.mux_mask = 0x1ff,

	.div_reg = 0xd000,
	.div_mask = 0xf,
	.div_val = 4,
};

static struct debug_mux gpu_cc = {
	.phys = 0x3d90000,
	.size = 0xa000,
	.block_name = "gpu",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x18d,

	.enable_reg = 0x9274,
	.enable_mask = BIT(0),

	.mux_reg = 0x9564,
	.mux_mask = 0xff,

	.div_reg = 0x9270,
	.div_mask = 0xf,
	.div_val = 2,
};

static struct debug_mux video_cc = {
	.phys = 0xaaf0000,
	.size = 0x10000,
	.block_name = "video",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x78,

	.enable_reg = 0x80fc,
	.enable_mask = BIT(0),

	.mux_reg = 0x9a4c,
	.mux_mask = 0x3f,

	.div_reg = 0x80f8,
	.div_mask = 0xf,
	.div_val = 3,
};

static struct measure_clk sm8650_clocks[] = {
	/* GCC entries */
	{ "gcc_aggre_noc_pcie_axi_clk", &gcc.mux, 0x3f },
	{ "gcc_aggre_ufs_phy_axi_clk", &gcc.mux, 0x41 },
	{ "gcc_aggre_usb3_prim_axi_clk", &gcc.mux, 0x40 },
	{ "gcc_boot_rom_ahb_clk", &gcc.mux, 0xe2 },
	{ "gcc_camera_ahb_clk", &gcc.mux, 0x62 },
	{ "gcc_camera_hf_axi_clk", &gcc.mux, 0x65 },
	{ "gcc_camera_sf_axi_clk", &gcc.mux, 0x67 },
	{ "gcc_camera_xo_clk", &gcc.mux, 0x69 },
	{ "gcc_cfg_noc_pcie_anoc_ahb_clk", &gcc.mux, 0x2c },
	{ "gcc_cfg_noc_usb3_prim_axi_clk", &gcc.mux, 0x1e },
	{ "gcc_cnoc_pcie_sf_axi_clk", &gcc.mux, 0x18 },
	{ "gcc_cpuss_ubwcp_clk", &gcc.mux, 0x132 },
	{ "gcc_ddrss_gpu_axi_clk", &gcc.mux, 0xfe },
	{ "gcc_ddrss_pcie_sf_qtb_clk", &gcc.mux, 0xff },
	{ "gcc_ddrss_ubwcp_clk", &gcc.mux, 0x107 },
	{ "gcc_disp_ahb_clk", &gcc.mux, 0x6b },
	{ "gcc_disp_hf_axi_clk", &gcc.mux, 0x6d },
	{ "gcc_disp_xo_clk", &gcc.mux, 0x6e },
	{ "gcc_gp1_clk", &gcc.mux, 0x144 },
	{ "gcc_gp2_clk", &gcc.mux, 0x145 },
	{ "gcc_gp3_clk", &gcc.mux, 0x146 },
	{ "gcc_gpu_cfg_ahb_clk", &gcc.mux, 0x18a },
	{ "gcc_gpu_gpll0_clk_src", &gcc.mux, 0x191 },
	{ "gcc_gpu_gpll0_div_clk_src", &gcc.mux, 0x192 },
	{ "gcc_gpu_memnoc_gfx_clk", &gcc.mux, 0x18e },
	{ "gcc_gpu_snoc_dvm_gfx_clk", &gcc.mux, 0x190 },
	{ "gcc_pcie_0_aux_clk", &gcc.mux, 0x14c },
	{ "gcc_pcie_0_cfg_ahb_clk", &gcc.mux, 0x14b },
	{ "gcc_pcie_0_mstr_axi_clk", &gcc.mux, 0x14a },
	{ "gcc_pcie_0_phy_rchng_clk", &gcc.mux, 0x14e },
	{ "gcc_pcie_0_pipe_clk", &gcc.mux, 0x14d },
	{ "gcc_pcie_0_slv_axi_clk", &gcc.mux, 0x149 },
	{ "gcc_pcie_0_slv_q2a_axi_clk", &gcc.mux, 0x148 },
	{ "gcc_pcie_1_aux_clk", &gcc.mux, 0x155 },
	{ "gcc_pcie_1_cfg_ahb_clk", &gcc.mux, 0x154 },
	{ "gcc_pcie_1_mstr_axi_clk", &gcc.mux, 0x153 },
	{ "gcc_pcie_1_phy_aux_clk", &gcc.mux, 0x156 },
	{ "gcc_pcie_1_phy_rchng_clk", &gcc.mux, 0x158 },
	{ "gcc_pcie_1_pipe_clk", &gcc.mux, 0x157 },
	{ "gcc_pcie_1_slv_axi_clk", &gcc.mux, 0x152 },
	{ "gcc_pcie_1_slv_q2a_axi_clk", &gcc.mux, 0x151 },
	{ "gcc_pdm2_clk", &gcc.mux, 0xd3 },
	{ "gcc_pdm_ahb_clk", &gcc.mux, 0xd1 },
	{ "gcc_pdm_xo4_clk", &gcc.mux, 0xd2 },
	{ "gcc_qmip_camera_nrt_ahb_clk", &gcc.mux, 0x63 },
	{ "gcc_qmip_camera_rt_ahb_clk", &gcc.mux, 0x64 },
	{ "gcc_qmip_disp_ahb_clk", &gcc.mux, 0x6c },
	{ "gcc_qmip_gpu_ahb_clk", &gcc.mux, 0x18b },
	{ "gcc_qmip_pcie_ahb_clk", &gcc.mux, 0x147 },
	{ "gcc_qmip_video_cv_cpu_ahb_clk", &gcc.mux, 0x74 },
	{ "gcc_qmip_video_cvp_ahb_clk", &gcc.mux, 0x71 },
	{ "gcc_qmip_video_v_cpu_ahb_clk", &gcc.mux, 0x73 },
	{ "gcc_qmip_video_vcodec_ahb_clk", &gcc.mux, 0x72 },
	{ "gcc_qupv3_i2c_core_clk", &gcc.mux, 0xa3 },
	{ "gcc_qupv3_i2c_s0_clk", &gcc.mux, 0xa4 },
	{ "gcc_qupv3_i2c_s1_clk", &gcc.mux, 0xa5 },
	{ "gcc_qupv3_i2c_s2_clk", &gcc.mux, 0xa6 },
	{ "gcc_qupv3_i2c_s3_clk", &gcc.mux, 0xa7 },
	{ "gcc_qupv3_i2c_s4_clk", &gcc.mux, 0xa8 },
	{ "gcc_qupv3_i2c_s5_clk", &gcc.mux, 0xa9 },
	{ "gcc_qupv3_i2c_s6_clk", &gcc.mux, 0xaa },
	{ "gcc_qupv3_i2c_s7_clk", &gcc.mux, 0xab },
	{ "gcc_qupv3_i2c_s8_clk", &gcc.mux, 0xac },
	{ "gcc_qupv3_i2c_s9_clk", &gcc.mux, 0xad },
	{ "gcc_qupv3_i2c_s_ahb_clk", &gcc.mux, 0xa2 },
	{ "gcc_qupv3_wrap1_core_2x_clk", &gcc.mux, 0xb1 },
	{ "gcc_qupv3_wrap1_core_clk", &gcc.mux, 0xb0 },
	{ "gcc_qupv3_wrap1_qspi_ref_clk", &gcc.mux, 0xba },
	{ "gcc_qupv3_wrap1_s0_clk", &gcc.mux, 0xb2 },
	{ "gcc_qupv3_wrap1_s1_clk", &gcc.mux, 0xb3 },
	{ "gcc_qupv3_wrap1_s2_clk", &gcc.mux, 0xb4 },
	{ "gcc_qupv3_wrap1_s3_clk", &gcc.mux, 0xb5 },
	{ "gcc_qupv3_wrap1_s4_clk", &gcc.mux, 0xb6 },
	{ "gcc_qupv3_wrap1_s5_clk", &gcc.mux, 0xb7 },
	{ "gcc_qupv3_wrap1_s6_clk", &gcc.mux, 0xb8 },
	{ "gcc_qupv3_wrap1_s7_clk", &gcc.mux, 0xb9 },
	{ "gcc_qupv3_wrap2_core_2x_clk", &gcc.mux, 0xbe },
	{ "gcc_qupv3_wrap2_core_clk", &gcc.mux, 0xbd },
	{ "gcc_qupv3_wrap2_ibi_ctrl_2_clk", &gcc.mux, 0xc9 },
	{ "gcc_qupv3_wrap2_ibi_ctrl_3_clk", &gcc.mux, 0xca },
	{ "gcc_qupv3_wrap2_s0_clk", &gcc.mux, 0xbf },
	{ "gcc_qupv3_wrap2_s1_clk", &gcc.mux, 0xc0 },
	{ "gcc_qupv3_wrap2_s2_clk", &gcc.mux, 0xc1 },
	{ "gcc_qupv3_wrap2_s3_clk", &gcc.mux, 0xc2 },
	{ "gcc_qupv3_wrap2_s4_clk", &gcc.mux, 0xc3 },
	{ "gcc_qupv3_wrap2_s5_clk", &gcc.mux, 0xc4 },
	{ "gcc_qupv3_wrap2_s6_clk", &gcc.mux, 0xc5 },
	{ "gcc_qupv3_wrap2_s7_clk", &gcc.mux, 0xc6 },
	{ "gcc_qupv3_wrap3_core_2x_clk", &gcc.mux, 0xce },
	{ "gcc_qupv3_wrap3_core_clk", &gcc.mux, 0xcd },
	{ "gcc_qupv3_wrap3_qspi_ref_clk", &gcc.mux, 0xd0 },
	{ "gcc_qupv3_wrap3_s0_clk", &gcc.mux, 0xcf },
	{ "gcc_qupv3_wrap_1_m_ahb_clk", &gcc.mux, 0xae },
	{ "gcc_qupv3_wrap_1_s_ahb_clk", &gcc.mux, 0xaf },
	{ "gcc_qupv3_wrap_2_ibi_2_ahb_clk", &gcc.mux, 0xc7 },
	{ "gcc_qupv3_wrap_2_ibi_3_ahb_clk", &gcc.mux, 0xc8 },
	{ "gcc_qupv3_wrap_2_m_ahb_clk", &gcc.mux, 0xbb },
	{ "gcc_qupv3_wrap_2_s_ahb_clk", &gcc.mux, 0xbc },
	{ "gcc_qupv3_wrap_3_m_ahb_clk", &gcc.mux, 0xcb },
	{ "gcc_qupv3_wrap_3_s_ahb_clk", &gcc.mux, 0xcc },
	{ "gcc_sdcc2_ahb_clk", &gcc.mux, 0x9d },
	{ "gcc_sdcc2_apps_clk", &gcc.mux, 0x9c },
	{ "gcc_sdcc4_ahb_clk", &gcc.mux, 0xa0 },
	{ "gcc_sdcc4_apps_clk", &gcc.mux, 0x9f },
	{ "gcc_ufs_phy_ahb_clk", &gcc.mux, 0x15d },
	{ "gcc_ufs_phy_axi_clk", &gcc.mux, 0x15c },
	{ "gcc_ufs_phy_ice_core_clk", &gcc.mux, 0x163 },
	{ "gcc_ufs_phy_phy_aux_clk", &gcc.mux, 0x164 },
	{ "gcc_ufs_phy_rx_symbol_0_clk", &gcc.mux, 0x15f },
	{ "gcc_ufs_phy_rx_symbol_1_clk", &gcc.mux, 0x165 },
	{ "gcc_ufs_phy_tx_symbol_0_clk", &gcc.mux, 0x15e },
	{ "gcc_ufs_phy_unipro_core_clk", &gcc.mux, 0x162 },
	{ "gcc_usb30_prim_master_clk", &gcc.mux, 0x8f },
	{ "gcc_usb30_prim_mock_utmi_clk", &gcc.mux, 0x91 },
	{ "gcc_usb30_prim_sleep_clk", &gcc.mux, 0x90 },
	{ "gcc_usb3_prim_phy_aux_clk", &gcc.mux, 0x92 },
	{ "gcc_usb3_prim_phy_com_aux_clk", &gcc.mux, 0x93 },
	{ "gcc_usb3_prim_phy_pipe_clk", &gcc.mux, 0x94 },
	{ "gcc_video_ahb_clk", &gcc.mux, 0x70 },
	{ "gcc_video_axi0_clk", &gcc.mux, 0x75 },
	{ "gcc_video_axi1_clk", &gcc.mux, 0x76 },
	{ "gcc_video_xo_clk", &gcc.mux, 0x77 },
	{ "mc_cc_debug_mux", &gcc.mux, 0x10b },
	{ "measure_only_cnoc_clk", &gcc.mux, 0x16 },
	{ "measure_only_ipa_2x_clk", &gcc.mux, 0x176 },
	{ "measure_only_memnoc_clk", &gcc.mux, 0x103 },
	{ "measure_only_snoc_clk", &gcc.mux, 0xb },
	{ "pcie_0_pipe_clk", &gcc.mux, 0x14f },
	{ "pcie_1_phy_aux_clk", &gcc.mux, 0x15a },
	{ "pcie_1_pipe_clk", &gcc.mux, 0x159 },
	{ "ufs_phy_rx_symbol_0_clk", &gcc.mux, 0x161 },
	{ "ufs_phy_rx_symbol_1_clk", &gcc.mux, 0x167 },
	{ "ufs_phy_tx_symbol_0_clk", &gcc.mux, 0x160 },
	{ "usb3_phy_wrapper_gcc_usb30_pipe_clk", &gcc.mux, 0x98 },
	/* CAMCC Entries */
	{ "cam_cc_bps_ahb_clk", &cam_cc, 0x17 },
	{ "cam_cc_bps_clk", &cam_cc, 0x18 },
	{ "cam_cc_bps_fast_ahb_clk", &cam_cc, 0x16 },
	{ "cam_cc_bps_shift_clk", &cam_cc, 0x7b },
	{ "cam_cc_camnoc_axi_nrt_clk", &cam_cc, 0x86 },
	{ "cam_cc_camnoc_axi_rt_clk", &cam_cc, 0x49 },
	{ "cam_cc_camnoc_dcd_xo_clk", &cam_cc, 0x4a },
	{ "cam_cc_camnoc_xo_clk", &cam_cc, 0x60 },
	{ "cam_cc_cci_0_clk", &cam_cc, 0x44 },
	{ "cam_cc_cci_1_clk", &cam_cc, 0x45 },
	{ "cam_cc_cci_2_clk", &cam_cc, 0x61 },
	{ "cam_cc_core_ahb_clk", &cam_cc, 0x4d },
	{ "cam_cc_cpas_ahb_clk", &cam_cc, 0x46 },
	{ "cam_cc_cpas_bps_clk", &cam_cc, 0x19 },
	{ "cam_cc_cpas_cre_clk", &cam_cc, 0x5d },
	{ "cam_cc_cpas_fast_ahb_clk", &cam_cc, 0x47 },
	{ "cam_cc_cpas_ife_0_clk", &cam_cc, 0x25 },
	{ "cam_cc_cpas_ife_1_clk", &cam_cc, 0x2a },
	{ "cam_cc_cpas_ife_2_clk", &cam_cc, 0x2f },
	{ "cam_cc_cpas_ife_lite_clk", &cam_cc, 0x34 },
	{ "cam_cc_cpas_ipe_nps_clk", &cam_cc, 0x1b },
	{ "cam_cc_cpas_sbi_clk", &cam_cc, 0x22 },
	{ "cam_cc_cpas_sfe_0_clk", &cam_cc, 0x39 },
	{ "cam_cc_cpas_sfe_1_clk", &cam_cc, 0x3d },
	{ "cam_cc_cpas_sfe_2_clk", &cam_cc, 0x78 },
	{ "cam_cc_cre_ahb_clk", &cam_cc, 0x5e },
	{ "cam_cc_cre_clk", &cam_cc, 0x5c },
	{ "cam_cc_csi0phytimer_clk", &cam_cc, 0x9 },
	{ "cam_cc_csi1phytimer_clk", &cam_cc, 0xc },
	{ "cam_cc_csi2phytimer_clk", &cam_cc, 0xe },
	{ "cam_cc_csi3phytimer_clk", &cam_cc, 0x10 },
	{ "cam_cc_csi4phytimer_clk", &cam_cc, 0x12 },
	{ "cam_cc_csi5phytimer_clk", &cam_cc, 0x14 },
	{ "cam_cc_csi6phytimer_clk", &cam_cc, 0x58 },
	{ "cam_cc_csi7phytimer_clk", &cam_cc, 0x5a },
	{ "cam_cc_csid_clk", &cam_cc, 0x48 },
	{ "cam_cc_csid_csiphy_rx_clk", &cam_cc, 0xb },
	{ "cam_cc_csiphy0_clk", &cam_cc, 0xa },
	{ "cam_cc_csiphy1_clk", &cam_cc, 0xd },
	{ "cam_cc_csiphy2_clk", &cam_cc, 0xf },
	{ "cam_cc_csiphy3_clk", &cam_cc, 0x11 },
	{ "cam_cc_csiphy4_clk", &cam_cc, 0x13 },
	{ "cam_cc_csiphy5_clk", &cam_cc, 0x15 },
	{ "cam_cc_csiphy6_clk", &cam_cc, 0x59 },
	{ "cam_cc_csiphy7_clk", &cam_cc, 0x5b },
	{ "cam_cc_drv_ahb_clk", &cam_cc, 0x75 },
	{ "cam_cc_drv_xo_clk", &cam_cc, 0x74 },
	{ "cam_cc_gdsc_clk", &cam_cc, 0x4e },
	{ "cam_cc_icp_ahb_clk", &cam_cc, 0x43 },
	{ "cam_cc_icp_clk", &cam_cc, 0x42 },
	{ "cam_cc_ife_0_clk", &cam_cc, 0x24 },
	{ "cam_cc_ife_0_fast_ahb_clk", &cam_cc, 0x28 },
	{ "cam_cc_ife_0_shift_clk", &cam_cc, 0x7f },
	{ "cam_cc_ife_1_clk", &cam_cc, 0x29 },
	{ "cam_cc_ife_1_fast_ahb_clk", &cam_cc, 0x2d },
	{ "cam_cc_ife_1_shift_clk", &cam_cc, 0x80 },
	{ "cam_cc_ife_2_clk", &cam_cc, 0x2e },
	{ "cam_cc_ife_2_fast_ahb_clk", &cam_cc, 0x32 },
	{ "cam_cc_ife_2_shift_clk", &cam_cc, 0x81 },
	{ "cam_cc_ife_lite_ahb_clk", &cam_cc, 0x37 },
	{ "cam_cc_ife_lite_clk", &cam_cc, 0x33 },
	{ "cam_cc_ife_lite_cphy_rx_clk", &cam_cc, 0x36 },
	{ "cam_cc_ife_lite_csid_clk", &cam_cc, 0x35 },
	{ "cam_cc_ipe_nps_ahb_clk", &cam_cc, 0x1e },
	{ "cam_cc_ipe_nps_clk", &cam_cc, 0x1a },
	{ "cam_cc_ipe_nps_fast_ahb_clk", &cam_cc, 0x1f },
	{ "cam_cc_ipe_pps_clk", &cam_cc, 0x1c },
	{ "cam_cc_ipe_pps_fast_ahb_clk", &cam_cc, 0x20 },
	{ "cam_cc_ipe_shift_clk", &cam_cc, 0x7c },
	{ "cam_cc_jpeg_1_clk", &cam_cc, 0x5f },
	{ "cam_cc_jpeg_clk", &cam_cc, 0x40 },
	{ "cam_cc_mclk0_clk", &cam_cc, 0x1 },
	{ "cam_cc_mclk1_clk", &cam_cc, 0x2 },
	{ "cam_cc_mclk2_clk", &cam_cc, 0x3 },
	{ "cam_cc_mclk3_clk", &cam_cc, 0x4 },
	{ "cam_cc_mclk4_clk", &cam_cc, 0x5 },
	{ "cam_cc_mclk5_clk", &cam_cc, 0x6 },
	{ "cam_cc_mclk6_clk", &cam_cc, 0x7 },
	{ "cam_cc_mclk7_clk", &cam_cc, 0x8 },
	{ "cam_cc_qdss_debug_clk", &cam_cc, 0x4b },
	{ "cam_cc_qdss_debug_xo_clk", &cam_cc, 0x4c },
	{ "cam_cc_sbi_clk", &cam_cc, 0x21 },
	{ "cam_cc_sbi_fast_ahb_clk", &cam_cc, 0x23 },
	{ "cam_cc_sbi_shift_clk", &cam_cc, 0x7e },
	{ "cam_cc_sfe_0_clk", &cam_cc, 0x38 },
	{ "cam_cc_sfe_0_fast_ahb_clk", &cam_cc, 0x3b },
	{ "cam_cc_sfe_0_shift_clk", &cam_cc, 0x82 },
	{ "cam_cc_sfe_1_clk", &cam_cc, 0x3c },
	{ "cam_cc_sfe_1_fast_ahb_clk", &cam_cc, 0x3f },
	{ "cam_cc_sfe_1_shift_clk", &cam_cc, 0x83 },
	{ "cam_cc_sfe_2_clk", &cam_cc, 0x76 },
	{ "cam_cc_sfe_2_fast_ahb_clk", &cam_cc, 0x7a },
	{ "cam_cc_sfe_2_shift_clk", &cam_cc, 0x84 },
	{ "cam_cc_sleep_clk", &cam_cc, 0x4f },
	{ "cam_cc_titan_top_shift_clk", &cam_cc, 0x85 },
	/* DISPCC Entries */
	{ "disp_cc_mdss_accu_clk", &disp_cc, 0x46 },
	{ "disp_cc_mdss_ahb1_clk", &disp_cc, 0x37 },
	{ "disp_cc_mdss_ahb_clk", &disp_cc, 0x33 },
	{ "disp_cc_mdss_byte0_clk", &disp_cc, 0x14 },
	{ "disp_cc_mdss_byte0_intf_clk", &disp_cc, 0x15 },
	{ "disp_cc_mdss_byte1_clk", &disp_cc, 0x16 },
	{ "disp_cc_mdss_byte1_intf_clk", &disp_cc, 0x17 },
	{ "disp_cc_mdss_dptx0_aux_clk", &disp_cc, 0x20 },
	{ "disp_cc_mdss_dptx0_crypto_clk", &disp_cc, 0x1D },
	{ "disp_cc_mdss_dptx0_link_clk", &disp_cc, 0x1A },
	{ "disp_cc_mdss_dptx0_link_intf_clk", &disp_cc, 0x1C },
	{ "disp_cc_mdss_dptx0_pixel0_clk", &disp_cc, 0x1E },
	{ "disp_cc_mdss_dptx0_pixel1_clk", &disp_cc, 0x1F },
	{ "disp_cc_mdss_dptx0_usb_router_link_intf_clk", &disp_cc, 0x1B },
	{ "disp_cc_mdss_dptx1_aux_clk", &disp_cc, 0x27 },
	{ "disp_cc_mdss_dptx1_crypto_clk", &disp_cc, 0x26 },
	{ "disp_cc_mdss_dptx1_link_clk", &disp_cc, 0x23 },
	{ "disp_cc_mdss_dptx1_link_intf_clk", &disp_cc, 0x25 },
	{ "disp_cc_mdss_dptx1_pixel0_clk", &disp_cc, 0x21 },
	{ "disp_cc_mdss_dptx1_pixel1_clk", &disp_cc, 0x22 },
	{ "disp_cc_mdss_dptx1_usb_router_link_intf_clk", &disp_cc, 0x24 },
	{ "disp_cc_mdss_dptx2_aux_clk", &disp_cc, 0x2D },
	{ "disp_cc_mdss_dptx2_crypto_clk", &disp_cc, 0x2C },
	{ "disp_cc_mdss_dptx2_link_clk", &disp_cc, 0x2A },
	{ "disp_cc_mdss_dptx2_link_intf_clk", &disp_cc, 0x2B },
	{ "disp_cc_mdss_dptx2_pixel0_clk", &disp_cc, 0x28 },
	{ "disp_cc_mdss_dptx2_pixel1_clk", &disp_cc, 0x29 },
	{ "disp_cc_mdss_dptx3_aux_clk", &disp_cc, 0x31 },
	{ "disp_cc_mdss_dptx3_crypto_clk", &disp_cc, 0x32 },
	{ "disp_cc_mdss_dptx3_link_clk", &disp_cc, 0x2F },
	{ "disp_cc_mdss_dptx3_link_intf_clk", &disp_cc, 0x30 },
	{ "disp_cc_mdss_dptx3_pixel0_clk", &disp_cc, 0x2E },
	{ "disp_cc_mdss_esc0_clk", &disp_cc, 0x18 },
	{ "disp_cc_mdss_esc1_clk", &disp_cc, 0x19 },
	{ "disp_cc_mdss_mdp1_clk", &disp_cc, 0x34 },
	{ "disp_cc_mdss_mdp_clk", &disp_cc, 0x11 },
	{ "disp_cc_mdss_mdp_lut1_clk", &disp_cc, 0x35 },
	{ "disp_cc_mdss_mdp_lut_clk", &disp_cc, 0x12 },
	{ "disp_cc_mdss_non_gdsc_ahb_clk", &disp_cc, 0x38 },
	{ "disp_cc_mdss_pclk0_clk", &disp_cc, 0xF },
	{ "disp_cc_mdss_pclk1_clk", &disp_cc, 0x10 },
	{ "disp_cc_mdss_rscc_ahb_clk", &disp_cc, 0x3A },
	{ "disp_cc_mdss_rscc_vsync_clk", &disp_cc, 0x39 },
	{ "disp_cc_mdss_vsync1_clk", &disp_cc, 0x36 },
	{ "disp_cc_mdss_vsync_clk", &disp_cc, 0x13 },
	{ "disp_cc_sleep_clk", &disp_cc, 0x47 },
	{ "disp_cc_xo_clk", &disp_cc, 0x45 },
	/* GPUCC entries */
	{ "gpu_cc_ahb_clk", &gpu_cc, 0x18 },
	{ "gpu_cc_crc_ahb_clk", &gpu_cc, 0x19 },
	{ "gpu_cc_cx_ff_clk", &gpu_cc, 0x22 },
	{ "gpu_cc_cx_gmu_clk", &gpu_cc, 0x1f },
	{ "gpu_cc_cxo_aon_clk", &gpu_cc, 0xc },
	{ "gpu_cc_cxo_clk", &gpu_cc, 0x20 },
	{ "gpu_cc_demet_clk", &gpu_cc, 0xe },
	{ "gpu_cc_freq_measure_clk", &gpu_cc, 0xd },
	{ "gpu_cc_gx_ff_clk", &gpu_cc, 0x14 },
	{ "gpu_cc_gx_gfx3d_rdvm_clk", &gpu_cc, 0x16 },
	{ "gpu_cc_gx_gmu_clk", &gpu_cc, 0x13 },
	{ "gpu_cc_gx_vsense_clk", &gpu_cc, 0x10 },
	{ "gpu_cc_hub_aon_clk", &gpu_cc, 0x31 },
	{ "gpu_cc_hub_cx_int_clk", &gpu_cc, 0x21 },
	{ "gpu_cc_memnoc_gfx_clk", &gpu_cc, 0x23 },
	{ "gpu_cc_mnd1x_gfx3d_clk", &gpu_cc, 0x2d },
	{ "gpu_cc_sleep_clk", &gpu_cc, 0x1d },
	{ "measure_only_gpu_cc_cx_gfx3d_clk", &gpu_cc, 0x28 },
	{ "measure_only_gpu_cc_cx_gfx3d_slv_clk", &gpu_cc, 0x21 },
	{ "measure_only_gpu_cc_gx_gfx3d_clk", &gpu_cc, 0xf },
	/* VIDEOCC Entries */
	{ "video_cc_ahb_clk", &video_cc, 0x7 },
	{ "video_cc_mvs0_clk", &video_cc, 0x3 },
	{ "video_cc_mvs0c_clk", &video_cc, 0x1 },
	{ "video_cc_mvs1_clk", &video_cc, 0x5 },
	{ "video_cc_mvs1c_clk", &video_cc, 0x9 },
	{ "video_cc_sleep_clk", &video_cc, 0xc },
	{ "video_cc_xo_clk", &video_cc, 0xb },
	{}
};

struct debugcc_platform sm8650_debugcc = {
	"sm8650",
	sm8650_clocks,
};
