/*
 *  br_aes_glue.c
 *
 *  Written by Torrey Hoffman, March 2002
 *
 *  Copyright 2002 by Torrey Hoffman
 *  Redistribution of this file is permitted under the GNU Public License version 2.
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/if_ether.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include <linux/in.h>
#include <net/checksum.h>
#include "br_private.h"
#include "aes.h"

void br_aes_deletecontext(aes_ctx *context)
{
	/* Frees an aes_ctx */    
	if (context) {
		memset(context, 0, sizeof(aes_ctx));
		kfree(context);
	}
}

aes_ctx * br_aes_enc_context(unsigned char *key)
{
	/* Allocate, set up, and return a new encryption context (aes_ctx)
	 * from the key.  Returns NULL on error.
	 */
	aes_ctx *context  = (aes_ctx *) kmalloc(sizeof(aes_ctx), GFP_KERNEL);

	if (context) 
		aes_enc_key(key, BR_CRYPT_KEYLEN, context);

	return context;
}

aes_ctx * br_aes_dec_context(unsigned char *key)
{
	/* Allocate, set up, and return a new decryption context (aes_ctx)
	 * from the key.  Returns NULL on error.
	 */
	aes_ctx *context  = (aes_ctx *) kmalloc(sizeof(aes_ctx), GFP_KERNEL);

	if (context) 
		aes_dec_key(key, BR_CRYPT_KEYLEN, context);

	return context;
}

int br_aes_encrypt_frame(aes_ctx *context, struct sk_buff *skb)
{
	/* returns 0 if frame was encrypted, 
	 * negative if frame should be dropped, 
	 * positive if frame was not encrypted but should be forwarded anyway 
	 */
	static unsigned int sample_sec;
	static unsigned int sample_count;
	unsigned int avail;
	struct iphdr *iph;
	struct udphdr *uh;

	unsigned char *datastart;
	unsigned char *dataend;
	unsigned int blocks;

	unsigned char pad;
	u16 datalen;
	u16 check;

	cycles_t c_end;
	cycles_t c_start = get_cycles();

	/* drop all non-IP packets */
	if (ntohs(skb->protocol) != ETH_P_IP)
		return -1;

	/* Pass non-UDP IP without encryption. (TCP, IGMP, ICMP, etc) */
	if (skb->nh.iph->protocol != IPPROTO_UDP) 
		return 1;

	/* So it's UDP/IP, we will encrypt it. */
	iph = skb->nh.iph;
	uh = (struct udphdr*)(skb->data+(iph->ihl<<2));

	datastart = ((unsigned char *)uh) + 8;
	dataend = ((unsigned char *)uh) + ntohs(uh->len);
	datalen = ntohs(uh->len) - 8;

	/* Need to pad UDP data to multiple of AES block size.
	 *
	 * Padding strategy:
	 * -  Every packet is padded by 1 to BLOCK_SIZE bytes.
	 * -  Padding is zeros, but the number of bytes of padding 
	 *    is saved in last byte of the pad
	 * -  Padding increases packet length, we update UDP, IP, and 
	 *    ethernet headers to match.   
	 * -  Note that UDP header gets extra +8 as it includes the 
	 *    UDP header length subtracted above.
	 */

	blocks = (datalen / BLOCK_SIZE) + 1;
	pad = BLOCK_SIZE - (datalen % BLOCK_SIZE);

	/* how much room available at end of buffer? */
	avail = skb->end - skb->tail;

	if ( pad > avail ) {
		printk(KERN_WARNING "bridge encryption: sk_buff too small, passing.\n");
		return 1;
	}

	if ( (skb->len + pad) > ETH_DATA_LEN ) {
		printk(KERN_WARNING "bridge encryption: packet too big, rejecting!\n");
		return -1;
	}

	memset(dataend, 0, pad);
	dataend += pad - 1;
	*dataend = pad;

	uh->len = htons(datalen + pad + 8);
	iph->tot_len = htons(ntohs(iph->tot_len) + pad);
	skb->len += pad;

	/* AES-encrypt the UDP data in place using EBC mode */
	while (blocks--) {
		aes_enc_blk(datastart, datastart, context);
		datastart += BLOCK_SIZE;
	}

	/* if packet had a UDP checksum, recalculate it */
	if ((check = uh->check) != 0) {
		/* FIXME */
		printk("bridge encryption: UDP checksums in use. Doh!\n");
		return -1;
	}
	
	/* recalculate IP header checksum */
	iph->check = 0;
	iph->check = ip_fast_csum((unsigned char *)iph, iph->ihl);

	/* dump sample every 1024 jiffies (100 jiffies / second). */
	if ( (jiffies >> 10) == sample_sec ) sample_count++;
	else { 
		c_end = get_cycles();
		printk(KERN_DEBUG "br: IP src=%#lx dst=%#lx %d pkt %ld cyc\n", 
		       (unsigned long int)ntohl(iph->saddr), 
		       (unsigned long int)ntohl(iph->daddr), 
		       sample_count, (long)(c_end - c_start));

		printk(KERN_DEBUG "br: UDP srcpt=%#hx dstpt=%#hx len=%#hx av=%#hx\n",
		       ntohs(uh->source), ntohs(uh->dest), uh->len, avail);

		sample_sec = (jiffies >> 10);
		sample_count = 0;
	}

	return 0;
}

int br_aes_decrypt_frame(aes_ctx *context, struct sk_buff *skb)
{
	struct iphdr *iph;
	struct udphdr *uh;

	unsigned char *datastart;
	unsigned char *dataend;
	unsigned int blocks;

	unsigned char pad;
	u16 datalen;
	u16 check;

	/* drop all non-IP packets */
	if (ntohs(skb->protocol) != ETH_P_IP) {
		return -1;
	}

	/* Pass non-UDP IP without decryption. (TCP, IGMP, ICMP, etc) */
	if (skb->nh.iph->protocol != IPPROTO_UDP) return 1;

	/* So it's UDP/IP, we will decrypt it. */
	iph = skb->nh.iph;
	uh = (struct udphdr*)(skb->data+(iph->ihl<<2));

	/* data length should be a multiple of BLOCK_SIZE */
	datastart = ((unsigned char *)uh) + 8;
	datalen = ntohs(uh->len) - 8;
	blocks = datalen / BLOCK_SIZE;
	if (datalen % BLOCK_SIZE) {
		printk("br_aes_decrypt_frame: pkt length not multiple of BLOCK_SIZE, discarding\n");
		return -1;
	}

	/* AES-decrypt the UDP data using EBC mode */
	while (blocks--) {
		aes_enc_blk(datastart, datastart, context);
		datastart += BLOCK_SIZE;
	}

	/* depad UDP data after decryption, 
	   bytes of padding is stored in last byte */
	dataend = ((unsigned char *)uh) + ntohs(uh->len);
	pad = *dataend;
	/* decrease packet length - don't forget to 
	   add 8 for the UDP header length */
	uh->len = htons(datalen - pad + 8);
	iph->tot_len = htons(ntohs(iph->tot_len) - pad);
	skb->len -= pad;

	/* recalculate UDP checksum, if checksums are being used */
	if ((check = uh->check) != 0) {
		/* FIXME */
		printk("bridge decryption: UDP checksums in use. Doh!\n");
		return -1;
	}
	
	/* recalculate IP header checksum */
	iph->check = 0;
	iph->check = ip_fast_csum((unsigned char *)iph, iph->ihl);

	return 0;
}
