/* evfs 0.3 2003 sd */

#define	MINLEN			6	/* minimum password len */
#define	MAX_MOUNTS_PER_USER	3	/* maximum evfs mounts per user */
#define MOUNT_FLAGS		0	/* mount flags */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/fsuid.h>
#include <sys/mount.h>

#include "getpw.h"
#include "evfs.h"
#define __KERNEL__
#include "sha256.h"
#undef __KERNEL__

int	yes_no(void)
{
	int	c;
	while ((c = getchar()) != EOF) {
		if (toupper(c) == 'Y')
			return 1;
		if (toupper(c) == 'N')
			return 0;
	}
	/* shutup gcc */
	return 0;
}

void	do_sha256(void *dest, void *src, int len)
{
	struct sha256_ctx *ctx = salloc(sizeof *ctx);
	sha256_init(ctx);
	sha256_update(ctx, src, len);
	sha256_final(ctx, dest);
}

void	do_sha256_iter(void *dest, void *src, int len, int num)
{
	int	i;
	do_sha256(dest, src, len);

	for (i = 0; i < num; i++)
		do_sha256(dest, dest, 32);
}

int	isempty(char *path)
{
	DIR	*d = opendir(path);
	struct	dirent *de;
	int	count = 0;
	
	if (!d) {
		perror("path");
		return -1;
	}
	while ((de = readdir(d))) {
		if (strcmp(de->d_name, "."))
			continue;
		if (strcmp(de->d_name, ".."))
			continue;
		count++;
	}
	closedir(d);
	return !count;
}

int	isdir(char *file)
{
	int err;
	struct	stat st;
	err = stat(file, &st);
	if (err) return 0;
	return S_ISDIR(st.st_mode);
}

int	fuid(char *file, int *uid, int *gid)
{
	int err;
	struct	stat st;
	err = stat(file, &st);
	if (!err) {
		*uid = st.st_uid;
		*gid = st.st_gid;
	}
	return err;
}

int	do_umount(char *from)
{
	int	uid, gid;

	if (fuid(from, &uid, &gid)) {
		perror(from);
		return 1;
	}
	if ((uid != getuid()) || (gid != getgid())) {
		printf("uid/gid mismatch for '%s'\n", from);
		return 2;
	}
	if (umount(from)) {
		perror("umount");
		printf("umount of '%s' failed\n", from);
		return 3;
	} else {
		printf("'%s' was sucesfully unmounted!\n", from);
	}
	return 0;
}

/* get number of mounts for given uid */
int	get_mount_count(int uid)
{
	char	from[8192], to[8192], type[8192];
	char	buf[8192];
	int	count = 0;

	FILE	*fd = fopen("/proc/mounts", "r");
	if (!fd) {
		perror("/proc/mounts");
		return -1;
	}
	while (fgets(buf, sizeof(buf)-1, fd)) {
		int	uid, gid;

		if (sscanf(buf, "%s %s %s", from, to, type) != 3)
			continue;
		if (!strcasecmp(type, "evfs")) {
			if (fuid(to, &uid, &gid)) {
				perror(to);
				return -1;
			}
			if ((uid == getuid()) || (gid == getgid()))
				count++;
		}
	}
	fclose(fd);
	return count;
}

int	do_create(char *mark)
{
	char	*pwd1, *pwd2;
	int	fd;
	char	*sha;
	int	error = 0;

	fd = creat(mark, 0600);

	if (fd < 0) {
		perror(mark);
		return 1;
	}
	while (1) {
		pwd1 = getpwd("Enter your new password:");
		pwd2 = getpwd("Retype your new password:");

		if ((!pwd1 || !pwd2) || (strlen(pwd1) < MINLEN)) {
			printf("Sorry, but minimal length for password is %d characters\n", MINLEN);
			goto out;
		}

		if (strcmp(pwd1, pwd2)) {
			printf("Sorry, passwords do not match.\n");
			goto out;
		}
		putpwd(pwd2);
		break;
	out:
		putpwd(pwd1);
		putpwd(pwd2);
	}
	sha = salloc(32);
	/* k, we've passwords */
	do_sha256_iter(sha, pwd1, strlen(pwd1), 32);
	if (write(fd, sha, 2) != 2) {
		perror(mark);
		printf("Can't create evfs cookie!\n");
		error = 1;
	}
	close(fd);
	putpwd(pwd1);
	sfree(sha);
	return error;
}

int	verify_cookie(char *path, char *pass)
{
	char	*sha = salloc(32);
	char	*cookie = salloc(32);
	int	fd = open(path, O_RDONLY);
	int	error = 0;

	if (fd < 0) {
		perror(path);
		return 1;
	}

	if (read(fd, cookie, 2) != 2) {
		printf("Can't read cookie file '%s'\n", path);
		close(fd);
		return 1;
	}
	close(fd);
	do_sha256_iter(sha, pass, strlen(pass), 32);
	if (memcmp(sha, cookie, 2)) {
		printf("Negative on that, houston!\n");
		error = 1;
	}
	sfree(sha);
	sfree(cookie);
	return error;
}

int	do_mount(char *from, char *to)
{
	int	mc;
	int	uid, gid, i;
	char	*pwd;
	char	*buf = salloc(strlen(from) + 1024);

	/* first check if someone is trying to reach it's limit */
	mc = get_mount_count(getuid());
	if (mc < 0)
		return 1;
	if (mc > MAX_MOUNTS_PER_USER) {
		printf("Sorry, but you've reached maximum mount count (%d) for your uid (%d)\n",
			MAX_MOUNTS_PER_USER, getuid());
		return 2;
	}

	/* check if it's directories */
	if (!isdir(from)) {
nodir:
		printf("No way man, but '%s' have to be a directory\n", from);
		return 3;
	}
	if (!isdir(to)) {
		from = to;
		goto nodir;
	}

	/* he seem to be friend, now check if really - look perms on stuff he gave us */
	if (fuid(from, &uid, &gid)) {
		perror(from);
		return 4;
	}
	if (uid != getuid() || gid != getgid()) {
badperm:
		printf("Sorry, but '%s' permissions doesn't match your uid/gid\n", from);
		return 4;
	}
	if (fuid(to, &uid, &gid)) {
		perror(to);
		return 4;
	}
	if (uid != getuid() || gid != getgid()) {
		from = to;
		goto badperm;
	}

	/* ok, now look if 'from' directory isn't empty, if so, run
	   us in "create" mode, i.e. requiere password twice and create
	   .evfs mark file there */
	sprintf(buf, "%s/%s", from, EVFS_COOKIE);
	if (fuid(buf, &uid, &gid)) {
		if (errno == ENOENT) {
			if (isempty(from)) {
				printf("'%s' is empty directory, "
					"do you want to create new evfs there [y/n]?", from);
				if (!yes_no()) return 0;
				if (do_create(buf))
					return 5;
			} else {
				printf(	"*DANGEROUS*"
					"is not empty - most probably *regular* directory,\n"
					"if you're sure that you lost password cookie from\n"
					"there (%s), then type *exactly same* password you've\n"
					"used for it last time, otherwise you see crap!\n"
					"*DANGEROURS*\n"
					"Do you wan't use this directory as evfs source [y/n]?",
					EVFS_COOKIE);
				if (!yes_no()) return 0;
				if (do_create(buf))
					return 5;
			}
			printf("New evfs created in '%s', DO NOT DELETE '%s' from there!\n", from, EVFS_COOKIE);
		}
	} else {
		if ((uid != getuid()) || (gid != getgid())) {
			printf("Sorry, you do not match the cookie uid/gid!\n");
			return 6;
		}
	}
	/* now we can be sure that there is (probably that right) cookie */

	/* huh, this person is really right one ? hrmm. let him type his password,
	   if it will not be right, he'll see only crap anyway :) */ 
	printf("Mounting '%s' to '%s'\n", from, to);
	pwd = getpwd("Enter password:");
	/* don't allow simple b/f attacks, he'll get pause even with good password */
	sleep(1);
	if (!pwd) {
		printf("Sorry, you've supplied no password\n");
		return 1;
	}
	/* we've some pass, ok, ok now verify it against against cookie
	   to be sure that user made no mistypes */
	if (verify_cookie(buf, pwd)) {
		putpwd(pwd);
		return 10;
	}
	
	/* uh, uh, everything is really ok ? hard to believe, but ok,
	  we'll mount it now */
	sfree(buf);
	buf = salloc(strlen(from) + strlen(pwd) + strlen(to) + 3);	/* thanx Paul Lasarev! */
	sprintf(buf, "%s %s %s", from, to, pwd);
	i = mount(from, to, "evfs", MOUNT_FLAGS, buf);
	if (!i) {
		printf("'%s' mounted sucesfully to '%s'\n", from, to);
	} else {
		perror("mount");
		printf("mounting from '%s' to '%s' failed\n", from, to);
	}
	sfree(buf);
	return i?1:0;
}

/*
	efs /from /to
	efs /to
 */



int	main(int argc, char *argv[])
{
	printf(
		"Encrypted Virtual File System v0.3 (linux release)\n"
		"(c) 2003 sd <sd@hysteria.sk> http://hysteria.sk/evfs\n");
	if (argc != 2 && argc != 3) {
		printf("use:\n"
			"%s /from/where /mount/point - to mount\n"
			"%s /mount/point             - to unmount\n\n"
			"note that both, /from/where and /mount/point must\n"
			"be directories and *must* match your uid/gid\n", argv[0], argv[0]);
		return 1;
	}
	setfsgid(getgid());	/* uhh uhh. this is essential *g* */
	setfsuid(getuid());
	if (argc == 3) {
		char *from = argv[1];
		/* relative path ? */
		if (from[0] != '/') {
			from = malloc(strlen(argv[1]) + PATH_MAX + 1);
			getcwd(from, PATH_MAX);
			strcat(from, "/");
			strcat(from, argv[1]);
		}
		return do_mount(from, argv[2]);
	}
	return do_umount(argv[1]);
}
