/*
 *  linux/drivers/video/fbcon-splash16.c -- Low level frame buffer operations for 16 bpp
 *                                          framebuffer operation. Modified to present
 *					    boot splash screen. 2002/11/7 stepan@suse.de
 * 
 * Based on linux/drivers/video/fbcon-cfb16.c, which is
 *
 *	Created 5 Apr 1997 by Geert Uytterhoeven
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING in the main directory of this archive for
 *  more details.
 */

#include <linux/module.h>
#include <linux/config.h>
#include <linux/tty.h>
#include <linux/console.h>
#include <linux/string.h>
#include <linux/fb.h>
#include <asm/io.h>

#include <video/fbcon.h>
#include <video/fbcon-cfb16.h>

#include "fbcon-splash.h"

    /*
     *  16 bpp packed pixels
     */

static u32 tab_cfb16[] = {
#if defined(__BIG_ENDIAN)
    0x00000000, 0x0000ffff, 0xffff0000, 0xffffffff
#elif defined(__LITTLE_ENDIAN)
    0x00000000, 0xffff0000, 0x0000ffff, 0xffffffff
#else
#error FIXME: No endianness??
#endif
};

void fbcon_splash16_bmove(struct display *p, int sy, int sx, int dy, int dx,
		       int height, int width)
{
    int bytes = p->next_line, linesize = bytes * fontheight(p), rows;
    u8 *src, *dst;

    if (sx == 0 && dx == 0 && width * fontwidth(p) * 2 == bytes) {
	fb_memmove(p->screen_base + dy * linesize,
		  p->screen_base + sy * linesize,
		  height * linesize);
	return;
    }
    if (fontwidthlog(p)) {
	sx <<= fontwidthlog(p)+1;
	dx <<= fontwidthlog(p)+1;
	width <<= fontwidthlog(p)+1;
    } else {
	sx *= fontwidth(p)*2;
	dx *= fontwidth(p)*2;
	width *= fontwidth(p)*2;
    }
    sx += splash_data.splash_text_xo*2 + splash_data.splash_text_yo * bytes;
    dx += splash_data.splash_text_xo*2 + splash_data.splash_text_yo * bytes;
    if (dy < sy || (dy == sy && dx < sx)) {
	src = p->screen_base + sy * linesize + sx;
	dst = p->screen_base + dy * linesize + dx;
	for (rows = height * fontheight(p); rows--;) {
	    fb_memmove(dst, src, width);
	    src += bytes;
	    dst += bytes;
	}
    } else {
	src = p->screen_base + (sy+height) * linesize + sx - bytes;
	dst = p->screen_base + (dy+height) * linesize + dx - bytes;
	for (rows = height * fontheight(p); rows--;) {
	    fb_memmove(dst, src, width);
	    src -= bytes;
	    dst -= bytes;
	}
    }
}

static inline void rectfill(u8 *dest, int width, int height, u32 data,
			    int linesize)
{
    int i;

    data |= data<<16;

    while (height-- > 0) {
	u32 *p = (u32 *)dest;
	for (i = 0; i < width/4; i++) {
	    fb_writel(data, p++);
	    fb_writel(data, p++);
	}
	if (width & 2)
	    fb_writel(data, p++);
	if (width & 1)
	    fb_writew(data, (u16*)p);
	dest += linesize;
    }
}

void splashfill(u8 *dest, u8 *src, int width, int height,
                       int dest_linesize, int src_linesize)
{

    int i;

    while (height-- > 0) {
	u32 *p = (u32 *)dest;
	u32 *q = (u32 *)src;

	for (i=0; i < width/4; i++) {
	    fb_writel(*q++,p++);
	    fb_writel(*q++,p++);
	}
	if (width & 2)
	    fb_writel(*q++,p++);
	if (width & 1)
	    fb_writew(*(u16*)q,(u16*)p);
	dest += dest_linesize;
	src  += src_linesize;
    }
}

void fbcon_splash16_clear(struct vc_data *conp, struct display *p, int sy, int sx,
		       int height, int width)
{
    u8 *dest;
    int bytes = p->next_line, lines = height * fontheight(p);
    u32 bgx;
    int offset, transparent=0;

    dest = p->screen_base + sy * fontheight(p) * bytes + sx * fontwidth(p) * 2;

    bgx = ((u16 *)p->dispsw_data)[attr_bgcol_ec(p, conp)];

    width *= fontwidth(p)/4;

    dest   += splash_data.splash_text_yo * bytes  +
		  splash_data.splash_text_xo * 2;

    transparent = (splash_data.splash_color == attr_bgcol_ec(p, conp));
    
    if (transparent) {
	offset = (sy * fontheight(p) + splash_data.splash_text_yo) * splash_bytes +
		 (sx * fontwidth(p) + splash_data.splash_text_xo) *  2;
	
	if ((width * 8 == bytes && splash_bytes == bytes))
	    splashfill(dest,linux_splash + offset, lines * width * 4, 
			    1, bytes, splash_bytes);
	else
	    splashfill(dest,linux_splash + offset, width*4, lines,
			    bytes, splash_bytes);
    } else {
	 if (width * 8 == bytes)
	    rectfill(dest, lines * width * 4, 1, bgx, bytes);
	 else
	    rectfill(dest, width * 4, lines, bgx, bytes);
    }
}


/*
 *  Helper function to read the background from the splashscreen
 */
# define SPLASH_BGX(off)			\
	if (transparent) {			\
	    bgx = *(u32*)(splashbgx + (off));	\
	    eorx = fgx ^ bgx;			\
	}


void fbcon_splash16_putc(struct vc_data *conp, struct display *p, int c, int yy,
		      int xx)
{
    u8 *dest, *cdat, bits;
    int bytes = p->next_line, rows;
    u32 eorx, fgx, bgx;
    int transparent = 0;
    u8 *splashbgx = 0;
 
    dest = p->screen_base + yy * fontheight(p) * bytes + xx * fontwidth(p) * 2;

    fgx = ((u16 *)p->dispsw_data)[attr_fgcol(p, c)];
    bgx = ((u16 *)p->dispsw_data)[attr_bgcol(p, c)];

    transparent = (splash_data.splash_color == attr_bgcol(p, c));
    
    dest += splash_data.splash_text_xo * 2 + splash_data.splash_text_yo * bytes;
    splashbgx = linux_splash +
	    (yy * fontheight(p) + splash_data.splash_text_yo) * splash_bytes + 
	    (xx * fontwidth(p) + splash_data.splash_text_xo) * 2;

    if (transparent && splash_data.splash_color == 0xf) {
	if (fgx == 0xffea)
	    fgx = 0xfe4a;
	else if (fgx == 0x57ea)
	    fgx = 0x0540;
	else if (fgx == 0xffff)
	    fgx = 0x52aa;
    }

    fgx |= (fgx << 16);
    bgx |= (bgx << 16);
    eorx = fgx ^ bgx;

    switch (fontwidth(p)) {
    case 4:
    case 8:
	cdat = p->fontdata + (c & p->charmask) * fontheight(p);
	for (rows = fontheight(p); rows--; dest += bytes) {
	    bits = *cdat++;
	    SPLASH_BGX(0);
	    fb_writel((tab_cfb16[bits >> 6] & eorx) ^ bgx, dest);
	    SPLASH_BGX(4);
	    fb_writel((tab_cfb16[bits >> 4 & 3] & eorx) ^ bgx, dest+4);
	    if (fontwidth(p) == 8) {
		SPLASH_BGX(8);
		fb_writel((tab_cfb16[bits >> 2 & 3] & eorx) ^ bgx, dest+8);
		SPLASH_BGX(12);
		fb_writel((tab_cfb16[bits & 3] & eorx) ^ bgx, dest+12);
	    }

	    splashbgx += splash_bytes;

	}
	break;
    case 12:
    case 16:
	cdat = p->fontdata + ((c & p->charmask) * fontheight(p) << 1);
	for (rows = fontheight(p); rows--; dest += bytes) {
	    bits = *cdat++;
	    SPLASH_BGX(0);
	    fb_writel((tab_cfb16[bits >> 6] & eorx) ^ bgx, dest);
	    SPLASH_BGX(4);
	    fb_writel((tab_cfb16[bits >> 4 & 3] & eorx) ^ bgx, dest+4);
	    SPLASH_BGX(8);
	    fb_writel((tab_cfb16[bits >> 2 & 3] & eorx) ^ bgx, dest+8);
	    SPLASH_BGX(12);
	    fb_writel((tab_cfb16[bits & 3] & eorx) ^ bgx, dest+12);
	    bits = *cdat++;
	    SPLASH_BGX(16);
	    fb_writel((tab_cfb16[bits >> 6] & eorx) ^ bgx, dest+16);
	    SPLASH_BGX(20);
	    fb_writel((tab_cfb16[bits >> 4 & 3] & eorx) ^ bgx, dest+20);
	    if (fontwidth(p) == 16) {
		SPLASH_BGX(24);
		fb_writel((tab_cfb16[bits >> 2 & 3] & eorx) ^ bgx, dest+24);
		SPLASH_BGX(28);
		fb_writel((tab_cfb16[bits & 3] & eorx) ^ bgx, dest+28);
	    }
	}
	break;
    }
}

void fbcon_splash16_putcs(struct vc_data *conp, struct display *p,
		       const unsigned short *s, int count, int yy, int xx)
{
    u8 *cdat, *dest, *dest0;
    u16 c;
    int rows, bytes = p->next_line;
    u32 eorx, fgx, bgx;
    int transparent = 0;
    u8 *splashbgx0 = 0, *splashbgx;

    dest0 = p->screen_base + yy * fontheight(p) * bytes + xx * fontwidth(p) * 2;
    c = scr_readw(s);
    fgx = ((u16 *)p->dispsw_data)[attr_fgcol(p, c)];
    bgx = ((u16 *)p->dispsw_data)[attr_bgcol(p, c)];

    transparent = (splash_data.splash_color == attr_bgcol(p, c));
    
    dest0 += splash_data.splash_text_xo * 2 + splash_data.splash_text_yo * bytes;
    splashbgx0 = linux_splash +
	      (yy * fontheight(p) + splash_data.splash_text_yo) * splash_bytes +
	      (xx * fontwidth(p) + splash_data.splash_text_xo) * 2;

    if (transparent && splash_data.splash_color == 0xf) {
	if (fgx == 0xffea)
	    fgx = 0xfe4a;
	else if (fgx == 0x57ea)
	    fgx = 0x0540;
	else if (fgx == 0xffff)
	    fgx = 0x52aa;
    }

    fgx |= (fgx << 16);
    bgx |= (bgx << 16);
    eorx = fgx ^ bgx;

    switch (fontwidth(p)) {
    case 4:
    case 8:
	while (count--) {
	    c = scr_readw(s++) & p->charmask;
	    cdat = p->fontdata + c * fontheight(p);

	    splashbgx = splashbgx0;

	    for (rows = fontheight(p), dest = dest0; rows--; dest += bytes) {
		u8 bits = *cdat++;
		SPLASH_BGX(0);
	        fb_writel((tab_cfb16[bits >> 6] & eorx) ^ bgx, dest);
		SPLASH_BGX(4);
	        fb_writel((tab_cfb16[bits >> 4 & 3] & eorx) ^ bgx, dest+4);
		if (fontwidth(p) == 8) {
		    SPLASH_BGX(8);
		    fb_writel((tab_cfb16[bits >> 2 & 3] & eorx) ^ bgx, dest+8);
		    SPLASH_BGX(12);
		    fb_writel((tab_cfb16[bits & 3] & eorx) ^ bgx, dest+12);
		}
		splashbgx += splash_bytes;
	    }

	    dest0 += fontwidth(p)*2;
	    splashbgx0 += fontwidth(p) * 2;
	}

	break;
    case 12:
    case 16:
	while (count--) {
	    c = scr_readw(s++) & p->charmask;
	    cdat = p->fontdata + (c * fontheight(p) << 1);

	    splashbgx = splashbgx0;

	    for (rows = fontheight(p), dest = dest0; rows--; dest += bytes) {
		u8 bits = *cdat++;
		SPLASH_BGX(0);
	        fb_writel((tab_cfb16[bits >> 6] & eorx) ^ bgx, dest);
		SPLASH_BGX(4);
	        fb_writel((tab_cfb16[bits >> 4 & 3] & eorx) ^ bgx, dest+4);
		SPLASH_BGX(8);
	        fb_writel((tab_cfb16[bits >> 2 & 3] & eorx) ^ bgx, dest+8);
		SPLASH_BGX(12);
	        fb_writel((tab_cfb16[bits & 3] & eorx) ^ bgx, dest+12);
		bits = *cdat++;
		SPLASH_BGX(16);
	        fb_writel((tab_cfb16[bits >> 6] & eorx) ^ bgx, dest+16);
		SPLASH_BGX(20);
	        fb_writel((tab_cfb16[bits >> 4 & 3] & eorx) ^ bgx, dest+20);
		if (fontwidth(p) == 16) {
		    SPLASH_BGX(24);
		    fb_writel((tab_cfb16[bits >> 2 & 3] & eorx) ^ bgx, dest+24);
		    SPLASH_BGX(28);
		    fb_writel((tab_cfb16[bits & 3] & eorx) ^ bgx, dest+28);
		}
		splashbgx += splash_bytes;
	    }

	    dest0 += fontwidth(p)*2;
	    splashbgx0 += fontwidth(p) * 2;
	}

	break;
    }
}

void fbcon_splash16_revc(struct display *p, int xx, int yy)
{
    u8 *dest;
    int bytes = p->next_line, rows;

    dest = p->screen_base + yy * fontheight(p) * bytes + xx * fontwidth(p)*2;
    dest += splash_data.splash_text_yo * bytes + splash_data.splash_text_xo * 2;

    for (rows = fontheight(p); rows--; dest += bytes) {
	switch (fontwidth(p)) {
	case 16:
	    fb_writel(fb_readl(dest+24) ^ 0xffffffff, dest+24);
	    fb_writel(fb_readl(dest+28) ^ 0xffffffff, dest+28);
	    /* FALL THROUGH */
	case 12:
	    fb_writel(fb_readl(dest+16) ^ 0xffffffff, dest+16);
	    fb_writel(fb_readl(dest+20) ^ 0xffffffff, dest+20);
	    /* FALL THROUGH */
	case 8:
	    fb_writel(fb_readl(dest+8) ^ 0xffffffff, dest+8);
	    fb_writel(fb_readl(dest+12) ^ 0xffffffff, dest+12);
	    /* FALL THROUGH */
	case 4:
	    fb_writel(fb_readl(dest+0) ^ 0xffffffff, dest+0);
	    fb_writel(fb_readl(dest+4) ^ 0xffffffff, dest+4);
	}
    }
}

void fbcon_splash16_clear_margins(struct vc_data *conp, struct display *p,
			       int bottom_only)
{
    int bytes = p->next_line;
    u32 bgx;

    unsigned int right_start = conp->vc_cols*fontwidth(p);
    unsigned int bottom_start = conp->vc_rows*fontheight(p);
    unsigned int right_width, bottom_width;

    int left_margin_width = splash_data.splash_text_xo;
    int text_width = conp->vc_cols * fontwidth(p);
    int right_margin_width = p->var.xres - text_width - left_margin_width;
    int top_margin_height = splash_data.splash_text_yo;
    int text_height = conp->vc_rows * fontheight(p);
    int bottom_margin_height = p->var.yres - text_height - top_margin_height;
 
    bgx = ((u16 *)p->dispsw_data)[attr_bgcol_ec(p, conp)];

    if (bottom_only == -1) {
	printk(KERN_DEBUG "Called with bottom-only\n");
	splashfill(p->screen_base, linux_splash, p->var.xres, p->var.yres, bytes, splash_bytes); 
	return;
    }

    if (!bottom_only && (right_width = p->var.xres-right_start)) {
  	/* left margin */
	splashfill(p->screen_base + top_margin_height * bytes,
		   linux_splash + top_margin_height * 
		   splash_bytes, left_margin_width,
		   text_height, bytes, splash_bytes);

	/* right margin */
	splashfill(p->screen_base + left_margin_width*2 + text_width*2 +
		    top_margin_height * bytes, linux_splash +
		    left_margin_width*2 + text_width*2 + top_margin_height *
		    splash_bytes, right_margin_width, text_height,
		    bytes, splash_bytes);
    }

    if ((bottom_width = p->var.yres-bottom_start))
	/* bottom margin */
	splashfill(p->screen_base + (top_margin_height + text_height) *
		    bytes, linux_splash + (top_margin_height +
		    text_height) * splash_bytes, p->var.xres, 
		    bottom_margin_height, bytes, splash_bytes);

    /* top margin */
    splashfill(p->screen_base, linux_splash,
	    p->var.xres, top_margin_height,
	    bytes, splash_bytes);

    /* leave function if work is done */
    return;
}


    /*
     *  `switch' for the low level operations
     */

struct display_switch fbcon_splash16 = {
    bmove:		fbcon_splash16_bmove,
    clear:		fbcon_splash16_clear,
    putc:		fbcon_splash16_putc,
    putcs:		fbcon_splash16_putcs,
    revc:		fbcon_splash16_revc,
    clear_margins:	fbcon_splash16_clear_margins,
    fontwidthmask:	FONTWIDTH(4)|FONTWIDTH(8)|FONTWIDTH(12)|FONTWIDTH(16)
};


#ifdef MODULE
MODULE_LICENSE("GPL");

int init_module(void)
{
    return 0;
}

void cleanup_module(void)
{}
#endif /* MODULE */


    /*
     *  Visible symbols for modules
     */

EXPORT_SYMBOL(fbcon_splash16);
EXPORT_SYMBOL(fbcon_splash16_bmove);
EXPORT_SYMBOL(fbcon_splash16_clear);
EXPORT_SYMBOL(fbcon_splash16_putc);
EXPORT_SYMBOL(fbcon_splash16_putcs);
EXPORT_SYMBOL(fbcon_splash16_revc);
EXPORT_SYMBOL(fbcon_splash16_clear_margins);
