#---------------------------------------------------------------------------
#
#	Implementation of an association list form, i.e. a widget with
#	two columns: a column of labels, and a column of associated
#	values. The value fields are mouse-sensitive and may produce a
#	menu or other selection mechanism when clicked on with the
#	left mouse button.
#
#	The 'alist' parameter of 'new(alist)' is a list of triples:
#	a label, an initial value, and a menu action.
#
#---------------------------------------------------------------------------

defwidget Alist _Filter

defmethod Alist new {name args} {

  args	text alist actions buttons {label1 Attributes} {label2 Values} \
	inputlabel layout embedded wait {textfont text} {labelfont bold} \
	{attributewidth 20} {valuewidth 32}

  if { $buttons == {} } {
    if { [assoc Ok+Save $actions] == {} } {
      set buttons { Ok }
    } {
      set buttons { Ok Ok+Save }
    }
  }

  if { [assoc Help $actions] == {} } {
    lappend actions {Help {} Library/alist}
  }

  if { $embedded == "true" } {
    Frame new $name -relief flat
  } {
    Toplevel new $name -title $text \
	-buttons $buttons \
	-handler [list $name _button] \
	-actions $actions
  }
  defsuper $name Alist

  set bm_menu [the(image) menu]
  set bm_empty [the(image) empty]
  set bm_input [the(image) input]
  set bm_action [the(image) action]

  Frame new $name.head \
	-layout {top padx 5 pady 5 fillx} \
	-relief raised
  Label new $name.head.a \
	-layout {left frame nw padx 5} \
	-relief flat -text $label1 -width $attributewidth \
	-textfont $labelfont
  Bitmap new $name.head.m \
	-layout {left frame nw padx 5} \
	-relief flat -bitmap @$bm_empty
  Label new $name.head.v \
	-layout {left frame nw padx 5} \
	-relief flat -text $label2 -width $valuewidth \
	-textfont $labelfont

  $name slot alist $alist
  if { $inputlabel != {} && [assoc input $alist] == "" } {
    set inputlabel {}
  }

  set start_a 5
  set start_m [expr 10+$start_a+[winfo reqwidth $name.head.a]]
  set start_v [expr 10+$start_m+[winfo reqwidth $name.head.m]]
  set start_x [expr 10+$start_v+[winfo reqwidth $name.head.v]]
  set size_y [expr [winfo reqheight $name.head.a]/2*2]

  set f $name.avm
  Frame new $f -layout {top padx 5 pady 5 expand fill} -relief raised
  set pos 0
  set ypos 5

  set fields {}

  foreach spec $alist {
    set need_a 1; set need_v 1; set need_m 1; set bonus 0.5
    set form Label
    set parameter [list -relief sunken -text [lindex $spec 2] \
			-width $valuewidth -textfont $textfont]

    case [lindex $spec 0] in {
    {menu} {
	# Fixed menu of options
	set m [Menubutton new $f.m$pos -bitmap @$bm_menu]
	foreach item [lrange $spec 3 end] {
	  $m addItem -text $item -action [list $f.v$pos set $item]
	}
      }
    {input} {
	# Input from input line - display in label
	if { $inputlabel == {} } {
	  Bitmap new $f.m$pos -bitmap @$bm_empty
	  set form Entry
          set parameter [list -text [lindex $spec 2] \
		-width [expr $valuewidth-2] -textfont $textfont \
		-action [list $name _entry $pos [lrange $spec 3 end]]]
	  lappend fields [lreplace $spec 0 0 $f.v$pos]
	} {
	  Button new $f.m$pos -bitmap @$bm_input \
		-action [list $name _input $f.v$pos [lrange $spec 3 end]]
	}
      }
    {text} {
	# Input from input line - display in text item
	if { $inputlabel == {} } {
	  Bitmap new $f.m$pos -bitmap @$bm_empty
	  set form Text
          set parameter [list -text [lindex $spec 2] -state normal \
		-width [expr $valuewidth-1] -height 3 -textfont $textfont \
		-action [list $name _entry $pos [lrange $spec 3 end]]]
	  lappend fields [lreplace $spec 0 0 $f.v$pos]
	} {
	  Button new $f.m$pos -bitmap @$bm_input \
		-action [list $name _input $f.v$pos [lrange $spec 3 end]]
	}
	set need_v 2
      }
    {consttext} {
	# Constant line of text
	set need_m 0; set need_v 2
      }
    {command} {
	# Special command activation - allows dynamic recomputation of options
        if { [lindex $spec 3] == "-" } {
	  Button new $f.m$pos -bitmap @$bm_action \
		-action [concat [list $name] [lrange $spec 4 end] $f.v$pos]
	} {
	  Button new $f.m$pos -bitmap @$bm_action \
		-action [concat [lrange $spec 3 end] $f.v$pos]
	}
      }
    {const} {
	# Constant label
	set need_m 0
      }
    {*} {
	# Space? The final frontier?
	set need_a 0; set need_m 0; set need_v 0
	set bonus 0
      }
    }

    if { $need_a == 1 } {
      Label new $f.a$pos -relief flat -text [lindex $spec 1] -width 20 \
		-textfont $labelfont
      place $f.a$pos -in $f -anchor nw -y $ypos -x $start_a
    }
    if { $need_m == 1 } {
      place $f.m$pos -in $f -anchor nw -y $ypos -x $start_m
    }

    if { $need_v > 0 } {
      if { $need_v == 1 } {
	eval [concat [list $form new $f.v$pos] $parameter]
      } {
	if { [streq $form "Label"] } {
	  lappend parameter -edittext true
	}
	eval [concat [list $form new $f.v$pos] $parameter]
	set bonus -1
      }
      if { $inputlabel != {} } {
        bind $f.v$pos <Button-1> [list $name _setinput $f.v$pos]
      }
      place $f.v$pos -in $f -anchor nw -y $ypos -x $start_v
      $f.v$pos slot _alistvalue [lindex $spec 2]
    }

    if { $bonus == -1 } {
      set ypos [expr $ypos+[winfo reqheight $f.v$pos]+$size_y/2]
    } {
      set ypos [expr $ypos+$size_y*(1+$bonus)]
    }
    incr pos
  }

  # The maximum number of items in this alist.
  $name slot _max $pos
  $name slot _fields $fields

  $f configure -width $start_x -height $ypos
  if { $inputlabel != {} } {
    Inputline new $name.input -layout {bottom padx 5 pady 5 fillx} \
	-textfont $textfont -edittext true -width 32 \
	-label $inputlabel -input any
    $name.input resize $start_x
    focus $name.input
  } {
    $name _entry {} {} {}
  }

  $name layout $layout

  if { $embedded != "true" && $wait == "true" } {
    tkwait window $name
  }

  return $name
}

#---------------------------------------------------------------------------

defmethod Alist clear {{value {}}} {

  if { [is $self.input Window] } {
    $self.input set $value
  }
}

defmethod Alist items {{checked 0}} {

  set f $self.avm
  set list {}
  for {set i 0} {$i < [$self slot _max]} {incr i} {
    if { [is $f.a$i Window] } {
      set label [$f.a$i get]
      if { $checked } {
	set value [$f.v$i slot _alistvalue]
      } {
	set value [$f.v$i get]
      }
      lappend list [concat [list $label] $value]
    }
  }
  return $list
}

defmethod Alist _button {button action} {

  case $button in {

  {Help Dismiss} {
	$self $button $action
    }
  {Ok Ok+Save} {
	foreach field [$self slot _fields] {
	  set win [lindex $field 0]
	  set label [lindex $field 1]
	  set initial [lindex $field 2]
	  set checker [lrange $field 3 end]
	  if { ![streq $checker ""] } {
	    set value [$win get]
	    lappend checker $initial $value
	    if { [lindex $checker 0] == "-" } {
	      set new [uplevel #0 [concat $self [lrange $checker 1 end]]]
	    } {
	      set new [uplevel #0 $checker]
	    }
	puts stdout ">> $label: {$initial} {$value} {$new}"
	    if { ![streq $value $new] } {
	      $win set $initial
	      Dialog new * -text "The value in $label is invalid.\n\nPlease correct the value and try again."
	      return
	    }
	  }
	}
	if { $action != {} } {
	  set res [uplevel #0 $action]
	}
	if { [streq $res {}] } {
	  $self Dismiss
	}
    }
  default {
	if { $action != {} } {
	  set res [uplevel #0 $action]
	  if { $res != {} } {
	    Dialog new * -help [list Library/failed $res]
	  }
	}
    }
  }
}

defmethod Alist _setinput {value} {

  $self.input set [$value get]
  focus [$self.input entry]
}

defmethod Alist _input {value form} {

  set new [$self.input get]
  set old [$value get]

  if { $form != {} } {
    lappend form $old $new
    if { [lindex $form 0] == "-" } {
      set new [uplevel #0 [concat $self [lrange $form 1 end]]]
    } {
      set new [uplevel #0 $form]
    }
  }
  if { ![streq $new $old] } {
    $value set $new
    $value slot _alistvalue $new
  }
}

defmethod Alist _entry {pos check new} {

  set root $self.avm.v

  if { [streq $pos {}] } {
    # Initial setup (focus on first item)
    set pos [expr [$self slot _max]-1]
  } {
    # Check value
    set old [$root$pos slot _alistvalue]
    if { ![streq $check {}] } {
      lappend check $old $new
      if { [lindex $check 0] == "-" } {
        set new [uplevel #0 [concat $self [lrange $check 1 end]]]
      } {
        set new [uplevel #0 $check]
      }
    }
    # Enter value as confirmed
    $root$pos slot _alistvalue $new
    $root$pos set $new
  }

  # Advance cursor to the next writable field
  set npos [expr $pos+1]
  while { $npos != $pos } {
    if { [is $root$npos Entry] || [is $root$npos Text] } {
      break
    }
    incr npos
    if { $npos >= [$self slot _max] } {
      set npos 0
    }
  }
  focus $root$npos
  return
}

#---------------------------------------------------------------------------

Window addDemo Alist

defmethod Alist demo {} {

  Alist new * -alist [list \
	{const {Activity Name} {Assemble e-box}} \
	{space} \
	{menu {Start Earliest} 0 0 10 20 30 40 50 60 70 80} \
	{command {Start Latest} * puts stdout} \
	{const {Duration} 10} \
	{space} \
	{input {Activity Id} 8732 - chooseNumber} \
	{input {Another Id} 8732 - chooseNumberX} \
	{text {Description} "Get parts and assemble electronics box."} \
	] \
	-layout +50+50 \
	-text		{Attribute Definition Form} \
	-label1		ATTRIBUTES \
	-label2		VALUES \
	-inputlabel	{New Value} \
	-buttons	{Constraints Effects Resources {} Ok Ok+Save} \
	-actions	{ \
	{Ok		Ok - showAction Ok} \
	{Ok+Save	{Ok/Speichern} - showAction Ok+Save} \
	{Constraints	{} - showAction Constraints} \
	{Effects	{} - showAction Effects} \
	{Resources	{} - showAction Resources} \
	}

  Alist new * -alist [list \
	{const {Activity Name} {Assemble e-box}} \
	{space} \
	{menu {Start Earliest} 0 0 10 20 30 40 50 60 70 80} \
	{command {Start Latest} * puts stdout} \
	{const {Duration} 10} \
	{space} \
	{input {Activity Id} 8732 - chooseNumber} \
	{input {Another Id} 8732 - chooseNumber} \
	{text {Description} "Get parts and assemble electronics box."} \
	] \
	-layout		-50-50 \
	-text		{Attribute Definition Form} \
	-label1		ATTRIBUTES \
	-label2		VALUES \
	-buttons	{Constraints Effects Resources {} Ok Ok+Save} \
	-actions	{ \
	{Ok		Ok - showAction Ok} \
	{Ok+Save	{Ok/Speichern} - showAction Ok+Save} \
	{Constraints	{} - showAction Constraints} \
	{Effects	{} - showAction Effects} \
	{Resources	{} - showAction Resources} \
	}
}
