/*
 * tmExpand.c --
 *	This module handles % expansions of callback data
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "tm.h"
#include "tmFuncs.h"

#if XmVersion >= 1002
#include <Xm/XmAll.h>
#else
#include <Xm/ArrowB.h>
#include <Xm/Command.h>
#include <Xm/DrawnB.h>
#include <Xm/FileSB.h>
#include <Xm/List.h>
#include <Xm/PushB.h>
#include <Xm/Scale.h>
#include <Xm/ScrollBar.h>
#include <Xm/SelectioB.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#ifndef MOTIF11
#include <Xm/DragDrop.h>
#endif
#endif

static Tm_ReasonType Tm_Reasons[] = {
	    {XmCR_ACTIVATE,			"activate"},
	    {XmCR_APPLY,			"apply"},
	    {XmCR_ARM,				"arm"},
	    {XmCR_BROWSE_SELECT,		"browse_select"},
	    {XmCR_CANCEL,			"cancel"},
	    {XmCR_CASCADING,			"cascading"},
	    {XmCR_CLIPBOARD_DATA_DELETE,	"clipboard_data_delete"},
	    {XmCR_CLIPBOARD_DATA_REQUEST,	"clipboard_data_request"},
	    {XmCR_COMMAND_CHANGED,		"command_changed"},
	    {XmCR_COMMAND_ENTERED,		"command_entered"},
	    {XmCR_CREATE,			"create"},
	    {XmCR_DECREMENT,			"decrement"},
	    {XmCR_DEFAULT_ACTION,		"default_action"},
	    {XmCR_DISARM,			"disarm"},
	    {XmCR_DRAG,				"drag"},
	    {XmCR_EXECUTE,			"execute"},
	    {XmCR_EXPOSE,			"expose"},
	    {XmCR_EXTENDED_SELECT,		"extended_select"},
	    {XmCR_FOCUS,			"focus"},
	    {XmCR_GAIN_PRIMARY,			"gain_primary"},
	    {XmCR_HELP,				"help"},
	    {XmCR_INCREMENT,			"increment"},
	    {XmCR_INPUT,			"input"},
	    {XmCR_LOSE_PRIMARY,			"lose_primary"},
	    {XmCR_LOSING_FOCUS,			"losing_focus"},
	    {XmCR_MAP,				"map"},
	    {XmCR_MODIFYING_TEXT_VALUE,		"modifying_text_value"},
	    {XmCR_MOVING_INSERT_CURSOR,		"moving_insert_cursor"},
	    {XmCR_MULTIPLE_SELECT,		"multiple_select"},
	    {XmCR_NONE,				"none"},
	    {XmCR_NO_MATCH,			"no_match"},
	    {XmCR_OBSCURED_TRAVERSAL,		"obscured_traversal"},
	    {XmCR_OK,				"ok"},
	    {XmCR_PAGE_DECREMENT,		"page_decrement"},
	    {XmCR_PAGE_INCREMENT,		"page_increment"},
	    {XmCR_RESIZE,			"resize"},
	    {XmCR_SINGLE_SELECT,		"single_select"},
	    {XmCR_TEAR_OFF_ACTIVATE,		"tear_off_activate"},
	    {XmCR_TEAR_OFF_DEACTIVATE,		"tear_off_deactivate"},
	    {XmCR_TO_BOTTOM,			"to_bottom"},
	    {XmCR_TO_TOP,			"to_top"},
	    {XmCR_UNMAP,			"unmap"},
	    {XmCR_VALUE_CHANGED,		"value_changed"},
#if XmVersion >= 2000
	    {XmCR_COLLAPSED,			"collapsed"},
	    {XmCR_EXPANDED,			"expanded"},
	    {XmCR_MAJOR_TAB,			"major_tab"},
	    {XmCR_MINOR_TAB,			"minor_tab"},
	    {XmCR_PAGE_SCROLLER_INCREMENT,	"page_scroller_increment"},
	    {XmCR_SPIN_NEXT,			"spin_next"},
	    {XmCR_SPIN_PRIOR,			"spin_prior"},
	    {XmCR_SPIN_FIRST,			"spin_first"},
	    {XmCR_SPIN_LAST,			"spin_last"},
	    {XmCR_SELECT,			"select"},
	    {-1,				NULL},
#endif
};

/* the set of reasons for externally defined widgets */
extern Tm_ReasonType Tm_ExternReasons[];

/*
 *--------------------------------------------------------------
 *
 * make2list --
 *
 *	make a list of 2 elmts
 *
 * Results:
 *
 *	returns the list of 2 elmts
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static char *
make2list(first, second)
    char *first;
    char *second;
{
    char *argv[2];

    argv[0] = first;
    argv[1] = second;
    return Tcl_Merge(2, argv);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_Merge --
 *
 *	make a list reusing static space
 *
 * Results:
 *
 *	returns a list in static space
 *
 * Side effects:
 *	frees elmts of argv
 *
 *--------------------------------------------------------------
 */

static char *
Tm_Merge(argc, argv)
    int argc;
    char *argv[];
{
    char *all;
    static char *result = NULL;
    int n;

    all = Tcl_Merge(argc, argv);

    XtFree(result);
    result = XtNewString(all);

    for (n = 0; n < argc; n++)
	free(argv[n]);
    free(all);

    return result;
}

/*
 *--------------------------------------------------------------
 *
 * items2String --
 *
 *	convert a list of items to a string, 
 *	storing in static space
 *
 * Results:
 *
 *	returns the static string
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static char *
items2String(count, selected_items)
    int count;
    XmString *selected_items;
{
        String *items;
	static String items_list = NULL;
	static int items_list_len = 0;
	int n, len;
	char *p, *end;

	items = (char **) XtMalloc(sizeof(char *) * count);
	len= 0;
	for (n = 0; n < count; n++) {
	    items[n] = XtNewString(
			Tm_XmStringToString(selected_items[n]));
	    len += strlen(items[n]) + 2; /* for , */
	}

	/* reuse the static space we malloc for this list */
	if (items_list == NULL) {
	    items_list = XtMalloc(len + 1); /* we overshoot by 1 */
	    items_list_len = len + 1;
	} else
	if (items_list_len < len) {
	   items_list = XtRealloc(items_list, len + 1);
	   items_list_len = len + 1;
	}

	if (len == 0) {
	    *items_list = '\0';
	    return items_list;
	}
#ifdef TM_MOTIF_XMSTRINGTABLE
	end = items_list;
	for (n = 0; n < count; n++) {
	    p = items[n];
	    while (*end++ = *p++)
		;
	    *(end - 1) = ',';
	    *end++ = ' ';
	    XtFree(items[n]);
	}
	*(end - 2) = '\0';
#else
	items_list = Tcl_Merge(count, items);
        for (n = 0; n < count; n++) {
            XtFree(items[n]);
        }
#endif
	XtFree((char *) items);
	return items_list;
}

 
/*
 *--------------------------------------------------------------
 *
 * widgets2String --
 *
 *	convert a list of widgets to a string, 
 *	storing in static space
 *
 * Results:
 *
 *	returns the static string
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static char *
widgets2String(count, w_items)
    int count;
    WidgetList w_items;
{
    String *items;
    static String items_list = NULL;
    static int items_list_len = 0;
    int n, len;
    char *p, *end;
    Tm_Widget *wPtr;

    items = (char **) XtMalloc(sizeof(char *) * count);
    len= 0;
    for (n = 0; n < count; n++) {
    	XtVaGetValues(w_items[n], 
			XmNuserData, &wPtr, NULL);
        items[n] = wPtr->pathName;
        len += strlen(items[n]) + 2; /* for , */
    }

    /* reuse the static space we malloc for this list */
    if (items_list == NULL) {
	    items_list = XtMalloc(len + 1); /* we overshoot by 1 */
	    items_list_len = len + 1;
    } else
    if (items_list_len < len) {
	   items_list = XtRealloc(items_list, len + 1);
	   items_list_len = len + 1;
    }

    if (len == 0) {
        *items_list = '\0';
        return items_list;
    }
    items_list = Tcl_Merge(count, items);
    for (n = 0; n < count; n++) {
        XtFree(items[n]);
    }
    XtFree((char *) items);
    return items_list;
	    
}

/*
 *--------------------------------------------------------------
 *
 * getValue --
 *
 *	Get the value of a field from out of the callback data.
 *
 * Results:
 *
 *	places the string value in "value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static char *
getValue(pathName, w, event, call_data, value, length)
    char *pathName;
    Widget w;
    XEvent *event;
    XtPointer call_data;
    char *value;
    int length;
{
    char *result;
    XrmValue from, to;
    static char buf[32];
    static char event_buf[32];
    static char *reason;
    WidgetClass class;
    XmAnyCallbackStruct *c_data = (XmAnyCallbackStruct *) call_data;
    char *all[32];	/* max elmts in callback struct */
    int n_all = 0;
    static char x[32], y[32];
    int n;

    if (length == strlen("w") &&
	    strncmp(value, "w", length) == 0) {
	return pathName;
    }

    if (length == strlen("dragContext") &&
	    strncmp(value, "dragContext", length) == 0) {
	return pathName;
    }

    if (event != NULL) {
        sprintf(event_buf, "event-%lu", (long) event);
        if (length == strlen("event") &&
	    strncmp(value, "event", length) == 0) {
  	    return event_buf;
        }
    }

    /* extract the reason field because we may have a "call_data"
     */
    reason = NULL;
    for (n = 0; Tm_Reasons[n].str != NULL; n++) {
	if (c_data->reason == Tm_Reasons[n].reason) {
	    reason = Tm_Reasons[n].str;
	    break;
	}
    }
    if (reason == NULL) {
	/* try the external widgets instead */
        for (n = 0; Tm_ExternReasons[n].str != NULL; n++) {
            if (c_data->reason == Tm_ExternReasons[n].reason) {
                reason = Tm_ExternReasons[n].str;
                break;
            }
        }
    }
    if (reason == NULL) {
	/* nowhere to be found */
	reason = "none";
    }
#if 0
    switch (c_data->reason) {
	    case XmCR_ACTIVATE:			reason = "activate"; 	break;
	    case XmCR_APPLY:			reason = "apply"; 	break;
	    case XmCR_ARM:			reason = "arm"; 	break;
	    case XmCR_BROWSE_SELECT:		reason = "browse_select"; 	break;
	    case XmCR_CANCEL:			reason = "cancel"; 	break;
	    case XmCR_CASCADING:		reason = "cascading"; 	break;
	    case XmCR_CLIPBOARD_DATA_DELETE:	reason = "clipboard_data_delete"; 	break;
	    case XmCR_CLIPBOARD_DATA_REQUEST:	reason = "clipboard_data_request"; 	break;
	    case XmCR_COMMAND_CHANGED:		reason = "command_changed"; 	break;
	    case XmCR_COMMAND_ENTERED:		reason = "command_entered"; 	break;
	    case XmCR_CREATE:			reason = "create"; 	break;
	    case XmCR_DECREMENT:		reason = "decrement"; 	break;
	    case XmCR_DEFAULT_ACTION:		reason = "default_action"; 	break;
	    case XmCR_DISARM:			reason = "disarm"; 	break;
	    case XmCR_DRAG:			reason = "drag"; 	break;
	    case XmCR_EXECUTE:			reason = "execute"; 	break;
	    case XmCR_EXPOSE:			reason = "expose"; 	break;
	    case XmCR_EXTENDED_SELECT:		reason = "extended_select"; 	break;
	    case XmCR_FOCUS:			reason = "focus"; 	break;
	    case XmCR_GAIN_PRIMARY:		reason = "gain_primary"; 	break;
	    case XmCR_HELP:			reason = "help"; 	break;
	    case XmCR_INCREMENT:		reason = "increment"; 	break;
	    case XmCR_INPUT:			reason = "input"; 	break;
	    case XmCR_LOSE_PRIMARY:		reason = "lose_primary"; 	break;
	    case XmCR_LOSING_FOCUS:		reason = "losing_focus"; 	break;
	    case XmCR_MAP:			reason = "map"; 	break;
	    case XmCR_MODIFYING_TEXT_VALUE:	reason = "modifying_text_value"; 	break;
	    case XmCR_MOVING_INSERT_CURSOR:	reason = "moving_insert_cursor"; 	break;
	    case XmCR_MULTIPLE_SELECT:		reason = "multiple_select"; 	break;
	    case XmCR_NONE:			reason = "none"; 	break;
	    case XmCR_NO_MATCH:			reason = "no_match"; 	break;
	    case XmCR_OBSCURED_TRAVERSAL:	reason = "obscured_traversal"; 	break;
	    case XmCR_OK:			reason = "ok"; 	break;
	    case XmCR_PAGE_DECREMENT:		reason = "page_decrement"; 	break;
	    case XmCR_PAGE_INCREMENT:		reason = "page_increment"; 	break;
	    case XmCR_RESIZE:			reason = "resize"; 	break;
	    case XmCR_SINGLE_SELECT:		reason = "single_select"; 	break;
	    case XmCR_TEAR_OFF_ACTIVATE:	reason = "tear_off_activate"; 	break;
	    case XmCR_TEAR_OFF_DEACTIVATE:	reason = "tear_off_deactivate"; 	break;
	    case XmCR_TO_BOTTOM:		reason = "to_bottom"; 	break;
	    case XmCR_TO_TOP:			reason = "to_top"; 	break;
	    case XmCR_UNMAP:			reason = "unmap"; 	break;
	    case XmCR_VALUE_CHANGED:		reason = "value_changed"; 	break;
#if XmVersion >= 2000
	    case XmCR_COLLAPSED:		reason = "collapsed"; 	break;
	    case XmCR_EXPANDED:			reason = "expanded"; 	break;
	    case XmCR_MAJOR_TAB:		reason = "major_tab"; 	break;
	    case XmCR_MINOR_TAB:		reason = "minor_tab"; 	break;
	    case XmCR_PAGE_SCROLLER_INCREMENT:	reason = "page_scroller_increment"; 	break;
	    case XmCR_SPIN_NEXT:		reason = "spin_next"; 	break;
	    case XmCR_SPIN_PRIOR:		reason = "spin_prior"; 	break;
	    case XmCR_SPIN_FIRST:		reason = "spin_first"; 	break;
	    case XmCR_SPIN_LAST:		reason = "spin_last"; 	break;
	    case XmCR_SELECT:			reason = "select"; 	break;
#endif
	    default: 				reason = "none";
    }
#endif /* 0 */

    if (length == strlen("reason") &&
	    strncmp(value, "reason", length) == 0) {
	return reason;
    }

    if (length == strlen("call_data") &&
	    strncmp(value, "call_data", length) == 0) {

	all[n_all++] = make2list("reason", reason);
	if (event != NULL) {
	    all[n_all++] = make2list("event", event_buf);
	}
    }

    class = XtClass(w);
    if (XtIsSubclass(w, xmArrowButtonWidgetClass)) {
	XmArrowButtonCallbackStruct *c_data = (XmArrowButtonCallbackStruct *) call_data;

	if (length == strlen("click_count") &&
		strncmp(value, "click_count", length) == 0) {
	    sprintf(buf, "%d", c_data->click_count);
	    return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    if (c_data->reason == XmCR_ACTIVATE) {
	        sprintf(buf, "%d", c_data->click_count);
	        all[n_all++] = make2list("click_count", buf);
	        return Tm_Merge(n_all, all);
	    } else {
		return Tm_Merge(n_all, all);
	    }

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmCommandWidgetClass)) {
	XmCommandCallbackStruct *c_data = (XmCommandCallbackStruct *) call_data;

	if (length == strlen("value") &&
		strncmp(value, "value", length) == 0) {
	    result = Tm_XmStringToString(c_data->value);
	    return result;
	} else

	if (length == strlen("length") &&
		strncmp(value, "length", length) == 0) {
	    sprintf(buf, "%d", c_data->length);
	    return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    all[n_all++] = make2list("value", Tm_XmStringToString(c_data->value));

	    sprintf(buf, "%d", c_data->length);
	    all[n_all++] = make2list("length", buf);

	    return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

#ifndef MOTIF11
    if (XtIsSubclass(w, xmDragContextClass)) {
	if (length == strlen("type") &&
		strncmp(value, "type", length) == 0) {
	    return TM_CONVERT_TYPE;
	} else

	if (length == strlen("value") &&
		strncmp(value, "value", length) == 0) {
	    return TM_CONVERT_VALUE;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    all[n_all++] = make2list("type", TM_CONVERT_TYPE);
	    all[n_all++] = make2list("value", TM_CONVERT_VALUE);

            return Tm_Merge(n_all, all);
	} else {
	    return "ERROR in substitution";
	}	
    } else
#endif

    if (XtIsSubclass(w, xmDrawnButtonWidgetClass)) {
	XmDrawnButtonCallbackStruct *c_data = (XmDrawnButtonCallbackStruct *) call_data;

	if (length == strlen("click_count") &&
		strncmp(value, "click_count", length) == 0) {
	    sprintf(buf, "%d", c_data->click_count);
	    return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    sprintf(buf, "%d", c_data->click_count);
	    all[n_all++] = make2list("click_count", buf);
	    return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmFileSelectionBoxWidgetClass)) {
	XmFileSelectionBoxCallbackStruct *c_data =
			(XmFileSelectionBoxCallbackStruct *) call_data;

	if (length == strlen("value") &&
		strncmp(value, "value", length) == 0) {
	    result = Tm_XmStringToString(c_data->value);
	    return result;
	} else

	if (length == strlen("length") &&
		strncmp(value, "length", length) == 0) {
	    sprintf(buf, "%d", c_data->length);
	    return buf;
	} else

	if (length == strlen("mask") &&
		strncmp(value, "mask", length) == 0) {
	    result = Tm_XmStringToString(c_data->mask);
	    return result;
	} else

	if (length == strlen("mask_length") &&
		strncmp(value, "mask_length", length) == 0) {
	    sprintf(buf, "%d", c_data->mask_length);
	    return buf;
	} else

	if (length == strlen("dir") &&
		strncmp(value, "dir", length) == 0) {
	    result = Tm_XmStringToString(c_data->dir);
	    return result;
	} else

	if (length == strlen("dir_length") &&
		strncmp(value, "dir_length", length) == 0) {
	    sprintf(buf, "%d", c_data->dir_length);
	    return buf;
	} else

	if (length == strlen("pattern") &&
		strncmp(value, "pattern", length) == 0) {
	   result = Tm_XmStringToString(c_data->pattern);
	    return result;
	} else

	if (length == strlen("pattern_length") &&
		strncmp(value, "pattern_length", length) == 0) {
	    sprintf(buf, "%d", c_data->pattern_length);
	    return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    all[n_all++] = make2list("value", Tm_XmStringToString(c_data->value));

	    sprintf(buf, "%d", c_data->length);
	    all[n_all++] = make2list("length", buf);

	    all[n_all++] = make2list("mask", Tm_XmStringToString(c_data->mask));

	    sprintf(buf, "%d", c_data->mask_length);
	    all[n_all++] = make2list("mask_length", buf);

	    all[n_all++] = make2list("dir", Tm_XmStringToString(c_data->dir));

	    sprintf(buf, "%d", c_data->dir_length);
	    all[n_all++] = make2list("dir_length", buf);

	    all[n_all++] = make2list("pattern", Tm_XmStringToString(c_data->pattern));

	    sprintf(buf, "%d", c_data->pattern_length);
	    all[n_all++] = make2list("pattern_length", buf);

	    return Tm_Merge(n_all, all);
	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmListWidgetClass)) {
	XmListCallbackStruct *c_data = (XmListCallbackStruct *) call_data;

	if (length == strlen("item") &&
		strncmp(value, "item",length) == 0) {
	    result = Tm_XmStringToString(c_data->item);
	    return result;
	} else
	if (length == strlen("item_length") &&
		strncmp(value, "item_length", length) == 0) {
	    sprintf(buf, "%d", c_data->item_length);
	    return buf;
	} else
	if (length == strlen("item_position") &&
		strncmp(value, "item_position", length) == 0) {
	    sprintf(buf, "%d", c_data->item_position);
	    return buf;
	} else
	if (length == strlen("selected_items") &&
		strncmp(value, "selected_items", length) == 0) {
	    if (c_data->reason == XmCR_MULTIPLE_SELECT ||
	    	c_data->reason == XmCR_EXTENDED_SELECT) {

		return items2String(c_data->selected_item_count,
				    c_data->selected_items);
#if 0
		String *items;
		static String items_list = NULL;
		static int items_list_len = 0;
		int n, len;
		char *p, *end;

		items = (char **) XtMalloc(sizeof(char *) *
					c_data->selected_item_count);
		len= 0;
		for (n = 0; n < c_data->selected_item_count; n++) {
		    items[n] = XtNewString(
				Tm_XmStringToString(c_data->selected_items[n]));
		    len += strlen(items[n]) + 2; /* for , */
		}

		/* reuse the static space we malloc for this list */
		if (items_list == NULL) {
		    items_list = XtMalloc(len + 1); /* we overshoot by 1 */
		    items_list_len = len + 1;
		} else
		if (items_list_len < len) {
		   items_list = XtRealloc(items_list, len + 1);
		   items_list_len = len + 1;
		}

		if (len == 0) {
		    *items_list = '\0';
		    return items_list;
		}
#ifdef TM_MOTIF_XMSTRINGTABLE
		end = items_list;
		for (n = 0; n < c_data->selected_item_count; n++) {
		    p = items[n];
		    while (*end++ = *p++)
			;
		    *(end - 1) = ',';
		    *end++ = ' ';
		    XtFree(items[n]);
		}
		*(end - 2) = '\0';
#else
		items_list = Tcl_Merge(c_data->selected_item_count, items);
                for (n = 0; n < c_data->selected_item_count; n++) {
                    XtFree(items[n]);
                }
#endif
		XtFree((char *) items);
		return items_list;
#endif /* 0 */
	    }
	} else
	if (length == strlen("selection_type") &&
		strncmp(value, "selection_type", length) == 0) {
	    switch (c_data->selection_type) {
		case XmINITIAL:		return "initial";
		case XmMODIFICATION:	return "modification";
		case XmADDITION:	return "addition";
		default:		return "ERROR!";
	    }
	} else
	if (length == strlen("selected_item_count") &&
		strncmp(value, "selected_item_count", length) == 0) {
	    sprintf(buf, "%d", c_data->selected_item_count);
	    return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    char *type;

	    all[n_all++] = make2list("item",
				Tm_XmStringToString(c_data->item));

	    sprintf(buf, "%d", c_data->item_length);
	    all[n_all++] = make2list("item_length", buf);

	    sprintf(buf, "%d", c_data->item_position);
	    all[n_all++] = make2list("item_position", buf);

	    all[n_all++] = make2list("selected_items", 
				items2String(c_data->selected_item_count,
                                    c_data->selected_items));
	    switch (c_data->selection_type) {
		case XmINITIAL:		type = "initial";
		case XmMODIFICATION:	type = "modification";
		case XmADDITION:	type = "addition";
		default:		type = "ERROR!";
	    }
	    all[n_all++] = make2list("selection_type", type);

	    sprintf(buf, "%d", c_data->selected_item_count);
	    all[n_all++] = make2list("selected_item_count", buf);

            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmPushButtonWidgetClass)) {
        XmPushButtonCallbackStruct *c_data = (XmPushButtonCallbackStruct *) call_data;

        if (length == strlen("click_count") &&
                strncmp(value, "click_count", length) == 0) {
            sprintf(buf, "%d", c_data->click_count);
            return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    if (c_data->reason == XmCR_ACTIVATE) {
	        sprintf(buf, "%d", c_data->click_count);
	        all[n_all++] = make2list("click_count", buf);
	        return Tm_Merge(n_all, all);
	    } else {
		return Tm_Merge(n_all, all);
	    }

        } else {
            return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmScaleWidgetClass)) {
	XmScaleCallbackStruct *c_data = (XmScaleCallbackStruct *) call_data;

	if (length == strlen("value") &&
		strncmp(value, "value", length) == 0) {
	    sprintf(buf, "%d", c_data->value);
	    return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    sprintf(buf, "%d", c_data->value);
	    all[n_all++] = make2list("value", buf);
	    return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmScrollBarWidgetClass)) {
	XmScrollBarCallbackStruct *c_data = (XmScrollBarCallbackStruct *) call_data;

	if (length == strlen("value") &&
		strncmp(value, "value", length) == 0) {
	    sprintf(buf, "%d", c_data->value);
	    return buf;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    sprintf(buf, "%d", c_data->value);
	    all[n_all++] = make2list("value", buf);
	    return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmTextWidgetClass) ||
	XtIsSubclass(w, xmTextFieldWidgetClass)) {
	XmTextVerifyCallbackStruct *c_data =
			(XmTextVerifyCallbackStruct *) call_data;

	if (length == strlen("doit") &&
		strncmp(value, "doit", length) == 0) {
		return TM_TEXT_DOIT;
	} else
	if (length == strlen("currInsert") &&
		strncmp(value, "currInsert", length) == 0) {
	    sprintf(buf, "%d", c_data->currInsert);
	    return buf;
	} else
	if (length == strlen("newInsert") &&
		strncmp(value, "newInsert", length) == 0) {
	    sprintf(buf, "%d", c_data->newInsert);
	    return buf;
	} else
	if (length == strlen("startPos") &&
		strncmp(value, "startPos", length) == 0) {
		return TM_TEXT_STARTPOS;
	} else
	if (length == strlen("endPos") &&
		strncmp(value, "endPos", length) == 0) {
		return TM_TEXT_ENDPOS;
	} else
	if (length == strlen("ptr") &&
		strncmp(value, "ptr", length) == 0) {
		return TM_TEXT_PTR;
	} else
	if (length == strlen("length") &&
		strncmp(value, "length", length) == 0) {
		return TM_TEXT_LENGTH;
	} else

	if (length == strlen("call_data") &&
		strncmp(value, "call_data", length) == 0) {
	    all[n_all++] = make2list("doit", TM_TEXT_DOIT);

	    sprintf(buf, "%d", c_data->currInsert);
	    all[n_all++] = make2list("currInsert", buf);

	    sprintf(buf, "%d", c_data->newInsert);
	    all[n_all++] = make2list("newInsert", buf);

	    all[n_all++] = make2list("startPos", TM_TEXT_STARTPOS);

	    all[n_all++] = make2list("endPos", TM_TEXT_ENDPOS);

	    all[n_all++] = make2list("ptr", TM_TEXT_PTR);

	    all[n_all++] = make2list("length", TM_TEXT_LENGTH);

	    return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmSelectionBoxWidgetClass)) {
	XmSelectionBoxCallbackStruct *c_data =
			(XmSelectionBoxCallbackStruct *) call_data;

	if (length == strlen("value") &&
		strncmp(value, "value",length) == 0) {
	    result = Tm_XmStringToString(c_data->value);
	    return result;
	} else
	if (length == strlen("length") &&
		strncmp(value, "length", length) == 0) {
	    sprintf(buf, "%d", c_data->length);
	    return buf;
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
            all[n_all++] = make2list("value", Tm_XmStringToString(c_data->value));

	    sprintf(buf, "%d", c_data->length);
            all[n_all++] = make2list("value", buf);

            return Tm_Merge(n_all, all);


	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmToggleButtonWidgetClass)) {
	XmToggleButtonCallbackStruct *c_data =
			(XmToggleButtonCallbackStruct *) call_data;

	if (length == strlen("set") &&
		strncmp(value, "set", length) == 0) {
	    if (c_data->set) {
		return "true";
	    } else {
		return "false";
	    }
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
	    if (c_data->set) {
                all[n_all++] = make2list("set", "true");
	    } else {
                all[n_all++] = make2list("set", "false");
	    }
 
            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else
#ifndef MOTIF11
    if (XtIsSubclass(w, xmDropTransferObjectClass)) {
	Tm_TransferStruct *c_data =
			(Tm_TransferStruct *) call_data;

	if (length == strlen("value") &&
		strncmp(value, "value", length) == 0) {
	    return c_data->value;
	} else
	if (length == strlen("closure") &&
	 	strncmp(value, "closure", length) == 0) {
	    return c_data->closure;
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
            all[n_all++] = make2list("value", c_data->value);
            all[n_all++] = make2list("closure", c_data->closure);
            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else
#endif /* MOTIF11 */

#if XmVERSION >= 2
    if (XtIsSubclass(w, xmContainerWidgetClass) &&
	(((XmAnyCallbackStruct *)call_data)->reason == XmCR_COLLAPSED ||
	 ((XmAnyCallbackStruct *)call_data)->reason == XmCR_EXPANDED)) {
	XmContainerOutlineCallbackStruct *c_data =
			(XmContainerOutlineCallbackStruct *) call_data;

	if (length == strlen("item") &&
		strncmp(value, "item", length) == 0) {
    	    Tm_Widget *wPtr;

	    XtVaGetValues(c_data->item, XmNuserData, &wPtr, NULL);
	    return wPtr->pathName;
	} else
	if (length == strlen("new_outline_state") &&
		strncmp(value, "new_outline_state", length) == 0) {
	    if (c_data->new_outline_state == XmCOLLAPSED)
		return "collapsed";
	    else
		return "expanded";
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
    	    Tm_Widget *wPtr;

	    XtVaGetValues(c_data->item, XmNuserData, &wPtr, NULL);
            all[n_all++] = make2list("item", wPtr->pathName);

	    if (c_data->new_outline_state == XmCOLLAPSED)
                all[n_all++] = make2list("new_outline_state", "collapsed");
	    else
                all[n_all++] = make2list("new_outline_state", "expanded");

            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmContainerWidgetClass)) { 
	XmContainerSelectCallbackStruct *c_data =
			(XmContainerSelectCallbackStruct *) call_data;

	if (length == strlen("selected_items") &&
		strncmp(value, "selected_items", length) == 0) {
	    return widgets2String(c_data->selected_item_count,
				  c_data->selected_items);
#if 0
	    String *items;
	    static String items_list = NULL;
	    static int items_list_len = 0;
	    int n, len;
	    char *p, *end;
    	    Tm_Widget *wPtr;

	    items = (char **) XtMalloc(sizeof(char *) *
					c_data->selected_item_count);
	    len= 0;
	    for (n = 0; n < c_data->selected_item_count; n++) {
	    	XtVaGetValues(c_data->selected_items[n], 
				XmNuserData, &wPtr, NULL);
	        items[n] = wPtr->pathName;
	        len += strlen(items[n]) + 2; /* for , */
	    }

	    /* reuse the static space we malloc for this list */
	    if (items_list == NULL) {
		    items_list = XtMalloc(len + 1); /* we overshoot by 1 */
		    items_list_len = len + 1;
	    } else
	    if (items_list_len < len) {
		   items_list = XtRealloc(items_list, len + 1);
		   items_list_len = len + 1;
	    }

	    if (len == 0) {
	        *items_list = '\0';
	        return items_list;
	    }
	    items_list = Tcl_Merge(c_data->selected_item_count, items);
            for (n = 0; n < c_data->selected_item_count; n++) {
                XtFree(items[n]);
            }
	    XtFree((char *) items);
	    return items_list;
#endif /* 0 */
	    
	} else
	if (length == strlen("selected_item_count") &&
		strncmp(value, "selected_item_count", length) == 0) {
	    sprintf(buf, "%d", c_data->selected_item_count);
	    return buf;
	} else
	if (length == strlen("auto_selection_type") &&
		strncmp(value, "auto_selection_type", length) == 0) {
	   switch (c_data->auto_selection_type) {
		case XmAUTO_UNSET:	return "auto_unset";
		case XmAUTO_BEGIN:	return "auto_begin";
		case XmAUTO_MOTION:	return "auto_motion";
		case XmAUTO_CANCEL:	return "auto_cancel";
		case XmAUTO_NO_CHANGE:	return "auto_no_change";
		case XmAUTO_CHANGE:	return "auto_change";
		default:		return "ERROR in substitution";
	    }
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
	    char *type;

	    all[n_all++] = make2list("selected_items",
				 widgets2String(c_data->selected_item_count,
				  		c_data->selected_items));

	    sprintf(buf, "%d", c_data->selected_item_count);
            all[n_all++] = make2list("selected_item_count", buf);

 	    switch (c_data->auto_selection_type) {
		case XmAUTO_UNSET:	type = "auto_unset";
		case XmAUTO_BEGIN:	type = "auto_begin";
		case XmAUTO_MOTION:	type = "auto_motion";
		case XmAUTO_CANCEL:	type = "auto_cancel";
		case XmAUTO_NO_CHANGE:	type = "auto_no_change";
		case XmAUTO_CHANGE:	type = "auto_change";
		default:		type = "ERROR in substitution";
	    }
            all[n_all++] = make2list("auto_selection_type", type);

            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmComboBoxWidgetClass)) {
	XmComboBoxCallbackStruct *c_data =
			(XmComboBoxCallbackStruct *) call_data;

	if (length == strlen("item_position") &&
		strncmp(value, "item_position", length) == 0) {
	    sprintf(buf, "%d", c_data->item_position);
	    return buf;
	} else
	if (length == strlen("item_or_text") &&
		strncmp(value, "item_or_text", length) == 0) {
	    return Tm_XmStringToString(c_data->item_or_text);
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
            all[n_all++] = make2list("item_position", c_data->item_position);
            all[n_all++] = make2list("item_or_text", Tm_XmStringToString(c_data->item_or_text));
            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmNotebookWidgetClass)) {
	XmNotebookCallbackStruct *c_data =
			(XmNotebookCallbackStruct *) call_data;

	if (length == strlen("page_number") &&
		strncmp(value, "page_number", length) == 0) {
	    sprintf(buf, "%d", c_data->page_number);
	    return buf;
	} else
	if (length == strlen("prev_page_number") &&
		strncmp(value, "prev_page_number", length) == 0) {
	    sprintf(buf, "%d", c_data->prev_page_number);
	    return buf;
	} else
	if (length == strlen("page_widget") &&
		strncmp(value, "page_widget", length) == 0) {
    	    Tm_Widget *wPtr;

	    XtVaGetValues(c_data->page_widget, XmNuserData, &wPtr, NULL);
	    return wPtr->pathName;
	} else
	if (length == strlen("prev_page_widget") &&
		strncmp(value, "prev_page_widget", length) == 0) {
    	    Tm_Widget *wPtr;

	    XtVaGetValues(c_data->prev_page_widget, XmNuserData, &wPtr, NULL);
	    return wPtr->pathName;
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
    	    Tm_Widget *wPtr;

	    sprintf(buf, "%d", c_data->page_number);
            all[n_all++] = make2list("page_number", buf);

	    sprintf(buf, "%d", c_data->prev_page_number);
            all[n_all++] = make2list("prev_page_number", buf);

	    XtVaGetValues(c_data->page_widget, XmNuserData, &wPtr, NULL);
	    all[n_all++] = make2list("page_widget", wPtr->pathName);

	    XtVaGetValues(c_data->prev_page_widget, XmNuserData, &wPtr, NULL);
	    all[n_all++] = make2list("prev_page_widget", wPtr->pathName);

            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmSpinBoxWidgetClass)) {
	XmSpinBoxCallbackStruct *c_data =
			(XmSpinBoxCallbackStruct *) call_data;

	if (length == strlen("position") &&
		strncmp(value, "position", length) == 0) {
	    sprintf(buf, "%d", c_data->position);
	    return buf;
	} else
	if (length == strlen("crossed_boundary") &&
		strncmp(value, "crossed_boundary", length) == 0) {
	    if (c_data->crossed_boundary)
		return "true";
	    else
		return "false";
	} else
	if (length == strlen("doit") &&
		strncmp(value, "doit", length) == 0) {
		return TM_TEXT_DOIT;
	} else
	if (length == strlen("widget") &&
		strncmp(value, "widget", length) == 0) {
    	    Tm_Widget *wPtr;

	    XtVaGetValues(c_data->widget, XmNuserData, &wPtr, NULL);
	    return wPtr->pathName;
	} else
	if (length == strlen("value") &&
		strncmp(value, "value", length) == 0) {
	    return Tm_XmStringToString(c_data->value);
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {
    	    Tm_Widget *wPtr;

	    sprintf(buf, "%d", c_data->position);
            all[n_all++] = make2list("position", buf);

	    if (c_data->crossed_boundary)
		all[n_all++] = make2list("crossed_boundary", "true");
	    else
		all[n_all++] = make2list("crossed_boundary", "false");

            all[n_all++] = make2list("doit", TM_TEXT_DOIT);

	    XtVaGetValues(c_data->widget, XmNuserData, &wPtr, NULL);
            all[n_all++] = make2list("widget", wPtr->pathName);

            all[n_all++] = make2list("value", Tm_XmStringToString(c_data->value));

            return "ERROR in substitution";

            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

    if (XtIsSubclass(w, xmCSTextWidgetClass)) {
	XmCSTextVerifyCallbackStruct *c_data =
			(XmCSTextVerifyCallbackStruct *) call_data;

	if (length == strlen("doit") &&
		strncmp(value, "doit", length) == 0) {
		return TM_TEXT_DOIT;
	} else
	if (length == strlen("currInsert") &&
		strncmp(value, "currInsert", length) == 0) {
	    sprintf(buf, "%d", c_data->currInsert);
	    return buf;
	} else
	if (length == strlen("newInsert") &&
		strncmp(value, "newInsert", length) == 0) {
	    sprintf(buf, "%d", c_data->newInsert);
	    return buf;
	} else
	if (length == strlen("startPos") &&
		strncmp(value, "startPos", length) == 0) {
		return TM_TEXT_STARTPOS;
	} else
	if (length == strlen("endPos") &&
		strncmp(value, "endPos", length) == 0) {
		return TM_TEXT_ENDPOS;
	} else
	if (length == strlen("text") &&
		strncmp(value, "text", length) == 0) {
		return TM_TEXT_PTR;
        } else

        if (length == strlen("call_data") &&
                strncmp(value, "call_data", length) == 0) {

	    all[n_all++] = make2list("doit", TM_TEXT_DOIT);

	    sprintf(buf, "%d", c_data->currInsert);
	    all[n_all++] = make2list("currInsert", buf);

	    sprintf(buf, "%d", c_data->newInsert);
	    all[n_all++] = make2list("newInsert", buf);

	    all[n_all++] = make2list("startPos", TM_TEXT_STARTPOS);

	    all[n_all++] = make2list("endPos", TM_TEXT_ENDPOS);

	    all[n_all++] = make2list("ptr", TM_TEXT_PTR);

	    all[n_all++] = make2list("length", TM_TEXT_LENGTH);

            return Tm_Merge(n_all, all);

	} else {
	    return "ERROR in substitution";
	}
    } else

#endif /* XmVERSION >= 2 */

    /*
     * oh well, it wasnt one of the standard widgets -
     * does it belong in what someone else added?
     */
    return Tm_ExternExpandPercent(pathName, w, event,
                		call_data, value, length);
}


/*
 *--------------------------------------------------------------
 *
 * addChars --
 *
 *	add a list of characters to a string.
 *	grows space if neccessary
 *
 * Results:
 *
 *	"from" as appended to "after"
 *
 * Side effects:
 *
 *	may reallocate memory
 *--------------------------------------------------------------
 */

static char *
addChars(after, afterSize, end, spaceLeft, fromSize, from)
    char *after;
    int *afterSize;
    char **end;
    int *spaceLeft;
    int fromSize;
    char *from;
{
    char *newSpace;
    int extra;

    if (*spaceLeft < fromSize) {
	if (fromSize > *afterSize) {
	    extra = fromSize + 50;
	} else {
	    extra = *afterSize;
	}
	newSpace = XtRealloc(after, *afterSize + extra);
	*spaceLeft += extra;
	*afterSize += extra;
	*end = newSpace + (*end - after);
	after = newSpace;
    }
    *spaceLeft -= fromSize;

    while (fromSize) {
	**end = *from;
	from++;
	(*end)++;
	fromSize--;
    }
    return after;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ExpandPercents --
 *
 *	expand out the %field in patterns by their callback values
 *
 * Results:
 *
 *	A new string with the expansions is returned
 * Side effects:
 *
 *	none.
 *--------------------------------------------------------------
 */

char *
Tm_ExpandPercents(pathName, w, event, call_data, before)
    char *pathName;
    Widget w;
    XEvent *event;
    XtPointer call_data;
    char *before;
{
    char *after, *end;
    char *value;
#   define TM_STR_SIZE 128
    int afterSize = TM_STR_SIZE;
    int spaceLeft = TM_STR_SIZE;
    int wordSize;

    if (strchr(before, '%') == NULL) {
	after = XtNewString(before);
	return after;
    }

    end = after = XtMalloc(TM_STR_SIZE);

    while (*before != 0) {
	if (*before != '%') {
	    after = addChars(after, &afterSize, &end, &spaceLeft, 1, before);
	    before++;
	} else {
	    before++;
	    wordSize = 0;
	    while (isalpha(before[wordSize]) || before[wordSize] == '_') {
		wordSize++;
	    }
	    if (wordSize == 0) {
	        after = addChars(after, &afterSize, &end, &spaceLeft, 1, "%");
	    } else {
		value = getValue(pathName, w,
				event, call_data, before, wordSize);
		after = addChars(after, &afterSize, &end, &spaceLeft,
				 strlen(value), value);
		before += wordSize;
	    }
	}
    }
    *end = '\0';

    return after;
}
