
/*
 * TmResources.c --
 *	This module contains the functions to implement resource
 *	handling such as getValues, setValues, and resources
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "tm.h"
#include "tmFuncs.h"
#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/Text.h>

/* this is a hack because some converters need info about what
   types they are converting, and Xt does not give such info.
   So we need to place this in globals before calling converters.
   This is to replace the earlier, worse, Tm_MaybeSetTableSize
 */
char *Tm_Resource = NULL;


/*
 *--------------------------------------------------------------
 *
 * CopyToArgs --
 *
 *	copy to arg of correct size
 *	borrowed from _XtCopyToArgs from Xt lib
 *
 * Results:
 *
 *	new value (of whatever type) stored in dst
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */
static void 
CopyToArg(src, dst, size)
    char* src;
    XtArgVal *dst;
    register unsigned int size;
{
    if (! (*dst)) {
        if      (size == sizeof(long))     *dst = (XtArgVal)*(long*)src;
        else if (size == sizeof(short))    *dst = (XtArgVal)*(short*)src;
        else if (size == sizeof(char))     *dst = (XtArgVal)*(char*)src;
        else if (size == sizeof(XtPointer)) *dst = (XtArgVal)*(XtPointer*)src;
        else if (size == sizeof(char*))    *dst = (XtArgVal)*(char**)src;
        else if (size == sizeof(XtArgVal)) *dst = *(XtArgVal*)src;
        else bcopy((char*)src, (char*)dst, (int)size);
    }
} 

/*
 *--------------------------------------------------------------
 *
 * These functions are all to handle resources that got created
 * and need to be reclaimed
 *
 *--------------------------------------------------------------
 */

static Tm_FreeResourceType *Tm_FreeResources = NULL;
static int Tm_FreeResourcesCount;
static int Tm_MaxFreeResourcesCount;

typedef struct Tm_FreeResourceStackElmt {
    Tm_FreeResourceType *freeResources;
    int			freeResourcesCount;
    int 		maxFreeResourcesCount;
} Tm_FreeResourceStackElmt;

static int Tm_FreeResourceStackDepth = 0;
static Tm_FreeResourceStackElmt *Tm_FreeResourceStack = NULL;

/*
 *--------------------------------------------------------------
 *
 * Tm_InitFreeResourceList
 *
 *	initialise the list of resources for which space must
 *	be reclaimed after XtSet/GetValues
 *
 *	to allow this to be reentrant, place old entries on stack
 *
 *--------------------------------------------------------------
 */
void Tm_InitFreeResourceList(size)
    int size;
{

    /* store any old values if there were any */
    if (Tm_FreeResourceStackDepth > 0) {
	Tm_FreeResourceStack = (Tm_FreeResourceStackElmt *)
		XtRealloc((char *)Tm_FreeResourceStack,
			Tm_FreeResourceStackDepth * 
			sizeof(Tm_FreeResourceStackElmt));
	Tm_FreeResourceStack[Tm_FreeResourceStackDepth - 1].freeResources = 
				Tm_FreeResources;
	Tm_FreeResourceStack[Tm_FreeResourceStackDepth - 1].freeResourcesCount =
				Tm_FreeResourcesCount;
	Tm_FreeResourceStack[Tm_FreeResourceStackDepth - 1].maxFreeResourcesCount =
				Tm_MaxFreeResourcesCount;

    }
    Tm_FreeResourceStackDepth++;

    if (size <= 0)
	size = 1;
    Tm_FreeResources = (Tm_FreeResourceType *)
			XtMalloc(size * sizeof(Tm_FreeResourceType));
    Tm_FreeResourcesCount = 0;
    Tm_MaxFreeResourcesCount = size;
}


/*
 *--------------------------------------------------------------
 *
 *  Tm_AddToFreeResourceList
 *
 *	add some data and its corresponding free proc to the list
 *
 *--------------------------------------------------------------
 */
void  Tm_AddToFreeResourceList(data, free)
    char *data;
    Tm_FreeProc free;
{
    /* we only setup the list for adding during a Tm_Set/GetValues
       sequence. We init at the start and free at the end. This
       func gets called by various resource converters. However,
       Motif (or someone else) may be calling our converter outside
       of the Tm_Get/SetValues. We have to assume that they are doing
       their own garbage collect, so need to make sure that we don't
       add for their calls
     */
    if (Tm_FreeResources == NULL)
	return;

    /* things like XmStringTables may add many elmts. Grow if needed */
    if (Tm_FreeResourcesCount >= Tm_MaxFreeResourcesCount) {
	Tm_MaxFreeResourcesCount *= 2;
	Tm_FreeResources = (Tm_FreeResourceType *) 
				XtRealloc((char *)Tm_FreeResources,
					Tm_MaxFreeResourcesCount * 
					sizeof(Tm_FreeResourceType));
    }
    Tm_FreeResources[Tm_FreeResourcesCount].data = data;
    Tm_FreeResources[Tm_FreeResourcesCount++].free = free;
}


/*
 *--------------------------------------------------------------
 * Tm_FreeResourceList
 *
 *	call the free proc on each data elmt of the list
 *
 *--------------------------------------------------------------
 */
void Tm_FreeResourceList()
{
    int n;
    char *data;
    Tm_FreeProc free;

    for (n = 0; n < Tm_FreeResourcesCount; n++) {
	free =  Tm_FreeResources[n].free;
	data = Tm_FreeResources[n].data;
	(free) (data);
    }
    XtFree((char *) Tm_FreeResources);
    Tm_FreeResources = NULL;

    Tm_FreeResourceStackDepth--;
    if (Tm_FreeResourceStackDepth < 1)
	return;

    /* nested calls to Tm_InitFreeResourceList are unwinding - 
       restore values from the stack
     */
    Tm_FreeResources =
        Tm_FreeResourceStack[Tm_FreeResourceStackDepth - 1].freeResources;
    Tm_FreeResourcesCount =
        Tm_FreeResourceStack[Tm_FreeResourceStackDepth - 1].freeResourcesCount;
    Tm_MaxFreeResourcesCount =
        Tm_FreeResourceStack[Tm_FreeResourceStackDepth - 1].maxFreeResourcesCount;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValue --
 *
 *	convert a value from one format to another
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValue(w, fromType, fromValue, fromSize,
		   toType, toValue, toSize)
    Widget w;
    char *fromType;
    char *fromValue;
    unsigned int fromSize;
    char *toType;
    XtArgVal *toValue;
    unsigned int toSize;
{
    XrmValue from, to;

    if (strcmp(fromType, toType) == 0) {
	*toValue = (XtArgVal) fromValue;
	return TRUE;
    }

    from.size = fromSize;
    from.addr = fromValue;
    to.size = toSize;
    to.addr = XtMalloc(toSize);
/*
    to.addr = NULL;
*/

    if (!XtConvertAndStore(w, fromType, &from, toType, &to))  {
	fprintf(stderr, "failed conversion %s to %s\n",
				fromType, toType);
	return FALSE;
    }
    *toValue = 0;
    CopyToArg(to.addr, toValue, to.size);
    XtFree(to.addr);

    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueFromString --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants.
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValueFromString(w, resources, num_resources,
		resource, orig_value, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n;
    XrmValue from, converted;
    XtArgVal tmpval;
    Boolean result;

#   ifdef DEBUG
    fprintf(stderr, "converting from string \"%s\" of type %s\n", orig_value, resource);
#   endif

    for (n = 0; n < num_resources; n++) {
	if (strcmp(resource, resources->resource_name) == 0) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    /* we have a match - convert and install the resource */
    Tm_Resource = resource;
    result = Tm_ConvertValue(w, XtRString, orig_value,
				strlen(orig_value)+1,
				resources->resource_type,
				new_value,
				resources->resource_size);
    return result;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueFromStringQuark --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants. In this form the resources have been
 *	quark'ed so we don't have string compares but int ones
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValueFromStringQuark(w, resources, num_resources,
		resource, orig_value, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n;
    XrmValue from, converted;
    XtArgVal tmpval;
    int resourceQ;

    resourceQ = XrmStringToQuark(resource);
#   ifdef DEBUG
    fprintf(stderr, "converting from string \"%s\" of type %s\n",
			orig_value, resource);
#   endif
    for (n = 0; n < num_resources; n++) {
	if (resourceQ == (int) resources->resource_name) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    /* we have a match - convert and install the resource */
    /* special case - same type */
    if (strcmp(XrmQuarkToString((int) resources->resource_type), XtRString) == 0) {
	*new_value =  (XtArgVal) orig_value;
	return TRUE;
    }
    from.size = strlen(orig_value) + 1;
    from.addr = orig_value;
    converted.size = resources->resource_size;
    converted.addr = NULL;

    if (!XtConvertAndStore(w, XtRString, &from,
		XrmQuarkToString((int) resources->resource_type), &converted)) {
	return FALSE;
    }
    *new_value = 0;
    CopyToArg(converted.addr, new_value, converted.size);
/* for now
    XtFree(converted.addr);
*/
/*
    bzero(new_value, sizeof(XtArgVal));
    bcopy(converted.addr, new_value, converted.size);
    *new_value = *(XtArgVal *) (converted.addr);
*/
    
    return TRUE;
}
		

/*
 *--------------------------------------------------------------
 *
 * Tm_MaybeSetStringTableSize --
 *
 *	Hack for Motif XmStringTables.
 *	Motif *sometimes* NULL terminates its XmStringTables
 *	most times doesnt.
 *	XmStringGet...() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	_XmStringIsXmString() sometimes returns False on running off
 *	the end of a table, sometimes crashes
 *	i.e. there is no internal way of finding the size of one
 *	of these things. So we have to check if we have one and
 *	then look up the resource that tells us the size :-(
 *
 * Results:
 *
 * Side effects:
 *
 *	Sets global vbl Tm_XmStringTableSize
 *--------------------------------------------------------------
 */

static void
Tm_MaybeSetStringTableSize(w, resource)
    Widget w;
    String resource;
{
    extern int Tm_XmStringTableSize;
    WidgetClass class;
    Arg arg;
    char *size_resource;

#if 0
    class = XtClass(w);
    if (class == xmListWidgetClass) {
	if (strcmp (resource, XmNitems) == 0) {
	    size_resource = XmNitemCount;
	} else
	if (strcmp (resource, XmNselectedItems) == 0) {
	    size_resource = XmNselectedItemCount;
	} else
	return;
    } else
    return;

    /* Sigh, we have a StringTable, find its size */
    XtSetArg(arg, size_resource, &Tm_XmStringTableSize);
    XtGetValues(w, &arg, 1);
#endif
}

/*
 *--------------------------------------------------------------
 *
 * Tm_CovertResourceToString --
 *	convert a single resource to a string
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertResourceToString(w, resource, resource_size, resource_type, value)
    Widget w;
    char *resource;
    Cardinal resource_size;
    String resource_type;
    char **value;
{
    XrmValue from, converted;
    Arg args[1];

    Tm_MaybeSetStringTableSize(w, resource);

    from.size = resource_size;
    from.addr = (XtPointer) XtMalloc(from.size);
    XtSetArg(args[0], resource, from.addr);
    XtGetValues(w, args, 1);
    
    /* special case - same type */
    if (strcmp(resource_type, XtRString) == 0) {
	*value =  XtNewString(*(char **) from.addr);
	/* check if it is Text's value */
	if (XtIsSubclass(w, xmTextWidgetClass) &&
	    strcmp(resource, XmNvalue) == 0)  {
	    XtFree(*(char **) from.addr);
	}
	XtFree(from.addr);

	/* allow this to be gc later */
	Tm_AddToFreeResourceList(*value, (Tm_FreeProc) XtFree);

	return TRUE;
    }

    converted.addr = NULL;
    if (!XtConvertAndStore(w, resource_type, &from,
			XtRString, &converted)) {
	XtFree(from.addr);
	return FALSE;
    }
    bzero(value, sizeof(XtArgVal));
    bcopy(converted.addr, value, converted.size);
    
    XtFree(from.addr);
    return TRUE;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Boolean
Tm_ConvertValueToString(w, resources, num_resources,
		resource, new_value)
    Widget w;
    XtResourceList resources;
    int num_resources;
    char *resource;
    char **new_value;
{
    int n;

    for (n = 0; n < num_resources; n++) {
	if (strcmp(resource, resources->resource_name) == 0) {
	    break;
	}
	resources++;
    }
    if (n == num_resources) {
	return FALSE;
    }

    Tm_Resource = resource;
    return Tm_ConvertResourceToString(w, resource,
		resources->resource_size, resources->resource_type, new_value);

}
		
/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertSubValueFromString --
 *
 *	convert a value from its Tcl string format to the one
 *	the widget wants.
 *
 * Results:
 *
 *	new value (of whatever type) stored in "new_value"
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertSubValueFromString(w, sec_resources, num_sec_resources,
		resource, orig_value, new_value)
    Widget w;
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
    char *resource;
    char *orig_value;
    XtArgVal *new_value;
{
    int n, length;

    for (n = 0; n < num_sec_resources; n++) {
       	if (Tm_ConvertValueFromString(w, sec_resources[n]->resources, 
            sec_resources[n]->num_resources, resource,
       	    orig_value, new_value)) {
	    return True;
	}
    }
    return False;
}
		
/*
 *--------------------------------------------------------------
 *
 * Tm_ConvertSubValueToString --
 *
 *	converts from the internal widget form to a string for Tcl
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

static Boolean
Tm_ConvertSubValueToString(w, sec_resources, num_sec_resources,
		resource, new_value)
    Widget w;
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
    char *resource;
    char **new_value;
{
    int n, length;

    for (n = 0; n < num_sec_resources; n++) {
       	if (Tm_ConvertValueToString(w, sec_resources[n]->resources, 
            sec_resources[n]->num_resources, resource,
       	    new_value)) {
	    return True;
	}
    }
    return False;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_FreeSecondaryResources --
 *
 *	Motif gives us a copy of these, so we have to tidy up
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_FreeSecondaryResources(sec_resources, num_sec_resources)
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
{
    int n;

    for (n = 0; n < num_sec_resources; n++) {
	XtFree((char *) sec_resources[n]->resources);
	XtFree((char *) sec_resources[n]);
    }

    if (num_sec_resources > 0) {
        XtFree((char *) sec_resources);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tm_SetValues --
 *
 *	set resource values on a widget
 *
 * Results:
 * 	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

int
Tm_SetValues(pathName, interp, w, parent, class, argv, argc, args, num_args)
    char *pathName;
    Tcl_Interp *interp;
    Widget w;
    Widget parent;
    WidgetClass class;
    char **argv;
    int argc;
    Arg args[];
    int *num_args;
{
    XtResourceList resources, constraint_resources;
#ifndef MOTIF11
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
#endif
    Cardinal num_resources;
    Cardinal num_constraint_resources;
    int num_values = 0;
    XrmValue from, converted;
    XtArgVal new_value;
    char *resource;
    char *value;
    Tm_ClientData *client_data;

    *num_args = 0;

    if (argc > TM_MAXARGS*2) {
	return TCL_ERROR;
    }

    XtGetResourceList(class, &resources, &num_resources);
#ifndef MOTIF11
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
#endif
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    while (argc >= 2) {
	if (argv[0][0] != '-') {
	    sprintf(interp->result, "illegal setValues option \"%50s\"", argv[0]);
	    return TCL_ERROR;
	}
	resource = argv[0]+1;
        value = argv[1];

        /* we cant allow callbacks to be converted - too messy */
        if (strstr(resource, "Callback") != NULL) {
	    sprintf(interp->result, "setValues doesn't support callbacks.\
Use \"%s\" method\n", resource);
  	    return TCL_ERROR;
        }

        if (Tm_ConvertValueFromString(w, resources, num_resources,
		resource, value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else
        if (num_constraint_resources > 0 &&
	    Tm_ConvertValueFromString(w, constraint_resources, 
	            num_constraint_resources, resource,
        	    value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else 
#ifndef MOTIF11
        if (num_sec_resources > 0 &&
	    Tm_ConvertSubValueFromString(w, sec_resources, 
	            num_sec_resources, resource,
        	    value, &new_value)) {
            XtSetArg(args[num_values], resource, new_value);
	    num_values++;
	} else
#endif
	{
	    sprintf(interp->result, "Conversion from String to \"%50s\" failed\n",
                                resource);
	    return TCL_ERROR;
	}
	argc -= 2;
	argv += 2;
    }

#ifndef MOTIF11
    if (num_sec_resources > 0)
	Tm_FreeSecondaryResources(sec_resources, num_sec_resources);
#endif /* MOTIF11 */
    if (num_resources > 0) 
	XtFree((char *) resources);
    if (num_constraint_resources > 0) 
	XtFree((char *) constraint_resources);

    *num_args = num_values;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_GetValues --
 *
 *	get the resource values out of a widget
 *
 * Results:
 *	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

int
Tm_GetValues(pathName, interp, w, class, argv, argc)
    char *pathName;
    Tcl_Interp *interp;
    Widget w;
    WidgetClass class;
    char **argv;
    int argc;
{
    XtResourceList resources, constraint_resources;
#ifndef MOTIF11
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
#endif
    Cardinal num_resources;
    Cardinal num_constraint_resources;
    int num_values = 0;
    XrmValue from, converted;
    char *new_value;
#   define MAXARGS 100
    char *resource;
    Tm_ClientData *client_data;
    Widget parent;

    if (argc > MAXARGS*2) {
	return TCL_ERROR;
    }

    parent = XtParent(w);	/* this is NULL for w == "." */
    XtGetResourceList(class, &resources, &num_resources);

#ifndef MOTIF11
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
#endif
  
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    while (argc >= 2) {
	if (argv[0][0] != '-') {
	    sprintf(interp->result, "illegal setValues option \"%50s\"", argv[0]);
	    return TCL_ERROR;
	}
	resource = argv[0]+1;

        if (Tm_ConvertValueToString(w, resources, num_resources,
		resource, &new_value)) {
	    /* store new_value in variable in argv[1] now */
#	    ifdef DEBUG
	    fprintf(stderr, "Got value: %s\n", (char *) new_value);
#	    endif
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else
        if (num_constraint_resources > 0 &&
	    Tm_ConvertValueToString(w, constraint_resources, 
	            num_constraint_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else 
#ifndef MOTIF11
        if (num_sec_resources > 0 &&
	    Tm_ConvertSubValueToString(w, sec_resources, 
	            num_sec_resources, resource,
        	    &new_value)) {
	    /* store in a variable now */
	    if (Tcl_SetVar(interp, argv[1], (char *) new_value,
			TCL_LEAVE_ERR_MSG) == NULL)
		fprintf(stderr, "%s\n", interp->result);
	} else
#endif
	{
	    sprintf(interp->result, "Conversion from %s to String failed\n",
					resource);
	    return TCL_ERROR;
	}
	argc -= 2;
	argv += 2;
    }
#ifndef MOTIF11
    if (num_sec_resources > 0)
	Tm_FreeSecondaryResources(sec_resources, num_sec_resources);
#endif
    if (num_resources > 0) 
	XtFree((char *) resources);
    if (num_constraint_resources > 0) 
	XtFree((char *) constraint_resources);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ResourceInfo --
 *
 *	get the resource info out of a widget
 *
 * Results:
 *	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_GetResourceInfo(resourceInfo, w, resource)
    Tcl_DString *resourceInfo;
    Widget w;
    XtResource *resource;
{
    char *resource_value;

    Tcl_DStringAppend(resourceInfo, "-", 1);
    Tcl_DStringAppend(resourceInfo, resource->resource_name, -1);
    Tcl_DStringAppendElement(resourceInfo, resource->resource_name);
    Tcl_DStringAppendElement(resourceInfo, resource->resource_class);
    Tcl_DStringAppendElement(resourceInfo, resource->resource_type);

    /* hack to let resource converter know what this is */
    Tm_Resource = resource->resource_name;

    if (Tm_ConvertResourceToString(w, resource->resource_name,
		resource->resource_size, resource->resource_type,
		&resource_value)) {;
        Tcl_DStringAppendElement(resourceInfo, resource_value);
    } else {
        Tcl_DStringAppendElement(resourceInfo, "");
    }
}
/*
 *--------------------------------------------------------------
 *
 * Tm_NoErrorMsg --
 *
 * Results:
 * 	none
 *
 * Side effects:
 *	does nothing
 *
 *--------------------------------------------------------------
 */

void
Tm_NoErrorMsg(name, type, Class, defalt, params, num_params)
    String name;
    String type;
    String Class;
    String defalt;
    String *params;
    Cardinal *num_params;
{
    /* void */
}


/*
 *--------------------------------------------------------------
 *
 * Tm_ResourceList --
 *
 *	get the resource list out of a widget
 *
 * Results:
 *	a tcl result
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

int
Tm_ResourceList(interp, w, class)
    Tcl_Interp *interp;
    Widget w;
    WidgetClass class;
{
    XtResourceList resources, constraint_resources;
#ifndef MOTIF11
    XmSecondaryResourceData *sec_resources;
    int num_sec_resources;
#endif
    Cardinal num_resources;
    Cardinal num_constraint_resources;
    int num_values = 0;
    char *new_value;
    char *resource;
    Tm_ClientData *client_data;
    Widget parent;
    int m, n;
    Tcl_DString resourceList, resourceInfo;
    XtErrorHandler oldWarningHandler;

    parent = XtParent(w);	/* this is NULL for w == "." */
    XtGetResourceList(class, &resources, &num_resources);

#ifndef MOTIF11
    num_sec_resources = XmGetSecondaryResourceData(class, &sec_resources);
#endif
  
    if (parent != NULL && XtIsConstraint(parent)) {
	XtGetConstraintResourceList(XtClass(parent), 
	    &constraint_resources, &num_constraint_resources);
    } else {
	num_constraint_resources = 0;
    }

    /* turn off error messages about converters that we don't support */
    oldWarningHandler = XtAppSetWarningHandler(XtWidgetToApplicationContext(w), 
					Tm_NoErrorMsg);

    /* set up for garbage collect of resource values */
    Tm_InitFreeResourceList(128);

    Tcl_DStringInit(&resourceList);

    Tcl_DStringInit(&resourceInfo);
    for (n = 0; n < num_resources; n++) {
	Tm_GetResourceInfo(&resourceInfo, w, resources+n);
	/* hack to get string out of D-string */
/*
	Tcl_DStringResult(interp, &resourceInfo);
	Tcl_DStringAppendElement(&resourceList, interp->result);
*/

	/* no, do it the proper way (accessing internals of DString :-(  ) */
	Tcl_DStringAppendElement(&resourceList, resourceInfo.string);
	Tcl_DStringFree(&resourceInfo);
	Tcl_DStringInit(&resourceInfo);
    }

#   ifndef MOTIF11
    Tcl_DStringInit(&resourceInfo);
    for (n = 0; n < num_sec_resources; n++) {
	for (m = 0; m < sec_resources[n]->num_resources; m++) {
	    Tm_GetResourceInfo(&resourceInfo, w, &(sec_resources[n]->resources[m]));
	    /* hack to get string out of D-string */
	    Tcl_DStringResult(interp, &resourceInfo);
	    Tcl_DStringAppendElement(&resourceList, interp->result);
	}
    }
#   endif

    Tcl_DStringInit(&resourceInfo);
    for (n = 0; n < num_constraint_resources; n++) {
	Tm_GetResourceInfo(&resourceInfo, w, constraint_resources+n);
	/* hack to get string out of D-string */
	Tcl_DStringResult(interp, &resourceInfo);
	Tcl_DStringAppendElement(&resourceList, interp->result);
    }

    /* turn error reporting back on */
    XtAppSetWarningHandler(XtWidgetToApplicationContext(w), oldWarningHandler);

    Tcl_DStringResult(interp, &resourceList);

#ifndef MOTIF11
    if (num_sec_resources > 0)
	Tm_FreeSecondaryResources(sec_resources, num_sec_resources);
#endif
    if (num_resources > 0) 
	XtFree((char *) resources);
    if (num_constraint_resources > 0) 
	XtFree((char *) constraint_resources);
    Tm_FreeResourceList();

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Tm_GetAppResources -- get the resources specific to the
 *	application level
 *
 * Results:
 * 	standard tcl result
 *
 * Side effects:
 *	sets tcl vbls with the application resources
 *
 *--------------------------------------------------------------
 */

int
Tm_GetAppResources(interp, w, resource_str)
    Tcl_Interp *interp;
    Widget w;
    String resource_str;
{
    char **resource_list;
    int num_resources;
    char **res_elmts;
    int num_elmts;
    XtResourceList resources;
    String *vbls;
    String *values;
    int n;

    Tcl_SplitList(interp, resource_str, &num_resources, &resource_list);
    resources = (XtResource *) XtMalloc(num_resources * sizeof(XtResource));
    vbls = (String *) XtMalloc(num_resources * sizeof(String));
    values = (String *) XtMalloc(num_resources * sizeof(String));

    for (n = 0; n < num_resources; n++) {
	Tcl_SplitList(interp, resource_list[n], &num_elmts, &res_elmts);
	if (num_elmts != 4) {
	    sprintf(interp->result, "wrong getAppResources \"%50s\"",
			resource_list[n]);
	   return TCL_ERROR;
	}

	resources[n].resource_name = res_elmts[0];
	resources[n].resource_class = res_elmts[1];
	resources[n].resource_type = XtRString;
	resources[n].resource_offset = n * sizeof(String);
	resources[n].default_type = XtRString;
	resources[n].default_addr = res_elmts[2];

	vbls[n] = res_elmts[3];
    }

    XtGetApplicationResources(w, values, resources, num_resources, NULL, 0);

    for (n = 0; n < num_resources; n++) {
	Tcl_SetVar(interp, vbls[n], values[n], 0);
    }

    return TCL_OK;
}
