#
# HyperThing Conference Application
# ---------------------------------
# 
# This application extends the Drawing application (see draw.tcl) so that
# rather than drawing rectangles, we're now drawing ovals.  Furthermore,
# these ovals can be connected together with directed lines, producing a
# graph like display.  When the ovals are moved, any lines connected to them
# are moved accordingly.

#source "conf.tcl"
initConf $argv

#
# Set up the basic window, some globals, canvas level bindings, etc.
#

gkDefaultMenu .menu
canvas .c 
pack append . .menu {top fillx} .c top
set startX 0
set startY 0

set nextID [expr {[keylget ouropts usernum] * 100}]

set cursors(0) ""
bind .c <1> { start %x %y }
bind .c <B1-Motion> {doit %x %y; movecursor %x %y }
bind .c <Motion> {movecursor %x %y }


#
# This is called on a mouse down on the canvas, signifying the start
# of a new oval.  Assign it an id number, create it locally and 
# tell the other users to create it
#

proc start {x y} {global startX startY others theid id global nextID .c
    set startX $x;     set startY $y
    set theid $nextID; incr nextID
    setnode $theid $x $y $x $y
    foreach i $others {
	RDO [keylget i filedesc] setnode $theid $x $y $x $y
    }
}


#
# This does the work of "resizing" the oval on the local display.
# It also creates the object if necessary.  The id() array holds the mappings
# between object ids and canvas ids.
#
# This routine also does all the work of making sure any edges attached
# to the node are updated appropriately.  Besides a canvasid, each object
# also stores a list of all edges attached to the object.  This list is
# traversed so that each edge is updated.
#

proc setnode {objid x0 y0 x1 y1} { global id .c
    if {[info exists id($objid)] != 1} {
	keylset id($objid) canvasid [.c create oval $x0 $y0 $x1 $y1 \
				     -tags {node}] edgelist ""
	.c bind [keylget id($objid) canvasid] <3> \
	    "global lastX lastY; set lastX %x; set lastY %y"
	.c bind [keylget id($objid) canvasid] <B3-Motion> \
	    " domove $objid %x %y; movecursor %x %y "
    } else {
	eval ".c coords [keylget id($objid) canvasid] $x0 $y0 $x1 $y1"
    }
    set edges ""
    catch {set edges [keylget id($objid) edgelist]}
    foreach i $edges {
	set cid [keylget id([lindex $i 2]) canvasid]
	set otherid [idfromcanvasid [keylget id([lindex $i 1]) canvasid]]
	if {[lindex $i 0] == "to"} {
	    eval ".c coords $cid [center [keylget id($objid) canvasid]] [center [keylget id($otherid) canvasid]]"
	} else {
	    eval ".c coords $cid [center [keylget id($otherid) canvasid]]   [center [keylget id($objid) canvasid]]"
	}
    }
}


#
# This is called when the mouse is moved while pressing button 1.  Essentially
# we're resizing the oval when we first create it.  Display locally
# and tell everyone else.
#

proc doit {x y} { global startX startY others theid
    setnode $theid $startX $startY $x $y
    foreach i $others { 
	RDO [keylget i filedesc] setnode $theid $startX $startY $x $y
    }
}


#
# Called when the local cursor moves; tell everyone else about this
#

proc movecursor {x y} { global others .c ouropts
    update
    foreach i $others {
	RDO [keylget i filedesc] moveit [keylget ouropts usernum] $x $y
    }
}


#
# Called when a remote cursor is moved.  Move the bitmap for the remote
# cursor.
#

proc moveit {who x y} { global .c cursors gk_library
    if {[info exists cursors($who)] != 1} {
	set cursors($who) [.c create bitmap 2c 2c -bitmap @$gk_library/bitmaps/cursor.bit]
    }
    .c coords $cursors($who) $x $y
}



#
# When a user leaves, get rid of their cursor bitmap on the canvas.
#

trace variable deleted_user w nuked_user

proc nuked_user args { global deleted_user cursors
    if {[info exists cursors([keylget deleted_user usernum])]} {
        .c delete $cursors([keylget deleted_user usernum])
        unset cursors([keylget deleted_user usernum])
    }
}



set lastX 0
set lastY 0


#
# Called when the oval is grabbed with the right mouse button and
# moved.  Move the oval on the local display, and broadcast its new
# coordinates to the other users.  
#

proc domove {theid x y} { global .c lastX lastY id others
    .c move [keylget id($theid) canvasid] [expr $x-$lastX] [expr $y-$lastY]
    eval "setnode $theid [.c coords [keylget id($theid) canvasid]]"
    foreach i $others {
	RDO [keylget i filedesc] eval "setnode $theid [.c coords [keylget id($theid) canvasid]]"
    }
    set lastX $x
    set lastY $y
}


#
# Bindings and globals for drawing the lines between nodes; use the middle
# mouse button for drawing lines
#

bind .c <2> { startline %x %y }
bind .c <B2-Motion> { continueline %x %y; movecursor %x %y} 
bind .c <ButtonRelease-2> { endline %x %y }

set line_id 0
set line_cid 0
set startwidget 0


#
# Start creating the line.  Force the line to start at the closest node
# to the point the user clicked on (we constrain all lines to go from the
# middle of the nodes - later we'd change this to be from the perimeter of
# nodes, but thats easy to fix).  Create the local line and tell everyone
# else about it.
#
# (Structurally, this should just call setedge to do most of the work...)
#

proc startline {x y} {  global line_id startwidget id nextID others line_cid
    set startwidget [closestnode $x $y]
    set line_cid [eval ".c create line [center $startwidget] $x $y -arrow last"]
    keylset id($nextID) canvasid $line_cid
    set line_id $nextID
    foreach i $others {
	RDO [keylget i filedesc] eval "setedge $nextID [center $startwidget] $x $y"
    }
    incr nextID
}


#
# This is called as the user drags out the line; change the local copy
# and pass the new coordinates on to everyone else
#

proc continueline {x y} { global line_id startwidget others line_cid
    eval ".c coords $line_cid [center $startwidget] $x $y"
    foreach i $others {
	RDO [keylget i filedesc] eval "setedge $line_id [center $startwidget] $x $y"
    }
}


#
# Change the start and end points of a line, creating it if necessary.
#

proc setedge {objid x0 y0 x1 y1} { 
    global id
    if {[info exists id($objid)] != 1} {
        keylset id($objid) canvasid [.c create line $x0 $y0 $x1 $y1 -arrow last]
    } else {
        eval ".c coords [keylget id($objid) canvasid] $x0 $y0 $x1 $y1"
    }
}


#
# This is called when the user releases the line after initially dragging
# it out.  Several things happen.  The endpoint of the line is constrained
# to the middle of the closest node, just to make things pretty.  The
# other users are told about the final position of the line.  Finally,
# the completed edge is recorded in the edgelist (see proc setnode, above)
# of both the starting and ending nodes, so that when either node moves,
# the edge will follow.
#

proc endline {x y} { global line_id startwidget id line_cid others
    set endwidget [closestnode $x $y]
    set startid [idfromcanvasid $startwidget]
    set endid [idfromcanvasid $endwidget]
    eval ".c coords $line_cid [center $startwidget] [center $endwidget]"
    recordedge $startid $endid $line_id
    foreach i $others {
	RDO [keylget i filedesc] eval "setedge $line_id [center $startwidget] [center $endwidget]"
	RDO [keylget i filedesc] recordedge $startid $endid $line_id
    }
}


#
# This does the actual job of recording the edge in the start and end
# widgets.  It appends to each node's edgelist a list of the form
# "to endnode edgeid" for the start node, or "from startnode edgeid" for
# the end node.
#

proc recordedge {start end edgeid} { global id
    set edges ""
    catch {set edges [keylget id($start) edgelist]}
    lappend edges [list "to" $end $edgeid]
    keylset id($start) edgelist $edges
    set edges ""
    catch {set edges [keylget id($end) edgelist]}
    lappend edges [list "from" $start $edgeid]
    keylset id($end) edgelist $edges
}


#
# This proc gets the application level object id given a canvas id
#

proc idfromcanvasid {canvasid} { global id
    foreach i [array names id] {
	if {[keylget id($i) canvasid] == $canvasid} {
	    return $i
	}
    }
    return ""
}


#
# This proc returns the x,y coordinates of the center of the given widget
# on the canvas (specified by canvas id)
#

proc center {widget} {
    set c [.c coords $widget]
    set x [expr {([lindex $c 0] + [lindex $c 2])*0.5}]
    set y [expr {([lindex $c 1] + [lindex $c 3])*0.5}]
    return [list $x $y]
}


#
# this routine returns the closest node to the given points.  what we're
# doing is basically ".c find closest x y withtag node" except that the
# canvas find command can't have multiple conditions on it...
#

proc closestnode {x y} {
    set l [eval ".c find withtag node"]
    set min 9999999
    set item 0
    foreach i $l {
	set xx [center $i]; set x0 [lindex $xx 0]; set y0 [lindex $xx 1]
	set dist [expr {($x-$x0)*($x-$x0)+($y-$y0)*($y-$y0)}]
	if {$dist < $min} {
	    set min $dist
	    set item $i
	}
    }
    return $item
}
