/*
 *	$Source: /user/nlfm/Working/Frink/RCS/token.c,v $
 *	$Date: 1994/09/28 09:18:24 $
 *	$Revision: 1.2.1.4 $
 *
 *------------------------------------------------------------------------
 *   AUTHOR:  Lindsay Marshall <lindsay.marshall@newcastle.ac.uk>
 *------------------------------------------------------------------------
 *    Copyright 1994 The University of Newcastle upon Tyne (see COPYRIGHT)
 *========================================================================
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "frink.h"

static char *tokenName[] =
{
    "ENDF",
    "ENDLINE",
    "ENDC",
    "LIST",
    "STRING",
    "CONC",
    "CALL",
    "VAR",
    "CPART",
    "SPART",
    "CONST",
    "ARRAY",
    "NL",
    "SP",
    "COMMENT",
    "HEAD",
    "LBRACK",
    "LBRACE",
    "LPAREN",
    "DOLLAR",
    "VNAME",
    "OSTART",
    "START",
    "XCONT",
    "CONT",
    "RBRACK",
    "RBRACE",
    "BLANK",
    "DQSTART",
    "DQEND",
    "SEMI"
};


static Token *concToken(char, char, Input *);
static Token *callToken(Input *);
static Token *varToken(Input *);

void dumpToken(Token *tk)
{
    Token *tp;
    static int depth = -1;
    if (tk == 0)
    {
	fprintf(stderr, "Null Token Pointer\n");
    }
    else
    {
	depth += 1;
	fprintf(stderr, "{%s {", tokenName[tk->type]);
	if (tk->text)
	{
	    fprintf(stderr, "%s", tk->text);
	}
	fprintf(stderr, "} {");
	tp = tk->sequence;
	while (tp != (Token *) 0)
	{
	    dumpToken(tp);
	    tp = tp->next;
	}
	fprintf(stderr,"} }");
	if ((depth -= 1) < 0) { fprintf(stderr, "\n"); }
    }
}

int lengthText(Token *hd)
{
    int leng;
    if (hd == (Token *) 0) { return 0; }

    switch (hd->type)
    {
    case VAR :
	leng = lengthText(hd->sequence) + 1;
	break;
    case LIST :
    case ARRAY :
	leng = lengthText(hd->sequence) + 2;
	break;
    case NL :
	leng = 1;
	break;
    case CONC :
    case STRING :
    case CALL :
	leng = lengthText(hd->sequence) + 2;
	break;
    default :
	leng = hd->length;
    }
    return leng + lengthText(hd->next);
}

void makeText(Token *hd, char *buff)
{
    if (hd == (Token *) 0) { return; }

    switch (hd->type)
    {
    case VAR :
	strcat(buff, "$");
	makeText(hd->sequence, buff);
	break;
    case ARRAY :
	strcat(buff, "(");
	makeText(hd->sequence, buff);
	strcat(buff, ")");
	break;
    case NL :
	strcat(buff, "\n");
	break;
    case CONC :
	makeText(hd->sequence, buff);
	break;
    case STRING :
	strcat(buff, "\"");
	makeText(hd->sequence, buff);
	strcat(buff, "\"");
	break;
    case CALL :
	strcat(buff, "[");
	makeText(hd->sequence, buff);
	strcat(buff, "]");
	break;
    case LIST :
	strcat(buff, "{");
	if (hd->text != (char *) 0) { strcat(buff, hd->text); }
	strcat(buff, "}");
	break;
    default :
	if (hd->text != (char *) 0) { strcat(buff, hd->text); }
	break;
    }
    makeText(hd->next, buff);
}

void streamMore(Input *file)
{
    file->pushed = 0;
    file->position = file->text;
    if (file->stream != NULL)
    {
	file->remaining = fread(file->text, 1, 64*1024, file->stream);
    }
}

int isBlank(char ch)
{
    return ch == ' ' || ch == '\t' || ch == '\n';
}

char reallyGet(Input *file)
{
    if (file->remaining <= 0) {	streamMore(file); }
    if (file->remaining <= 0) { return '\0'; }
    file->remaining -= 1;
    return *file->position++;
}

void push(Input *file, char ch)
{
    if (ch != '\0')
    {
	file->back[file->pushed] = ch;
	file->pushed += 1;
    }
}

char chGet(Input *file)
{
    char ch;
    if (file->pushed)
    {
	file->pushed -= 1;
	return file->back[file->pushed];
    }
    else if ((ch = reallyGet(file)) == '\\')
    {
	if ((ch = reallyGet(file)) == '\n')
	{
	    while (isBlank(ch = reallyGet(file)))
	    {
	    }
	    push(file, ch);
	    ch = ' ';
	}
	else
	{
	    push(file, ch);
	    ch = '\\';
	}
    }
    return ch;
}

Token *newToken(TokenType t)
{
    Token *tp = malloc(sizeof(Token));
    tp->type = t;
    tp->text = (char *) 0;
    tp->length = 0;
    tp->sequence = tp->next = (Token *) 0;
    return tp;
}

void freeToken(Token *tp)
{
    if (tp != (Token *) 0)
    {
	if (tp->text != (char *) 0 && tp->text != tp->little)
	{
	    free(tp->text);
	}
	if (tp->sequence) freeToken(tp->sequence);
	if (tp->next) freeToken(tp->next);
	free(tp);
    }
}

Input *tokenise(char *string, int length)
{
    Input *file = (Input *) malloc(sizeof(Input));
    file->position = string;
    file->remaining = length;
    file->pushed = 0;
    file->text = string;
    file->stream = NULL;
    file->tcall = 0;
    return file;
}

void untokenise(Input *file)
{
    free(file);
}

Token *tokenPop(Token **tp)
{
    Token *sp;
    if ((sp = *tp) != (Token *) 0)
    {
	*tp = sp->next;
        sp->next = (Token *) 0;
    }
    return sp;
}

void tokenPush(Token **tp, Token *v)
{
    Token * ptr;
    v->next = (Token *) 0;
    if ((ptr = *tp) == (Token *) 0)
    {
	*tp = v;
    }
    else
    {
	while (ptr->next != (Token *) 0)
	{
	    ptr = ptr->next;
	}
	ptr->next = v;
    }
}

Token *fillToken(TokenType t, int leng, char *string)
{
    Token *tp = malloc(sizeof(Token));
    tp->type = t;
    if ((tp->length = leng) < sizeof(tp->little))
    {
	strcpy(tp->little, string);
	free(string);
	tp->text = tp->little;
    }
    else
    {
	tp->text = realloc(string, leng + 1);
    }
    tp->length = leng;
    tp->sequence = tp->next = (Token *) 0;
    return tp;
}

char *newString(char *str, int length)
{
    return strcpy((char *) malloc(length +1), str);
}

Token *spaceToken(char ch, Input *file)
{
    Token *tp = newToken(SP);
    int leng = 0;
    char buff[512], *cp = buff;
    do
    {
	*cp++ = ch;
	 leng += 1;
    }
    while ((ch = chGet(file)) == ' ' || ch == '\t');
    push(file, ch);
    *cp = '\0';
    tp->text = newString(buff, leng);
    tp->length = leng;
    return tp;    
}

Token *listToken(Input *file)
{
    char ch, *buff = malloc(16*1024), *cp = buff;
    int bcount = 0, leng = 0;
    for (;;)
    {
	switch (ch = chGet(file))
	{
	case '\\' :
	    *cp++ = ch;
	    leng += 1;
	    ch = chGet(file);
	    break;
	case '{' :
	    bcount += 1;
	    break;
	case '}' :
	    if ((bcount -= 1) < 0) { goto done; }
	    break;
	case '\0' :
	    fprintf(stderr, "Missing }\n");
	    goto done;
	}
	*cp++ = ch;
	leng += 1;
    }
done:
    *cp = '\0';
    return fillToken(LIST, leng, buff);
}

Token *stringToken(Input *file, char lst, TokenType res)
{
    Token *tp = newToken(res), *nt;
    char ch, *buff = malloc(1024), *cp = buff;
    int leng = 0;

    for(;;)
    {
	if ((ch = chGet(file)) == lst) { goto done; }
	switch (ch)
	{
	case '\n' :
	    nt =  newToken(NL);
	    break;
	case ';' :
	    nt = newToken(SEMI);
	    break;
	case ' ' :
	case '\t':
	    nt = spaceToken(ch, file);
	    break;
	case '[' :
	    nt = callToken(file);
	    break;
	case '$' :
	    nt = varToken(file);
	    break;
	case '\0' :
	    fprintf(stderr, "Missing %c\n", lst);
	    goto done;
	case '\\' :
	    *cp++ = ch;
	    leng += 1;
	    ch = chGet(file);
	default :
	    *cp++ = ch;
	    leng += 1;
	    continue;
	}
        if (leng != 0)
	{
	    *cp = '\0';
	    tokenPush(&tp->sequence, fillToken(SPART, leng, buff));
	    cp = buff = malloc(1024);
	    leng = 0;
	}
	tokenPush(&tp->sequence, nt);
    }
done:
    *cp = '\0';
    if (leng != 0)
    {
	tokenPush(&tp->sequence, fillToken(SPART, leng, buff));
    }
    else
    {
	free(buff);
    }
    return tp;
}

Token *getToken(Input *file)
{
    char ch;
    Token *tp;
    extern int trace;

    switch (ch = chGet(file))
    {
    case '\0' : tp = newToken(ENDF); break;
    case ';' :
    case '\n' : tp = newToken(NL); break;
    case ' ' :
    case '\t': tp = spaceToken(ch, file); break;
    case ']':
	tp = (file->tcall) ? newToken(ENDC) : concToken(ch, '\0', file);
	break;
    case '{' : tp = listToken(file); break;
    case '"' : tp = stringToken(file, '"', STRING); break;
    default  : tp = concToken(ch, '\0', file); break;
    }
    if (trace) dumpToken(tp);
    return tp;
}

static Token *callToken(Input *file)
{
    Token *tp = newToken(CALL), *stp;
    file->tcall += 1;
    for(;;)
    {
	stp = getToken(file);
	switch (stp->type)
	{
	case SP :
	    freeToken(stp);
	    continue;
	case ENDF :
	    fprintf(stderr, "Missing ]\n");
	case ENDC :
	    freeToken(stp);
	    file->tcall -= 1;
	    return tp;
	default :
	    break;
	}
	tokenPush(&tp->sequence, stp);
    }
}

static Token *varToken(Input *file)
{
    Token *tp = newToken(VAR), *stp;
    char ch, buff[512], *cp = buff;
    int leng = 0;
    
    if ((ch = chGet(file)) == '{')
    {
	stp = listToken(file);
	tp->sequence = stp;
	return tp;
    }

    for(;;)
    {
	switch (ch)
	{
	case '.' :
	case ')' :
	case '$' :
	case ';' :
	case ' ' :
	case '\t' :
	case '\n' :
	case '\0' :
	case '[' :
	case ']' :
	case '{' :
	case '}' :
	case '"' :
	case '\\' :
	    *cp = '\0';
	    stp = newToken(VNAME);
	    stp->length = leng;
	    stp->text = newString(buff, leng);
	    tokenPush(&tp->sequence, stp);
	    push(file, ch);
	    return tp;
	case '(' :
	    if (leng != 0)
	    {
		*cp = '\0';
		stp = newToken(VNAME);
		stp->length = leng;
		stp->text = newString(buff, leng);
		tokenPush(&tp->sequence, stp);
	    }
	    else
	    {
		fprintf(stderr, "Variable error\n");
		exit(1);
	    }
	    tokenPush(&tp->sequence, stringToken(file, ')', ARRAY));
	    return tp;
	}
	*cp++ = ch;
	leng += 1;
	ch = chGet(file);
    }
}

static Token *concToken(char fst, char lst, Input *file)
{
    Token *tp = newToken(CONC);
    char ch, *buff = malloc(512), *cp = buff;
    int leng = 0;

    ch = fst;
    for(;;)
    {
	if (ch == lst) { goto done; }
	switch (ch)
	{
	case ';' :
	case ' ' :
	case '\t' :
	case '\n' :
	case '\0' :
	    goto done;
	case ']' :
	    if (file->tcall) { goto done; }
	    break;
	case '\\' :
	    *cp++ = ch;
	    leng += 1;
	    ch = chGet(file);
	    break;
	case '[' :
	    if (leng != 0)
	    {
		*cp = '\0';
		tokenPush(&tp->sequence, fillToken(CPART, leng, buff));
		leng = 0;
		cp = buff = malloc(512);
	    }
	    tokenPush(&tp->sequence, callToken(file));
	    ch = chGet(file);
	    continue;
	case '$' :
	    if (leng != 0)
	    {
		*cp = '\0';
		tokenPush(&tp->sequence, fillToken(CPART, leng, buff));
		leng = 0;
		cp = buff = malloc(512);
	    }
	    tokenPush(&tp->sequence, varToken(file));
	    ch = chGet(file);
	    continue;
	}
	*cp++ = ch;
	leng += 1;
	ch = chGet(file);
    }
done :
    push(file, ch);
    if (leng != 0)
    {
	*cp = '\0';
	if (tp->sequence == (Token *) 0)
	{
	    freeToken(tp);
	    return fillToken(CONST, leng, buff);
	}
        tokenPush(&tp->sequence, fillToken(CPART, leng, buff));
    }
    else
    {
	free(buff);
    }
    return tp;
}

int handle (Token *line)
{
    Token *hd;

    extern int tclop(Token*, Token*);
    extern void comment(Token *);

    if (line == (Token *) 0)
    {
	if (!minimise) { blank(); }
	return 1;
    }
    if (line->type == ENDF) { freeToken(line) ; return 0; }
    hd = line;
    line = line->next;
    switch (hd->type)
    {
    case CONST :
	if (tclop(hd, line)) { break; }
    case VAR :
    case CALL :
    case CONC :
	makeCall(hd, line);
	break;
    case COMMENT :
	comment(hd);
	break;
    case NL:
	blank();
	break;
    default :
	fprintf(stderr, "Invalid tcl program. - %s %s\n", tokenName[hd->type],
	  hd->text);
	exit(1);
    }
    freeToken(line);
    return 1;
}

static Token *getComment(Input *file, char *buff, int leng)
{
    Token *tp;
    char ch, *cp = &buff[leng];

    while ((ch = chGet(file)) != '\n' && ch != '\0')
    {
	*cp++ = ch;
	leng += 1;
    }
    push(file, ch);
    *cp = '\0';
    tp = newToken(COMMENT);
    tp->length = leng;
    tp->text = newString(buff, leng);
    return tp;
}

Token *collect(Input *file)
{
    Token *line = (Token *) 0, *tp;
    char buff[512];
    for(;;)
    {
	tp = getToken(file);
	switch (tp->type)
	{
	case NL :
	case ENDF :
	    if (line == (Token *) 0) { return tp; }
	    freeToken(tp);
	    return line;
	case CONST :
	    if (line == (Token *) 0 && tp->text[0] == '#')
	    {
		strcpy(buff, tp->text);
		freeToken(tp);
		tp = getComment(file, buff, tp->length);
	    }
	    break;
	case CONC :
	    if (line == (Token *) 0 && tp->sequence->type == CPART &&
		tp->sequence->text[0] == '#')
	    {
		*buff = '\0';
		makeText(tp, buff);
		freeToken(tp);
		tp = getComment(file, buff, strlen(buff));
	    }
	    break;
	case SP :
	    freeToken(tp);
	    continue;
	default :
	    break;
	}
	tokenPush(&line, tp);
    }
}

void sconv(Token **conc, Token **line)
{
    Token *tp;
    if (*conc != (Token *) 0)
    {
	tp = (*conc)->sequence;
	if (tp->type == CPART && tp->next == (Token *) 0)
	{
	    tp->type = CONST;
	    (*conc)->sequence = (Token *) 0;
	    freeToken(*conc);
	    *conc = tp;
	}
	tokenPush(line, *conc);
	*conc = (Token *) 0;
    }
}

void sprocess(Token *lst, int nls)
{
    Token *line = (Token *) 0, *conc = (Token *) 0;
    while (lst != (Token *) 0)
    {
	switch (lst->type)
	{
	case SPART :
	    lst->type = CPART;
	    break;
	case SEMI:
	    lst->type = NL;
	case NL :
	    sconv(&conc, &line);
	    if (nls || line != (Token *) 0) { handle(line); }
	    line = (Token *) 0;
	    freeToken(tokenPop(&lst));
	    continue;
	case SP :
	    sconv(&conc, &line);
	    freeToken(tokenPop(&lst));
	    continue;
	default :
	    break;
	}
	if (conc == (Token *) 0) { conc = newToken(CONC); }
	tokenPush(&conc->sequence, tokenPop(&lst));
    }
    sconv(&conc, &line);
    if (line != (Token *) 0) { handle(line); }
}

void lprocess(Token *lst, int nls)
{
    Token *line = (Token *) 0;
    while (lst != (Token *) 0)
    {
	switch (lst->type)
	{
	case ENDF :
	    if (line != (Token *) 0) { handle(line); }
	    return;
	case NL :
	    if (nls || line != (Token *) 0) { handle(line); }
	    line = (Token *) 0;
	case SP :
	    freeToken(tokenPop(&lst));
	    continue;
	default :
	    break;
	}
	tokenPush(&line, tokenPop(&lst));
    }
    if (line != (Token *) 0) { handle(line); }
}

Token *accumulate(Input *file, int nl)
{
    Token *line = (Token *) 0, *hd;
    TokenType last = NL;
    char buff[512];
    for(;;)
    {
	hd = getToken(file);
	switch (hd->type)
	{
	case ENDF :
	    if (line == (Token *) 0) { line = hd;}
	    return line;
	case NL :
	    if (!nl) { freeToken(hd) ; continue; }
	    break;
	case CONST :
	    if (last == NL && hd->text[0] == '#')
	    {
		strcpy(buff, hd->text);
		freeToken(hd);
		hd = getComment(file, buff, hd->length);
	    }
	    break;
	case CONC :
	    if (last == NL && hd->sequence->type == CPART &&
		hd->sequence->text[0] == '#')
	    {
		*buff = '\0';
		makeText(hd, buff);
		freeToken(hd);
		hd = getComment(file, buff, strlen(buff));
	    }
	    break;
	case SP :
	    freeToken(hd);
	    continue;
	default :
	    break;
	}
	tokenPush(&line, hd);
	last = hd->type;
    }
}
