/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1994  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.7 of 15-Sep-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	sunsite.unc.edu:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	ftp.cygnus.com:/pub/forth/pfe-VERSION.tar.gz
 *
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * misc.c ---	Compatiblity with former standards, miscellaneous useful words.
 * (duz 09Jul93)
 */

#include <stdlib.h>
#include <string.h>
#include <setjmp.h>
#include <errno.h>

#include "config.h"
#include "term.h"
#include "forth.h"
#include "support.h"
#include "compiler.h"
#include "dblsub.h"


/*****************************************************************************/
/* FIG Forth compatibility						     */
/*****************************************************************************/

Code (cold)
{
  close_all_files_();
  initialize_system ();
  longjmp (abort_dest, 1);
}

Code (dot_line)			/* .LINE (line# block# --) */
{
  dot_line (sp [0], sp [1]);
  sp += 2;
}

code (store_csp)		/* !CSP */
{
  CSP = sp;
}

code (question_csp)		/* ?CSP */
{
  if (sp != CSP)
    tHrow (THROW_CONTROL_MISMATCH);
}

code (question_comp)		/* ?COMP */
{
  if (!STATE)
    tHrow (THROW_COMPILE_ONLY);
}

code (question_exec)		/* ?EXEC */
{
  if (STATE)
    tHrow (THROW_COMPILER_NESTING);
}

code (question_file)		/* ?FILE */
{
  int ior = *sp++;
  if (ior)
    tHrow (-256 - ior);
}

code (question_loading)		/* ?LOADING */
{
  if (BLK == 0)
    tHrow (THROW_INVALID_BLOCK);
}

code (question_pairs)		/* ?PAIRS */
{
  question_comp_();
  question_pairs (*sp++);
}

code (question_stack)		/* ?STACK */
{
  if (sp > sys.s0)		tHrow (THROW_STACK_UNDER);
  if (sp < membot.stack)	tHrow (THROW_STACK_OVER);
  if (fp > sys.f0)		tHrow (THROW_FSTACK_UNDER);
  if (fp < membot.fstack)	tHrow (THROW_FSTACK_OVER);
  if (rp > sys.r0)		tHrow (THROW_RSTACK_UNDER);
  if (rp < membot.rstack)	tHrow (THROW_RSTACK_OVER);
}

Code (toggle)			/* TOGGLE */
{
  *(Byte *)sp [1] ^= (Byte)sp [0];
  sp += 2;
}

Code (latest)			/* LATEST */
{
  *--sp = (Cell)latest ();
}

code (smudge)			/* SMUDGE (modified from FIG definition) */
{				/* FIG definition toggles the bit! */
  *LAST |= SMUDGED;
}

code (unsmudge)			/* UNSMUDGE (turn smudge-bit off) */
{				/* neccessary because SMUDGE modified */
  *LAST &= ~SMUDGED;
}


/*****************************************************************************/
/* some well known words with no clear pedigree				     */
/*****************************************************************************/

Code (u_d_dot_r)		/* UD.R */
{
  Cell w = *sp++;
  less_number_sign_();
  number_sign_s_();
  number_sign_greater_();
  spaces (w - *sp);
  type_();
}

Code (u_d_dot)			/* UD. */
{
  *--sp = 0;
  u_d_dot_r_();
  space_();
}

Code (dot_name)			/* .NAME */
{
  dot_name ((char *)*sp++);
}

Code (dash_roll)		/* -ROLL */
{
  Cell n = *sp++;
  Cell h, i;

  h = sp [0];
  for (i = 0; i < n; i++)
    sp [i] = sp [i + 1];
  sp [i] = h;
}

Code (r_from_drop)		/* R>DROP	shortcut I saw in CSI-Forth */
{
  rp++;
}

Code (dup_to_r)			/* DUP>R	dito */
{
  RPUSH (*sp);
}

Code (random)			/* RANDOM ( n1 --- n2 ) */
{				/* returns random number within [0,n1) */

# ifdef BSD			/* BSD has a better random number generator */
# define rand random		/* than standard unix' rand() */
# endif

  if (*sp == 0)
    *sp = rand ();
  else
    *sp = ummul (*sp, rand () << 1).hi;

# undef rand
}

Code (srand)			/* SRAND ( n --- ) */
{
#ifdef BSD
  srandom (*sp++);
#else
  srand (*sp++);
#endif
}

Code (under_plus)		/* n1 n2 --- n1+n2 n2 */
{				/* same as TUCK + SWAP */
  sp [1] += sp [0];
}


/*****************************************************************************/
/* more local variables							     */
/*****************************************************************************/

code (plus_to_execution)	/* add to current contents of inline */
{				/* following VALUE */
  *TO_BODY (*ip++) += *sp++;
}

code (plus_to_local_execution)	/* add to current value of local variable */
{
  lp [(Cell)*ip++] += *sp++;
}

Code (plus_to)
{
  if (STATE)
    {
      char *p;
      int l, n;

      p = word (' ');
      l = *(Byte *)p++;
      if (sys.locals && (n = find_local (p, l)) != 0)
	{
	  compile2 ();
	  COMMA (n);
	}
      else
	{
	  if ((p = find (p, l)) == NULL)
	    tHrow (THROW_UNDEFINED);
	  compile1 ();
	  COMMA (name_from (p));
	}
    }
  else
    {
      Xt xt;

      tick (&xt);
      *TO_BODY (xt) += *sp++;
    }
}
COMPILES2 (plus_to, plus_to_execution, plus_to_local_execution,
	   SKIPS_CELL, DEFAULT_STYLE);


/*****************************************************************************/
/* data structures							     */
/*****************************************************************************/

Code (build_array)		/* n1 n2 ... nX X --- n */
{				/* writes X, n1, ... nX into the dictionary */
  Cell i = *sp++;		/* returns product n1 * n2 * ... * nX */
  uCell n = 1;

  COMMA (i);
  while (--i >= 0)
    {
      COMMA (*sp);
      n *= *sp++;
    }
  *--sp = n;
}

Code (access_array)		/* i1 i2 ... iX addr1 --- addr2 n */
{
  uCell *p = (uCell *)*sp++, n = 0;
  Cell i = *p++;

  for (;;)
    {
      if (*p++ <= *sp)
	tHrow (THROW_INDEX_RANGE);
      n += *sp++;
      if (--i <= 0)
	break;
      n *= *p;
    }
  *--sp = (Cell)p;
  *--sp = n;
}


/*****************************************************************************/
/* more comparision operators						     */
/*****************************************************************************/

Code (zero_less_equal)
{
  *sp = FLAG (*sp <= 0);
}

Code (zero_greater_equal)
{
  *sp = FLAG (*sp >= 0);
}

Code (less_equal)
{
  sp [1] = FLAG (sp [1] <= sp [0]);
  sp++;
}

Code (greater_equal)
{
  sp [1] = FLAG (sp [1] >= sp [0]);
  sp++;
}

Code (u_less_equal)
{
  sp [1] = FLAG ((uCell)sp [1] <= (uCell)sp [0]);
  sp++;
}

Code (u_greater_equal)
{
  sp [1] = FLAG ((uCell)sp [1] >= (uCell)sp [0]);
  sp++;
}

Code (u_max)
{
  if ((uCell)sp [0] > (uCell)sp [1])
    sp [1] = sp [0];
  sp++;
}

Code (u_min)
{
  if ((uCell)sp [0] < (uCell)sp [1])
    sp [1] = sp [0];
  sp++;
}


/*****************************************************************************/
/* implementation							     */
/*****************************************************************************/

Code (source_line)		/* SOURCE-LINE (--- n) */
{
  switch (SOURCE_ID)
    {
    case 0:
      if (BLK)
	*--sp = TO_IN / 64 + 1;	/* source line from BLOCK */
      else
    case -1:			/* string from EVALUATE */
	*--sp = 0;		/* or from QUERY (0/BLK==0) */
      break;
    default:			/* source line from text file */
      *--sp = SOURCE_FILE->n + 1;
    }
}

code (pocket)			/* POCKET ( n --- addr u) */
{				/* returns string in the specified pocket */
  int n = *sp;
  sp -= 1;
  sp [1] = (Cell)membot.pocket [n] + 1;
  sp [0] = *(Byte *)membot.pocket [n];
}

Code (wl_hash)			/* WL-HASH (addr len -- n) */
{				/* calc hash-code for selection of thread */
  sp [1] = wl_hash ((char *)sp [1], sp [0]);
  sp++;
}

Code (wwords)			/* WWORDS */
{				/* words with wildcards */
  char *pattern = word (' ');
  Wordl *wl = CONTEXT [0] ? CONTEXT [0] : ONLY;

  if (LOWER_CASE)
    upper (pattern + 1, *pattern);
  wild_words (wl, pattern + 1);
}

Code (w_fetch)			/* W@ */
{
  *sp = *(short *)*sp;
}

Code (w_store)			/* W! */
{
  *(short *)sp [0] = (short)sp [1];
  sp += 2;
}

Code (w_plus_store)		/* W+! */
{
  *(short *)sp [0] += (short)sp [1];
  sp += 2;
}

Code (paren_forget)		/* (FORGET) (addr ---) */
{				/* forgets everything above addr */
  forget ((char *)*sp++);
}

Code (tab)			/* TAB (n --)	jump to next column */
{				/*		divisible by n */
  tab (*sp++);
}

code (backspace)		/* BACKSPACE (--) reverse of SPACE */
{
#if !defined (EMX)
  outs ("\b \b");
#else
  goleft ();
  outc (' ');
  goleft ();
#endif
}

Code (question_stop)		/* ?STOP (-- flag) check for 'q' pressed */
{
  *--sp = FLAG (question_stop ());
}

code (start_question_cr)	/* START?CR */
{				/* initialize for more-like effect */
  sys.more = rows - 2;
  sys.lines = 0;
}

Code (question_cr)		/* ?CR */
{				/* like CR, stop 25 lines past START?CR */
  *--sp = question_cr ();
}

code (close_all_files)		/* CLOSE-ALL-FILES */
{
  File *f;

  for (f = membot.files; f < memtop.files - 3; f++)
    if (f->f)
      {
	if (f->updated)
	  read_write (f, f->buffer, f->n, FALSE);
	fclose (f->f);
      }
}

code (dot_memory)
{
  outf ("\nDictionary space:    %7ld Bytes, in use: %7ld Bytes\n"
	"Stack space:         %7ld cells\n"
	"Floating stack space:%7ld floats\n"
	"Return stack space:  %7ld cells\n",
	(long)memsiz.dict,
	(long)((char *)DP - (char *)sys.dict),
	(long)(memsiz.stack / sizeof (Cell)),
	(long)(memsiz.fstack / sizeof (double)),
	(long)(memsiz.rstack / sizeof (void *)));
}

Code (dot_version)	{ outs (version_string); }
Code (dot_pfe_date)	{ outf ("PFE compiled %s, %s ",
				compile_date, compile_time); }
Code (license)		{ outs (license_string); }
Code (warranty)		{ outs (warranty_string); }


/*****************************************************************************/
/* vectorized I/O							     */
/*****************************************************************************/

Code (paren_emit)
{
  outc ((char)*sp++);
}

Code (paren_expect)
{
  expect ((char *)sp [1], sp [0]);
  sp += 2;
}

Code (paren_key)
{
  int c;
  do c = getekey ();
  while (c >= 0x100);
  *--sp = c;
}

Code (paren_type)
{
  type ((char *)sp [1], sp [0]);
  sp += 2;
}

code (standard_io)
{
  static pCode
    paren_emit_cfa = paren_emit_,
    paren_expect_cfa = paren_expect_,
    paren_key_cfa = paren_key_,
    paren_type_cfa = paren_type_;

  sys.emit = &paren_emit_cfa;
  sys.expect = &paren_expect_cfa;
  sys.key = &paren_key_cfa;
  sys.type = &paren_type_cfa;
}


/*****************************************************************************/
/* more advanced screen control						     */
/*****************************************************************************/

Code (gotoxy)			/* GOTOXY (x y --) */
{
  gotoxy (sp [1], sp [0]);
  sp += 2;
}

Code (question_xy)		/* ?XY (-- x y) */
{				/* returns cursor position on screen */
  int x, y;

  wherexy (&x, &y);
  sp -= 2;
  sp [1] = x;
  sp [0] = y;
}

/* these are defined in the driver term.c. */
/* They are renamed here to satisfy the macro CO(char *, pCode) */

#define	cls_			clrscr
#define	clreol_			clreol
#define	home_			home
#define bell_			bing
#define highlight_		standout_on
#define minus_highlight_	standout_off
#define underline_		underline_on
#define minus_underline_	underline_off
#define intensity_		bright
#define minus_intensity_	normal
#define blink_			blinking
#define minus_blink_		normal
#define reverse_		reverse
#define minus_reverse_		normal
#define normal_			normal


/*****************************************************************************/
/* Function keys on the commandline					     */
/*****************************************************************************/

static Xt fkey_executes_xt [10] = { NULL };

void
accept_executes_xt (int n)
{
  if (fkey_executes_xt [n])
    run_forth (fkey_executes_xt [n]);
}

static void
store_execution (Xt xt, int key)
{
  if (key < EKEY_k1 || EKEY_k0 < key)
    tHrow (THROW_ARG_TYPE);
  fkey_executes_xt [key - EKEY_k1] = xt;
}

Code (executes_execution)
{
  store_execution (*ip++, *sp++);
}
Code (executes)
{
  if (STATE)
    {
      compile1 ();
      bracket_compile_();
    }
  else
    {
      Xt xt;

      tick (&xt);
      store_execution (xt, *sp++);
    }
}
COMPILES (executes, executes_execution,
	  SKIPS_NOTHING, DEFAULT_STYLE);


/*****************************************************************************/
/* hooks to editors and os services					     */
/*****************************************************************************/

Code (expand_fn)		/* EXPAND-FN */
{				/* ( addr1 cnt1 addr2 --- addr2 cnt2 ) */
  char *nm = (char *)sp [2];
  char *fn = (char *)sp [0];
  int len = sp [1];
  char buf [0x100];

  store_filename (nm, len, buf, sizeof buf);
  expand_filename (buf, option.incpaths, option.incext, fn);
  sp++;
  sp [1] = (Cell)fn;
  sp [0] = strlen (fn);
}

Code (using)
{
  char *p, *name;
  int len;

  p = word (' ');
  name = p + 1;
  len = *(Byte *)p;
  if (len == 0)
    tHrow (THROW_INVALID_NAME);
  if (!use_block_file (name, len))
    tHrow (THROW_FILE_NEX);
}

Code (using_new)
{
  char *p, *name;
  int len;
  File *f;

  p = word (' ');
  name = p + 1;
  len = *(Byte *)p;
  if (len == 0)
    tHrow (THROW_INVALID_NAME);
  f = open_file (name, len, FMODE_ROB);
  if (f == NULL)
    {
      f = create_file (name, len, FMODE_RWB);
      if (f == NULL)
	tHrow (THROW_FILE_ERROR);
    }
  close_file (f);
  if (!use_block_file (name, len))
    tHrow (THROW_FILE_NEX);
}

void edit (int n, int r, int c);

Code (edit_block)
{
  edit (*sp++, 0, 0);
}

Code (edit_text)
{
  char *nm, fn [0x100];

  nm = word (' ');
  if (*nm == '\0')
    tHrow (THROW_FILE_NEX);
  expand_filename (nm + 1, option.incpaths, option.incext, fn);
  systemf ("%s %s", option.editor, fn);
}

Code (edit_error)
{
  switch (sys.input_err.source_id)
    {
    case 0:
      if (sys.input_err.blk)
	{
	  edit (sys.input_err.blk,
		sys.input_err.to_in / 64,
		sys.input_err.to_in % 64);
	  break;
	}
    case -1:
      bing ();
      break;
    default:
      {
	File *f = (File *)sys.input_err.source_id;
	systemf ("%s +%d %s", option.editor, (int)f->n + 1, f->name);
	break;
      }
    }
}

Code (include)
{
  char *fn = word (' ');
  included (fn + 1, *(Byte *)fn);
}

Code (system)
{
  sp [1] = systemf ("%.*s", (int)sp [0], (char *)sp [1]);
  sp++;
}

code (system_quote_execution)
{
  char *p = (char *)ip;

  SKIP_STRING;
  *--sp = systemf ("%.*s", *p, p + 1);
}

Code (system_quote)
{
  if (STATE)
    {
      compile1 ();
      alloc_parsed_string ('"');
    }
  else
    {
      char *p;
      uCell l;

      parse ('"', &p, &l, 0);
      *--sp = systemf ("%.*s", l, p);
    }
}
COMPILES (system_quote, system_quote_execution,
	  SKIPS_STRING, DEFAULT_STYLE);

Code (signal)			/* xt1 n --- xt2 ; install signal handler */
{				/* return old signal handler */
  sp [1] = (Cell)forth_signal (sp [0], (Xt)sp [1]);
  sp++;
}


LISTWORDS (misc) =
{
  /* FIG-Forth */
  OC ("0",		0),
  OC ("1",		1),
  OC ("2",		2),
  OC ("3",		3),
  CO ("COLD",		cold),
  CO ("LIT",		literal_execution),
  CO (".LINE",		dot_line),
  SV ("CSP",		CSP),
  CO ("!CSP",		store_csp),
  CO ("?CSP",		question_csp),
  CO ("?COMP",		question_comp),
  CO ("?EXEC",		question_exec),
  CO ("?FILE",		question_file),
  CO ("?LOADING",	question_loading),
  CO ("?PAIRS",		question_pairs),
  CO ("?STACK",		question_stack),
  CO ("TOGGLE",		toggle),
  CO ("LATEST",		latest),
  DV ("LAST",		last),
  SV ("OUT",		OUT),
  DV ("HLD",		hld),
  SV ("R0",		sys.r0),
  SV ("S0",		sys.s0),
  CO ("SMUDGE",		smudge),
  CO ("UNSMUDGE",	unsmudge),

  /* words without pedigree */
  CO ("UD.R",		u_d_dot_r),
  CO ("UD.",		u_d_dot),
  CO (".NAME",		dot_name),
  CO ("-ROLL",		dash_roll),
  CO ("R>DROP",		r_from_drop),
  CO ("DUP>R",		dup_to_r),
  CO ("RANDOM",		random),
  CO ("SRAND",		srand),
  CO ("UNDER+",		under_plus),

  /* more local variables */
  CS ("+TO",		plus_to),
  /* data structures */
  CO ("BUILD-ARRAY",	build_array),
  CO ("ACCESS-ARRAY",	access_array),

  /* more comparision */
  CO ("0<=",		zero_less_equal),
  CO ("0>=",		zero_greater_equal),
  CO ("<=",		less_equal),
  CO (">=",		greater_equal),
  CO ("U<=",		u_less_equal),
  CO ("U>=",		u_greater_equal),
  CO ("UMIN",		u_min),
  CO ("UMAX",		u_max),

  /* implementation */
  OC ("EXCEPTION_MAGIC",EXCEPTION_MAGIC),
  OC ("INPUT_MAGIC",	INPUT_MAGIC),
  OC ("DEST_MAGIC",	DEST_MAGIC),
  OC ("ORIG_MAGIC",	ORIG_MAGIC),
  OC ("LOOP_MAGIC",	LOOP_MAGIC),
  OC ("CASE_MAGIC",	CASE_MAGIC),
  OC ("OF_MAGIC",	OF_MAGIC),
  CO ("FLIT",		f_literal_execution),
  SV ("F0",		sys.f0),
  SV ("LOWER-CASE",	LOWER_CASE),
  SV ("LOWER-CASE-FN",	LOWER_CASE_FN),
  SV ("REDEFINED-MSG",	REDEFINED_MSG),
  SV ("FLOAT-INPUT",	FLOAT_INPUT),
  DV ("APPLICATION",	application),
  CO ("SOURCE-LINE",	source_line),
  CO ("POCKET",		pocket),
  OC ("WSIZE",		sizeof (Cell)),
  CO ("W@",		w_fetch),
  CO ("W!",		w_store),
  CO ("W+!",		w_plus_store),
  CO ("WL-HASH",	wl_hash),
  CO ("WWORDS",		wwords),
  CO ("(FORGET)",	paren_forget),
  CO ("BELL",		bell),
  CO ("TAB",		tab),
  CO ("BACKSPACE",	backspace),
  CO ("?STOP",		question_stop),
  CO ("START?CR",	start_question_cr),
  CO ("?CR",		question_cr),
  CO ("CLOSE-ALL-FILES",close_all_files),
  CO (".MEMORY",	dot_memory),
  CO (".VERSION",	dot_version),
  CO (".PFE-DATE",	dot_pfe_date),
  CO ("LICENSE",	license),
  CO ("WARRANTY",	warranty),

  /* vectorized i/o */
  SV ("*EMIT*",		sys.emit),
  SV ("*EXPECT*",	sys.expect),
  SV ("*KEY*",		sys.key),
  SV ("*TYPE*",		sys.type),
  CO ("(EMIT)",		paren_emit),
  CO ("(EXPECT)",	paren_expect),
  CO ("(KEY)",		paren_key),
  CO ("(TYPE)",		paren_type),
  CO ("STANDARD-I/O",	standard_io),

  /* more advanced screen control */
  SC ("ROWS",		rows),
  SC ("COLS",		cols),
  SC ("XMAX",		xmax),
  SC ("YMAX",		ymax),
  CO ("GOTOXY",		gotoxy),
  CO ("?XY",		question_xy),
  CO ("CLS",		cls),
  CO ("CLREOL",		clreol),
  CO ("HOME",		home),
  CO ("HIGHLIGHT",	highlight),
  CO ("-HIGHLIGHT",	minus_highlight),
  CO ("UNDERLINE",	underline),
  CO ("-UNDERLINE",	minus_underline),
  CO ("INTENSITY",	intensity),
  CO ("-INTENSITY",	minus_intensity),
  CO ("BLINKING",	blink),
  CO ("-BLINKING",	minus_blink),
  CO ("REVERSE",	reverse),
  CO ("-REVERSE",	minus_reverse),
  CO ("NORMAL",		normal),

  /* EKEY return codes for function keys: */
  OC ("K-LEFT",		EKEY_kl),
  OC ("K-RIGHT",	EKEY_kr),
  OC ("K-UP",		EKEY_ku),
  OC ("K-DOWN",		EKEY_kd),
  OC ("K-HOME",		EKEY_kh),
  OC ("K-END",		EKEY_kH),
  OC ("K-PRIOR",	EKEY_kP),
  OC ("K-NEXT",		EKEY_kN),
  OC ("K1",		EKEY_k1),
  OC ("K2",		EKEY_k2),
  OC ("K3",		EKEY_k3),
  OC ("K4",		EKEY_k4),
  OC ("K5",		EKEY_k5),
  OC ("K6",		EKEY_k6),
  OC ("K7",		EKEY_k7),
  OC ("K8",		EKEY_k8),
  OC ("K9",		EKEY_k9),
  OC ("K10",		EKEY_k0),
  CS ("EXECUTES",	executes),

  /* editors and system hooks: */
  SC ("STDIN",		sys.stdIn),	/* --- fid */
  SC ("STDOUT",		sys.stdOut),	/* --- fid */
  SC ("STDERR",		sys.stdErr),	/* --- fid */
  SC ("BLOCK-FILE",	BLOCK_FILE),	/* --- fid */
  CO ("EXPAND-FN",	expand_fn),
  CO ("USING",		using),
  CO ("USING-NEW",	using_new),
  CO ("EDIT-BLOCK",	edit_block),
  CO ("EDIT-TEXT",	edit_text),
  CO ("EDIT-ERROR",	edit_error),
  CO ("INCLUDE",	include),
  CO ("SYSTEM",		system),
  CS ("SYSTEM\"",	system_quote),
  CO ("SIGNAL",		signal),
};
COUNTWORDS (misc, "Compatibility + miscellaneous");
