package Mojo::Twist::Date;
use Mojo::Base -base;

use overload '""' => \&to_string, fallback => 1;

require Time::Local;
require Time::Piece;
require Encode;

my $DATE_RE      = qr/(\d\d\d\d)-?(\d?\d)-?(\d?\d)/;
my $TIME_SEP_RE  = qr/_|:/;
my $TIME_RE      = qr/T(\d\d)$TIME_SEP_RE?(\d\d)$TIME_SEP_RE?(\d\d)?/;
my $TIMESTAMP_RE = qr/$DATE_RE$TIME_RE?/;

sub new {
    my $self   = shift->SUPER::new;
    my %params = @_;

    if (exists $params{epoch}) {
        $self->{epoch} = $params{epoch};
    }
    else {
        my $epoch = $self->_to_epoch($params{timestamp});
        $self->{epoch} = $epoch;
    }

    return $self;
}

sub epoch { @_ > 1 ? $_[0]->{epoch} = $_[0] : $_[0]->{epoch} }


sub year {
    my $self = shift;

    return Time::Piece->gmtime($self->epoch)->year;
}

sub month {
    my $self = shift;

    return Time::Piece->gmtime($self->epoch)->mon;
}

sub timestamp {
    my $self = shift;

    my $t = Time::Piece->gmtime($self->epoch);

    return $t->strftime('%Y%m%dT%H:%M:%S');
}

sub strftime {
    my $self = shift;
    my $fmt  = shift;

    my $t = Time::Piece->gmtime($self->epoch);

    return $t->strftime($fmt);
}

sub is_date {
    my $self = shift;
    my ($date) = @_;

    return $date =~ qr/^$TIMESTAMP_RE$/ ? 1 : 0;
}

sub to_string {
    my $self = shift;

    my $fmt = "%a, %d %b %Y";

    return Encode::decode('UTF-8', $self->strftime($fmt));
}

sub to_rss {
    my $self = shift;

    my $fmt = "%a, %d %b %Y %T GMT";

    return Encode::decode('UTF-8', $self->strftime($fmt));
}

sub _to_epoch {
    my $self      = shift;
    my $timestamp = shift;

    return unless $timestamp =~ qr/^$TIMESTAMP_RE$/;

    my ($year, $month, $day, $hour, $minute, $second) =
      ($1, $2, $3, ($4 || 0), ($5 || 0), ($6 || 0));

    my $epoch = 0;
    eval {
        $epoch =
          Time::Local::timegm($second, $minute, $hour, $day, $month - 1,
            $year - 1900);
    };

    return if $@ || $epoch < 0;

    return $epoch;
}

1;
